#
# Rhythmlet - gDesklets sensor for Rhythmbox
# Copyright (c) 2004 Alex Revo
# ----------------------------------------------------------------------
#  Covers.py - cover/image-related functions
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
########################################################################


# Import the required (system) modules.
import md5
import os
import string
import types
import urllib
from   shutil				import copy
from   xml.sax.saxutils		import escape

# Import Rhythmlet-specific modules.
import RLBase

# Import optional modules.
#  * (SOAPpy is needed for Amazon)
try:
    import SOAPpy
    HAVE_SOAP = 1
except:
    HAVE_SOAP = 0
# end try
#  * Python Imaging Library is needed for resizing
try:
	import Image, ImageOps
	HAVE_PIL = 1
except:
	HAVE_PIL = 0

	# If PIL isn't available, fallback to GTK
	try:
		import gtk
		HAVE_GTK = 1
	except:
		HAVE_GTK = 0
# end try


class RLCovers(RLBase.RLBase):
#-----------------------------------------------------------------------
	fileNames		= [ 'cover', 'folder', 'albumartsmall']
	fileExtensions	= [ 'png', 'jpg','gif','jpeg' ]

	def __init__(self, *args):
	#-------------------------------------------------------------------
		"""Constructor for RLCovers class."""

		RLBase.RLBase.__init__(self)

		self.dPrint("RLCovers class instantiated.")
	#-------------------------------------------------------------------


	def getCover_local(self, artist, album, filePath):
	# (written by Ola Ormset)
	#-------------------------------------------------------------------
		"""Searches for cover image locally.
		Takes the following arguments:
		  - artist: artist name
		  - album:  album
		  - filePath: location of file
		Returns:
		  - path to cover
		  - candidates
		  - preference
		  - song path"""

		# Find any local covers
		# Return existing covers, list of valid names, list of valid prefixes in preferred order and the song path
		localCover = ""
		songPath = ""

		try:
			songPath = filePath.replace('file://', '')
			songPath = urllib.unquote_plus(songPath)
			songPath = os.path.dirname(songPath)

			if (os.path.isdir(songPath)):
				files = os.listdir(songPath)

				lowerFiles = []
				for file in files:
					lowerFiles.append(file.lower())
				# end for

				# Look for images in this order:
				#  * <artist name>-<album name>.ext
				#  * <artist name>.ext
				#  * <any self.fileNames>.ext
				preference = [
					"%s-%s" % (self.normalizeString(artist),
							self.normalizeString(album) ),
					"%s" % self.normalizeString(album)
				]
				preference += self.fileNames

				albumFiles = [ "%s.%s" % (prefix, ext) for prefix in preference for ext in self.fileExtensions ]

				candidates  = filter(lambda x: x in lowerFiles, albumFiles)
				
				if (candidates):
					localCover = os.path.join(songPath, files[lowerFiles.index(candidates[0])])
				# end if

				return (localCover, candidates, preference, songPath)
			# end if
		except Exception, error:
			self.handleException("RLCovers::getCover_local()", error)
		# end try

		return ("", [], [], "")
	#-------------------------------------------------------------------


	def getCover_Amazon(self, artist, album, filePath, saveImage=0, imagePath='/tmp/rhythmlet.jpg'):
	#-------------------------------------------------------------------
		"""Searches for cover using the Amazon Web Services API.
		Takes the following arguments:
		  - artist: artist name
		  - album:  album
		  - filePath: location of file
		  - saveImage: (bool) whether to save image (if found to be exact)
		  - imagePath: where to save image
		Returns: path to new cover image"""

		details		= ""
		foundExact	= 0
		myArtist	= self.normalizeString(artist)
		myAlbum		= self.normalizeString(album)
		myLink		= "http://www.allmusic.com"
		myCover		= "gfx/coverDefault.png"

		# SOAPpy is required; if not found don't continue.
		if (not HAVE_SOAP):
			return(myCover, myLink)

		try:
			# Build and execute SOAP request.
			# ----------------------------------------------------------
			request = {
				"artist":	myArtist,
				"keywords":	myAlbum,
				"page":		1,
				"mode":		"music",
				"sort":		"+pmrank",
				"type":		"lite",
				"tag":		"accessonline",
				"devtag":	"D1R5WXCELBX4NF"
			}

			try:
				proxy = SOAPpy.WSDL.Proxy("http://soap.amazon.com/schemas3/AmazonWebServices.wsdl")
				results = proxy.ArtistSearchRequest(request)
			except:
				self.dPrint("  Amazon search results not found, or the search has timed out. ")
				return(myCover, myLink)


			# Process results of SOAP search.
			# ----------------------------------------------------------
			self.dPrint("  " + str(results.TotalResults) + " result(s) from Amazon:")
			if (int(results.TotalResults) > 0):
				# Limit results to top 10.
				if (int(results.TotalResults) > 10):
					self.dPrint("  (Limiting to 10 results)")
					results.TotalResults = 10
				# end if

				self.dPrint("  Looking for: " + myArtist + " - " + myAlbum)

				# Loop through results, look for exact match.
				for i in range(int(results.TotalResults)):
					details = results.Details[i]

					myAmazonArtist = self.normalizeString(details['Artists'][0])
					myAmazonAlbum = self.normalizeString(details['ProductName'])

					self.dPrint("    * " + details['Artists'][0] + " - " + details['ProductName'])
					#self.dPrint("      (" + myAmazonArtist + " - " + myAmazonAlbum + ")")

					if ((myAmazonArtist == myArtist) and (myAmazonAlbum == myAlbum)):
						# Found exact match.
						foundExact = 1

						# No need to continue looping after we have what we want.
						if (foundExact == 1):
							break
						# end if
					# end if
				# end for

				# No exact match, use the first
				if (foundExact == 0):
					self.dPrint("  Exact match not found; Using first result instead.")

					details = results.Details[0]
				else:
					self.dPrint("      (EXACT)")
				# end if

				myLink = details["Url"]

				# Fetch images:
				# ------------------------------------------------------
				# Try getting larger images first.
				urllib.urlretrieve(details["ImageUrlLarge"], imagePath)
				if (self.isBlankImage(imagePath)):
					urllib.urlretrieve(details["ImageUrlMedium"], imagePath)
					if (self.isBlankImage(imagePath)):
						urllib.urlretrieve(details["ImageUrlSmall"], imagePath)
					# end if
				# end if

				# Make sure it's not Amazon's blank image.
				if (not self.isBlankImage(imagePath)):
					self.dPrint("  Retrieved image.")


					# Save only if we're absolutely sure that it's a match.
					# --------------------------------------------------
					if (saveImage and (foundExact == 1)):
						# No cover images are available; save cover as artist-album.jpg
						# (Saving as "cover.jpg" gives false positives on a directory containing
						#  files belonging to more than one album.)
						filenames = ["%s-%s.jpg" % (artist.replace("/", ""),
													album.replace("/", "")),
						#			 "cover.jpg"
						]

						# First fetch the song path and do an extra check
						# that we really have _no_ cover art...
						(localCover, candidates, preference, songPath) = self.getCover_local(artist, album, filePath)
						if (localCover or candidates):
							filenames = []
						# end if

						for file in filenames:
							self.copyFile(imagePath, os.path.join(songPath, file))
						# end for
					# end if

					myCover = imagePath
				else:
					self.dPrint("  Resulting image is blank; Discarding it.")
				# end if
			# end if
		except Exception, error:
			self.handleException("RLCovers::getCover_Amazon(): Fetch from Amazon", error)
		# end try

		return(myCover, myLink)
	#-------------------------------------------------------------------


	def copyFile(self, origFile, newFile, deleteAfterCopy=0):
	#-------------------------------------------------------------------
		try:
			self.dPrint("Copying \"" + origFile + "\" to: \"" + newFile + "\"")

			# Check for write permission.
			if (os.access(os.path.dirname(newFile), os.W_OK)):
				copy(origFile, newFile)
				self.dPrint("  Success!")
				if (deleteAfterCopy):
					self.dPrint("  Deleting original file (" + origFile + ").")
					os.unlink(origFile)
				# end if
				return(newFile)
			else:
				self.dPrint("  [!!!] Permission denied")
			# end if
		except Exception, error:
			self.handleException("RLCovers::copyFile()", error)
		# end try

		return("")
	#-------------------------------------------------------------------


	def isBlankImage(self, imagePath):
	#-------------------------------------------------------------------
		"""Return 1 if image is Amazon.com's blank image."""
		# (code from Python source distribution: Tools/scripts/md5sum.py)

		f = open(imagePath, "r")
		sum = md5.new()
		for i in f.readlines():
			sum.update(i)
		# end for
		f.close()

		if ((sum.hexdigest()) == "10e0c38f29fb91bed65e950b022e5054"):
			return 1
		else:
			return 0
		# end if
	#-------------------------------------------------------------------


	def resizeImage(self, imagePath, newSize, drawBorder=1):
	#-------------------------------------------------------------------
		"""Resize image to newSize."""

		if (HAVE_PIL):
			return(self.resizeImagePil(imagePath, newSize, drawBorder))
		if (HAVE_GTK):
			return(self.resizeImageGtk(imagePath, newSize, drawBorder))

		return((0, 0))
	#-------------------------------------------------------------------


	def resizeImagePil(self, imagePath, newSize, drawBorder=1):
	#-------------------------------------------------------------------
		"""Resize image to newSize using Python Imaging Library."""

		self.dPrint("(PIL) Resizing " + imagePath)

		try:
			image = Image.open(imagePath)

			self.dPrint("  Old size: " + str(image.size[0]) + "x" + str(image.size[1]))
			
			image.draft("RGB", image.size)
			image = image.convert("RGB")

			if (drawBorder):
				newSize -= 2

			image.thumbnail((newSize, newSize), Image.ANTIALIAS)

			if (drawBorder):
				image = ImageOps.expand(image, border=1)
			# end if

			self.dPrint("  New size: " + str(image.size[0]) + "x" + str(image.size[1]))
			image.save(imagePath)

			return(image.size)
		except Exception, error:
			self.handleException("RLCovers::resizeImage()", error)

			return((0, 0))
	#-------------------------------------------------------------------


	def resizeImageGtk(self, imagePath, newSize, drawBorder=1):
	#-------------------------------------------------------------------
		"""Resize image to newSize using GTK."""

		if (HAVE_GTK == 0):
			return((0, 0))

		self.dPrint("(GTK) Resizing " + imagePath)

		try:
			pixbuf = gtk.gdk.pixbuf_new_from_file(imagePath)
			self.dPrint(" Old size: " + str(pixbuf.get_width()) + "x" + str(pixbuf.get_height()))

			newWidth = pixbuf.get_width()
			newHeight= pixbuf.get_height()

			# Do not resize if image is already desired size.
			if ((newWidth == newSize) or (newHeight == newSize)):
				self.dPrint("No need to resize.")
				return(newWidth, newHeight)

			# Retain aspect ratio.
			if (newWidth > newSize):
				newHeight = newHeight * newSize / newWidth
				newWidth = newSize
			if (newHeight > newSize):
				newWidth = newWidth * newSize / newHeight
				newHeight = newSize
			
			# Draw border.
			if (drawBorder):
				# Create new blank image with black background newWidth*newHeight in size.
				GDK_BLACK = long(0)
				bg = gtk.gdk.pixbuf_new_from_file(imagePath)
				bg = bg.composite_color_simple(newWidth, newHeight, gtk.gdk.INTERP_BILINEAR, 0, 2, GDK_BLACK, GDK_BLACK)

				# Readjust for borders.
				newWidth	-= 2
				newHeight	-= 2

				# Resize original file to newWidth*newHeight.
				pixbuf = pixbuf.scale_simple(newWidth, newHeight, gtk.gdk.INTERP_BILINEAR)

				# Paste pixbuf over bg at offset 1x1 will full opacity (255.)
				pixbuf.composite(bg, 1, 1, newWidth, newHeight, 0, 0, 1, 1, gtk.gdk.INTERP_BILINEAR, 255)

				# Copy resulting image (now with borders) to pixbuf variable.
				pixbuf = bg.copy()
			else:
				# Resize original file to newWidth*newHeight.
				pixbuf = pixbuf.scale_simple(newWidth, newHeight, gtk.gdk.INTERP_BILINEAR)
			# end if

			# Save the resized image.
			self.dPrint(" New size: " + str(pixbuf.get_width()) + "x" + str(pixbuf.get_height()))
			pixbuf.save(imagePath, "jpeg")

			return((newWidth, newHeight))
		except Exception, error:
			self.handleException("resizeImageGTK()", error)

			return((0, 0))
	#-------------------------------------------------------------------


	def getImageSize(self, imagePath):
	#-------------------------------------------------------------------
		"""Return image size."""

		try:
			if (HAVE_PIL):
				image = Image.open(imagePath)
				return(image.size)
			if (HAVE_GTK):
				pixbuf = gtk.gdk.pixbuf_new_from_file(imagePath)
				return((pixbuf.get_width(), pixbuf.get_height()))
		except:
			return((0, 0))
		# end try

		return((0, 0))
	#-------------------------------------------------------------------
#-----------------------------------------------------------------------
