/* sqldb.c
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqldb.h"
#include <string.h>
#include "gasql_conf.h"

/*
 *
 * Main object: SqlDb
 *
 */

static void sql_db_class_init (SqlDbClass * class);
static void sql_db_init (SqlDb * db);
static void sql_db_destroy (GtkObject * object);

/* "fault" signal callback */
static void sql_db_cb (GtkObject * obj, gpointer data);

/* widget inner callbacks to signals */
static void s_table_created_dropped (SqlDb * db, SqlMemTable * t);
static void s_seq_created_dropped (SqlDb * db, SqlMemSeq * t);
static void s_field_created_dropped (SqlDb * db, SqlMemTable * t,
				     SqlMemField * field);
static void s_ff_link_created_dropped (SqlDb * db, SqlMemFlink * f);
static void s_fs_link_created_dropped (SqlDb * db, SqlMemSeq * seq,
				       SqlMemField * f);
/* updates the DB structure */
static void sql_db_load_tables (SqlDb * db, SqlAccess * srv);
static void sql_db_load_views (SqlDb * db, SqlAccess * srv);
static void sql_db_load_clean_unused_tables_views (SqlDb * db,
						   SqlAccess * srv);
static void sql_db_load_sequences (SqlDb * db, SqlAccess * srv);

/* loads or reloads the fields of a table */
static void sql_mem_table_load_fields (SqlMemTable * t,
				       SqlAccess * srv, SqlDb * db);
static void sql_mem_field_load_contents (SqlAccess * srv, SqlMemField * field,
					 GdaRecordset * recset);
static void sql_db_catch_field_create_cb (GtkObject * obj,
					  SqlMemField * new_field,
					  gpointer data);
static void sql_db_catch_field_drop_cb (GtkObject * obj, SqlMemField * field,
					gpointer data);

static gpointer sqlaccess_binding_func (GtkObject * obj);

/* signals */
enum
{
	UPDATED,
	STRUCT_SAVED,
	FAULT,
	TABLE_CREATED,
	TABLE_CREATED_FILLED,
	TABLE_DROPPED,
	SEQ_CREATED,
	SEQ_DROPPED,
	FIELD_CREATED,
	FIELD_DROPPED,
	FF_LINK_CREATED,
	FF_LINK_DROPPED,
	FS_LINK_CREATED,
	FS_LINK_DROPPED,
	PROGRESS,
	LASTDB_SIGNAL
};

static gint sql_db_signals[LASTDB_SIGNAL] = { 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0
};

GtkType
sql_db_get_type (void)
{
	static GtkType f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Db",
			sizeof (SqlDb),
			sizeof (SqlDbClass),
			(GtkClassInitFunc) sql_db_class_init,
			(GtkObjectInitFunc) sql_db_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_db_class_init (SqlDbClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
	sql_db_signals[FAULT] =
		gtk_signal_new ("fault",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, fault),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);
	sql_db_signals[STRUCT_SAVED] =
		gtk_signal_new ("struct_saved", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, struct_saved),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);
	sql_db_signals[UPDATED] =
		gtk_signal_new ("updated", GTK_RUN_FIRST, object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, updated),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_db_signals[TABLE_CREATED] =
		gtk_signal_new ("table_created",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, table_created),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_db_signals[TABLE_CREATED_FILLED] =
		gtk_signal_new ("table_created_filled",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass,
						   table_created_f),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_db_signals[TABLE_DROPPED] =
		gtk_signal_new ("table_dropped",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, table_dropped),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_db_signals[SEQ_CREATED] =
		gtk_signal_new ("seq_created",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, seq_created),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_db_signals[SEQ_DROPPED] =
		gtk_signal_new ("seq_dropped",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, seq_dropped),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_db_signals[FIELD_CREATED] =
		gtk_signal_new ("field_created",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, field_created),
				gtk_marshal_NONE__POINTER_POINTER,
				GTK_TYPE_NONE, 2, GTK_TYPE_POINTER,
				GTK_TYPE_POINTER);

	sql_db_signals[FIELD_DROPPED] =
		gtk_signal_new ("field_dropped",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, field_dropped),
				gtk_marshal_NONE__POINTER_POINTER,
				GTK_TYPE_NONE, 2, GTK_TYPE_POINTER,
				GTK_TYPE_POINTER);

	sql_db_signals[FF_LINK_CREATED] =
		gtk_signal_new ("ff_link_created",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass,
						   ff_link_created),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_db_signals[FF_LINK_DROPPED] =
		gtk_signal_new ("ff_link_dropped",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass,
						   ff_link_dropped),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_db_signals[FS_LINK_CREATED] =
		gtk_signal_new ("fs_link_created",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass,
						   fs_link_created),
				gtk_marshal_NONE__POINTER_POINTER,
				GTK_TYPE_NONE, 2, GTK_TYPE_POINTER,
				GTK_TYPE_POINTER);

	sql_db_signals[FS_LINK_DROPPED] =
		gtk_signal_new ("fs_link_dropped",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass,
						   fs_link_dropped),
				gtk_marshal_NONE__POINTER_POINTER,
				GTK_TYPE_NONE, 2, GTK_TYPE_POINTER,
				GTK_TYPE_POINTER);

	sql_db_signals[PROGRESS] =
		gtk_signal_new ("progress",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlDbClass, progress),
				gtk_marshal_NONE__POINTER_UINT_UINT,
				GTK_TYPE_NONE, 3, GTK_TYPE_POINTER,
				GTK_TYPE_UINT, GTK_TYPE_UINT);

	gtk_object_class_add_signals (object_class, sql_db_signals,
				      LASTDB_SIGNAL);
	class->updated = NULL;
	class->struct_saved = NULL;
	class->fault = NULL;
	class->table_created = s_table_created_dropped;
	class->table_created_f = NULL;	/*s_table_created_dropped; */
	class->table_dropped = s_table_created_dropped;
	class->seq_created = s_seq_created_dropped;
	class->seq_dropped = s_seq_created_dropped;
	class->field_created = s_field_created_dropped;
	class->field_dropped = s_field_created_dropped;
	class->ff_link_created = s_ff_link_created_dropped;
	class->ff_link_dropped = s_ff_link_created_dropped;
	class->fs_link_created = s_fs_link_created_dropped;
	class->fs_link_dropped = s_fs_link_created_dropped;
	class->progress = NULL;
	object_class->destroy = sql_db_destroy;
}

static void
sql_db_init (SqlDb * db)
{
	db->name = NULL;
	db->is_fault = FALSE;
	db->tables = NULL;
	db->sequences = NULL;
	db->field_links = NULL;
	db->users = NULL;
}

GtkObject *
sql_db_new (SqlAccess * srv)
{
	GtkObject *obj;
	SqlDb *db;

	obj = gtk_type_new (sql_db_get_type ());
	db = SQL_DB (obj);

	db->srv = srv;
	gtk_signal_connect (GTK_OBJECT (srv), "conn_closed",
			    GTK_SIGNAL_FUNC (sql_db_empty_all_cb), db);

	/* declaring that tables' fields are objects that one can bind plugins to */
	sql_access_declare_object_bindable (srv, sqlaccess_binding_func);

	return obj;
}

static void
sql_db_destroy (GtkObject * object)
{
	SqlDb *db;
	GtkObject *parent_class = NULL;
	GSList *list;

	parent_class = gtk_type_class (gtk_object_get_type ());
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_DB (object));

	db = SQL_DB (object);

	/* destroying Tables list */
	if (db->tables) {
		list = db->tables;
		while (list) {
			gtk_object_destroy (GTK_OBJECT (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (db->tables);
		db->tables = NULL;
	}
	/* destroying Sequences list */
	if (db->sequences) {
		list = db->sequences;
		while (list) {
			gtk_object_destroy (GTK_OBJECT (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (db->sequences);
		db->sequences = NULL;
	}

	/* Field links list */
	if (db->field_links) {
		list = db->field_links;
		while (list) {
			gtk_object_destroy (GTK_OBJECT (list->data));
			list = g_slist_next (list);
		}
		g_slist_free (db->field_links);
		db->field_links = NULL;
	}

	/* the name */
	if (db->name) {
		g_free (db->name);
		db->name = NULL;
	}

	/* the users list */
	while (db->users) {
		GSList *hold = db->users;
		g_free (db->users->data);
		db->users = g_slist_remove_link (db->users, db->users);
		g_slist_free_1 (hold);
	}

	/* for the parent class */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static gpointer
sqlaccess_binding_func (GtkObject * obj)
{
	g_return_val_if_fail ((obj != NULL), NULL);
	if (IS_SQL_MEM_FIELD (obj))
		return SQL_MEM_FIELD (obj)->type;
	else
		return NULL;
}


void
sql_db_empty_all_cb (GtkObject * obj, SqlDb * db)
{
	gpointer data;
	GSList *hold;

	/* first everything should be saved */
	/* MODIFY */

	while (db->tables) {
		hold = db->tables;
		data = db->tables->data;
		db->tables = g_slist_remove_link (db->tables, db->tables);
		g_slist_free_1 (hold);
		gtk_signal_emit (GTK_OBJECT (db),
				 sql_db_signals[TABLE_DROPPED],
				 SQL_MEM_TABLE (data));
		gtk_object_destroy (GTK_OBJECT (data));
	}
	db->tables = NULL;

	while (db->sequences) {
		hold = db->sequences;
		data = db->sequences->data;
		db->sequences =	g_slist_remove_link (db->sequences, db->sequences);
		g_slist_free_1 (hold);
		gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[SEQ_DROPPED],
				 SQL_MEM_SEQ (data));
		gtk_object_destroy (GTK_OBJECT (data));
	}
	db->sequences = NULL;

	while (db->field_links) {
		hold = db->field_links;
		data = db->field_links->data;
		db->field_links =
			g_slist_remove_link (db->field_links,
					     db->field_links);
		g_slist_free_1 (hold);
		gtk_signal_emit (GTK_OBJECT (db),
				 sql_db_signals[FF_LINK_DROPPED],
				 SQL_MEM_FLINK (data));
		gtk_object_destroy (GTK_OBJECT (data));
	}
	db->field_links = NULL;
}

static void
s_ff_link_created_dropped (SqlDb * db, SqlMemFlink * f)
{
#ifdef debug_signal
	g_print (">> 'UPDATED' from sqldb->s_ff_link_created_dropped\n");
#endif
	gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[UPDATED]);
#ifdef debug_signal
	g_print ("<< 'UPDATED' from sqldb->s_ff_link_created_dropped\n");
#endif
}

static void
s_fs_link_created_dropped (SqlDb * db, SqlMemSeq * seq, SqlMemField * f)
{
#ifdef debug_signal
	g_print (">> 'UPDATED' from sqldb->s_fs_link_created_dropped\n");
#endif
	gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[UPDATED]);
#ifdef debug_signal
	g_print ("<< 'UPDATED' from sqldb->s_fs_link_created_dropped\n");
#endif
}

static void
s_table_created_dropped (SqlDb * db, SqlMemTable * t)
{
#ifdef debug_signal
	g_print (">> 'UPDATED' from sqldb->s_table_created_dropped\n");
#endif
	gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[UPDATED]);
#ifdef debug_signal
	g_print ("<< 'UPDATED' from sqldb->s_table_created_dropped\n");
#endif
}

static void
s_seq_created_dropped (SqlDb * db, SqlMemSeq * t)
{
#ifdef debug_signal
	g_print (">> 'UPDATED' from sqldb->s_seq_created_dropped\n");
#endif
	gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[UPDATED]);
#ifdef debug_signal
	g_print ("<< 'UPDATED' from sqldb->s_seq_created_dropped\n");
#endif
}

static void
s_field_created_dropped (SqlDb * db, SqlMemTable * t, SqlMemField * field)
{
#ifdef debug_signal
	g_print (">> 'UPDATED' from sqldb->s_field_created_dropped\n");
#endif
	gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[UPDATED]);
#ifdef debug_signal
	g_print ("<< 'UPDATED' from sqldb->s_field_created_dropped\n");
#endif
}

/****************************************/
/*                                      */
/* Callbacks                            */
/*                                      */
/****************************************/

/* this callback is called whenever a table or sequence 
   emits the signal "load_fault" */
static void
sql_db_cb (GtkObject * obj, gpointer data)
{
	SQL_DB (data)->is_fault = TRUE;
#ifdef debug_signal
	g_print (">> 'FAULT' from sqldb->sql_db_cb\n");
#endif
	gtk_signal_emit (GTK_OBJECT (data), sql_db_signals[FAULT]);
#ifdef debug_signal
	g_print ("<< 'FAULT' from sqldb->sql_db_cb\n");
#endif
	/* note: The default signal handler for that signal will ecit(1) the
	   application */
}

/* This CB is called whenever a table emits the "field_created" signal, 
   so the DB object can forward that signal and emit the "field_created" signal
   for its listeners. Data points to the DB object */
static void
sql_db_catch_field_create_cb (GtkObject * obj,
			      SqlMemField * new_field, gpointer data)
{
#ifdef debug_signal
	g_print (">> 'FIELD_CREATED' from sqldb->sql_db_catch_field_create_cb\n");
#endif
	gtk_signal_emit (GTK_OBJECT (data), sql_db_signals[FIELD_CREATED],
			 obj, new_field);
#ifdef debug_signal
	g_print ("<< 'FIELD_CREATED' from sqldb->sql_db_catch_field_create_cb\n");
#endif
}

/* This CB is called whenever a table emits the "field_dropped" signal, 
   so the DB object can forward that signal and emit the "field_dropped" signal
   for its listeners. Data points to the DB object */
static void
sql_db_catch_field_drop_cb (GtkObject * obj,
			    SqlMemField * field, gpointer data)
{
	/* emit signal */
#ifdef debug_signal
	g_print (">> 'FIELD_DROPPED' from sqldb->sql_db_catch_field_drop_cb\n");
#endif
	gtk_signal_emit (GTK_OBJECT (data), sql_db_signals[FIELD_DROPPED],
			 obj, field);
#ifdef debug_signal
	g_print ("<< 'FIELD_DROPPED' from sqldb->sql_db_catch_field_drop_cb\n");
#endif
}

/****************************************/
/*                                      */
/* Tables and fields manipulation       */
/*                                      */
/****************************************/
GSList *
find_user_name (SqlDb * db, gchar * uname)
{
	GSList *list;
	GSList *found = NULL;

	g_return_val_if_fail ((!db->is_fault), NULL);

	list = db->users;
	while (!found && list) {
		if (!strcmp (uname, (gchar *) (list->data))) {
			found = list;
		}
		list = g_slist_next (list);
	}
	if (!found) {
		db->users = g_slist_append (db->users, g_strdup (uname));
		found = g_slist_last (db->users);
	}

	return found;
}

SqlMemTable *
sql_db_find_table_by_name (SqlDb * db, gchar * name)
{
	GSList *list = db->tables;

	g_return_val_if_fail ((!db->is_fault), NULL);

	while (list && (strcmp (name, ((SqlMemTable *) (list->data))->name))) {
		list = g_slist_next (list);
	}
	if (list)
		return (SqlMemTable *) (list->data);
	else
		return NULL;
}

SqlMemTable *
sql_db_find_table_by_xml_name (SqlDb * db, gchar * xmlname)
{
	return sql_db_find_table_by_name (db, xmlname + 2);
}

SqlMemSeq *
sql_db_find_sequence_by_name (SqlDb * db, gchar * name)
{
	GSList *list = db->sequences;

	g_return_val_if_fail ((!db->is_fault), NULL);

	while (list && (strcmp (name, ((SqlMemSeq *) (list->data))->name))) {
		list = g_slist_next (list);
	}
	if (list)
		return (SqlMemSeq *) (list->data);
	else
		return NULL;
}

SqlMemSeq *
sql_db_find_sequence_by_xml_name (SqlDb * db, gchar * xmlname)
{
	return sql_db_find_sequence_by_name (db, xmlname + 2);
}

SqlMemSeq *
sql_db_find_sequence_to_field (SqlDb * db, SqlMemField * field)
{
	GSList *list;
	SqlMemSeq *seq = NULL;

	g_return_val_if_fail ((!db->is_fault), NULL);

	list = db->sequences;
	while (list && !seq) {
		if (g_slist_find
		    (SQL_MEM_SEQ (list->data)->field_links, field))
			seq = SQL_MEM_SEQ (list->data);
		else
			list = g_slist_next (list);
	}
	return seq;
}

SqlMemField *
sql_db_find_field_from_names (SqlDb * db, gchar * table, char *field)
{
	SqlMemTable *tab;
	SqlMemField *f = NULL;
	GSList *list;

	g_return_val_if_fail ((!db->is_fault), NULL);

	tab = sql_db_find_table_by_name (db, table);
	if (tab) {
		list = tab->fields;
		while (list && !f) {
			if (!strcmp (SQL_MEM_FIELD (list->data)->name, field))
				f = SQL_MEM_FIELD (list->data);
			list = g_slist_next (list);
		}
	}
	return f;
}

SqlMemField *
sql_db_find_field_from_xml_name (SqlDb * db, char *xmlname)
{
	gchar *ptr, *cpy;
	SqlMemField *field;
	cpy = g_strdup (xmlname);
	ptr = strstr (cpy, ":FI");
	*ptr = 0;
	field = sql_db_find_field_from_names (db, cpy + 2, ptr + 3);
	g_free (cpy);
	return field;
}


SqlMemTable *
sql_db_find_table_from_field (SqlDb * db, SqlMemField * field)
{
	SqlMemTable *table = NULL;
	GSList *tlist = db->tables;
	GSList *flist;

	g_return_val_if_fail ((!db->is_fault), NULL);

	while (tlist && !table) {
		flist = ((SqlMemTable *) (tlist->data))->fields;
		while (flist && !table) {
			if (flist->data == field)
				table = (SqlMemTable *) (tlist->data);
			flist = g_slist_next (flist);
		}
		tlist = g_slist_next (tlist);
	}

	return table;
}

guint
sql_db_get_field_order (SqlDb * db, SqlMemTable * table, SqlMemField * field)
{
	SqlMemTable *tab;
	GSList *list;
	guint order, i;

	g_return_val_if_fail ((!db->is_fault), 0);

	if (!table)
		tab = sql_db_find_table_from_field (db, field);
	else
		tab = table;

	if (!tab)		/* field does not belong to any table */
		return 0;

	list = tab->fields;
	order = 0;
	i = 0;
	while (list && !order) {
		i++;
		if (list->data == field)
			order = i;
		list = g_slist_next (list);
	}
	return order;
}

void
sql_db_refresh (SqlDb * db, SqlAccess * srv)
{
	g_return_if_fail ((!db->is_fault));

	/* the update procedure */
	if (sql_access_is_open (srv)) {
		sql_db_load_tables (db, srv);
		sql_db_load_views (db, srv);
		sql_db_load_clean_unused_tables_views (db, srv);
		sql_db_load_sequences (db, srv);
	}
	else
		g_warning ("SqlDb::sql_db_refresh() -> connection closed!\n");

#ifdef debug_signal
	g_print (">> 'UPDATED' from sql_db_refresh\n");
#endif
	gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[UPDATED]);
#ifdef debug_signal
	g_print ("<< 'UPDATED' from sql_db_refresh\n");
#endif
}

static void
sql_db_load_tables (SqlDb * db, SqlAccess * srv)
{
	SqlMemTable *table;
	GSList *list, *list2;
	gboolean created;
	GdaRecordset *rst;
	GdaField *field;
	gchar *str;
	guint now, total;

	g_return_if_fail ((!db->is_fault));

	if (db->name) {
		g_free (db->name);
		db->name = NULL;
	}
	db->name = g_strdup (srv->gda_datasource->str);

	/* getting the tables list */
	rst = gda_connection_open_schema (GDA_CONNECTION (srv),
					  GDA_Connection_GDCN_SCHEMA_TABLES,
					  GDA_Connection_EXTRA_INFO, "t",
					  GDA_Connection_no_CONSTRAINT);

	if (rst) {
		/* FIXME: use the supports() function to see if the total number of
		   tuples can be got that way */
		total = rst->affected_rows;

		now = 0;
		gda_recordset_move (rst, 1, 0);
		while (!gda_recordset_eof (rst)) {
			now++;
			field = gda_recordset_field_idx (rst, 0);
			str = gda_stringify_value (NULL, 0, field);	/* table name */
			table = sql_db_find_table_by_name (db, str);
			if (!table) {	/* the table doesn't already exist, create it */
				table = SQL_MEM_TABLE (sql_mem_table_new ());
				table->name = str;
				str = NULL;

				gtk_signal_connect (GTK_OBJECT (table),
						    "load_fault",
						    GTK_SIGNAL_FUNC
						    (sql_db_cb), db);
				gtk_signal_connect (GTK_OBJECT (table),
						    "field_created",
						    GTK_SIGNAL_FUNC
						    (sql_db_catch_field_create_cb),
						    db);
				gtk_signal_connect (GTK_OBJECT (table),
						    "field_dropped",
						    GTK_SIGNAL_FUNC
						    (sql_db_catch_field_drop_cb),
						    db);
				db->tables =
					g_slist_append (db->tables, table);
				created = TRUE;
			}
			else {
				created = FALSE;
			}
			if (str)
				g_free (str);

			/* (re)computes the comments */
			field = gda_recordset_field_idx (rst, 2);	/* comments */
			if (field && gda_field_get_string_value (field)) {
				str = gda_stringify_value (NULL, 0, field);
				sql_mem_table_set_comments (table, NULL, str,
							    FALSE);
				g_free (str);
			}
			/* (re)computes the owner */
			field = gda_recordset_field_idx (rst, 1);	/* owner */
			if (field && gda_field_get_string_value (field)) {
				GSList *list;
				str = gda_stringify_value (NULL, 0, field);
				list = find_user_name (db, str);
				table->owner = (gchar *) (list->data);
				g_free (str);
			}

			if (created) {	/* to emit signals now because then owner and comments filled */
#ifdef debug_signal
				g_print (">> 'TABLE_CREATED' from sql_db_load_tables\n");
#endif
				gtk_signal_emit (GTK_OBJECT (db),
						 sql_db_signals
						 [TABLE_CREATED], table);
#ifdef debug_signal
				g_print ("<< 'TABLE_CREATED' from sql_db_load_tables\n");
#endif
			}

			sql_mem_table_load_fields (table, srv, db);
			if (created) {
#ifdef debug_signal
				g_print (">> 'TABLE_CREATED_FILLED' from sql_db_load_tables\n");
#endif
				gtk_signal_emit (GTK_OBJECT (db),
						 sql_db_signals
						 [TABLE_CREATED_FILLED],
						 table);
#ifdef debug_signal
				g_print ("<< 'TABLE_CREATED_FILLED' from sql_db_load_tables\n");
#endif
			}
			/* mark the table as updated */
			((SqlMemItem *) table)->updated = TRUE;

			gtk_signal_emit_by_name (GTK_OBJECT (db), "progress",
						 _
						 ("Updating the list of tables..."),
						 now, total);
			gda_recordset_move_next (rst);
		}
		gda_recordset_free (rst);

		if (!(db->is_fault)) {
	  /***************************************************************/
			/* 2nd step: re-compute the parents table list for every table */
	  /***************************************************************/
			list = db->tables;
			while (list) {
				if ((SQL_MEM_TABLE (list->data)->is_view ==
				     FALSE)
				    && (SQL_MEM_ITEM (list->data)->updated ==
					TRUE)) {
					g_slist_free (SQL_MEM_TABLE
						      (list->data)->parents);
					SQL_MEM_TABLE (list->data)->parents =
						NULL;
					/* get the parents' list */
					rst = gda_connection_open_schema
						(GDA_CONNECTION (srv),
						 GDA_Connection_GDCN_SCHEMA_TAB_PARENTS,
						 GDA_Connection_OBJECT_NAME,
						 SQL_MEM_TABLE (list->data)->
						 name,
						 GDA_Connection_no_CONSTRAINT);

					if (rst) {
						list2 = NULL;
						gda_recordset_move (rst, 1,
								    0);
						while (!gda_recordset_eof
						       (rst)) {
							field = gda_recordset_field_idx (rst, 0);
							str = gda_stringify_value (NULL, 0, field);
							table = sql_db_find_table_by_name (db, str);
							g_free (str);
							if (table)
								list2 = g_slist_append (list2, table);
							gda_recordset_move_next
								(rst);
						}
						SQL_MEM_TABLE (list->data)->
							parents = list2;
						gda_recordset_free (rst);
					}
				}
				list = g_slist_next (list);
			}
		}
	}
	else {
		/* An error occured, the db emits the fault signal */
#ifdef debug_signal
		g_print (">> 'FAULT' from sql_db_load_tables\n");
#endif
		gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[FAULT]);
#ifdef debug_signal
		g_print ("<< 'FAULT' from sql_db_load_tables\n");
#endif
	}
	/* if an error occurs, the DB remains non accessible */
	gtk_signal_emit_by_name (GTK_OBJECT (db), "progress", NULL, 0, 0);
}

static void
sql_db_load_views (SqlDb * db, SqlAccess * srv)
{
	SqlMemTable *table;
	gboolean created;
	GdaRecordset *rst;
	GdaField *field;
	gchar *str;
	guint now, total;

	g_return_if_fail ((!db->is_fault));

	if (db->name) {
		g_free (db->name);
		db->name = NULL;
	}
	db->name = g_strdup (srv->gda_datasource->str);

	/* getting the tables list */
	rst = gda_connection_open_schema (GDA_CONNECTION (srv),
					  GDA_Connection_GDCN_SCHEMA_VIEWS,
					  GDA_Connection_EXTRA_INFO, "t",
					  GDA_Connection_no_CONSTRAINT);

	if (rst) {
		/* FIXME: use the supports() function to see if the total number of
		   tuples can be got that way */
		total = rst->affected_rows;

		now = 0;
		gda_recordset_move (rst, 1, 0);
		while (!gda_recordset_eof (rst)) {
			now++;
			field = gda_recordset_field_idx (rst, 0);
			str = gda_stringify_value (NULL, 0, field);	/* table name */
			table = sql_db_find_table_by_name (db, str);
			if (!table) {	/* the table doesn't already exist, create it */
				table = SQL_MEM_TABLE (sql_mem_table_new ());
				table->name = str;
				str = NULL;

				gtk_signal_connect (GTK_OBJECT (table),
						    "load_fault",
						    GTK_SIGNAL_FUNC
						    (sql_db_cb), db);
				gtk_signal_connect (GTK_OBJECT (table),
						    "field_created",
						    GTK_SIGNAL_FUNC
						    (sql_db_catch_field_create_cb),
						    db);
				gtk_signal_connect (GTK_OBJECT (table),
						    "field_dropped",
						    GTK_SIGNAL_FUNC
						    (sql_db_catch_field_drop_cb),
						    db);
				db->tables =
					g_slist_append (db->tables, table);
				created = TRUE;
			}
			else {
				created = FALSE;
			}
			if (str)
				g_free (str);

			/* set it here because otherwise if an object was a table and 
			   becomes a view, this won't be set correctly */
			table->is_view = TRUE;

			/* (re)computes the comments */
			field = gda_recordset_field_idx (rst, 2);	/* comments */
			if (gda_field_get_string_value (field)) {
				str = gda_stringify_value (NULL, 0, field);
				sql_mem_table_set_comments (table, NULL, str,
							    FALSE);
				g_free (str);
			}
			/* (re)computes the owner */
			field = gda_recordset_field_idx (rst, 1);	/* owner */
			if (gda_field_get_string_value (field)) {
				GSList *list;
				str = gda_stringify_value (NULL, 0, field);
				list = find_user_name (db, str);
				table->owner = (gchar *) (list->data);
				g_free (str);
			}

			if (created) {	/* to emit signals now because then owner and comments filled */
#ifdef debug_signal
				g_print (">> 'TABLE_CREATED' from sql_db_load_tables\n");
#endif
				gtk_signal_emit (GTK_OBJECT (db),
						 sql_db_signals
						 [TABLE_CREATED], table);
#ifdef debug_signal
				g_print ("<< 'TABLE_CREATED' from sql_db_load_tables\n");
#endif
			}
			sql_mem_table_load_fields (table, srv, db);
			if (created) {
#ifdef debug_signal
				g_print (">> 'TABLE_CREATED_FILLED' from sql_db_load_tables\n");
#endif
				gtk_signal_emit (GTK_OBJECT (db),
						 sql_db_signals
						 [TABLE_CREATED_FILLED],
						 table);
#ifdef debug_signal
				g_print ("<< 'TABLE_CREATED_FILLED' from sql_db_load_tables\n");
#endif
			}
			/* mark the table as updated */
			((SqlMemItem *) table)->updated = TRUE;

			gtk_signal_emit_by_name (GTK_OBJECT (db), "progress",
						 _
						 ("Updating the list of views..."),
						 now, total);
			gda_recordset_move_next (rst);
		}
		gda_recordset_free (rst);
	}
	gtk_signal_emit_by_name (GTK_OBJECT (db), "progress", NULL, 0, 0);
}

static void
sql_db_load_clean_unused_tables_views (SqlDb * db, SqlAccess * srv)
{
	GSList *list, *list2, *list_hold;

	g_return_if_fail ((!db->is_fault));

	list = db->tables;
	while (list) {
		if (!SQL_MEM_ITEM (list->data)->updated) {
			GtkObject *obj;
			/* we want to emit the signal "table_dropped" with a pointer to 
			   a table which still exists, but is not anymore in the list 
			   of tables in the DB. */
			obj = GTK_OBJECT (list->data);
			/* we first remove all the links to any field of that table */
			list2 = SQL_MEM_TABLE (obj)->fields;
			while (list2) {
				sql_db_delete_seq_field_link (db, NULL,
							      SQL_MEM_FIELD
							      (list2->data));
				sql_db_delete_field_field_link (db, NULL,
								SQL_MEM_FIELD
								(list2->
								 data));
				sql_db_delete_field_field_link (db,
								SQL_MEM_FIELD
								(list2->data),
								NULL);
				list2 = g_slist_next (list2);
			}
			list_hold = g_slist_next (list);
			list2 = list;
			db->tables = g_slist_remove_link (db->tables, list);
			g_slist_free_1 (list2);
			list = list_hold;
#ifdef debug_signal
			g_print (">> 'TABLE_DROPPED' from sql_db_load_tables\n");
#endif
			gtk_signal_emit (GTK_OBJECT (db),
					 sql_db_signals[TABLE_DROPPED], obj);
#ifdef debug_signal
			g_print ("<< 'TABLE_DROPPED' from sql_db_load_tables\n");
#endif
			gtk_object_destroy (obj);
		}
		else {
			SQL_MEM_ITEM (list->data)->updated = FALSE;
			list = g_slist_next (list);
		}
	}
}


static void
sql_db_load_sequences (SqlDb * db, SqlAccess * srv)
{
	SqlMemSeq *seq;
	GSList *list, *list_hold, *hold;
	GdaRecordset *rs;
	GdaField *field;
	gchar *str;
	guint now, total;
	gboolean created;

	g_return_if_fail ((!db->is_fault));

	/* 1: getting the sequence list */
	rs = gda_connection_open_schema (GDA_CONNECTION (srv),
					 GDA_Connection_GDCN_SCHEMA_SEQUENCES,
					 GDA_Connection_EXTRA_INFO, "t",
					 GDA_Connection_no_CONSTRAINT);
	if (rs) {
		/* FIXME: use the supports() function to see if the total number of
		   tuples can be got that way */
		total = rs->affected_rows;

		now = 0;
		gda_recordset_move (rs, 1, 0);
		while (!gda_recordset_eof (rs)) {
			now++;
			field = gda_recordset_field_idx (rs, 0);
			str = gda_stringify_value (NULL, 0, field);
			created = FALSE;
			seq = sql_db_find_sequence_by_name (db, str);
			if (!seq) {
				seq = SQL_MEM_SEQ (sql_mem_seq_new ());
				seq->name = str;
				str = NULL;
				gtk_signal_connect (GTK_OBJECT (seq),
						    "load_fault",
						    GTK_SIGNAL_FUNC
						    (sql_db_cb), db);
				db->sequences =
					g_slist_append (db->sequences, seq);
				created = TRUE;
			}
			if (str)
				g_free (str);

			/* (re)computes the comments */
			field = gda_recordset_field_idx (rs, 2);	/* comments */
			if (gda_field_get_string_value (field)) {
				str = gda_stringify_value (NULL, 0, field);	/* str allocated */
				if (seq->comments)
					g_free (seq->comments);
				seq->comments = str;
			}

			/* (re)computes the owner */
			field = gda_recordset_field_idx (rs, 1);	/* owner */
			if (gda_field_get_string_value (field)) {
				GSList *list;
				str = gda_stringify_value (NULL, 0, field);
				list = find_user_name (db, str);
				seq->owner = (gchar *) (list->data);
				g_free (str);
			}

			if (created) {	/* then emit a signal */
#ifdef debug_signal
				g_print (">> 'SEQ_CREATED' from sql_db_load_sequences (%s)\n", seq->name);
#endif
				gtk_signal_emit (GTK_OBJECT (db),
						 sql_db_signals[SEQ_CREATED],
						 seq);
#ifdef debug_signal
				g_print ("<< 'SEQ_CREATED' from sql_db_load_sequences\n");
#endif
			}

			/* mark the sequence as updated */
			SQL_MEM_ITEM (seq)->updated = TRUE;
			gtk_signal_emit_by_name (GTK_OBJECT (db), "progress",
						 _
						 ("Updating the list of sequences..."),
						 now, total);
			gda_recordset_move_next (rs);
		}
		gda_recordset_free (rs);

		/* 2nd step: destroy sequences that do not exist anymore */
		list = db->sequences;
		while (list) {
			if (!SQL_MEM_ITEM (list->data)->updated) {
				GtkObject *obj;

				/* we want to emit the "seq_dropped" signal with a SqlMemSeq 
				   object
				   which still exists but is not anymore in the SqlDb 
				   structure */
				obj = GTK_OBJECT (list->data);
				/* we want to remove all the links from that sequence first */
				sql_db_delete_seq_field_link (db,
							      SQL_MEM_SEQ
							      (obj), NULL);

				list_hold = g_slist_next (list);
				hold = list;
				db->sequences = g_slist_remove_link (db->sequences, list);
				g_slist_free_1 (hold);
				list = list_hold;
				/* let's first remove any link from that sequence */
				sql_db_delete_seq_field_link (db,
							      SQL_MEM_SEQ
							      (obj), NULL);
#ifdef debug_signal
				g_print (">> 'SEQ_DROPPED' from sql_db_load_sequences (%s)\n", SQL_MEM_SEQ (obj)->name);
#endif
				gtk_signal_emit (GTK_OBJECT (db),
						 sql_db_signals[SEQ_DROPPED],
						 obj);
#ifdef debug_signal
				g_print ("<< 'SEQ_DROPPED' from sql_db_load_sequences\n");
#endif
				gtk_object_destroy (obj);
			}
			else {
				SQL_MEM_ITEM (list->data)->updated = FALSE;
				list = g_slist_next (list);
			}
		}
	}
	else {
		/* An error occured, the db emits the fault signal, if the
		 * sequences are supported */
		if (srv->features.sequences) {
#ifdef debug_signal
			g_print (">> 'FAULT' from sql_db_load_sequences\n");
#endif
			gtk_signal_emit (GTK_OBJECT (db),
					 sql_db_signals[FAULT]);
#ifdef debug_signal
			g_print ("<< 'FAULT' from sql_db_load_sequences\n");
#endif
		}
	}
	/* if an error occurs, the DB remains non accessible */
	gtk_signal_emit_by_name (GTK_OBJECT (db), "progress", NULL, 0, 0);
}


#ifdef debug
/****************************************/
/*                                      */
/* Dumping the tables and structure     */
/*                                      */
/****************************************/

void
sql_db_dump_tables (SqlDb * db)
{
	GSList *list, *plist;

	g_return_if_fail ((!db->is_fault));

	list = db->tables;
	g_print ("*** Dumping Database Structure ***\n");
	while (list) {
		g_print ("  TABLE %s", SQL_MEM_TABLE (list->data)->name);
		if (SQL_MEM_TABLE (list->data)->parents) {
			g_print (", parents= ");
			plist = SQL_MEM_TABLE (list->data)->parents;
			while (plist) {
				g_print ("%s ",
					 SQL_MEM_TABLE (plist->data)->name);
				plist = g_slist_next (plist);
			}
			g_print (".\n");
		}
		else
			g_print (", no known parents.\n");
		sql_mem_table_dump_fields (SQL_MEM_TABLE (list->data));
		list = g_slist_next (list);
	}
}

void
sql_db_dump_links (SqlDb * db)
{
	GSList *list, *list2;
	gchar *str;

	g_return_if_fail ((!db->is_fault));

	g_print ("*** Dumping Sequences Relations ***\n");
	list = db->sequences;
	while (list) {
		list2 = SQL_MEM_SEQ (list->data)->field_links;
		while (list2) {
			g_print ("\t%-25s --> %s.%s\n",
				 SQL_MEM_SEQ (list->data)->name,
				 sql_db_find_table_from_field (db,
							       SQL_MEM_FIELD
							       (list2->
								data))->name,
				 SQL_MEM_FIELD (list2->data)->name);
			list2 = g_slist_next (list2);
		}
		list = g_slist_next (list);
	}

	g_print ("*** Dumping Field to Field Relations ***\n");
	list = db->field_links;
	while (list) {
		str = g_strdup_printf ("%s.%s",
				       sql_db_find_table_from_field (db,
								     SQL_MEM_FIELD
								     (SQL_MEM_FLINK
								      (list->
								       data)->
								      from))->
				       name,
				       SQL_MEM_FIELD (SQL_MEM_FLINK
						      (list->data)->from)->
				       name);
		g_print ("\t%-25s --> %s.%s\n", str,
			 sql_db_find_table_from_field (db,
						       SQL_MEM_FIELD
						       (SQL_MEM_FLINK
							(list->data)->to))->
			 name,
			 SQL_MEM_FIELD (SQL_MEM_FLINK (list->data)->to)->
			 name);
		g_free (str);
		list = g_slist_next (list);
	}
}

void
sql_db_dump_struct_as_graph (SqlDb * db, FILE * st)
{
	GSList *list;

	g_return_if_fail ((!db->is_fault));

	fprintf (st, "graph: {\n");
	fprintf (st, "title: \"SqlDb tables dumping\"\n\n");

	/* sequences */
	list = db->sequences;
	while (list) {
		sql_mem_seq_dump_as_graph (SQL_MEM_SEQ (list->data), db, st);
		list = g_slist_next (list);
	}

	/* Field links */
	list = db->field_links;
	while (list) {
		sql_mem_flink_dump_as_graph (SQL_MEM_FLINK (list->data), db,
					     st);
		list = g_slist_next (list);
	}

	/* tables */
	list = db->tables;
	while (list) {
		sql_mem_table_dump_as_graph (SQL_MEM_TABLE (list->data), st);
		list = g_slist_next (list);
	}

	fprintf (st, "}\n");
}
#endif


/* if the link does not already exists, insert the link between the
   sequence and the field */
void
sql_db_insert_seq_field_link (SqlDb * db, SqlMemSeq * seq,
			      SqlMemField * field)
{
	GSList *list;
	gboolean found = FALSE;

	g_return_if_fail ((!db->is_fault));

	if ((sql_db_find_sequence_by_name (db, seq->name) == seq) &&
	    (sql_db_find_table_from_field (db, field) != NULL)) {
		list = seq->field_links;
		while (list && !found) {
			if (list->data == field)
				found = TRUE;
			list = g_slist_next (list);
		}
		if (!found) {	/* then insert it */
			seq->field_links =
				g_slist_append (seq->field_links, field);
			/* emit a signal */
#ifdef debug_signal
			g_print (">> 'FS_LINK_CREATED' from sql_db_insert_seq_field_link\n");
#endif
			gtk_signal_emit (GTK_OBJECT (db),
					 sql_db_signals[FS_LINK_CREATED], seq,
					 field);
#ifdef debug_signal
			g_print ("<< 'FS_LINK_CREATED' from sql_db_insert_seq_field_link\n");
#endif
		}
	}
	else
		g_warning
			("SqlDb::sql_db_insert_seq_field_link\tyou are trying to access "
			 "the fields of the wrong SqlDb structure!\n");
}

/* removes from memory the specified link or all links if seq==NULL or
 field==NULL */
void
sql_db_delete_seq_field_link (SqlDb * db, SqlMemSeq * seq,
			      SqlMemField * field)
{
	GSList *list;
	SqlMemField *f;

	g_return_if_fail ((!db->is_fault));

	if (field)
		if (((seq
		      && (sql_db_find_sequence_by_name (db, seq->name) == seq)
		      && (sql_db_find_table_from_field (db, field) != NULL)))
		    || (!seq
			&& (sql_db_find_table_from_field (db, field) !=
			    NULL))) {
			if (seq) {
				seq->field_links =
					g_slist_remove (seq->field_links,
							field);
#ifdef debug_signal
				g_print (">> 'FS_LINK_DROPPED' from sql_db_delete_seq_field_link " "1 (%s->%s)\n", seq->name, field->name);
#endif
				gtk_signal_emit (GTK_OBJECT (db),
						 sql_db_signals
						 [FS_LINK_DROPPED], seq,
						 field);
#ifdef debug_signal
				g_print ("<< 'FS_LINK_DROPPED' from sql_db_delete_seq_field_link\n");
#endif
			}
			else {
				list = db->sequences;
				while (list) {
					if (g_slist_find
					    (SQL_MEM_SEQ (list->data)->
					     field_links, field)) {
						SQL_MEM_SEQ (list->data)->
							field_links =
							g_slist_remove
							(SQL_MEM_SEQ
							 (list->data)->
							 field_links, field);
#ifdef debug_signal
						g_print (">> 'FS_LINK_DROPPED' from sql_db_delete_seq_field_link " "2 (%s->%s)\n", SQL_MEM_SEQ (list->data)->name, field->name);
#endif
						gtk_signal_emit (GTK_OBJECT
								 (db),
								 sql_db_signals
								 [FS_LINK_DROPPED],
								 SQL_MEM_SEQ
								 (list->data),
								 field);
#ifdef debug_signal
						g_print ("<< 'FS_LINK_DROPPED' " "from sql_db_delete_seq_field_link\n");
#endif
					}
					list = g_slist_next (list);
				}
			}
		}
		else
			g_warning
				("SqlDb::sql_db_delete_seq_field_link\nyou are "
				 "trying to access "
				 "the fields of the wrong SqlDb structure!\n");
	else /* field = NULL */ if (seq) {
		while (seq->field_links) {
			GSList *hold;
			f = SQL_MEM_FIELD (seq->field_links->data);
			hold = seq->field_links;
			seq->field_links =
				g_slist_remove_link (seq->field_links, seq->field_links);
			g_slist_free_1 (hold);
#ifdef debug_signal
			g_print (">> 'FS_LINK_DROPPED' from sql_db_delete_seq_field_link " "(%s->%s)\n", seq->name, f->name);
#endif
			gtk_signal_emit (GTK_OBJECT (db),
					 sql_db_signals[FS_LINK_DROPPED], seq,
					 f);
#ifdef debug_signal
			g_print ("<< 'FS_LINK_DROPPED' from sql_db_delete_seq_field_link\n");
#endif
		}
		seq->field_links = NULL;
	}
}

void
sql_db_insert_field_field_link (SqlDb * db, SqlMemField * from,
				SqlMemField * to)
{
	GSList *list = db->field_links;
	gboolean found = FALSE;
	SqlMemFlink *item;

	g_return_if_fail ((!db->is_fault));

	while (list && !found) {
		if ((SQL_MEM_FLINK (list->data)->from == from) &&
		    (SQL_MEM_FLINK (list->data)->to == to))
			found = TRUE;
		list = g_slist_next (list);
	}
	if (!found) {
		item = SQL_MEM_FLINK (sql_mem_flink_new ());
		item->from = from;
		item->to = to;
		db->field_links = g_slist_append (db->field_links, item);
		/* emit a signal */
#ifdef debug_signal
		g_print (">> 'FF_LINK_CREATED' from sql_db_insert_field_field_link\n");
#endif
		gtk_signal_emit (GTK_OBJECT (db),
				 sql_db_signals[FF_LINK_CREATED], item);
#ifdef debug_signal
		g_print ("<< 'FF_LINK_CREATED' from sql_db_insert_field_field_link\n");
#endif
	}
}

void
sql_db_delete_field_field_link (SqlDb * db, SqlMemField * from,
				SqlMemField * to)
{
	GSList *list, *hlist;
	gboolean found = FALSE;
	SqlMemFlink *item;
	guint mode = 0;

	g_return_if_fail ((!db->is_fault));

	if (!to)
		mode += 1;
	if (!from)
		mode += 2;

	list = db->field_links;
	while (list && !found) {
		item = NULL;
		switch (mode) {
		case 0:
			if ((SQL_MEM_FLINK (list->data)->from == from) &&
			    (SQL_MEM_FLINK (list->data)->to == to)) {
				found = TRUE;
				item = SQL_MEM_FLINK (list->data);
			}
			break;
		case 1:
			if (SQL_MEM_FLINK (list->data)->from == from)
				item = SQL_MEM_FLINK (list->data);
			break;
		case 2:
			if (SQL_MEM_FLINK (list->data)->to == to)
				item = SQL_MEM_FLINK (list->data);
			break;
		}
		if (item) {
			hlist = g_slist_next (list);
			db->field_links =
				g_slist_remove (db->field_links, item);
#ifdef debug_signal
			g_print (">> 'FF_LINK_DROPPED' from sql_db_delete_field_field_link\n");
#endif
			gtk_signal_emit (GTK_OBJECT (db),
					 sql_db_signals[FF_LINK_DROPPED],
					 item);
#ifdef debug_signal
			g_print ("<< 'FF_LINK_DROPPED' from sql_db_delete_field_field_link\n");
#endif
			gtk_object_destroy (GTK_OBJECT (item));
			list = hlist;
		}
		else
			list = g_slist_next (list);
	}
	if (found) {		/* emit a signal */
#ifdef debug_signal
		g_print (">> 'UPDATED' from sql_db_delete_field_field_link\n");
#endif
		gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[UPDATED]);
#ifdef debug_signal
		g_print ("<< 'UPDATED' from sql_db_delete_field_field_link\n");
#endif
	}
}

SqlMemFlink *
sql_db_find_flink_from_xml_name (SqlDb * db, gchar * xmlname)
{
	guint i;
	i = atoi (xmlname + 2);
	return SQL_MEM_FLINK (g_slist_nth_data (db->field_links, i));
}

SqlMemFlink *
sql_db_find_flink_from_fields (SqlDb * db, SqlMemField * fromf,
			       SqlMemField * tof)
{
	GSList *list;
	SqlMemFlink *retval, *fl;

	g_return_val_if_fail ((!db->is_fault), NULL);

	retval = NULL;
	list = db->field_links;
	while (list && !retval) {
		fl = SQL_MEM_FLINK (list->data);
		if ((fl->from == fromf) && (fl->to == tof))
			retval = fl;
		list = g_slist_next (list);
	}

	return retval;
}

/*************************************************/
/*                                               */
/* XML storage of the SqlDb object               */
/*                                               */
/*************************************************/
static void sql_mem_table_load_from_xml_node (SqlDb * db,
					      xmlNodePtr tabletree);
static void sql_mem_seq_load_from_xml_node (SqlDb * db, xmlNodePtr tabletree);
/* to store SqlDb object */
void
sql_db_build_xml_tree (SqlDb * db, xmlDocPtr doc)
{
	xmlNodePtr links, toptree, tree, subtree, subsubtree;
	GSList *list, *list2;
	SqlMemTable *table, *table2;
	SqlMemSeq *seq;
	SqlMemField *field;
	SqlMemFlink *flink;
	gint order;
	gchar *str;
	SqlDataType *datatype = NULL;

	g_return_if_fail ((!db->is_fault));

	/* main node */
	toptree = xmlNewChild (doc->xmlRootNode, NULL, "SQLDB", NULL);
	xmlSetProp (toptree, "name", db->name);

	/* tables */
	tree = xmlNewChild (toptree, NULL, "TABLES", NULL);
	list = db->tables;
	while (list) {
		table = SQL_MEM_TABLE (list->data);
		subtree = xmlNewChild (tree, NULL, "table", NULL);
		str = g_strdup_printf ("TV%s", table->name);
		xmlSetProp (subtree, "name", str);
		g_free (str);
		if (table->comments)
			xmlSetProp (subtree, "comments", table->comments);
		if (table->is_user_comments)
			xmlSetProp (subtree, "user_comments", "t");
		if (table->owner)
			xmlSetProp (subtree, "owner", table->owner);
		if (table->is_view)
			xmlSetProp (subtree, "is_view", "t");

		/* table's inheritance */
		list2 = table->parents;
		order = 1;
		while (list2) {
			subsubtree =
				xmlNewChild (subtree, NULL, "parent", NULL);
			str = g_strdup_printf ("TV%s",
					       SQL_MEM_TABLE (list2->data)->
					       name);
			xmlSetProp (subsubtree, "table", str);
			g_free (str);
			str = g_strdup_printf ("%d", order);
			xmlSetProp (subsubtree, "order", str);
			g_free (str);
			order++;
			list2 = g_slist_next (list2);
		}

		/* table's fields */
		list2 = table->fields;
		while (list2) {
			SqlDataDisplayFns *fns;

			field = SQL_MEM_FIELD (list2->data);
			subsubtree =
				xmlNewChild (subtree, NULL, "field", NULL);
			str = g_strdup_printf ("TV%s:FI%s", table->name,
					       field->name);
			xmlSetProp (subsubtree, "name", str);
			g_free (str);
			datatype = field->type;
			str = g_strdup_printf ("DT%s", datatype->sqlname);
			xmlSetProp (subsubtree, "type", str);
			g_free (str);
			/* if ((datatype->numparams >= 1) && (field->length > 0)) { */
/* 	str = g_strdup_printf("%d", field->length); */
/* 	xmlSetProp(subsubtree, "length", str); */
/* 	g_free(str); */
/*       } */
			str = g_strdup_printf ("%d", field->length);
			xmlSetProp (subsubtree, "length", str);
			g_free (str);

			if (field->null_allowed)
				xmlSetProp (subsubtree, "notnull", "f");
			else
				xmlSetProp (subsubtree, "notnull", "t");
			if (field->is_key)
				xmlSetProp (subsubtree, "iskey", "t");
			else
				xmlSetProp (subsubtree, "iskey", "f");
			if (field->default_val)
				xmlSetProp (subsubtree, "default",
					    field->default_val);

			/* if there is a plugin for that field, write it */
			if ((fns =
			     g_hash_table_lookup (db->srv->types_objects_hash,
						  field))) {
				xmlSetProp (subsubtree, "plugin",
					    fns->plugin_name);
			}
			list2 = g_slist_next (list2);
		}
		list = g_slist_next (list);
	}

	/* Sequences */
	tree = xmlNewChild (toptree, NULL, "SEQUENCES", NULL);

	/* declaration of the LINKS node to be used afterwards */
	links = xmlNewChild (toptree, NULL, "LINKS", NULL);

	order = 0;
	list = db->sequences;
	while (list) {
		seq = SQL_MEM_SEQ (list->data);
		subtree = xmlNewChild (tree, NULL, "sequence", NULL);
		str = g_strdup_printf ("SE%s", seq->name);
		xmlSetProp (subtree, "name", str);
		g_free (str);
		if (seq->comments)
			xmlSetProp (subtree, "comments", seq->comments);
		if (seq->owner)
			xmlSetProp (subtree, "owner", seq->owner);

		/* sequence links */
		list2 = seq->field_links;
		while (list2) {
			field = SQL_MEM_FIELD (list2->data);
			table = sql_db_find_table_from_field (db, field);
			if (table) {
				subsubtree =
					xmlNewChild (links, NULL, "link",
						     NULL);
				str = g_strdup_printf ("LIS%d", order);	/* link dummy ID */
				xmlSetProp (subsubtree, "id", str);
				g_free (str);
				str = g_strdup_printf ("SE%s", seq->name);
				xmlSetProp (subsubtree, "from", str);
				g_free (str);
				str = g_strdup_printf ("TV%s:FI%s",
						       table->name,
						       field->name);
				xmlSetProp (subsubtree, "to", str);
				g_free (str);
			}
			list2 = g_slist_next (list2);
			order++;	/* for the next LINK ID */
		}
		list = g_slist_next (list);
	}

	/* Field links */
	list = db->field_links;
	order = 0;
	while (list) {
		flink = SQL_MEM_FLINK (list->data);

		table = sql_db_find_table_from_field (db, flink->from);
		table2 = sql_db_find_table_from_field (db, flink->to);
		if (table && table2) {
			subtree = xmlNewChild (links, NULL, "link", NULL);
			str = g_strdup_printf ("LI%d", order);	/* link ID */
			xmlSetProp (subtree, "id", str);
			g_free (str);
			if (flink->name)
				xmlSetProp (subtree, "name", flink->name);
			str = g_strdup_printf ("TV%s:FI%s", table->name,
					       flink->from->name);
			xmlSetProp (subtree, "from", str);
			g_free (str);
			str = g_strdup_printf ("TV%s:FI%s", table2->name,
					       flink->to->name);
			xmlSetProp (subtree, "to", str);
			g_free (str);
		}
		list = g_slist_next (list);
		order++;
	}
}

/* to load sqldb object 
   the SqlDb object MUST NOT have any table, sequence or field_links. 
   The XML document MUST be a valid one regarding the DTD.
   Otherwise the result is not known!
*/
gboolean
sql_db_build_db_from_xml_tree (SqlDb * db, xmlNodePtr node)
{
	xmlNodePtr tree, subtree;
	GSList *list, *sublist;
	gchar *str;

	g_return_val_if_fail ((!db->is_fault), FALSE);

	if (db->tables || db->sequences || db->field_links) {
		gnome_error_dialog (_
				    ("INTERNAL ERROR:\nTrying to load an SqlDb object "
				     "which already have some data inside. Clean it "
				     "before."));
		return FALSE;
	}
	/* sets the DB name */
	if (db->name)
		g_free (db->name);
	db->name = xmlGetProp (node, "name");

	tree = node->xmlChildrenNode;
	while (tree) {
		/* Tables loading */
		if (!strcmp (tree->name, "TABLES")) {
			subtree = tree->xmlChildrenNode;
			while (subtree) {
				sql_mem_table_load_from_xml_node (db,
								  subtree);
				subtree = subtree->next;
			}
		}

		/* Sequences Loading */
		if (tree && !strcmp (tree->name, "SEQUENCES")) {
			subtree = tree->xmlChildrenNode;
			while (subtree) {
				sql_mem_seq_load_from_xml_node (db, subtree);
				subtree = subtree->next;
			}
		}

		/* Links Loading */
		if (tree && !strcmp (tree->name, "LINKS")) {
			subtree = tree->xmlChildrenNode;
			while (subtree) {
				sql_mem_link_load_from_xml_node (db, subtree);
				subtree = subtree->next;
			}
		}

		if (tree)
			tree = tree->next;
	}


	/* sending ALL signals */
	list = db->tables;
	while (list) {
#ifdef debug_signal
		g_print (">> 'TABLE_CREATED_FILLED' from sql_db_build_db_from_xml_tree\n");
#endif
		gtk_signal_emit (GTK_OBJECT (db),
				 sql_db_signals[TABLE_CREATED_FILLED],
				 list->data);
#ifdef debug_signal
		g_print ("<< 'TABLE_CREATED_FILLED' from sql_db_build_db_from_xml_tree\n");
#endif
		list = g_slist_next (list);
	}

	list = db->sequences;
	while (list) {
#ifdef debug_signal
		g_print (">> 'SEQ_CREATED' from sql_db_build_db_from_xml_tree\n");
#endif
		gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[SEQ_CREATED],
				 list->data);
#ifdef debug_signal
		g_print ("<< 'SEQ_CREATED' from sql_db_build_db_from_xml_tree\n");
#endif
		sublist = SQL_MEM_SEQ (list->data)->field_links;
		while (sublist) {
#ifdef debug_signal
			g_print (">> 'FS_LINK_CREATED' from sql_db_build_db_from_xml_tree\n");
#endif
			gtk_signal_emit (GTK_OBJECT (db),
					 sql_db_signals[FS_LINK_CREATED],
					 list->data, sublist->data);
#ifdef debug_signal
			g_print ("<< 'FS_LINK_CREATED' from sql_db_build_db_from_xml_tree\n");
#endif
			sublist = g_slist_next (sublist);
		}
		list = g_slist_next (list);
	}

	list = db->field_links;
	while (list) {
#ifdef debug_signal
		g_print (">> 'FF_LINK_CREATED' from sql_db_build_db_from_xml_tree\n");
#endif
		gtk_signal_emit (GTK_OBJECT (db),
				 sql_db_signals[FF_LINK_CREATED], list->data);
#ifdef debug_signal
		g_print ("<< 'FF_LINK_CREATED' from sql_db_build_db_from_xml_tree\n");
#endif
		list = g_slist_next (list);
	}

	return TRUE;
}

/*
 *
 * Object common to memory representation of the DB: SqlMemItems
 * items such as tables, fields, sequences, links inherit this
 * object
 *
 */

static void sql_mem_item_class_init (SqlMemItemClass * class);
static void sql_mem_item_init (SqlMemItem * mi);
static void sql_mem_item_destroy (GtkObject * object);

/* signals */
enum
{
	LOAD_FAULT,
	LAST_SIGNAL
};

static gint sql_mem_item_signals[LAST_SIGNAL] = { 0 };


GtkType
sql_mem_item_get_type (void)
{
	static GtkType f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Mem_Item",
			sizeof (SqlMemItem),
			sizeof (SqlMemItemClass),
			(GtkClassInitFunc) sql_mem_item_class_init,
			(GtkObjectInitFunc) sql_mem_item_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_mem_item_class_init (SqlMemItemClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	sql_mem_item_signals[LOAD_FAULT] =
		gtk_signal_new ("load_fault",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlMemItemClass,
						   load_fault),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);
	gtk_object_class_add_signals (object_class, sql_mem_item_signals,
				      LAST_SIGNAL);
	class->load_fault = NULL;

	object_class->destroy = sql_mem_item_destroy;
}

static void
sql_mem_item_init (SqlMemItem * mi)
{
	mi->updated = FALSE;
}


GtkObject *
sql_mem_item_new (void)
{
	GtkObject *obj;

	obj = gtk_type_new (sql_mem_item_get_type ());
	return obj;
}

static void
sql_mem_item_destroy (GtkObject * object)
{
	GtkObject *parent_class = NULL;

	parent_class = gtk_type_class (gtk_object_get_type ());
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_MEM_ITEM (object));

	/* does not destroy anything here, just a link to the parent object */

	/* for the parent class */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


/*
 *
 * Object representing a table
 *
 */

static void sql_mem_table_class_init (SqlMemTableClass * class);
static void sql_mem_table_init (SqlMemTable * table);
static void sql_mem_table_destroy (GtkObject * object);
static void sql_mem_table_cb (GtkObject * obj, gpointer data);

enum
{
	TAB_FIELD_CREATED,
	TAB_FIELD_DROPPED,
	TAB_COMMENTS_CHANGED,
	LASTTA_SIGNAL
};

static gint sql_mem_table_signals[LASTTA_SIGNAL] = { 0, 0, 0 };


GtkType
sql_mem_table_get_type (void)
{
	static GtkType f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Mem_Table",
			sizeof (SqlMemTable),
			sizeof (SqlMemTableClass),
			(GtkClassInitFunc) sql_mem_table_class_init,
			(GtkObjectInitFunc) sql_mem_table_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (sql_mem_item_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_mem_table_class_init (SqlMemTableClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	sql_mem_table_signals[TAB_FIELD_CREATED] =
		gtk_signal_new ("field_created",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlMemTableClass,
						   field_created),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);

	sql_mem_table_signals[TAB_FIELD_DROPPED] =
		gtk_signal_new ("field_dropped",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlMemTableClass,
						   field_dropped),
				gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1,
				GTK_TYPE_POINTER);
	sql_mem_table_signals[TAB_COMMENTS_CHANGED] =
		gtk_signal_new ("comments_changed", GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlMemTableClass,
						   comments_changed),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	gtk_object_class_add_signals (object_class, sql_mem_table_signals,
				      LASTTA_SIGNAL);

	object_class->destroy = sql_mem_table_destroy;
	class->field_created = NULL;
	class->field_dropped = NULL;
	class->comments_changed = NULL;
}

static void
sql_mem_table_init (SqlMemTable * table)
{
	table->name = NULL;
	table->parents = NULL;
	table->fields = NULL;
	table->comments = NULL;
	table->is_user_comments = FALSE;
	table->is_view = FALSE;
}


GtkObject *
sql_mem_table_new (void)
{
	GtkObject *obj;

	obj = gtk_type_new (sql_mem_table_get_type ());
	return obj;
}

static void
sql_mem_table_destroy (GtkObject * object)
{
	SqlMemTable *table;
	SqlMemItem *parent_class = NULL;
	GSList *list;

	parent_class = gtk_type_class (sql_mem_item_get_type ());
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_MEM_TABLE (object));

	table = SQL_MEM_TABLE (object);
	/* we do free the parent GSList 
	   (only the lists, not the objects refered to) */
	g_slist_free (table->parents);
	table->parents = NULL;
	/* we do free the fields GSList and destroy every field */
	list = table->fields;
	while (list) {
		gtk_object_destroy (GTK_OBJECT (list->data));
		list = g_slist_next (list);
	}
	g_slist_free (table->fields);
	table->fields = NULL;
	/* we free the name */
	if (table->name)
		g_free (table->name);
	table->name = NULL;

	/* free comments */
	if (table->comments)
		g_free (table->comments);
	table->comments = NULL;

	/* for the parent class */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

/* this callback is called whenever a field emits the signal "load_fault" */
static void
sql_mem_table_cb (GtkObject * obj, gpointer data)
{
#ifdef debug_signal
	g_print (">> 'LOAD_FAULT' from sqldb->sql_mem_table_cb\n");
#endif
	gtk_signal_emit (GTK_OBJECT (data), sql_mem_item_signals[LOAD_FAULT]);
#ifdef debug_signal
	g_print ("<< 'LOAD_FAULT' from sqldb->sql_mem_table_cb\n");
#endif
}

void
sql_mem_table_set_comments (SqlMemTable * table,
			    gASQL_Main_Config * conf,
			    gchar * comments, gboolean is_user_comments)
{
	if (is_user_comments ||
	    (!table->is_user_comments && !is_user_comments)) {
		if (table->comments)
			g_free (table->comments);
		table->comments = g_strdup (comments);
		table->is_user_comments = is_user_comments;
		if (conf)
			conf->save_up_to_date = FALSE;
		gtk_signal_emit (GTK_OBJECT (table),
				 sql_mem_table_signals[TAB_COMMENTS_CHANGED]);
	}

	if (*(table->comments) == '\0') {
		g_free (table->comments);
		table->comments = NULL;
		table->is_user_comments = FALSE;
	}
}

/****************************************/
/*                                      */
/* Fields manipulation                  */
/*                                      */
/****************************************/

SqlMemField *
sql_mem_table_find_field_by_name (SqlMemTable * t, gchar * name)
{
	GSList *list = t->fields;

	while (list && (strcmp (name, SQL_MEM_FIELD (list->data)->name))) {
		list = g_slist_next (list);
	}
	if (list)
		return SQL_MEM_FIELD (list->data);
	else
		return NULL;
}

static void
sql_mem_table_load_fields (SqlMemTable * t, SqlAccess * srv, SqlDb * db)
{
	gint i;
	SqlMemField *field;
	GSList *list, *list_hold, *hold;
	GdaRecordset *rs;
	GdaField *gfield;
	gchar *str;

	g_return_if_fail ((!db->is_fault));

	rs = gda_connection_open_schema (GDA_CONNECTION (srv),
					 GDA_Connection_GDCN_SCHEMA_COLS,
					 GDA_Connection_OBJECT_NAME, t->name,
					 GDA_Connection_no_CONSTRAINT);
	if (rs) {
		gda_recordset_move (rs, 1, 0);
		i = 0;
		while (!gda_recordset_eof (rs)) {
			gfield = gda_recordset_field_idx (rs, 0);
			str = gda_stringify_value (NULL, 0, gfield);
			field = sql_mem_table_find_field_by_name (t, str);
			g_free (str);
			if (field)
				sql_mem_field_load_contents (srv, field, rs);
			else {
				field = SQL_MEM_FIELD (sql_mem_field_new ());
				gtk_signal_connect (GTK_OBJECT (field),
						    "load_fault",
						    GTK_SIGNAL_FUNC
						    (sql_mem_table_cb), t);
				t->fields =
					g_slist_insert (t->fields, field, i);
				/* updating the field's contents */
				sql_mem_field_load_contents (srv, field, rs);
#ifdef debug_signal
				g_print (">> 'TAB_FIELD_CREATED' from sql_mem_table_load_fields\n");
#endif
				gtk_signal_emit (GTK_OBJECT (t),
						 sql_mem_table_signals
						 [TAB_FIELD_CREATED], field);
#ifdef debug_signal
				g_print ("<< 'TAB_FIELD_CREATED' from sql_mem_table_load_fields\n");
#endif
			}

			/* mark the field as updated */
			SQL_MEM_ITEM (field)->updated = TRUE;

			gda_recordset_move_next (rs);
			i++;
		}
		gda_recordset_free (rs);

		/* 2nd step: destroy fields that do not exist anymore */
		list = t->fields;
		while (list) {
			if (!(SQL_MEM_ITEM (list->data)->updated)) {
				GtkObject *obj;
				/* we want to emit the "field_dropped" signal with the field
				   structure still existing, but not anymore in the table's list
				   of fields. It is destroyed after the signal has been handled 
				   correctly */
				obj = GTK_OBJECT (list->data);

				/* we want to remove any link to that field here, will be done
				   by the SqlDb object at reception of the "field_dropped" signal 
				   from SqlMemTable */

				/* removing any link to or from that field, 
				   and any sequence link to that
				   field */
				sql_db_delete_seq_field_link (db, NULL,
							      SQL_MEM_FIELD
							      (obj));
				sql_db_delete_field_field_link (db, NULL,
								SQL_MEM_FIELD
								(obj));
				sql_db_delete_field_field_link (db,
								SQL_MEM_FIELD
								(obj), NULL);

				list_hold = g_slist_next (list);
				hold = list;
				t->fields = g_slist_remove_link (t->fields, list);
				list = list_hold;
				g_slist_free_1 (hold);
#ifdef debug_signal
				g_print (">> 'TAB_FIELD_DROPPED' from sql_mem_table_load_fields\n");
#endif
				gtk_signal_emit (GTK_OBJECT (t),
						 sql_mem_table_signals
						 [TAB_FIELD_DROPPED], obj);
#ifdef debug_signal
				g_print ("<< 'TAB_FIELD_DROPPED' from sql_mem_table_load_fields\n");
#endif
				gtk_object_destroy (obj);
			}
			else {
				SQL_MEM_ITEM (list->data)->updated = FALSE;
				list = g_slist_next (list);
			}
		}
	}

}


/****************************************/
/*                                      */
/* Dumping the fields and structure     */
/*                                      */
/****************************************/

#ifdef debug
void
sql_mem_table_dump_fields (SqlMemTable * t)
{
	GSList *list;

	list = t->fields;
	g_print ("\tField name                  Type Length  Other\n");
	while (list) {
		g_print ("\t+ ");
		sql_mem_field_dump (SQL_MEM_FIELD (list->data));
		list = g_slist_next (list);
	}
}
#endif

void
sql_mem_table_dump_as_graph (SqlMemTable * t, FILE * st)
{
	GSList *list = t->fields;
	fprintf (st, "node: { title:\"table:%s\"\n", t->name);
	fprintf (st, "        label: \"");

	/* this table's fields */
	fprintf (st, "\\fb\\fu%s\\fn", t->name);
	while (list) {
		fprintf (st, "\\n%s", SQL_MEM_FIELD (list->data)->name);
		list = g_slist_next (list);
	}
	fprintf (st, "\"\n      }\n");

	/* links from the parent tables */
	list = t->parents;
	while (list) {
		fprintf (st,
			 "edge: { sourcename: \"table:%s\" targetname: \"table:%s\""
			 " linestyle: dashed }\n",
			 SQL_MEM_TABLE (list->data)->name, t->name);
		list = g_slist_next (list);
	}
}


/* creates the SqlMemTable. 
   the table MUST NOT exist before. */
static void
sql_mem_table_load_from_xml_node (SqlDb * db, xmlNodePtr tabletree)
{
	gchar *str;
	xmlNodePtr node;
	SqlMemField *field;
	SqlDataType *type;
	SqlMemTable *table, *t2;
	gboolean used;

	g_return_if_fail ((!db->is_fault));

	table = SQL_MEM_TABLE (sql_mem_table_new ());
	db->tables = g_slist_append (db->tables, table);
	str = xmlGetProp (tabletree, "name");
	table->name = g_strdup (str + 2);
	g_free (str);

	used = FALSE;
	str = xmlGetProp (tabletree, "user_comments");
	if (str) {
		used = (*str == 't') ? TRUE : FALSE;
		g_free (str);
	}

	str = xmlGetProp (tabletree, "comments");
	if (str) {
		sql_mem_table_set_comments (table, NULL, str, used);
		g_free (str);
	}

	str = xmlGetProp (tabletree, "owner");
	table->owner = (gchar *) ((find_user_name (db, str))->data);
	g_free (str);

	str = xmlGetProp (tabletree, "is_view");
	if (str) {
		table->is_view = (*str == 't') ? TRUE : FALSE;
		g_free (str);
	}

	gtk_signal_connect (GTK_OBJECT (table), "load_fault",
			    GTK_SIGNAL_FUNC (sql_db_cb), db);
	gtk_signal_connect (GTK_OBJECT (table), "field_created",
			    GTK_SIGNAL_FUNC (sql_db_catch_field_create_cb),
			    db);
	gtk_signal_connect (GTK_OBJECT (table), "field_dropped",
			    GTK_SIGNAL_FUNC (sql_db_catch_field_drop_cb), db);
#ifdef debug_signal
	g_print (">> 'TABLE_CREATED' from sql_mem_table_load_from_xml_node\n");
#endif
	gtk_signal_emit (GTK_OBJECT (db), sql_db_signals[TABLE_CREATED],
			 table);
#ifdef debug_signal
	g_print ("<< 'TABLE_CREATED' from sql_mem_table_load_from_xml_node\n");
#endif

	node = tabletree->xmlChildrenNode;
	while (node) {
		if (!strcmp (node->name, "parent")) {
			xmlNodePtr pt;
			str = xmlGetProp (node, "table");
			pt = xmlGetID (node->doc, str)->node;
			g_free (str);
			str = xmlGetProp (pt, "name");
			t2 = sql_db_find_table_by_name (db, str + 2);
			g_free (str);
			if (!g_slist_find (table->parents, t2))
				table->parents =
					g_slist_append (table->parents, t2);
		}

		if (!strcmp (node->name, "field")) {
			gchar *ptr;

			str = xmlGetProp (node, "name");
			field = SQL_MEM_FIELD (sql_mem_field_new ());
			gtk_signal_connect (GTK_OBJECT (field), "load_fault",
					    GTK_SIGNAL_FUNC
					    (sql_mem_table_cb), table);
			table->fields = g_slist_append (table->fields, field);
			ptr = strstr (str, ":FI");
			field->name = g_strdup (ptr + 3);
			g_free (str);
#ifdef debug_signal
			g_print (">> 'TAB_FIELD_CREATED' from "
				 "sql_mem_table_load_from_xml_node\n");
#endif
			gtk_signal_emit (GTK_OBJECT (table),
					 sql_mem_table_signals
					 [TAB_FIELD_CREATED], field);
#ifdef debug_signal
			g_print ("<< 'TAB_FIELD_CREATED' from "
				 "sql_mem_table_load_from_xml_node\n");
#endif

			str = xmlGetProp (node, "type");
			/* type should not be NULL because of the DTD */
			type = sql_data_type_get_from_xml_id (db->srv, str);
			g_free (str);
			field->type = type;

			str = xmlGetProp (node, "length");
			field->length = atoi (str);
			g_free (str);

			str = xmlGetProp (node, "notnull");
			switch (*str) {
			case 't':
				field->null_allowed = FALSE;
				break;
			default:
				field->null_allowed = TRUE;
				break;
			}
			g_free (str);

			str = xmlGetProp (node, "iskey");
			switch (*str) {
			case 't':
				field->is_key = TRUE;
				break;
			default:
				field->is_key = FALSE;
				break;
			}
			g_free (str);

			str = xmlGetProp (node, "default");
			if (str)
				field->default_val = str;

			/* plugin for this data type */
			str = xmlGetProp (node, "plugin");
			if (str) {
				GSList *plist;
				gboolean found = FALSE;
				SqlDataDisplayFns *fns;
				plist = db->srv->data_types_display;
				while (plist && !found) {
					fns = (SqlDataDisplayFns *) (plist->
								     data);
					if ((fns->plugin_name)
					    && !strcmp (fns->plugin_name,
							str)) {
						sql_access_bind_object_display
							(db->srv,
							 GTK_OBJECT (field),
							 fns);
						found = TRUE;
					}
					plist = g_slist_next (plist);
				}
				g_free (str);
			}
		}

		node = node->next;
	}
}


/*
 *
 * Object representing a field
 *
 */

static void sql_mem_field_class_init (SqlMemFieldClass * class);
static void sql_mem_field_init (SqlMemField * field);
static void sql_mem_field_destroy (GtkObject * object);

/* No signals */
GtkType
sql_mem_field_get_type (void)
{
	static GtkType f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Mem_Field",
			sizeof (SqlMemField),
			sizeof (SqlMemFieldClass),
			(GtkClassInitFunc) sql_mem_field_class_init,
			(GtkObjectInitFunc) sql_mem_field_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (sql_mem_item_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_mem_field_class_init (SqlMemFieldClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	object_class->destroy = sql_mem_field_destroy;
}

static void
sql_mem_field_init (SqlMemField * field)
{
	field->name = NULL;
	field->type = NULL;
	field->type = NULL;
	field->length = 0;
	field->null_allowed = TRUE;
	field->is_key = FALSE;
	field->default_val = NULL;
}


GtkObject *
sql_mem_field_new (void)
{
	GtkObject *obj;

	obj = gtk_type_new (sql_mem_field_get_type ());
	return obj;
}

static void
sql_mem_field_destroy (GtkObject * object)
{
	SqlMemField *field;
	SqlMemItem *parent_class = NULL;

	parent_class = gtk_type_class (sql_mem_item_get_type ());
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_MEM_FIELD (object));

	field = SQL_MEM_FIELD (object);

	/* we free the name */
	if (field->name)
		g_free (field->name);
	field->name = NULL;
	field->type = NULL;
	if (field->default_val) {
		g_free (field->default_val);
		field->default_val = NULL;
	}

	/* for the parent class */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


/****************************************/
/*                                      */
/* Managing the contents of the field   */
/*                                      */
/****************************************/

static void
sql_mem_field_load_contents (SqlAccess * srv, SqlMemField * field,
			     GdaRecordset * recset)
{
	gchar *str;
	GdaField *gfield;

	if (field->name)
		g_free (field->name);
	gfield = gda_recordset_field_idx (recset, 0);
	field->name = gda_stringify_value (NULL, 0, gfield);;

	gfield = gda_recordset_field_idx (recset, 1);
	str = gda_stringify_value (NULL, 0, gfield);
	field->type = sql_access_get_type_from_name (srv, str);
	g_free (str);
	if (!field->type)
		field->type = sql_access_get_type_from_name (srv, "varchar");

	gfield = gda_recordset_field_idx (recset, 2);
	str = gda_stringify_value (NULL, 0, gfield);
	field->length = atoi (str);
	g_free (str);

	gfield = gda_recordset_field_idx (recset, 4);
	if (gda_field_get_boolean_value (gfield))
		field->null_allowed = TRUE;
	else
		field->null_allowed = FALSE;

	/* FIXME: field->is_key needs to be set */
	gfield = gda_recordset_field_idx (recset, 5);
	if (gda_field_get_boolean_value (gfield))
		field->is_key = TRUE;
	else
		field->is_key = FALSE;

	/* Default value */
	gfield = gda_recordset_field_idx (recset, 6);
	if (field->default_val) {
		g_free (field->default_val);
		field->default_val = NULL;
	}
	if (gfield && gfield->real_value->_u.v._u.lvc) {
		str = gda_stringify_value (NULL, 0, gfield);
		field->default_val = str;
	}
}


/****************************************/
/*                                      */
/* Dumping the field                    */
/*                                      */
/****************************************/
#ifdef debug
void
sql_mem_field_dump (SqlMemField * field)
{
	/* prints all on one line */
/*  gchar*       sql_access_get_data_type (SqlAccess *srv, gchar *oid);*/

	g_print ("%-20s %-10s(%d)\t%3d   ", field->name,
		 field->type->sqlname, field->type->server_type,
		 field->length);
	if (field->null_allowed)
		g_print ("NULL allowed ");
	else
		g_print ("NOT NULL     ");
	if (field->is_key)
		g_print ("Key");
	if (field->default_val)
		g_print ("Defaults to %s", field->default_val);
	g_print ("\n");
}
#endif



/*
 *
 * Object representing a sequence
 *
 */

static void sql_mem_seq_class_init (SqlMemSeqClass * class);
static void sql_mem_seq_init (SqlMemSeq * seq);
static void sql_mem_seq_destroy (GtkObject * object);


/* No signals */
GtkType
sql_mem_seq_get_type (void)
{
	static GtkType f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Mem_Seq",
			sizeof (SqlMemSeq),
			sizeof (SqlMemSeqClass),
			(GtkClassInitFunc) sql_mem_seq_class_init,
			(GtkObjectInitFunc) sql_mem_seq_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (sql_mem_item_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_mem_seq_class_init (SqlMemSeqClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	object_class->destroy = sql_mem_seq_destroy;
}

static void
sql_mem_seq_init (SqlMemSeq * seq)
{
	seq->name = NULL;
	seq->field_links = NULL;
	seq->comments = NULL;
	seq->owner = NULL;
}


GtkObject *
sql_mem_seq_new (void)
{
	GtkObject *obj;

	obj = gtk_type_new (sql_mem_seq_get_type ());
	return obj;
}

static void
sql_mem_seq_destroy (GtkObject * object)
{
	SqlMemSeq *seq;
	SqlMemItem *parent_class = NULL;

	parent_class = gtk_type_class (sql_mem_item_get_type ());
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_MEM_SEQ (object));

	seq = SQL_MEM_SEQ (object);

	/* field links destroying */
	if (seq->field_links) {
		g_slist_free (seq->field_links);
		seq->field_links = NULL;
	}

	/* name destroying */
	if (seq->name)
		g_free (seq->name);
	seq->name = NULL;

	/* comments destroying */
	if (seq->comments)
		g_free (seq->comments);
	seq->comments = NULL;

	/* for the parent class */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


/****************************************/
/*                                      */
/* Dumping the sequence                 */
/*                                      */
/****************************************/

void
sql_mem_seq_dump_as_graph (SqlMemSeq * seq, SqlDb * db, FILE * st)
{
	GSList *list;

	g_return_if_fail ((!db->is_fault));

	/* sequence node */
	fprintf (st, "node: { title:\"seq:%s\"\n", seq->name);
	fprintf (st, "        label: \"%s\"\n", seq->name);
	fprintf (st, "        shape: ellipse\n");
	fprintf (st, "        color: pink\n");
	fprintf (st, "      }\n");

	/* links to fields */
	list = seq->field_links;
	while (list) {
		fprintf (st,
			 "edge: { sourcename: \"table:%s\" targetname: \"seq:%s\""
			 " anchor: %d color: pink arrowstyle: none backarrowstyle: solid "
			 "backarrowsize: 10}\n",
			 (sql_db_find_table_from_field
			  (db, SQL_MEM_FIELD (list->data)))->name, seq->name,
			 sql_db_get_field_order (db, NULL,
						 SQL_MEM_FIELD (list->data)) +
			 1);

		list = g_slist_next (list);
	}
}


/* creates the SqlMemSeq.
   the SqlMemSeq object MUST not exist before. */
static void
sql_mem_seq_load_from_xml_node (SqlDb * db, xmlNodePtr tabletree)
{
	gchar *str, *str2;
	xmlNodePtr node;
	SqlMemSeq *seq;
	SqlMemField *field;
	gboolean used;

	g_return_if_fail ((!db->is_fault));

	str = xmlGetProp (tabletree, "name");
	seq = SQL_MEM_SEQ (sql_mem_seq_new ());
	seq->name = g_strdup (str + 2);
	g_free (str);

	gtk_signal_connect (GTK_OBJECT (seq), "load_fault",
			    GTK_SIGNAL_FUNC (sql_db_cb), db);
	db->sequences = g_slist_append (db->sequences, seq);

	str = xmlGetProp (tabletree, "comments");
	if (str)
		seq->comments = str;

	str = xmlGetProp (tabletree, "owner");
	seq->owner = (gchar *) ((find_user_name (db, str))->data);
	g_free (str);
}


/*
 *
 * Object representing a field link
 *
 */

static void sql_mem_flink_class_init (SqlMemFlinkClass * class);
static void sql_mem_flink_init (SqlMemFlink * flink);
static void sql_mem_flink_destroy (GtkObject * object);


/* No signals */
GtkType
sql_mem_flink_get_type (void)
{
	static GtkType f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Mem_Flink",
			sizeof (SqlMemFlink),
			sizeof (SqlMemFlinkClass),
			(GtkClassInitFunc) sql_mem_flink_class_init,
			(GtkObjectInitFunc) sql_mem_flink_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (sql_mem_item_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_mem_flink_class_init (SqlMemFlinkClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	object_class->destroy = sql_mem_flink_destroy;
}

static void
sql_mem_flink_init (SqlMemFlink * flink)
{
	flink->name = NULL;
	flink->from = NULL;
	flink->to = NULL;
}


GtkObject *
sql_mem_flink_new (void)
{
	GtkObject *obj;

	obj = gtk_type_new (sql_mem_flink_get_type ());
	return obj;
}

static void
sql_mem_flink_destroy (GtkObject * object)
{
	SqlMemFlink *flink;
	SqlMemItem *parent_class = NULL;

	parent_class = gtk_type_class (sql_mem_item_get_type ());
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_MEM_FLINK (object));

	flink = SQL_MEM_FLINK (object);

	/* name destroying */
	if (flink->name)
		g_free (flink->name);
	flink->name = NULL;

	/* for the parent class */
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


/****************************************/
/*                                      */
/* Dumping the link                     */
/*                                      */
/****************************************/

void
sql_mem_flink_dump_as_graph (SqlMemFlink * flink, SqlDb * db, FILE * st)
{
	static guint num = 0;	/* numbering of the created nodes */

	g_return_if_fail ((!db->is_fault));

	/* creating a new node */
	if (flink->name)
		fprintf (st, "node: { title:\"fl%d\"\n"
			 "	label:\"%s\"\n"
			 "	borderwidth: 0\n"
			 "	color: aquamarine\n"
			 "      }\n", num, flink->name);
	else
		fprintf (st, "node: { title:\"fl%d\"\n"
			 "	label:\"N/A\"\n"
			 "	shape: ellipse\n"
			 "	color: aquamarine\n" "      }\n", num);

	/* creating the two edges */
	fprintf (st, "edge: { sourcename: \"table:%s\" targetname: \"fl%d\""
		 " anchor: %d arrowstyle: none backarrowstyle: solid "
		 "backarrowsize: 10 priority: 50}\n",
		 (sql_db_find_table_from_field (db, flink->to))->name, num,
		 sql_db_get_field_order (db, NULL, flink->to) + 1);

	fprintf (st, "edge: { sourcename: \"table:%s\" targetname: \"fl%d\""
		 " anchor: %d arrowstyle: none priority: 50}\n",
		 (sql_db_find_table_from_field (db, flink->from))->name, num,
		 sql_db_get_field_order (db, NULL, flink->from) + 1);

	/* prepares for the next one */
	num++;
}

/* the name of the node is a "link" by DTD definition */
void
sql_mem_link_load_from_xml_node (SqlDb * db, xmlNodePtr tree)
{
	gchar *str;
	gchar *from, *to;
	SqlMemField *tfield;

	g_return_if_fail ((!db->is_fault));

	str = xmlGetProp (tree, "id");
	from = xmlGetProp (tree, "from");
	to = xmlGetProp (tree, "to");
	tfield = sql_db_find_field_from_xml_name (db, to);
	g_free (to);

	if (*(str + 2) == 'S') {	/* seq. to field link */
		SqlMemSeq *seq;
		seq = sql_db_find_sequence_by_xml_name (db, from);
		seq->field_links = g_slist_append (seq->field_links, tfield);
	}
	else {			/* field to field link */

		SqlMemField *ffield;
		SqlMemFlink *flink;

		ffield = sql_db_find_field_from_xml_name (db, from);
		flink = SQL_MEM_FLINK (sql_mem_flink_new ());
		if (!sql_db_find_flink_from_fields (db, ffield, tfield)) {
			gchar *name;
			flink->from = ffield;
			flink->to = tfield;
			db->field_links =
				g_slist_append (db->field_links, flink);
			name = xmlGetProp (tree, "name");
			if (name)
				flink->name = name;
		}
	}
	g_free (str);
	g_free (from);
}
