/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: SSHPIX.cpp,v 1.5 2004/09/29 06:31:52 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/



#include "config.h"
#include "global.h"
#include "utils.h"

#include "SSHPIX.h"

#include <qobject.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qeventloop.h>

#include <iostream>

using namespace std;

SSHPIX::SSHPIX(const QString &_h,
               const QStringList &args,
               const QString &_p,
               const QString &_ep,
               const QStringList &_in) : SSHSession(_h,args,_p,_ep,_in)
{
    normal_prompt="> ";
    fwb_prompt="--**--**--";
    enable_prompt="# ";
    pwd_prompt="'s password: ";
    epwd_prompt="Password: ";
    scp_pwd_prompt="Password: ";
    ssh_pwd_prompt="'s password: ";
    ssoft_config_prompt="> ";
    sudo_pwd_prompt="Password:";
    putty_pwd_prompt="Password: ";
    passphrase_prompt="Enter passphrase for key ";

    errorsInit.push_back("Permission denied");
    errorsInit.push_back("Invalid password");
    errorsInit.push_back("Access denied");
    errorsInit.push_back("Unable to authenticate");
    errorsInit.push_back("Too many authentication failures");

    errorsLoggedin.push_back("Invalid password");
    errorsLoggedin.push_back("ERROR: ");
    errorsLoggedin.push_back("Not enough arguments");
    errorsLoggedin.push_back("cannot find");

    errorsEnabledState.push_back("ERROR: ");
    errorsEnabledState.push_back("Type help");
    errorsEnabledState.push_back("Not enough arguments");
    errorsEnabledState.push_back("Invalid");
    errorsEnabledState.push_back("invalid");
    errorsEnabledState.push_back("cannot find");
    errorsEnabledState.push_back("An object-group with the same id but different type");

}

SSHPIX::~SSHPIX()
{
}

QString SSHPIX::cmd(QProcess *proc,const QString &cmd)
{
    stdoutBuffer="";

    proc->writeToStdin( cmd  );
    proc->writeToStdin( "\n" );

    state=EXECUTING_COMMAND;
    QApplication::eventLoop()->enterLoop();

    return stdoutBuffer;
}



void SSHPIX::stateMachine()
{
    QStringList *errptr;

    switch (state)
    {
    case LOGGEDIN:  errptr= &errorsLoggedin;     break;
    case ENABLE:    errptr= &errorsEnabledState; break;
    default:        errptr= &errorsInit;         break;
    }

    for (QStringList::const_iterator i=errptr->begin();
         i!=errptr->end(); ++i)
    {
        if ( stdoutBuffer.findRev(*i,-1)!=-1 )
        {
            emit printStdout_sign( tr("*** Fatal error :") );
            emit printStdout_sign( stdoutBuffer+"\n" );
            stdoutBuffer="";
            terminate();
            return;
        }
    }

 entry:
    switch (state)
    {
    case NONE:
    {
        if ( cmpPrompt(stdoutBuffer,pwd_prompt) )
        {
            stdoutBuffer="";
            proc->writeToStdin( pwd );
            proc->writeToStdin( "\n" );
            break;
        }
/* we may get to LOGGEDIN state directly from NONE, for example when
 * password is supplied on command line to plink.exe
 */
        if (cmpPrompt(stdoutBuffer,normal_prompt) )
        {
            state=LOGGEDIN;
            emit printStdout_sign( "\n");
            emit printStdout_sign( tr("\nLogged in") );
            emit printStdout_sign( "\n");
            emit printStdout_sign( tr("Switching to enable mode..."));
            emit printStdout_sign( "\n");
            stdoutBuffer="";
            proc->writeToStdin( "enable\n" );
        }

        QString fingerprint;
        int n1,n2;
        if (stdoutBuffer.find(newKeyOpenSSH)!=-1 ||
            stdoutBuffer.find(newKeyPlink)!=-1)
        {
/* new key */

            n1=stdoutBuffer.find(fingerprintPrompt) + strlen(fingerprintPrompt);
            n2=stdoutBuffer.find(QRegExp("[^ 0-9a-f:]"), n1+4);
            fingerprint=stdoutBuffer.mid(n1,n2-n1);

            QString msg=newKeyMsg.arg(host).arg(fingerprint).arg(host);

            int res =QMessageBox::warning( NULL, tr("New RSA key"), msg,
                                           tr("Yes"), tr("No"), 0,
                                           0, -1 );

            stdoutBuffer="";
            if (res==0)
            {
                proc->writeToStdin( "yes\n" );
                break;
            } else
            {
                state=EXIT;
                goto entry;
            }
        }
    }
    break;

    case LOGGEDIN:
        if ( cmpPrompt(stdoutBuffer,epwd_prompt) )
        {
            stdoutBuffer="";
            proc->writeToStdin( epwd );
            proc->writeToStdin( "\n" );
            state=WAITING_FOR_ENABLE;
        }
        break;

    case WAITING_FOR_ENABLE:
        if ( cmpPrompt(stdoutBuffer,epwd_prompt) )
        {
            stdoutBuffer="";
            proc->writeToStdin( epwd );
            proc->writeToStdin( "\n" );
            state=WAITING_FOR_ENABLE;
            break;
        }
        if ( cmpPrompt(stdoutBuffer,enable_prompt) )
        {       
            emit printStdout_sign( "\n");
            emit printStdout_sign( tr("In enable mode."));
            emit printStdout_sign( "\n");
            state=ENABLE;  // and go to ENABLE target in switch
        }

    case ENABLE:
        if ( cmpPrompt(stdoutBuffer,enable_prompt) )
        {
            stdoutBuffer="";

            if (backup)
            {
/* the problem is that QProcess uses select and thus is tightly
 * integrated into event loop. QT uses internal private flag inside
 * QProcess to specifically prevent recursive calls to readyReadStdout
 * (look for d->socketReadCalled in kernel/qprocess_unix.cpp ). So, I
 * _must_ exit this callback before I can send commands to the process
 * and collect the output.
 */
                QTimer::singleShot( 0, this, SLOT(PIXbackup()) );
                break;
            }

            proc->writeToStdin( "config t" );
            proc->writeToStdin( "\n" );
            state=WAITING_FOR_CONFIG_PROMPT;
        }
        break;

    case EXECUTING_COMMAND:
        if ( cmpPrompt(stdoutBuffer,enable_prompt) )
        {       
            QApplication::eventLoop()->exitLoop();
            state=COMMAND_DONE;
        }
        break;

    case WAITING_FOR_CONFIG_PROMPT:
        if ( cmpPrompt(stdoutBuffer,enable_prompt) )
        {
            state=CONFIG;

/* install full policy */
            QString ff = wdir+"/"+conffile;
            config_file = new ifstream(ff.latin1());
            if ( ! *config_file)
            {
                emit printStdout_sign(
                    QObject::tr("Can not open file %1").arg(ff)
                );
                state=FINISH;
                break;
            } else
            {
/* read the whole file */
                string s0;
                nLines =0;
                bool   store=!incremental;
                while ( !config_file->eof() )
                {
                    getline( *config_file, s0);
                    if (!store)
                    {
                        store=(s0.find("!################")==0);
                    }
                    if (store)
                    {
                        QString s(s0.c_str());
                        s.stripWhiteSpace();
                        allConfig.push_back(s);
                        nLines++;
                    }
                }
                config_file->close();
                delete config_file;
                config_file=NULL;

                emit updateProgressBar_sign(nLines,true);
            }
            state=PUSHING_CONFIG; // and drop to PUSHING_CONFIG case
            if (!dry_run)
                emit printStdout_sign(tr("Pushing firewall configuration"));
            emit printStdout_sign( "\n");
            ncmd=0;
        }
#if 0
        else
        {
/* install incrementally */

            QTimer::singleShot( 1, this, SLOT(PIXincrementalInstall()) );
            break;
        }
#endif

    case PUSHING_CONFIG:
        if ( cmpPrompt(stdoutBuffer,enable_prompt) )
        {
        loop1:
            if ( allConfig.size()==0 )
            {
                state=EXIT_FROM_CONFIG;
                emit printStdout_sign( tr("*** End ") );
                proc->writeToStdin( "exit\n" );
                break;

            } else
            {
                QString s;

                do {
                    s = allConfig.front();
                    allConfig.pop_front();
                } while (stripComments && s[0]=='!');

                emit updateProgressBar_sign(allConfig.size(),false);

                s.replace('\"','\'');

                if (!verbose)
                {
                    QString rl="";
                    if (s.find("! Rule ")!=-1)  rl=s.mid(7);
                    if ( !rl.isEmpty())
                    {
                        emit printStdout_sign( tr("Rule %1").arg(rl) );
                        emit printStdout_sign( "\n");
                    }
                }

                if (!dry_run)
                {
                    if ( !s.isEmpty())  ncmd++;
                    stdoutBuffer="";
                    proc->writeToStdin( s+"\n"); // send even if s is empty
                    break;
                } else
                {
                    emit printStdout_sign( s+"\n" );
                    goto loop1;
                }
            }
        }
        break;

    case EXIT_FROM_CONFIG:
        if ( cmpPrompt(stdoutBuffer,enable_prompt) )
        {
/*
 * dry_run means run through the commands without executing them on
 * the firewall
 *
 * testRun means executing commands on the firewall without storing
 * them permanently (i.e. do not do "wr mem")
 */
            if (!dry_run && !testRun && ncmd!=0)
            {
                state=SAVE_CONFIG;
                proc->writeToStdin( "wr mem\n");
            } else
            {
                state=EXIT;
                proc->writeToStdin( "exit\n");
            }
        }
        break;

    case SAVE_CONFIG:
        if ( cmpPrompt(stdoutBuffer,enable_prompt) )
        {
            state=EXIT;
            proc->writeToStdin( "exit\n");
        }
        break;

    case EXIT:
        terminate();
        state=FINISH;
        break;

    case FINISH:
        break;

    default:    break;
    }
}

void SSHPIX::PIXbackup()
{
    if (fwbdebug) qDebug("SSHPIX::PIXbackup  ");

    bool sv=verbose;
    verbose=false;

    emit printStdout_sign(tr("Making backup copy of the firewall configuration"));
    emit printStdout_sign( "\n");

    QString cfg=cmd(proc,"show run");

    verbose=sv;

/* if state changed to FINISH, there was an error and ssh terminated */
    if (state==FINISH) return;
    if (state==COMMAND_DONE)
    {
        ofstream ofs(backupFile.latin1());
        ofs << cfg;
        ofs.close();

        backup=false;  // backup is done
        state=ENABLE;
    }

    proc->writeToStdin( "\n" );
}

void SSHPIX::PIXincrementalInstall()
{
    QString current_config;

    bool sv=verbose;
    verbose=false;

    emit printStdout_sign(tr("Reading current firewall configuration"));
    emit printStdout_sign( "\n");

    current_config =cmd(proc,"show run | grep ^telnet|^ssh|^icmp");
    if (state==FINISH) return;
    current_config+=cmd(proc,"show object-group");
    if (state==FINISH) return;
    current_config+=cmd(proc,"show access-list");
    if (state==FINISH) return;
    current_config+=cmd(proc,"show global");
    if (state==FINISH) return;
    current_config+=cmd(proc,"show nat");
    if (state==FINISH) return;
    current_config+=cmd(proc,"show static");
    if (state==FINISH) return;

    verbose=sv;

    if (state==COMMAND_DONE)
    {
        QString statefile = wdir+"/"+conffile + "_current";
        ofstream ofs(statefile.latin1());
        ofs << current_config;
        ofs.close();

        emit printStdout_sign(tr("Generating configuration diff"));
        emit printStdout_sign( "\n");

        QString cm = diff_pgm + " \"" + statefile + "\" \"" + wdir+"/"+conffile + "\"";

//        emit printStdout_sign(tr("Running command: %1\n").arg(cm));

#ifdef _WIN32
        FILE *f = _popen( cm.latin1(), "r");
#else
        FILE *f = popen( cm.latin1(), "r");
#endif
        if (f==NULL)
        {
            emit printStdout_sign(
                tr("Fork failed for %1").arg(diff_pgm));
            emit printStdout_sign( "\n");
            switch (errno)
            {
            case EAGAIN: 
            case ENOMEM:
                emit printStdout_sign(tr("Not enough memory."));
                break;
            case EMFILE:
            case ENFILE:
                emit printStdout_sign(
                    tr("Too many opened file descriptors in the system."));
                break;

            }
            emit printStdout_sign( "\n");
            state=FINISH;
            proc->writeToStdin( "\n" );
            return;
        }

        char buf[1024];
        int  nLines=0;
        while (fgets(buf,1024,f))
        {
            allConfig += buf;
            nLines++;
        }
#ifdef _WIN32
        _pclose(f);
#else
        pclose(f);
#endif

	if (allConfig.isEmpty())
	{
	    allConfig="";
            emit printStdout_sign(tr("Empty configuration diff"));
            emit printStdout_sign( "\n");
	}

        if (save_diff)
        {
            ofstream odiff(wdir+"/"+diff_file.latin1());
            odiff << allConfig.join("");
            odiff.close();
        }

        state=PUSHING_CONFIG;
        emit updateProgressBar_sign(nLines,true);
        if (!dry_run)
            emit printStdout_sign(tr("Pushing firewall configuration\n"));
    }
    proc->writeToStdin( "\n" );
}

