/*  Gpasman, a password manager
    Copyright (C) 1998-1999 Olivier Sessink, olivier@lx.student.wau.nl

    file.c, handles file opening and closing

    Other code contributors:    
    Dave Rudder 
    Chris Halverson
    Matthew Palmer
    Guide Berning
    Jimmy Mason
	 website at http://www.student.wau.nl/~olivier/gpasman/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include "file.h"
#include "rc2.h"

#define SAVE_BUFFER_LENGTH (1024*256)
#define LOAD_BUFFER_LENGTH (2048*256)

#ifndef S_IAMB
#define S_IAMB 00777
#endif

/* these three are needed in all save functions and initialized in save_init() */
FILE *fd;
unsigned short iv[4];
char *buffer;

/* these two are global because save_entry() and save_finalize() both need them */
int bufferIndex;
unsigned short plaintext[4];

int save_init(char *filename, char *password) {

/*
 * returncodes: 
 *  1 = success
 *  0 = can't open filedescriptor / can't create file
 * -1 = permissions are bad
 * -2 = is a symlink
 * -3 = can't get file status
 */

unsigned char key[128];
unsigned int j = 0;
unsigned int keylength;
int val, count2;

   /* first we should check the permissions of the filename */
   
   if (file_exists(filename)) {
      val = check_file(filename);
      if (val != 1) {
#ifdef DEBUG
      fprintf(stderr, "save_init, return %d\n", val);
#endif
        return val;
      }
   } else {
      val = creat (filename, (S_IRUSR | S_IWUSR));
      if (val == -1) {
#ifdef DEBUG
      fprintf(stderr, "save_init, return 0\n");
#endif
         return 0;
      } else {
         close(val);
      }
   }

   fd = fopen (filename, "wb");
   if (fd == NULL)
   {
      return 0;
   }
   buffer = malloc(SAVE_BUFFER_LENGTH);

   /* make the key ready */
#ifdef DEBUG
      fprintf(stderr, "save_init, password=%s\n", password);
#endif
   for (j = 0; password[j] != '\0'; j++)
   {
      key[j] = password[j];
   }
   keylength = j;
   rc2_expandkey (key, keylength, 128);
   
   /* First, we make the IV */
   for (count2 = 0; count2 < 4; count2++)
   {
      iv[count2] = rand ();
      putc ((unsigned char) (iv[count2] >> 8), fd);
      putc ((unsigned char) (iv[count2] & 0xff), fd);
   }
   
   bufferIndex = 0;   
   return 1;
}


int save_entry(char *entry[4]) {

char *text1;
int count2, count3;
unsigned short ciphertext[4];

   buffer = memset(buffer, '\0', SAVE_BUFFER_LENGTH);

   for (count2 = 0; count2 < 4; count2++) {
      text1 = entry[count2];
      if (strlen (text1) == 0)
      {
         strncpy (text1, " ", strlen (" "));
      }
      strncat (buffer, text1, strlen (text1));
      /* Use 255 as the marker.  \n is too tough to test for */
      buffer[strlen (buffer)] = 255;
      
   } /*for (count2 = 0; count2 < 4; count2++)*/
#ifdef DEBUG
   fprintf(stderr, "save_entry, buffer contains %s\n", buffer);
#endif
   count2 = 0;
      /* I'm using CBC mode and encrypting the data straight from top down.
       *  At the bottom, encrypted, I will append an MD5 hash of the file, eventually.
       *  PKCS 5 padding (explained at the code section
       */
   while (count2 < strlen (buffer)) {
#ifndef WORDS_BIGENDIAN
      plaintext[bufferIndex] = buffer[count2 + 1] << 8;
      plaintext[bufferIndex] += buffer[count2] & 0xff;
#endif
#ifdef WORDS_BIGENDIAN
      plaintext[bufferIndex] = buffer[count2] << 8;
      plaintext[bufferIndex] += buffer[count2 + 1] & 0xff;
#endif
      bufferIndex++;
      if (bufferIndex == 4)
      {
         rc2_encrypt (plaintext);

         for (count3 = 0; count3 < 4; count3++)
         {
            ciphertext[count3] = iv[count3] ^ plaintext[count3];

            /* Now store the ciphertext as the iv */
            iv[count3] = plaintext[count3];

            /* reset the buffer index */
            bufferIndex = 0;
            if (putc ((unsigned char) (ciphertext[count3] >> 8), fd) == EOF) return -1;
            if (putc ((unsigned char) (ciphertext[count3] & 0xff), fd) == EOF) return -1;
         } /*for (count3 = 0; count3 < 4; count3++)*/
      } /*if (bufferIndex == 4)*/
      /* increment a short, not a byte */
      count2 += 2;
   } /*while (count2 < strlen (buffer))*/
   return 1;
}

int save_finalize(void) {

int count1, retval=1;
unsigned short ciphertext[4];

  /* Tack on the PKCS 5 padding 
     How it works is we fill up the last n bytes with the value n

     So, if we have, say, 13 bytes, 8 of which are used, we have 5 left 
     over, leaving us 3 short, so we fill it in with 3's.

     If we come out even, we fill it with 8 8s

     um, except that in this instance we are using 4 shorts instead of 8 bytes.
     so, half everything
   */
    for (count1 = bufferIndex; count1 < 4; count1++)
    {
      plaintext[count1] = (4 - bufferIndex);
    }
#ifdef DEBUG
    fprintf(stderr, "save_finalize, 4 - bufferIndex = %d\n", (4 - bufferIndex));
    fprintf(stderr, "save_finalize, plaintext[3]=%c\n", plaintext[3]);
#endif
    rc2_encrypt (plaintext);
    for (count1 = 0; count1 < 4; count1++)
    {
      ciphertext[count1] = iv[count1] ^ plaintext[count1];
      if (putc ((unsigned char) (ciphertext[count1] >> 8), fd) == EOF) retval = -1;
      if (putc ((unsigned char) (ciphertext[count1] & 0xff), fd) == EOF) retval = -1;
    }

    fclose (fd);
#ifdef DEBUG
    fprintf(stderr, "save_finalize, fd is closed\n");
#endif
   free(buffer);
   return retval;

}

/* globals needed for file loading */
int lastcount, size;


int load_init(char* filename, char *password) {
/*
 * returncodes: 
 *  1 = success
 *  0 = can't open filedescriptor / can't create file
 * -1 = permissions are bad
 * -2 = is a symlink
 * -3 = can't get file status
 */
   unsigned int j = 0;
   unsigned int keylength=0;
   int count=0, count2=0, count3=0;
   unsigned char charbuf[8];
   unsigned short ciphertext[4];
   int val=0;
   unsigned char key[128];
   off_t filesize = LOAD_BUFFER_LENGTH;
  
   /* first we should check the file permissions */   
   if ( (filesize = file_exists(filename)) > 0) {
      val = check_file(filename);
      if (val != 1) {
        return val;
      }
   } else {
      return 0;
   }
   
   fd = fopen (filename, "rb");
   if (fd == NULL)
   {
      return 0;
   }
   
   filesize = filesize >= LOAD_BUFFER_LENGTH ? filesize : LOAD_BUFFER_LENGTH;
   buffer = malloc((size_t) filesize);
   if (buffer == NULL) {
#ifdef DEBUG
   fprintf(stderr, "load_init: failed to allocate buffer of size %ld\n", filesize);
#endif   
       return 0;
   }

#ifdef DEBUGload
   fprintf(stderr, "load_init, password=\"%s\"\n", password);
#endif   
   for (j = 0; password[j] != '\0'; j++)
   {
      key[j] = password[j];
   }
   keylength = j;
   rc2_expandkey (key, keylength, 128);
   
   size = read (fileno (fd), (unsigned char *) (charbuf + count), 8);
#ifdef DEBUGload
   fprintf(stderr, "load_init, size=%d, keylength=%d\n",size, keylength);
#endif

   if (size < 8)
       return -1;

   for (count = 0; count < 4; count++)
   {
      count2 = count << 1;
      iv[count] = charbuf[count2] << 8;
      iv[count] += charbuf[count2 + 1];
#ifdef DEBUGload
      fprintf(stderr, "load_init iv[%d]=%d\n",count,iv[count]);
#endif
   }

   size = 0;
   bufferIndex = 0;
   while ((count = read (fileno (fd), (unsigned char *) charbuf, 8)) > 0)
    {
#ifdef DEBUGload
      fprintf(stderr, "load_init A, count=%d, count2=%d\n",count,count2);
#endif
      while (count < 8)
	{
	  count2 = read (fileno (fd), (unsigned char *) (charbuf +
			 count), 8);
#ifdef DEBUGload
          fprintf(stderr, "load_init B, count=%d, count2=%d\n",count,count2);
#endif
	  if (count2 == 0)
	    {
	      printf ("bad EOF\n");
	      return -1;
	    }
	  count += count2;
	} /* while (count < 8) */

      size += 8;
#ifdef DEBUGload
      fprintf(stderr, "load_init charbuf[1]=%c\n",charbuf[1]);
#endif
      for (count2 = 0; count2 < 8; count2 += 2)
	{
	  count3 = count2 >> 1;
	  ciphertext[count3] = charbuf[count2] << 8;
	  ciphertext[count3] += charbuf[count2 + 1];

	  plaintext[count3] = ciphertext[count3] ^ iv[count3];
	  iv[count3] = plaintext[count3];
	}

      rc2_decrypt (plaintext);
      memcpy ((unsigned char *) (buffer + bufferIndex), plaintext, 8);
      bufferIndex += 8;
      buffer[bufferIndex + 1] = '\0';
#ifdef DEBUG
      fprintf(stderr, "bufferIndex=%d, buffer=%s\n", bufferIndex, buffer);
#endif
    } /* while ((count = read (fileno (fd), (unsigned char *) charbuf, 8)) > 0) */
#ifdef DEBUG
   fprintf(stderr, "load_init, size=%d, buffer[size-1]=%d, ", size, buffer[size-1]);
#endif
   size -= buffer[size - 1];
#ifdef DEBUG
   fprintf(stderr, "size=%d\n", size);
#endif
   lastcount = 0;
   
   /* This will point to the starting index */
   bufferIndex = 0;
   return 1;
}

int load_entry(char *entry[4]) {
/* Strip off PKCS 5 padding 
     Should check to make sure it's good here
 */
   int count, count1=0;
#ifdef DEBUG
   fprintf(stderr, "load_entry, lastcount=%d, size=%d, entry=%p\n", lastcount,size,entry);
#endif

  for (count = lastcount; count < size; count++)
    {
      if ((unsigned char) (buffer[count]) == 255)
	{
	  if (buffer[bufferIndex] == '\0')
	    {
	      bufferIndex++;
	    }
	  entry[count1] = (char *) malloc (count - bufferIndex + 1);
#ifdef DEBUGload
          fprintf(stderr, "load_entry, entry[%d]=%p\n",count1,entry[count1]);
#endif
	  memcpy (entry[count1],
		  (unsigned char *) (buffer + bufferIndex),
		  count - bufferIndex);
	  entry[count1][count - bufferIndex] = '\0';
#ifdef DEBUGload
          fprintf(stderr, "load_entry, entry[%d]=%s\n",count1,entry[count1]);
#endif
	  count++;
	  bufferIndex = count;
	  count1++;
	  if (count1 == 4)
	    {
             lastcount = count;
#ifdef DEBUGload
             fprintf(stderr, "load_entry, return 1, entry ready\n");
#endif
	     return 1;
	    }
	} /* if ((unsigned char) (buffer[count]) == 255) */
    } /* for (count = 0; count < size; count++) */

#ifdef DEBUGload
   fprintf(stderr, "load_entry, ended no entry anymore\n");
#endif
   return 2;
}

int load_finalize(void) {

  fclose (fd);
  free(buffer);
  return 1;
}

int
check_file (char *filename)
{
  struct stat naamstat;

  if (stat (filename, &naamstat) == -1)
    {
      return (-3);
    }

  if (((naamstat.st_mode & S_IAMB) | (S_IRUSR | S_IWUSR)) != (S_IRUSR |
    S_IWUSR))
    {
#ifdef DEBUGload
      fprintf (stderr,
	       "%s perms are bad, they are: %ld, should be -rw------\n", filename,
	       (naamstat.st_mode & (S_IREAD | S_IWRITE)));
#endif
      return (-1);
    }

  if (!S_ISREG (naamstat.st_mode))
    {
      lstat (filename, &naamstat);
      if (S_ISLNK (naamstat.st_mode))
	{
#ifdef DEBUGload
	  fprintf (stderr, "%s is a symlink\n", filename);
#endif
	  return (-2);
	}
    }

  return (1);
}


off_t
file_exists (char *tfile)
{
  struct stat naamstat;

  if ((stat (tfile, &naamstat) == -1) && (errno == ENOENT))
    {
#ifdef DEBUG
      fprintf (stderr, "file_exists, %s does NOT exist\n", tfile);
#endif
      return (0);
    }
  else
    {
#ifdef DEBUG
      fprintf (stderr, "file_exists, %s DOES exist, size is %ld\n", tfile, naamstat.st_size);
#endif
      return (naamstat.st_size);
    }
}
