/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <gnome.h>

#include "server.h"
#include "protocol.h"
#include "guiutils.h"
#include "guiprefs.h"
#include "tasks.h"
#include "news15.h"
#include "hldat.h"
#include "pixmap.h"

#define NEWS "News15"

typedef enum {NEWSFOLDER=2,NEWSGROUP=3} GroupType;

typedef struct {
  GroupType type;
  char *name;
  int postcount;
  GtkCTreeNode *node;
  GNode *self;
  gboolean have_children;
} Group;

typedef struct {
  Connection *c;
  NotebookPage *frame;
  GNode *cat_tree;
  HLObject **o;
  NewsGroup *ng;
  Task *task;
  Group *group_shown;
  GList *getgroups;
  GList *postlist;
  guint32 parent_id;
  guint32 current_id;
  guint32 next_id;
  guint32 prev_id;
  guint32 child_id;
  GtkWidget *groups_widget;
  GtkWidget *threads_widget;
  GtkWidget *news_widget;
} NewsPage;

typedef struct {
  NewsPage *np;
  HLObject *news_path;
  guint32 parent_post_id;
  HLTransaction *t;
  GtkWidget *frame;
  GtkWidget *text;
  GtkWidget *subject;
} PostWindow;
static void destroyPostWindow(PostWindow *);
static void newPost(NewsPage *);
static void replyTo(NewsPage *);

static NewsPage *get_news(Connection *c);
typedef struct {
  NewsPage *np;
  Path *path;
  Group *group;
  HLTransaction *t;
  Task *task;
} GetGroups;

static void delete_get_groups(GetGroups *gg);
static void news_show_message_options(NewsPage *np);


static gboolean destroy_one_group(GNode *n,gpointer data){
  Group *g=(Group *)n->data;
  if(g!=NULL){
    n->data=NULL;
    if(g->name!=NULL)
      free(g->name);
    free(g);
  }
  return FALSE;
}

static void destroy_groups_at(NewsPage *np,GNode *node){
  g_node_traverse(node,G_PRE_ORDER,G_TRAVERSE_ALL,
		  -1,destroy_one_group,NULL);
}

static void clear_groups(NewsPage *np){
  if(np->cat_tree!=NULL)
    destroy_groups_at(np,np->cat_tree);
  np->cat_tree=g_node_new(NULL);
  gtk_clist_clear(GTK_CLIST(np->groups_widget));
}

static void clear_posts(NewsPage *np){
  gtk_clist_clear(GTK_CLIST(np->threads_widget));
  gtk_editable_delete_text(GTK_EDITABLE(np->news_widget),0,-1);
  np->ng=NULL;
  if(np->o!=NULL)
    objects_destroy(np->o);
  np->o=NULL;
  np->parent_id=0;
  np->current_id=0;
  np->next_id=0;
  np->prev_id=0;
  np->child_id=0;
  news_show_message_options(np);
}

static void destroy_news(Connection *c,gpointer data){
  NewsPage *np=(NewsPage *)data;

  while(np->getgroups!=NULL)
    delete_get_groups((GetGroups *)np->getgroups->data);
  while(np->postlist!=NULL)
    destroyPostWindow((PostWindow *)np->postlist);
  hooks_set_data(c,NEWS,NULL);
  clear_posts(np);
  if(np->frame!=NULL)
    gutils_nbpage_destroy(np->frame);
  if(np->cat_tree!=NULL)
    destroy_groups_at(np,np->cat_tree);
  if(np->task!=NULL)
    task_remove(np->task);
  free(np);
}

/* ================================================== */

static void delete_get_groups(GetGroups *gg){
  if(gg==NULL) return;
  if(gg->np!=NULL)
    gg->np->getgroups=g_list_remove(gg->np->getgroups,gg);
  if(gg->task!=NULL)
    task_remove(gg->task);
  if(gg->t!=NULL)
    transaction_destroy(gg->t);
  if(gg->path!=NULL)
    path_destroy(gg->path);
  free(gg);
}

static gboolean got_get_groups(gpointer data){
  GetGroups *gg=(GetGroups *)data;
  HLObject **o;
  int count;
  int i;
  GtkCTreeNode *insert_node=NULL;
  GNode *node=NULL;
  Group *g;
  GdkPixmap *catP=NULL,*newsP=NULL;
  GdkBitmap *catM=NULL,*newsM=NULL;
  char buf[12],*p[]={"NONE",buf},*waiting[]={N_("Downloading list..."),""};

  gtk_clist_freeze(GTK_CLIST(gg->np->groups_widget));
  pixmap_get("news",&newsP,&newsM);
  pixmap_get("category",&catP,&catM);
  if(gg->t!=NULL){
    o=transaction_extract_objects(gg->t,HLO_CATEGORYITEM,&count);
    if(gg->path==NULL || gg->group==NULL){
      clear_groups(gg->np);
      insert_node=NULL;
      node=gg->np->cat_tree;
    } else {
      /* we must be certain that gg->group still exists at this time ... */
      node=g_node_find(gg->np->cat_tree,G_PRE_ORDER,G_TRAVERSE_ALL,gg->group);
      if(node!=NULL){
	insert_node=gg->group->node;
	gtk_ctree_remove_node(GTK_CTREE(gg->np->groups_widget),GTK_CTREE_ROW(insert_node)->children);
      }
    }
    if(node!=NULL){
      for(i=0;i<count;i++){
	g=(Group *)malloc(sizeof(Group));
	g->name=strdup(o[i]->data.catalogitem->name);
	g->postcount=o[i]->data.catalogitem->w.post_count;
	p[0]=g->name;
	sprintf(buf,"(%d)",g->postcount);
	g->type=o[i]->data.catalogitem->w.type;
	g->self=g_node_append_data(node,g);
	g->have_children=FALSE;
	if(g->type==3)
	  g->node=gtk_ctree_insert_node(GTK_CTREE(gg->np->groups_widget),
					insert_node,NULL,p,0,
					newsP,newsM,newsP,newsM,
					TRUE,FALSE);
	else
	  g->node=gtk_ctree_insert_node(GTK_CTREE(gg->np->groups_widget),
					insert_node,NULL,p,0,
					catP,catM,catP,catM,
					FALSE,FALSE);
	if(g->type==2){
	  GtkCTreeNode *tn=gtk_ctree_insert_node(GTK_CTREE(gg->np->groups_widget),g->node,
						 NULL,waiting,1,NULL,NULL,NULL,NULL,
						 TRUE,FALSE);
	  gtk_ctree_node_set_selectable(GTK_CTREE(gg->np->groups_widget),
					tn,FALSE);
	  gtk_ctree_node_set_selectable(GTK_CTREE(gg->np->groups_widget),
					g->node,FALSE);
	}
	gtk_ctree_node_set_row_data(GTK_CTREE(gg->np->groups_widget),
				    g->node,g);
      }
      if(insert_node!=NULL){
	gtk_ctree_expand(GTK_CTREE(gg->np->groups_widget),insert_node);
	if(gg->group!=NULL)
	  gg->group->have_children=TRUE;
      }
    }
    objects_destroy(o);
  } 
  gtk_clist_thaw(GTK_CLIST(gg->np->groups_widget));
  delete_get_groups(gg);
  return FALSE;
}

static void rcv_get_groups(Connection *c,
			   HLTransaction *t,
			   HLTransaction *r,
			   gpointer data){
  GetGroups *gg=(GetGroups *)data;
  if(r->w.error!=0){
    gg->t=NULL;
    gg->task=NULL;
    if(gg->group!=NULL)
      gg->group->have_children=FALSE;
    server_reply_default(c,t,r,strdup(_("Could not list newsgroups:\n")));
    gtk_idle_add(got_get_groups,gg);
    return;
  }
  transaction_destroy(t);
  gg->t=r;
  transaction_set_task(r,gg->task,0.5,1.0);
  transaction_read_objects(c,r);
  gtk_idle_add(got_get_groups,gg);
}

static void send_get_groups(Connection *c,gpointer p){
  GetGroups *gg=(GetGroups *)p;
  server_transaction_reply_set(c,gg->t,rcv_get_groups,gg);
  transaction_send(gg->t,c);
}

static void task_get_groups_abort(Task *t){
  GetGroups *gg=(GetGroups *)t->user_data;
  gg->task=NULL;
}

static void get_groups(NewsPage *np,Path *p,Group *g){
  GetGroups *gg=(GetGroups *)malloc(sizeof(GetGroups));
  gg->np=np;
  gg->path=p;
  gg->group=g;
  gg->t=transaction_new(HLCT_NEWSDIRLIST,FALSE,FALSE);
  gg->task=NULL;
  if(p!=NULL){
    HLObject *o=path_new(((Path *)p)->directories,
			 ((Path *)p)->dir_levels);
    o->type=HLO_NEWSPATH;
    transaction_add_object(gg->t,o);
  }
  gg->task=task_new(S_NAME(np->c));
  gg->task->user_data=gg;
  gg->task->user_cancel=task_get_groups_abort;
  gg->np->getgroups=g_list_append(gg->np->getgroups,gg);
  sprintf(gg->task->text,_("Retrieving Newsgroups ..."));
  task_update(gg->task,TRUE,0.0);
  transaction_set_task(gg->t,gg->task,0.0,0.5);
  task_add(gg->task);
  server_transaction_start(np->c,send_get_groups,gg);
}

static char **make_path(GNode *n,int *count){
  char **p;
  int c=*count;
  if(n==NULL)
    return NULL;
  if(n->parent==NULL){
    p=malloc(sizeof(char *)*(1+*count));
    p[*count]=NULL;
    return p;
  }
  *count+=1;
  p=make_path(n->parent,count);
  if(p!=NULL)
    p[*count-c-1]=strdup(((Group *)n->data)->name);
  return p;
}

static void news_folder_expanded(GtkCTree *tree,GtkCTreeNode *node,NewsPage *np){
  Group *g=(Group *)gtk_ctree_node_get_row_data(tree,node);
  int count=0;
  char **paths;
  Path *p;

  if(g->have_children==TRUE) return;
  g->have_children=TRUE;
  paths=make_path(g->self,&count);
  if(paths==NULL) return;
  p=calloc(1,sizeof(Path));
  p->dir_levels=count;
  p->directories=paths;
  get_groups(np,p,g);
}

/* ================================================== */

static void news_task_cancel(Task *task){
  HLTransaction *t=task->user_data;
  if(t!=NULL)
    t->task=NULL;
}

static gboolean got_news(gpointer data){
  HLTransaction *r=data;
  NewsPage *np=(NewsPage *)r->data;
  int i,j,count;
  GtkCTreeNode *node;
  char buf[12],*p[]={N_("Nothing"),N_("Nobody"),buf};
  struct newsptr {GtkCTreeNode *node; NewsItem *post;} *ptr=NULL;

  task_remove(r->task);
  np->task=NULL;
  gtk_clist_freeze(GTK_CLIST(np->threads_widget));
  clear_posts(np);
  np->o=transaction_extract_objects(r,HLO_CATLIST,&count);
  if(np->o!=NULL){
    np->ng=np->o[0]->data.newsgroup;
    count=np->ng->post_count;
    if(count>0)
      ptr=malloc(sizeof(struct newsptr)*count);
    for(i=0;i<count;i++){
      node=NULL;
      ptr[i].post=&(np->ng->posts[i]);
      for(j=0;j<i;j++)
	if(ptr[j].post->postid==ptr[i].post->parentid){
	  node=ptr[j].node;
	  break;
	}
      if(np->ng->posts[i].size<1024)
	sprintf(buf,"%d",np->ng->posts[i].size);
      else
	sprintf(buf,"%dK",(np->ng->posts[i].size+1023)/1024);
      p[0]=np->ng->posts[i].subject;
      p[1]=np->ng->posts[i].sender;
      ptr[i].node=gtk_ctree_insert_node(GTK_CTREE(np->threads_widget),
					node,NULL,p,0,NULL,NULL,NULL,NULL,FALSE,FALSE);
      gtk_ctree_node_set_row_data(GTK_CTREE(np->threads_widget),ptr[i].node,
				  GINT_TO_POINTER(i));
    }
    if(ptr!=NULL)
      free(ptr);
  }
  gtk_clist_thaw(GTK_CLIST(np->threads_widget));
  transaction_destroy(r);
  return FALSE;
}

static void rcvd_news(Connection *c,HLTransaction *t,
		      HLTransaction *r,gpointer p){
  if(r->w.error!=0){
    task_remove_when_idle(t->task);
    server_reply_default(c,t,r,strdup(_("Could not list news posts:\n")));
    ((NewsPage *)t->data)->group_shown=NULL;
    return;
  }
  transaction_set_task(r,t->task,0.1,1.0);
  t->task=NULL;
  r->task->user_data=r;
  r->data=t->data;
  transaction_destroy(t);
  transaction_read_objects(c,r);
  gtk_idle_add(got_news,r);
}

static void get_news_start(Connection *c,gpointer p){
  HLTransaction *t=p;
  server_transaction_reply_set(c,t,rcvd_news,t->data);
  transaction_send(t,c);
}

static void news_group_selected(GtkCTree *tree,GtkCTreeNode *node,int col,NewsPage *np){
  Group *g=(Group *)gtk_ctree_node_get_row_data(tree,node);
  int count=0;
  char **paths;
  int i;
  Task *task;
  HLTransaction *t;
  HLObject *o;

  if(np->task!=NULL) return;
  t=transaction_new(HLCT_NEWSCATLIST,np,FALSE);
  task=np->task=task_new(S_NAME(np->c));
  paths=make_path(g->self,&count);
  o=path_new(paths,count);
  o->type=HLO_NEWSPATH;
  transaction_add_object(t,o);
  for(i=0;i<count;i++) free(paths[i]);
  free(paths);
  task->user_data=t;
  task->user_cancel=news_task_cancel;
  sprintf(task->text,_("Retrieving posts..."));
  task_add(task);
  task_update(task,TRUE,0.0);
  transaction_set_task(t,task,0.0,0.1);
  clear_posts(np);
  gtk_editable_delete_text(GTK_EDITABLE(np->news_widget),0,-1);
  np->group_shown=g;
  server_transaction_start(np->c,get_news_start,t);
}

/* ================================================== */
static int find_news_by_id(NewsPage *np, int post){
  int i;
  for(i=0;i<np->ng->post_count;i++)
    if(np->ng->posts[i].postid==post)
      return i;
  return -1;
}

static void expand_post_and_parents(NewsPage *np,int id,gboolean do_select){
  int i;
  GtkCTreeNode *t;
  
  if(id<0) return;
  i=find_news_by_id(np,id);
  if(i==-1) return;
  if(np->ng->posts[i].parentid!=0)
    expand_post_and_parents(np,np->ng->posts[i].parentid,FALSE);
  t=gtk_ctree_find_by_row_data(GTK_CTREE(np->threads_widget),NULL,GINT_TO_POINTER(i));
  if(t==NULL) return;
  gtk_ctree_expand(GTK_CTREE(np->threads_widget),t);
  if(do_select==TRUE)
    gtk_ctree_select(GTK_CTREE(np->threads_widget),t);
  return;
}

static gboolean got_post(gpointer data){
  HLTransaction *r=data;
  NewsPage *np=(NewsPage *)r->data;
  HLObject *nd=transaction_find_object(r,HLO_NEWSDATA);
  HLObject *o;
  
  np->task=NULL;
  task_remove(r->task);
  gtk_text_freeze(GTK_TEXT(np->news_widget));
  gtk_editable_delete_text(GTK_EDITABLE(np->news_widget),0,-1);
  if(nd!=NULL){
    string_net_to_unix(nd);
     if((o=transaction_find_object(r,HLO_NEWSSUBJECT))!=NULL){
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,_("Subject: "),-1);
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,o->data.string,-1);
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,"\n",-1);
    }
    if((o=transaction_find_object(r,HLO_NEWSAUTHOR))!=NULL){
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,_("Author: "),-1);
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,o->data.string,-1);
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,"\n",-1);
    }
    if((o=transaction_find_object(r,HLO_NEWSDATE))!=NULL){
      time_t timet=date_to_unix(o->data.datetime);
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,_("Date: "),-1);
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,ctime(&timet),-1);
      gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,"\n\n",-1);
    }
    gtk_text_insert(GTK_TEXT(np->news_widget),NULL,NULL,NULL,
		    nd->data.string,-1);
  }
  gtk_text_thaw(GTK_TEXT(np->news_widget));

  if((o=transaction_find_object(r,HLO_NEXTTHREAD))!=NULL)
    np->next_id=o->data.number;
  else
    np->next_id=0;

  if((o=transaction_find_object(r,HLO_PREVTHREAD))!=NULL)
    np->prev_id=o->data.number;
  else
    np->prev_id=0;

  if((o=transaction_find_object(r,HLO_PARENT_POST))!=NULL)
    np->parent_id=o->data.number;
  else
    np->parent_id=0;

  if((o=transaction_find_object(r,HLO_CHILD_POST))!=NULL)
    np->child_id=o->data.number;
  else
    np->child_id=0;

  news_show_message_options(np);
  expand_post_and_parents(np,np->current_id,TRUE);
  transaction_destroy(r);
  return FALSE;
}

static void rcvd_post(Connection *c,HLTransaction *t,
		      HLTransaction *r,gpointer p){
  if(r->w.error!=0){
    task_remove_when_idle(t->task);
    ((NewsPage *)t->data)->task=NULL;
    server_reply_default(c,t,r,strdup(_("Could not list news posts:\n")));
    return;
  }
  transaction_set_task(r,t->task,0.1,1.0);
  t->task=NULL;
  r->task->user_data=r;
  r->data=t->data;
  transaction_destroy(t);
  transaction_read_objects(c,r);
  gtk_idle_add(got_post,r);
}

static void get_post_start(Connection *c,gpointer p){
  HLTransaction *t=p;
  server_transaction_reply_set(c,t,rcvd_post,t->data);
  transaction_send(t,c);
}

static HLObject *get_news_path(NewsPage *np){
  int count=0,i;
  char **paths=make_path(np->group_shown->self,&count);
  HLObject *o=path_new(paths,count);
  o->type=HLO_NEWSPATH;
  for(i=0;i<count;i++) free(paths[i]);
  free(paths);
  return o;
}

static void get_news_post(NewsPage *np, int post){
  int i;
  Task *task;
  HLTransaction *t;
  NewsItem *ni;

  if(np->group_shown==NULL) return;
  if(np->task!=NULL) return;
  if(post==np->current_id) return;
  i=find_news_by_id(np,post);
  if(i==-1) return;
  ni=&np->ng->posts[i];
  np->prev_id=0;
  np->next_id=0;
  np->parent_id=0;
  np->current_id=0;
  np->child_id=0;
  news_show_message_options(np);
  np->parent_id=ni->parentid;
  np->current_id=ni->postid;
  if(ni->partcount==0){
    gtk_editable_delete_text(GTK_EDITABLE(np->news_widget),0,-1);
    return;
  }
  t=transaction_new(HLCT_GETTHREAD,np,FALSE);
  task=np->task=task_new(S_NAME(np->c));
  transaction_add_object(t,get_news_path(np));
  transaction_add_object(t,create_number(HLO_THREADID,ni->postid));
  transaction_add_object(t,create_string(HLO_NEWSTYPE,ni->parts[0].mime_type));
  task->user_data=t;
  task->user_cancel=news_task_cancel;
  sprintf(task->text,_("Retrieving news post..."));
  task_add(task);
  task_update(task,TRUE,0.0);
  transaction_set_task(t,task,0.0,0.1);
  server_transaction_start(np->c,get_post_start,t);
}

static void news_post_selected(GtkCTree *tree,GtkCTreeNode *node,int col,NewsPage *np){
  int post=GPOINTER_TO_INT(gtk_ctree_node_get_row_data(tree,node));

  if(np->ng==NULL || post>=np->ng->post_count) return;
  get_news_post(np,np->ng->posts[post].postid);
}

/* ================================================== */


static void close_news(GtkButton *b,gpointer dummy){
  NewsPage *np=(NewsPage *)dummy;
  destroy_news(np->c,np);
}

static void refresh_news(GtkButton *b,gpointer dummy){
  NewsPage *np=(NewsPage *)dummy;
  get_groups(np,NULL,NULL);
  clear_posts(np);
}

static void show_parent(GtkButton *b,NewsPage *np){
  if(np==NULL || np->parent_id==0) return;
  get_news_post(np,np->parent_id);
}
static void show_next(GtkButton *b,NewsPage *np){
  if(np==NULL || np->next_id==0) return;
  get_news_post(np,np->next_id);
}
static void show_prev(GtkButton *b,NewsPage *np){
  if(np==NULL || np->prev_id==0) return;
  get_news_post(np,np->prev_id);
}
static void show_child(GtkButton *b,NewsPage *np){
  if(np==NULL || np->child_id==0) return;
  get_news_post(np,np->child_id);
}
static void new_posting(GtkButton *b,NewsPage *np){
  if(np==NULL) return;
  newPost(np);
}
static void reply_posting(GtkButton *b,NewsPage *np){
  if(np==NULL) return;
  replyTo(np);
}

static GnomeUIInfo news_toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("Close"),N_("Close news"),
			 close_news,
			 HL_STOCK_PIXMAP_CLOSE_PAGE),
  GNOMEUIINFO_ITEM_STOCK(N_("Refresh"),N_("Reload news"),
			 refresh_news,
			 HL_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_ITEM_STOCK(N_("Post"),N_("Post a Message"),
			 new_posting,
			 HL_STOCK_PIXMAP_NEWS_POST),
  GNOMEUIINFO_ITEM_STOCK(N_("Reply"),N_("Reply to this Message"),
			 reply_posting,
			 HL_STOCK_PIXMAP_REPLY),
  GNOMEUIINFO_ITEM_STOCK(N_("Prev"),N_("Previous Message in Thread"),
			 show_prev,
			 HL_STOCK_PIXMAP_NEWS_PREV),
  GNOMEUIINFO_ITEM_STOCK(N_("Next"),N_("Next Message in Thread"),
			 show_next,
			 HL_STOCK_PIXMAP_NEWS_NEXT),
  GNOMEUIINFO_ITEM_STOCK(N_("Up"),N_("Show the parent of this Message"),
			 show_parent,
			 HL_STOCK_PIXMAP_NEWS_PARENT),
  GNOMEUIINFO_ITEM_STOCK(N_("Down"),N_("Show the reply to this Message"),
			 show_child,
			 HL_STOCK_PIXMAP_NEWS_CHILD),
  GNOMEUIINFO_END
};

static void news_show_message_options(NewsPage *np){
  gutils_nbpage_toolbar_set_sensitive(np->frame,3,np->group_shown!=NULL?TRUE:FALSE);
  gutils_nbpage_toolbar_set_sensitive(np->frame,4,np->current_id!=0?TRUE:FALSE);
  gutils_nbpage_toolbar_set_sensitive(np->frame,5,np->prev_id!=0?TRUE:FALSE);
  gutils_nbpage_toolbar_set_sensitive(np->frame,6,np->next_id!=0?TRUE:FALSE);
  gutils_nbpage_toolbar_set_sensitive(np->frame,7,np->parent_id!=0?TRUE:FALSE);
  gutils_nbpage_toolbar_set_sensitive(np->frame,8,np->child_id!=0?TRUE:FALSE);
}

static void news_create_gui(NewsPage *np){
  GtkWidget *p1,*p2,*sw_g,*sw_t,*sw_p;
  static char *t1[]={N_("Name"),N_("Posts")},*t2[]={N_("Subject"),N_("Author"),N_("Size")};

  np->frame=gutils_nbpage_new("Main",news_toolbar,np->c,_("News"),"news",np);
  p1=gtk_hpaned_new();
  p2=gtk_vpaned_new();
  sw_p=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw_p),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  np->news_widget=gtk_text_new(NULL,NULL);
  gtk_container_add(GTK_CONTAINER(sw_p),np->news_widget);
  gtk_text_set_editable(GTK_TEXT(np->news_widget),FALSE);

  sw_g=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw_g),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  np->groups_widget=gtk_ctree_new_with_titles(2,0,t1);
  gtk_container_add(GTK_CONTAINER(sw_g),np->groups_widget);
  gtk_ctree_set_expander_style(GTK_CTREE(np->groups_widget),GTK_CTREE_EXPANDER_SQUARE);
  gtk_ctree_set_line_style(GTK_CTREE(np->groups_widget),GTK_CTREE_LINES_DOTTED);
  gtk_clist_set_column_justification(GTK_CLIST(np->groups_widget),1,GTK_JUSTIFY_RIGHT);
  gtk_clist_set_row_height(GTK_CLIST(np->groups_widget),16);

  sw_t=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw_t),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  np->threads_widget=gtk_ctree_new_with_titles(3,0,t2);
  gtk_container_add(GTK_CONTAINER(sw_t),np->threads_widget);
  gtk_ctree_set_expander_style(GTK_CTREE(np->threads_widget),GTK_CTREE_EXPANDER_SQUARE);
  gtk_ctree_set_line_style(GTK_CTREE(np->threads_widget),GTK_CTREE_LINES_DOTTED);
  gtk_clist_set_column_justification(GTK_CLIST(np->threads_widget),2,GTK_JUSTIFY_RIGHT);

  gtk_signal_connect(GTK_OBJECT(np->groups_widget),"tree-expand",
		     GTK_SIGNAL_FUNC(news_folder_expanded),np);
  gtk_signal_connect(GTK_OBJECT(np->groups_widget),"tree-select-row",
		     GTK_SIGNAL_FUNC(news_group_selected),np);
  gtk_signal_connect(GTK_OBJECT(np->threads_widget),"tree-select-row",
		     GTK_SIGNAL_FUNC(news_post_selected),np);

  guiprefs_add_clist(GTK_CLIST(np->groups_widget),
		     "News/List/Groups","140,0");
  guiprefs_add_clist(GTK_CLIST(np->threads_widget),
		     "News/List/Threads","150,75,0");
  if(news_big_messagewindow==TRUE){
    gtk_paned_add1(GTK_PANED(p2),p1);
    gtk_paned_add2(GTK_PANED(p1),sw_t);
    gtk_paned_add1(GTK_PANED(p1),sw_g);
  } else {
    gtk_paned_add2(GTK_PANED(p1),p2);
    gtk_paned_add1(GTK_PANED(p2),sw_t);
    gtk_paned_add1(GTK_PANED(p1),sw_g);
  }
  gtk_paned_add2(GTK_PANED(p2),sw_p);
  guiprefs_add_paned(GTK_PANED(p1),"News/Pane/PaneH");
  guiprefs_add_paned(GTK_PANED(p2),"News/Pane/PaneV");
  gutils_nbpage_set_main(np->frame,(news_big_messagewindow==TRUE)?p2:p1);
  refresh_news(NULL,np);
}

static NewsPage *get_news(Connection *c){
  NewsPage *np=(NewsPage *)hooks_get_data(c,NEWS);
  if(np==NULL){
    np=malloc(sizeof(NewsPage));
    np->c=c;
    np->frame=NULL;
    np->cat_tree=g_node_new(NULL);
    np->groups_widget=np->news_widget=np->threads_widget=NULL;
    np->o=NULL;
    np->ng=NULL;
    np->getgroups=NULL;
    np->task=NULL;
    np->postlist=NULL;
    hooks_create(c,NEWS,np,NULL,destroy_news);
  }
  return np;
}

void want_news_15(Connection *c){
  NewsPage *np;
  if(c->version<=150) return;
  np=get_news(c);
  if(np->frame==NULL)
    news_create_gui(np);
  gutils_nbpage_to_front(np->frame);
}

void check_news_15(Connection *c){
  if(auto_open_news==TRUE && c->version>150)
    want_news_15(c);
}

/* ===================================== */

static void destroyPostWindow(PostWindow *pw){
  if(pw->np!=NULL)
    pw->np->postlist=g_list_remove(pw->np->postlist,pw);
  if(pw->t!=NULL){
    pw->np=NULL;
    return;
  }
  if(pw->frame!=NULL)
    gtk_widget_destroy(pw->frame);
  if(pw->news_path!=NULL){
    HLObject **o=malloc(sizeof(HLObject *)*2);
    o[0]=pw->news_path;
    o[1]=NULL;
    objects_destroy(o);
  }
  free(pw);
}

static gboolean closePostWindow(GtkWidget *b,gpointer p){
  destroyPostWindow((PostWindow *)p);
  return FALSE;
}

static void transfer_post(Connection *c,gpointer p){
  HLTransaction *t=(HLTransaction *)p;
  server_transaction_reply_set(c,t,server_reply_default,
			       strdup(_("Could not post article:")));
  transaction_send(t,c);
}

static void send_post(GtkButton *b,PostWindow *pw){
  char *string;
  HLObject *o;
  HLTransaction *t=transaction_new(HLCT_POSTTHREAD,NULL,FALSE);
  transaction_add_object(t,pw->news_path);
  pw->news_path=NULL;
  transaction_add_object(t,create_number(HLO_THREADID,pw->parent_post_id));
  transaction_add_object(t,create_string(HLO_NEWSSUBJECT,
					 gtk_entry_get_text(GTK_ENTRY(pw->subject))));
  transaction_add_object(t,create_string(HLO_NEWSTYPE,"text/plain"));
  string=gtk_editable_get_chars(GTK_EDITABLE(pw->text),0,-1);
  o=create_string(HLO_NEWSDATA,string);
  free(string);
  string_unix_to_net(o);
  transaction_add_object(t,o);
  server_transaction_start(pw->np->c,transfer_post,t);
  destroyPostWindow(pw);
}

static PostWindow *newPostWindow(NewsPage *np){
  GtkWidget *hbox,*scrol;
  PostWindow *pw=malloc(sizeof(PostWindow));
  pw->np=np;
  np->postlist=g_list_prepend(np->postlist,pw);
  pw->news_path=get_news_path(np);
  pw->parent_post_id=0;
  pw->t=NULL;
  pw->frame=gnome_dialog_new(_("Post a Message"),
			     GNOME_STOCK_BUTTON_CANCEL,
			     "Send",NULL);
  gtk_widget_set_usize(pw->frame,250,250);
  gnome_dialog_set_parent(GNOME_DIALOG(pw->frame),
			  GTK_WINDOW(pw->np->c->gui->main_window));
  gnome_dialog_button_connect(GNOME_DIALOG(pw->frame),0,
			      GTK_SIGNAL_FUNC(closePostWindow),
			      (gpointer)pw);
  gnome_dialog_button_connect(GNOME_DIALOG(pw->frame),1,
			      GTK_SIGNAL_FUNC(send_post),
			      (gpointer)pw);
  gtk_signal_connect(GTK_OBJECT(pw->frame),"delete_event",
		     GTK_SIGNAL_FUNC(closePostWindow),(gpointer)pw);
  hbox=gtk_hbox_new(FALSE,10);
  gtk_box_pack_start(GTK_BOX(hbox),gtk_label_new(_("Subject:")),FALSE,FALSE,0);
  gtk_box_pack_end(GTK_BOX(hbox),pw->subject=gtk_entry_new(),TRUE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(pw->frame)->vbox),hbox,FALSE,FALSE,0);
  scrol=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrol),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(pw->frame)->vbox),scrol,TRUE,TRUE,0);
  pw->text=gtk_text_new(NULL,NULL);
  gtk_text_set_editable(GTK_TEXT(pw->text),TRUE);
  gtk_signal_connect(GTK_OBJECT(pw->text),"activate",
		     GTK_SIGNAL_FUNC(send_post),
		     (gpointer)pw);
  gtk_container_add(GTK_CONTAINER(scrol),pw->text);
  return pw;
}

static void newPost(NewsPage *np){
  PostWindow *pw;

  if(np->group_shown==NULL) return;
  pw=newPostWindow(np);
  gtk_widget_show_all(pw->frame);
}

static void replyTo(NewsPage *np){
  PostWindow *pw;
  int i;
  NewsItem *ni;

  if(np->group_shown==NULL || np->current_id==0) return;
  i=find_news_by_id(np,np->current_id);
  if(i==-1) return;
  ni=&np->ng->posts[i];
  pw=newPostWindow(np);
  pw->parent_post_id=np->current_id;
  if(ni->subject!=0){
    if(strncmp("Re: ",ni->subject,4)!=0)
      gtk_entry_set_text(GTK_ENTRY(pw->subject),_("Re: "));
    gtk_entry_append_text(GTK_ENTRY(pw->subject),ni->subject);
  }
  gtk_widget_show_all(pw->frame);
}

