# -*- coding: utf-8 -*-

"""
Parse a Python file and retrieve classes and methods.

Parse enough of a Python file to recognize class and method
definitions and to find out the superclasses of a class.

The interface consists of a single function:<br />
        readmodule(module, path)<br />
module is the name of a Python module, path is an optional list of
directories where the module is to be searched.  If present, path is
prepended to the system search path sys.path.
The return value is a dictionary.  The keys of the dictionary are
the names of the classes defined in the module (including classes
that are defined via the from XXX import YYY construct).  The values
are class instances of the class Class defined here.

A class is described by the class Class in this module.  Instances
of this class have the following instance variables:
        <ul>
        <li>name -- the name of the class</li>
        <li>super -- a list of super classes (Class instances)</li>
        <li>methods -- a dictionary of methods</li>
        <li>file -- the file in which the class was defined</li>
        <li>lineno -- the line in the file on which the class 
            statement occurred</li>
        </ul>
The dictionary of methods uses the method names as keys and a tuple
consisting of the line numbers on which the method was defined and a list
of the methods parameters as values.
If the name of a super class is not recognized, the corresponding
entry in the list of super classes is not a class instance but a
string giving the name of the super class.  Since import statements
are recognized and imported modules are scanned as well, this
shouldn't happen often.

<h4>BUGS</h4>
<ul>
<li>Code that doesn't pass tabnanny or python -t will confuse it, unless
  you set the module TABWIDTH vrbl (default 8) to the correct tab width
  for the file.</li>
</ul>

<h4>PACKAGE RELATED BUGS</h4>
<ul>
<li>If you have a package and a module inside that or another package
  with the same name, module caching doesn't work properly since the
  key is the base name of the module/package.</li>
<li>The only entry that is returned when you readmodule a package is a
  __path__ whose value is a list which confuses certain class browsers.</li>
<li>When code does:<br />
  from package import subpackage<br />
  class MyClass(subpackage.SuperClass):<br />
    ...<br />
  It can't locate the parent.  It probably needs to have the same
  hairy logic that the import locator already does.  (This logic
  exists coded in Python in the freeze package.)</li>
</ul>
  
This is a modified version of the one found in the Python 2.2.2
distribution.

"""

# Modifications Copyright (c) 2003 - 2005 Detlev Offenbach <detlev@die-offenbachs.de>


import sys
import os
import imp
import re
import string

__all__ = ["readmodule"]

TABWIDTH = 8

PTL_SOURCE = 128

SUPPORTED_TYPES = [imp.PY_SOURCE, PTL_SOURCE]
    
_getnext = re.compile(r"""
    (?P<String>
       \""" [^"\\]* (?:
                        (?: \\. | "(?!"") )
                        [^"\\]*
                    )*
       \"""

    |   ''' [^'\\]* (?:
                        (?: \\. | '(?!'') )
                        [^'\\]*
                    )*
        '''

    |   " [^"\\\n]* (?: \\. [^"\\\n]*)* "

    |   ' [^'\\\n]* (?: \\. [^'\\\n]*)* '
    )

|   (?P<Publics>
        ^
        [ \t]* __all__ [ \t]* = [ \t]* \[
        (?P<Identifiers> [^\]]*? )
        \]
    )

|   (?P<Method>
        ^
        (?P<MethodIndent> [ \t]* )
        def [ \t]+
        (?P<MethodName> [a-zA-Z_] \w* )
        (?: [ \t]* \[ (?: plain | html ) \] )?
        [ \t]* \(
        (?P<MethodSignature> (?: [^)] | \)[ \t]*,? )*? )
        \) [ \t]* :
    )

|   (?P<Class>
        ^
        (?P<ClassIndent> [ \t]* )
        class [ \t]+
        (?P<ClassName> [a-zA-Z_] \w* )
        [ \t]*
        (?P<ClassSupers> \( [^)]* \) )?
        [ \t]* :
    )

|   (?P<Attribute>
        ^
        (?P<AttributeIndent> [ \t]* )
        self [ \t]* \. [ \t]*
        (?P<AttributeName> [a-zA-Z_] \w* )
        [ \t]* =
    )

|   (?P<Import>
        ^ import [ \t]+
        (?P<ImportList> [^#;\n]+ )
    )

|   (?P<ImportFrom>
        ^ from [ \t]+
        (?P<ImportFromPath>
            [a-zA-Z_] \w*
            (?:
                [ \t]* \. [ \t]* [a-zA-Z_] \w*
            )*
        )
        [ \t]+
        import [ \t]+
        (?P<ImportFromList> [^#;\n]+ )
    )

|   (?P<ConditionalDefine>
        ^
        (?P<ConditionalDefineIndent> [ \t]* )
        (?: (?: if | elif ) [ \t]+ [^:]* | else [ \t]* ) : (?= \s* def)
    )
""", re.VERBOSE | re.DOTALL | re.MULTILINE).search

_commentsub = re.compile(r"""#[^\n]*\n|#[^\n]*$""").sub

_modules = {}                           # cache of modules we've seen

# each Python class is represented by an instance of this class
class Class:
    '''
    Class to represent a Python class.
    '''
    def __init__(self, module, name, super, file, lineno):
        """
        Constructor
        
        @param module name of the module containing this class
        @param name name of this class
        @param super list of class names this class is inherited from
        @param file filename containing this class
        @param lineno linenumber of the class definition
        """
        self.module = module
        self.name = name
        if super is None:
            super = []
        self.super = super
        self.methods = {}
        self.attributes = []
        self.classes = {}
        self.file = file
        self.lineno = lineno

    def _addmethod(self, name, function):
        """
        Method to add information about a method.
        
        @param name name of method to be added (string)
        @param function Function object to be added
        """
        self.methods[name] = function
        
    def _addattribute(self, name):
        """
        Method to add information about attributes.
        
        @param name name of the attribute to add (string)
        """
        if not name in self.attributes:
            self.attributes.append(name)
            
    def _addclass(self, name, _class):
        """
        Protected method to add a nested class to this class.
        
        @param name name of the class
        @param _class Class object to be added
        """
        self.classes[name] = _class
        
class Function(Class):
    '''
    Class to represent a top-level Python function
    '''
    def __init__(self, module, name, file, lineno, signature = ''):
        """
        Constructor
        
        @param module name of the module containing this function
        @param name name of this function
        @param file filename containing this class
        @param lineno linenumber of the class definition
        @param signature parameterlist of the method
        """
        Class.__init__(self, module, name, None, file, lineno)
        signature = _commentsub('', signature)
        self.parameters = [e.strip() for e in signature.split(',')]

class Publics:
    '''
    Class to represent the list of public  identifiers.
    '''
    def __init__(self, module, file, lineno, idents):
        """
        Constructor
        
        @param module name of the module containing this function
        @param file filename containing this class
        @param lineno linenumber of the class definition
        @param idents list of public identifiers
        """
        self.module = module
        self.name = '__all__'
        self.file = file
        self.lineno = lineno
        self.identifiers = [e.replace('"','').replace("'","").strip() \
                            for e in idents.split(',')]
    
def readmodule(module, path=[], inpackage=0):
    '''
    Backwards compatible interface.

    Like readmodule_ex() but strips Function objects from the
    resulting dictionary.
    
    @param path path the module should be searched in (list of strings)
    @param inpackage flag indicating a module inside a package is scanned
    @return the resulting dictionary
    '''

    dict = readmodule_ex(module, path, inpackage)
    res = {}
    for key, value in dict.items():
        if not isinstance(value, Function) and \
           not isinstance(value, Publics):
            res[key] = value
    return res

def readmodule_ex(module, path=[], inpackage=0):
    '''
    Read a module file and return a dictionary of classes.

    Search for MODULE in PATH and sys.path, read and parse the
    module and return a dictionary with one entry for each class
    found in the module.
    
    @param path path the module should be searched in (list of strings)
    @param inpackage flag indicating a module inside a package is scanned
    @return the resulting dictionary
    '''

    dict = {}
    dict_counts = {}

    if _modules.has_key(module):
        # we've seen this module before...
        return _modules[module]
    if module in sys.builtin_module_names:
        # this is a built-in module
        _modules[module] = dict
        return dict

    # search the path for the module
    f = None
    if inpackage:
        try:
            f, file, (suff, mode, type) = \
                    find_module(module, path)
        except ImportError:
            f = None
    if f is None:
        fullpath = list(path) + sys.path
        f, file, (suff, mode, type) = find_module(module, fullpath)
    if module.endswith(".py") and type == imp.PKG_DIRECTORY:
        return dict
    if type == imp.PKG_DIRECTORY:
        dict['__path__'] = [file]
        _modules[module] = dict
        path = [file] + path
        f, file, (suff, mode, type) = \
                        find_module('__init__', [file])
    if type not in SUPPORTED_TYPES:
        # not Python source, can't do anything with this module
        f.close()
        _modules[module] = dict
        return dict

    _modules[module] = dict
    classstack = [] # stack of (class, indent) pairs
    conditionalsstack = [] # stack of indents of conditional defines
    deltastack = []
    deltaindent = 0
    deltaindentcalculated = 0
    src = f.read()
    f.close()

    # To avoid having to stop the regexp at each newline, instead
    # when we need a line number we simply string.count the number of
    # newlines in the string since the last time we did this; i.e.,
    #    lineno = lineno + \
    #             string.count(src, '\n', last_lineno_pos, here)
    #    last_lineno_pos = here
    countnl = string.count
    lineno, last_lineno_pos = 1, 0
    i = 0
    while 1:
        m = _getnext(src, i)
        if not m:
            break
        start, i = m.span()

        if m.start("Method") >= 0:
            # found a method definition or function
            thisindent = _indent(m.group("MethodIndent"))
            meth_name = m.group("MethodName")
            meth_sig = m.group("MethodSignature")
            meth_sig = meth_sig.replace('\\\n', '')
            lineno = lineno + \
                     countnl(src, '\n',
                             last_lineno_pos, start)
            last_lineno_pos = start
            # modify indentation level for conditional defines
            if conditionalsstack:
                if thisindent > conditionalsstack[-1]:
                    if not deltaindentcalculated:
                        deltastack.append(thisindent - conditionalsstack[-1])
                        deltaindent = reduce(lambda x,y: x+y, deltastack)
                        deltaindentcalculated = 1
                    thisindent -= deltaindent
                else:
                    while conditionalsstack and \
                          conditionalsstack[-1] >= thisindent:
                        del conditionalsstack[-1]
                        if deltastack:
                            del deltastack[-1]
                    deltaindentcalculated = 0
            # close all classes indented at least as much
            while classstack and \
                  classstack[-1][1] >= thisindent:
                del classstack[-1]
            if classstack:
                # it's a class method
                cur_class = classstack[-1][0]
                if isinstance(cur_class, Class):
                    # it's a method
                    f = Function(None, meth_name,
                                 file, lineno, meth_sig)
                    cur_class._addmethod(meth_name, f)
                # else it's a nested def
            else:
                # it's a function
                f = Function(module, meth_name,
                             file, lineno, meth_sig)
                if dict.has_key(meth_name):
                    dict_counts[meth_name] += 1
                    meth_name = "%s_%d" % (meth_name, dict_counts[meth_name])
                else:
                    dict_counts[meth_name] = 0
                dict[meth_name] = f
            classstack.append((f, thisindent)) # Marker for nested fns

        elif m.start("String") >= 0:
            pass

        elif m.start("Class") >= 0:
            # we found a class definition
            thisindent = _indent(m.group("ClassIndent"))
            # close all classes indented at least as much
            while classstack and \
                  classstack[-1][1] >= thisindent:
                del classstack[-1]
            lineno = lineno + \
                     countnl(src, '\n', last_lineno_pos, start)
            last_lineno_pos = start
            class_name = m.group("ClassName")
            inherit = m.group("ClassSupers")
            if inherit:
                # the class inherits from other classes
                inherit = inherit[1:-1].strip()
                inherit = _commentsub('', inherit)
                names = []
                for n in inherit.split(','):
                    n = n.strip()
                    if dict.has_key(n):
                        # we know this super class
                        n = dict[n]
                    else:
                        c = n.split('.')
                        if len(c) > 1:
                            # super class
                            # is of the
                            # form module.class:
                            # look in
                            # module for class
                            m = c[-2]
                            c = c[-1]
                            if _modules.has_key(m):
                                d = _modules[m]
                                if d.has_key(c):
                                    n = d[c]
                    names.append(n)
                inherit = names
            # remember this class
            cur_class = Class(module, class_name, inherit,
                              file, lineno)
            if not classstack:
                if dict.has_key(class_name):
                    dict_counts[class_name] += 1
                    class_name = "%s_%d" % (class_name, dict_counts[class_name])
                else:
                    dict_counts[class_name] = 0
                dict[class_name] = cur_class
            else:
                classstack[-1][0]._addclass(class_name, cur_class)
            classstack.append((cur_class, thisindent))

        elif m.start("Attribute") >= 0:
            index = -1
            while index >= -len(classstack):
                if classstack[index][0] is not None and \
                   not isinstance(classstack[index][0], Function):
                    classstack[index][0]._addattribute(m.group("AttributeName"))
                    break
                else:
                    index -= 1

        elif m.start("Import") >= 0:
            # import module
            for n in m.group("ImportList").split(','):
                n = n.strip()
                try:
                    # recursively read the imported module
                    d = readmodule_ex(n, path, inpackage)
                except:
                    pass

        elif m.start("ImportFrom") >= 0:
            # from module import stuff
            mod = m.group("ImportFromPath")
            names = m.group("ImportFromList").split(',')
            try:
                # recursively read the imported module
                d = readmodule_ex(mod, path, inpackage)
            except:
                continue
            # add any classes that were defined in the
            # imported module to our name space if they
            # were mentioned in the list
            for n in names:
                n = n.strip()
                if d.has_key(n):
                    dict[n] = d[n]
                elif n == '*':
                    # only add a name if not
                    # already there (to mimic what
                    # Python does internally)
                    # also don't add names that
                    # start with _
                    for n in d.keys():
                        if n[0] != '_' and \
                           not dict.has_key(n):
                            dict[n] = d[n]
            
        elif m.start("Publics") >= 0:
            idents = m.group("Identifiers")
            lineno = lineno + \
                     countnl(src, '\n', last_lineno_pos, start)
            last_lineno_pos = start
            pubs = Publics(module, file, lineno, idents)
            dict['__all__'] = pubs
        
        elif m.start("ConditionalDefine") >= 0:
            # a conditional function/method definition
            thisindent = _indent(m.group("ConditionalDefineIndent"))
            while conditionalsstack and \
                  conditionalsstack[-1] >= thisindent:
                del conditionalsstack[-1]
                if deltastack:
                    del deltastack[-1]
            conditionalsstack.append(thisindent)
            deltaindentcalculated = 0
        else:
            assert 0, "regexp _getnext found something unexpected"

    return dict

def _indent(ws, _expandtabs=string.expandtabs):
    """
    Module function to return the indentation depth.
    
    @param ws the whitespace to be checked (string)
    @param _expandtabs callable object used to expand tab characters
    @return length of the whitespace string (integer)
    """
    return len(_expandtabs(ws, TABWIDTH))

def find_module(name, path):
    """
    Module function to extend the Python module finding mechanism.
    
    This function searches for files in the given path. If the filename
    doesn't have an extension or an extension of .py, the normal search
    implemented in the imp module is used. For all other supported files
    only path is searched.
    
    @param name filename or modulename to search for (string)
    @param path search path (list of strings)
    @return tuple of the open file, pathname and description. Description
        is a tuple of file suffix, file mode and file type)
    @exception ImportError The file or module wasn't found.
    """
    if name.lower().endswith('.ptl'):
        for p in path:      # only search in path
            if os.path.exists(os.path.join(p, name)):
                pathname = os.path.join(p, name)
                return (open(pathname), pathname, ('.ptl', 'r', PTL_SOURCE))
        raise ImportError
        
    if name.lower().endswith('.py'):
        name = name[:-3]
        
    return imp.find_module(name, path)
