#include <gtk/gtk.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"


static gint
rendgtk_wizard_realize_icon (gpointer user_data)
{
    ENode *node = user_data;

    GtkWidget *window;
    GdkPixmap *icon;

    gchar *filename;
    gchar *fpath;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return (FALSE);

    if (!window->window)
	gtk_widget_realize (window);

    filename = enode_attrib_str (node, "icon", NULL);
    fpath = eutils_file_search (node, filename);

    icon = gdk_pixmap_create_from_xpm (window->window, NULL, NULL, fpath);
    gdk_window_set_icon (window->window, NULL, icon, NULL);
    enode_set_kv (node, "icon-pixmap", icon);
    return (FALSE);
}

static gint
rendgtk_wizard_ondelete_callback (GtkWidget *widget,
				  GdkEvent *event, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;

    function = enode_attrib_str (node, "ondelete", NULL);

    enode_call_ignore_return (node, function, "");

    return (TRUE);
}

static gint
rendgtk_wizard_visible_attr (ENode *node, EBuf *attr, EBuf *value)
{
    rendgtk_widget_idle_visible (node);
    return (TRUE);
}

static gint
rendgtk_wizard_page_visible_attr (ENode *node, EBuf *attr, EBuf *value)
{
    GtkWidget *widget;

    EDEBUG (("Setting visibility of wizard-page to %s\n", value->str));

    widget = enode_get_kv (node, "top-widget");
    if (erend_value_is_true (value)) {
        gtk_widget_show (widget);
    } else {
        gtk_widget_hide (widget);
    }

    return (TRUE);
}


static gint
rendgtk_wizard_set_title_attr (ENode *node, EBuf *attr, EBuf *value)
{
    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return TRUE;

    gtk_window_set_title (GTK_WINDOW (window), value->str);

    /* 2nd arg is wmclass_name, 3rd arg is wmclass_class */
    /* May want to move this into a seperate attr */
    if (!GTK_WIDGET_REALIZED (window))
	gtk_window_set_wmclass (GTK_WINDOW (window), value->str, value->str);

    return (TRUE);
}

static gint
rendgtk_wizard_set_busy_attr (ENode *node, EBuf *attr, EBuf *value)
{
    GtkWidget *window;
    GdkCursor *cursor;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return TRUE;

    EDEBUG (("window-renderer", "setting busy flag on window to %s", value->str));

	
    if (window->window) {
	if (erend_value_is_true (value)) {
	    /* Set to watch */
	    cursor = gdk_cursor_new (150);
	    gdk_window_set_cursor (window->window, cursor);
	    gdk_cursor_destroy (cursor);
	} else {
	    /* Set to old value */
	    cursor = gdk_cursor_new (68);
	    gdk_window_set_cursor (window->window, cursor);
	    gdk_cursor_destroy (cursor);
	}
    
	while (gtk_events_pending ())
	    gtk_main_iteration ();
    
    }

    return (TRUE);
}


static gint
rendgtk_wizard_set_icon (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *window;

    window = enode_get_kv (node, "top-widget");
    if (!window)
	return (TRUE);

    gtk_idle_add (rendgtk_wizard_realize_icon, node);

    return (TRUE);
}

static void
rendgtk_wizard_pack (ENode * parent_node, ENode * child_node)
{
    GtkWidget *window;

    /* Only show window once something gets packed into it, should make it
     * pop up better */
    window = enode_get_kv (parent_node, "top-widget");

    rendgtk_box_pack (parent_node, child_node);
}

static void
rendgtk_wizard_destroy (ENode * node)
{
    GdkPixmap *icon;

    icon = enode_get_kv (node, "icon-pixmap");
    if (icon)
	gdk_pixmap_unref (icon);

    gdk_key_repeat_restore ();

    rendgtk_element_destroy (node);
}

/* Go through and see where we are and what buttons need to 
 * say what, or be insensitive etc. */
static void
update_buttons_status (ENode *wizard)
{
    GSList *children;
    GSList *tmp;
    ENode *node;
    EBuf *current_page;
    GtkWidget *widget;
    int first_page = TRUE;

    EDEBUG (("wizard", "Button sync.."));

    current_page = enode_attrib (wizard, "current-page", NULL);

    children = enode_children (wizard, NULL);
                   
    /* Set this up for the default */
    widget = enode_get_kv (wizard, "next-label-widget");
    gtk_label_set_text (GTK_LABEL (widget), "Next >");
    
    widget = enode_get_kv (wizard, "back-widget");
    gtk_widget_set_sensitive (widget, TRUE);

    widget = enode_get_kv (wizard, "next-button-widget");
    gtk_widget_set_sensitive (widget, TRUE);

    for (tmp = children; tmp; tmp = tmp->next) {
        node = tmp->data;

        if (ebuf_equal_str (enode_type (node), "wizard-page")) {
            if (ebuf_equal_ebuf (enode_attrib (node, "name", NULL), current_page)) {
                if (first_page) {
                    /* This is the first page in the wizard, so we have to set the
                     * back button insensitive */
                    widget = enode_get_kv (wizard, "back-widget");
                    gtk_widget_set_sensitive (widget, FALSE);
                }
                if (!tmp->next) {
                    /* This is the last page, so we have to change the next button
                     * to say finish. */
                    widget = enode_get_kv (wizard, "next-label-widget");
                    gtk_label_set_text (GTK_LABEL (widget), "Finish");
                }
                /* Check to see if the current page has next-sensitive set to false */
                if (ebuf_equal_str (enode_attrib (node, "next-sensitive", NULL), "false")) {
                    EDEBUG (("wizard", "next-sensitive is false, setting.."));
                    widget = enode_get_kv (wizard, "next-button-widget");
                    gtk_widget_set_sensitive (widget, FALSE);
                } else {
                    EDEBUG (("wizard", "next-sensitive is true, ignoring.."));
                }
            }

        }
        first_page = FALSE;
    }

    g_slist_free (children);
}


static void
rendgtk_wizard_cancel_clicked_callback (GtkWidget *widget, gpointer user_data)
{
    ENode *node = user_data;
    gchar *function = NULL;

    function = enode_attrib_str (node, "oncancel", NULL);

    if (function) {
        enode_call_ignore_return (node, function, "");
    }
}

static void
rendgtk_wizard_back_clicked_callback (GtkWidget *widget, gpointer user_data)
{
    GSList *children;
    GSList *tmp;
    ENode *wizard = user_data;
    ENode *node;
    EBuf *current_page;
    ENode *prev = NULL;
    gchar *function = NULL;

    current_page = enode_attrib (wizard, "current-page", NULL);

    children = enode_children (wizard, NULL);
    
    function = enode_attrib_str (wizard, "onback", NULL);
    if (function) {
        enode_call_ignore_return (wizard, function, "");
    }

    for (tmp = children; tmp; tmp = tmp->next) {
        node = tmp->data;

        if (ebuf_equal_str (enode_type (node), "wizard-page")) {
            if (ebuf_equal_ebuf (enode_attrib (node, "name", NULL), current_page)) {
                if (prev) {
                    function = enode_attrib_str (node, "onback", NULL);
                    if (function) {
                        enode_call_ignore_return (node, function, "");
                        goto done;
                    }
                    current_page = ebuf_new_with_ebuf (enode_attrib (prev, "name", NULL));
                    enode_attrib (wizard, "current-page", current_page);
                }
                goto done;
            }
        }
        prev = node;
    }
done:
    g_slist_free (children);
}


static void
rendgtk_wizard_next_clicked_callback (GtkWidget *widget, gpointer user_data)
{
    GSList *children;
    GSList *tmp;
    ENode *wizard = user_data;
    ENode *node;
    EBuf *current_page;
    int grab_next = FALSE;
    gchar *function = NULL;

    current_page = enode_attrib (wizard, "current-page", NULL);

    children = enode_children (wizard, NULL);

    function = enode_attrib_str (wizard, "onnext", NULL);
    if (function) {
        enode_call_ignore_return (wizard, function, "");
    }

    for (tmp = children; tmp; tmp = tmp->next) {
        node = tmp->data;

        if (grab_next) {
            current_page = ebuf_new_with_ebuf (enode_attrib (node, "name", NULL));
            enode_attrib (wizard, "current-page", current_page);
            goto done;
        }
        if (ebuf_equal_str (enode_type (node), "wizard-page")) {
            if (ebuf_equal_ebuf (enode_attrib (node, "name", NULL), current_page)) {
                function = enode_attrib_str (node, "onnext", NULL);
                if (function) {
                    enode_call_ignore_return (node, function, "");
                    goto done;
                }
                grab_next = TRUE;
            }
        }
    }
done:
    g_slist_free (children);
}

static gint
rendgtk_wizard_set_current_page_attr (ENode *node, EBuf *attr, EBuf *value)
{
    ENode *child;
    char *search;
    
    /* So, here we have to find the child wizard.value, and make it visible. */
    search = g_strconcat ("wizard-page.", value->str, NULL);
    child = enode_child (node, search);
    g_free (search);

    EDEBUG (("wizard", "Found child page %p", child));
    
    if (child) {
        ENode *pagenode;
        char *title;
        
        pagenode = enode_get_kv (node, "current-page-node");
        if (pagenode) {
            enode_attrib_str (pagenode, "_visible", "false");
            enode_unref (pagenode);
        }

       
        title = enode_attrib_str (child, "title", NULL);
        if (title) {
            GtkWidget *title_widget;
            title_widget = enode_get_kv (node, "title-widget");
            gtk_label_set_text (GTK_LABEL (title_widget), title);
        }
        enode_attrib_str (child, "_visible", "true");
        enode_set_kv (node, "current-page-node", child);
        enode_ref (child);
    }

    update_buttons_status (node);
    return (TRUE);
}

static gint
rendgtk_wizard_next_sensitive_attr (ENode *node, EBuf *attr, EBuf *value)
{
    ENode *wizard;

    wizard = enode_parent (node, "wizard");
    if (!wizard) {
        return (TRUE);
    }

    update_buttons_status (wizard);
    return (TRUE);
}

static void
rendgtk_wizard_render (ENode * node)
{
    GtkWidget *window;
    GtkWidget *vbox;
    GtkWidget *sep;
    GtkWidget *vbox2;
    GtkWidget *hbox;
    GtkWidget *hbox2;
    GtkWidget *label;
    GtkWidget *button;
    GtkWidget *ebox;
    GtkStyle *style;

    EDEBUG (("wizard", "Creating new wizard"));

    /* Setup the wizard window! */
    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), vbox);
 
    /* For the fancy title stuff at the top */
    vbox2 = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), vbox2, FALSE, FALSE, 0);
    gtk_widget_set_usize (vbox2, 0, 50);

    /* The black on white title strip */
    ebox = gtk_event_box_new ();
    gtk_widget_ensure_style (ebox);
    style = gtk_style_copy (ebox->style);
    style->bg[GTK_STATE_NORMAL] = style->white;
    gtk_widget_set_style (ebox, style);
    gtk_box_pack_start (GTK_BOX (vbox2), ebox, TRUE, TRUE, 0);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (ebox), hbox);
    
    hbox2 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 0);
    gtk_widget_set_usize (hbox2, 30, 0);
    
    label = gtk_label_new ("(No Page Set)");
    enode_set_kv (node, "title-widget", label);
    gtk_widget_ensure_style (label);
    style = gtk_style_copy (label->style);
    style->fg[GTK_STATE_NORMAL] = style->black; 
    gtk_widget_set_style (label, style);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

    sep = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox2), sep, FALSE, FALSE, 0);


    /* The container for child widgets */
    vbox2 = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox2), 20); 
    gtk_box_pack_start (GTK_BOX (vbox), vbox2, TRUE, TRUE, 0);

    /* Seperator */
    sep = gtk_hseparator_new ();
    gtk_box_pack_start (GTK_BOX (vbox), sep, TRUE, TRUE, 0);

    /* Buttons */
    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 10); 
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
    
    hbox2 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), hbox2, TRUE, TRUE, 0);

    button = gtk_button_new_with_label ("< Back");
    enode_set_kv (node, "back-widget", button);
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
    gtk_widget_set_usize (button, 100, 0);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			rendgtk_wizard_back_clicked_callback, node);

    button = gtk_button_new ();
    enode_set_kv (node, "next-button-widget", button);
    label = gtk_label_new ("Next >");
    gtk_container_add (GTK_CONTAINER (button), label);
    enode_set_kv (node, "next-label-widget", label);
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
    gtk_widget_set_usize (button, 100, 0);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			rendgtk_wizard_next_clicked_callback, node);

    hbox2 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 0);
    gtk_widget_set_usize (hbox2, 50, 0);

    button = gtk_button_new_with_label ("Cancel");
    gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 0);
    gtk_widget_set_usize (button, 100, 0);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			rendgtk_wizard_cancel_clicked_callback, node);
    
    hbox2 = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), hbox2, FALSE, FALSE, 0);
    gtk_widget_set_usize (hbox2, 40, 0);

    gtk_widget_show_all (vbox);

    enode_set_kv (node, "top-widget", window);
    enode_set_kv (node, "bottom-widget", vbox2);

    gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			GTK_SIGNAL_FUNC (rendgtk_wizard_ondelete_callback),
			(gpointer) node);

    gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, FALSE);
    enode_attribs_sync (node);
    gtk_widget_show (vbox);
	
    gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);

    /* Delay the display of the window until it's children have had a chance
     * to render. */
    rendgtk_widget_idle_visible (node);
}

static void
rendgtk_wizard_page_render (ENode * node)
{
    GtkWidget *vbox;

    vbox = gtk_vbox_new (FALSE, 0);
    enode_set_kv (node, "top-widget", vbox);
    enode_set_kv (node, "bottom-widget", vbox);

    enode_attribs_sync (node);

    /* We don't set the vbox as visible cause we want that to be done
     * when we switch to the page in question */
}

void
wizard_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    /* <wizard> element */
    element = g_new0 (Element, 1);
    element->render_func = rendgtk_wizard_render;
    element->destroy_func = rendgtk_wizard_destroy;
    element->parent_func = rendgtk_wizard_pack;
    element->tag = "wizard";
    element->description = "Create a new wizard - exists in its own window.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "current-page";
    e_attr->description = "get/set the current page.  Should be the name of the wizard-page node.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_wizard_set_current_page_attr; 
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondelete";
    e_attr->description = "Specify function to call when wizard is 'closed'.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(wizard_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "title";
    e_attr->description = "Set title and wmclass of wizard window.";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_wizard_set_title_attr;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "busy";
    e_attr->description = "Set to true and cursor will change to a watch in this wizard";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_attr_func = rendgtk_wizard_set_busy_attr;
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "icon";
    e_attr->description = "The xpm file to use as the icon";
    e_attr->value_desc = "string";
    e_attr->set_attr_func = rendgtk_wizard_set_icon;
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onback";
    e_attr->description = "Function to notify you when the back button is clicked.";
    e_attr->value_desc = "function";
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onnext";
    e_attr->description = "Function to notify you when the next button is clicked.";
    e_attr->value_desc = "function";
    element_register_attrib (element, e_attr);

    rendgtk_widget_attr_register (element, GTK_TYPE_WINDOW);

    /* Override the 'visible' attribute because windows in gtk are screwed and
     * will not show if you do an immediate hide/show pair */
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "visible";
    e_attr->description = "Toggle if the widget is to be visible on the screen.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_wizard_visible_attr;
    element_register_attrib (element, e_attr);
    
    rendgtk_containerbox_attr_register (element);
    
    /* <wizard-page> element */
    element = g_new0 (Element, 1);
    element->render_func = rendgtk_wizard_page_render;
    element->destroy_func = rendgtk_element_destroy;
    element->parent_func = rendgtk_box_pack;
    element->tag = "wizard-page";
    element->description = "Create a new wizard page within a wizard.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "title";
    e_attr->description = "Specify title of this page of the wizard.  This set the wizards windows title when this page is active.";
    e_attr->value_desc = "string";
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onnext";
    e_attr->description = "Specify the function to be called when the next page button is clicked.  Once specified, this function will be responsible for selecting the next page.";
    e_attr->value_desc = "function";
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "next-sensitive";
    e_attr->description = "Make the next button sensitive or insensitive.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_wizard_next_sensitive_attr;
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onback";
    e_attr->description = "Specify the function to be called when the back button is clicked.  Once specified, this function will be responsible for selecting the previous page.";
    e_attr->value_desc = "function";
    element_register_attrib (element, e_attr);
    
    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "_visible";
    e_attr->description = "Specify if the current page is visible (active). This shouldn't be used directly.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_wizard_page_visible_attr;
    element_register_attrib (element, e_attr);
}


