/**
 * Dictionary abstract data type. This dictionary implementation is a
 * map of strings to integers. The integers are unique in addition to
 * the strings.
 * @author Shaun Jackman <sdj@sfu.ca>
 * @copyright Copyright 2004 Shaun Jackman
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dictionary.h"
#include "util.h"


/** Compares two strings. */
static int
cmp( const void* pa, const void* pb)
{
	const char* a = *(const char**)pa;
	const char* b = *(const char**)pb;
	return strcmp( a, b);
}


/** Sorts the specified dictionary. */
#define sort_dictionary(d) qsort( (d)->entries, \
	(d)->count, sizeof *(d)->entries, cmp);


/** Searches the specified dictionary. */
#define search_dictionary(d,k) \
	bsearch( &(k), (d)->entries, (d)->count, sizeof *(d)->entries, cmp);


/** Prints this dictionary. */
void
print_dictionary( const Dictionary* dictionary)
{
	int i;
	printf( "%d/%d\n", dictionary->count, dictionary->max);
	for( i = 0; i < dictionary->count; i++)
		printf( "%s %d\n",
				dictionary->entries[i].key,
				dictionary->entries[i].id);
}


/** Clears this dictionary. */
void
clear_dictionary( Dictionary* dictionary)
{
	dictionary->max = 1;
	dictionary->count = 0;
	dictionary->entries = allocate_memory(
			sizeof *dictionary->entries);
	dictionary->keys = allocate_memory(
			sizeof *dictionary->keys);
}


/** Destroys this dictionary. */
void
destroy_dictionary( Dictionary* dictionary)
{
	int i;
	for( i = 0; i < dictionary->count; i++)
		free( (char*)dictionary->keys[i]);
	dictionary->max = dictionary->count = 0;
	free( dictionary->entries);
	free( dictionary->keys);
}


/** Allots an entry. */
static Entry*
allot_entry( Dictionary* dictionary)
{
	if( dictionary->count+1 > dictionary->max) {
		dictionary->max *= 2;
		dictionary->entries = reallocate_memory( dictionary->entries,
				dictionary->max * sizeof *dictionary->entries);
		dictionary->keys = reallocate_memory( dictionary->keys,
				dictionary->max * sizeof *dictionary->keys);
	}
	return &dictionary->entries[dictionary->count++];
}


/**
 * Adds the specified key to this dictionary and returns its ID.
 */
static int
add_entry( Dictionary* dictionary, const char* key)
{
	Entry* entry = allot_entry( dictionary);
	int id = dictionary->count;
	entry->key = key;
	entry->id = id;
	dictionary->keys[id-1] = key;
	sort_dictionary( dictionary);
	return dictionary->count;
}


/** Returns the ID of the specified key if found, otherwise zero. */
int
find( const Dictionary* dictionary, const char* key)
{
	Entry* entry = search_dictionary( dictionary, key);
	if( entry == NULL)
		return 0;
	return entry->id;
}


/** Returns the ID of the specified key. If the specified key is
 * not found, it is added to this dictionary and given a new ID. */
int
insert( Dictionary* dictionary, const char* key)
{
	Entry* entry = search_dictionary( dictionary, key);
	return entry != NULL ? entry->id : add_entry( dictionary, key);
}


/** Returns the key of the specified ID. */
const char*
get_key( const Dictionary* dictionary, int id)
{
	return dictionary->keys[id-1];
}
