/*
** Copyright (c) 2002 D. Richard Hipp
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public
** License as published by the Free Software Foundation; either
** version 2 of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
** 
** You should have received a copy of the GNU General Public
** License along with this library; if not, write to the
** Free Software Foundation, Inc., 59 Temple Place - Suite 330,
** Boston, MA  02111-1307, USA.
**
** Author contact information:
**   drh@hwaci.com
**   http://www.hwaci.com/drh/
**
*******************************************************************************
**
** Implementation of the Setup page
*/
#include <assert.h>
#include "config.h"
#include "setup.h"


/*
** Output a single entry for a menu generated using an HTML table.
** If zLink is not NULL or an empty string, then it is the page that
** the menu entry will hyperlink to.  If zLink is NULL or "", then
** the menu entry has no hyperlink - it is disabled.
*/
static void menu_entry(
  const char *zTitle,
  const char *zLink,
  const char *zDesc
){
  @ <tr><td valign="top">
  if( zLink && zLink[0] ){
    @ <a href="%s(zLink)"><b>%s(zTitle)</b></a></td>
  }else{
    @ %s(zTitle)</td>
  }
  @ <td valign="top">
  @ %s(zDesc)
  @ </td></tr>
  @
}

/*
** WEBPAGE: /setup
*/
void setup_page(void){
  /* The user must be at least the administrator in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okAdmin ){
    login_needed();
    return;
  }

  common_standard_menu("setup", 0);
  common_header("Setup Menu");
  @ <table cellpadding="10">
  if( g.okSetup ){
    menu_entry("CVS&nbsp;Repository", "setup_cvs",
      "Identify the CVS repository to which this CVSTrac server is linked.");
    menu_entry("User&nbsp;Database", "setup_user", 
      "Control how CVSTrac interacts with the CVS user and password database "
      "in CVSROOT/passwd");
    menu_entry("Log&nbsp;File", "setup_log",
      "Turn the CVSTrac log file on and off.");
    menu_entry("Attachments", "setup_attach",
      "Set the maximum allowable size for attachments.");
    menu_entry("Bandwidth Throttling", "setup_throttle",
      "Limit the number of hits that will be accepted from a single "
      "IP address in a single hour.");
  }
  menu_entry("Ticket&nbsp;Types", "setup_enum?e=type",
    "Enumerate the different types of tickets that can be entered into "
    "the system.");
  menu_entry("Ticket&nbsp;States", "setup_enum?e=status",
    "Configure the allowed values for the \"status\" attribute of tickets.");
  menu_entry("New&nbsp;Tickets&nbsp;Defaults", "setup_newtkt",
    "Specify the default values assigned to various ticket attributes when "
    "a new ticket is created.");
  menu_entry("Subsystem&nbsp;Names", "setup_enum?e=subsys",
    "List the names of subsystems that can be used in the \"subsystem\" "
    "attribute of tickets.");
  menu_entry("User-Defined&nbsp;Fields", "setup_udef",
    "Create user-defined database columns in the TICKET table");
  if( g.okSetup ){
    menu_entry("Diff&nbsp;Programs", "setup_diff",
      "Specify commands or scripts used to compute the difference between "
      "two versions of a file.");
    menu_entry("Change&nbsp;Notification", "setup_chng",
      "Define an external program to run whenever a ticket is created "
      "or modified.");
    menu_entry("Headers&nbsp;&amp;&nbsp;Footers", "setup_header",
      "Select alternative HTML to use at the top and bottom of every page "
      "generated by CVSTrac.");
    menu_entry("Backup&nbsp;&amp;&nbsp;Restore", "setup_backup",
      "Make a backup copy of the database or restore the database from a "
      "backup copy.");
  }
  @ </table>
  common_footer();
}

/*
** WEBPAGE: /setup_cvs
*/
void setup_cvs_page(void){
  const char *zRoot, *zOldRoot;
  const char *zModule, *zOldModule;

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  /*
  ** The "r" query parameter is the name of the CVS repository root
  ** directory.  Change it if it has changed.
  */
  zOldRoot = db_config("cvsroot","");
  zRoot = P("r");
  if( zRoot && strcmp(zOldRoot,zRoot)!=0 ){
    db_execute("REPLACE INTO config(name,value) VALUES('cvsroot','%q');",
      zRoot);
    zOldRoot = zRoot;
    db_config(0,0);
  }

  /*
  ** The "m" query parameter is the name of the module within the
  ** CVS repository that this CVSTrac instance is suppose to track.
  ** Change it if it has changed.
  */
  zOldModule = db_config("module","");
  zModule = P("m");
  if( zModule && strcmp(zOldModule,zModule)!=0 ){
    db_execute("REPLACE INTO config(name,value) VALUES('module','%q');",
      zModule);
    zOldModule = zModule;
    db_config(0,0);
  }

  /*
  ** The "rrh" query parameter is present if the user presses the
  ** "Reread Revision History" button.  This causes the CVSROOT/history
  ** file to be reread.  Do this with caution as it erases any edits
  ** to the history that are currently in the database.  Only the
  ** setup user can do this.
  */
  if( P("rrh") ){
    common_add_menu_item("setup_cvs", "Cancel");
    common_header("Confirm Reread Of CVSROOT/history");
    @ <h3">WARNING!</h3>
    @ <p>
    @ If have select to <b>Reconstruct</b> the change history database all
    @ of your check-ins will be renumbers.  This might break linkages between
    @ tickets and wiki pages and check-ins.  Any edits you may have made
    @ to check-in messages will be undo as well.</p>
    @
    @ <p> A safer alternative is to select <b>Rescan</b> which will attempt
    @ to preserve existing check-in numbers and check-in message changes.
    @ </p>
    @
    @ <p>In either case, you may want to make a <a href="setup_backup">
    @ backup copy</a> of the database so that you can recover if something
    @ goes wrong.</p>
    @
    @ <form action="%s(g.zPath)" method="POST">
    @ <p>
    @ <input type="submit" name="rrh2" value="Reconstruct">
    @ Reconstruct the check-in database from scratch.
    @ </p>
    @ <p>
    @ <input type="submit" name="rrh3" value="Rescan">
    @ Attempt to reuse existing check-in numbers.
    @ </p>
    @ <p>
    @ <input type="submit" name="cancel" value="Cancel">
    @ Do no do anything.
    @ </p>
    @ </form>
    common_footer();
    return;
  }
  if( P("rrh2") ){
    db_execute(
      "BEGIN;"
      "DELETE FROM chng WHERE not milestone;"
      "DELETE FROM filechng;"
      "DELETE FROM file;"
      "UPDATE config SET value=0 WHERE name='historysize';"
      "COMMIT;"
    );
    db_config(0,0);
    history_update(0);
  }
  if( P("rrh3") ){
    db_execute(
      "BEGIN;"
      "DELETE FROM filechng WHERE rowid NOT IN ("
         "SELECT min(rowid) FROM filechng "
         "GROUP BY filename, vers||'x'"
      ");"
      "DELETE FROM chng WHERE milestone=0 AND cn NOT IN ("
         "SELECT cn FROM filechng"
      ");"
      "UPDATE config SET value=0 WHERE name='historysize';"
      "COMMIT;"
    );
    db_config(0,0);
    history_update(1);
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure CVS Repository");
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ Enter the full pathname of the root directory of the CVS repository
  @ in the space provided below.  If you want to restrict this CVSTrac
  @ server to see only a subset of the files contained in the CVS repository
  @ (for example, if you want CVSTrac to see only one module in a 
  @ repository that contains many unrelated modules) then
  @ enter a pathname prefix for the files you want to see in the
  @ second entry box.</p>
  @ <p><table>
  @ <tr>
  @   <td align="right">CVS repository:</td>
  @   <td><input type="text" name="r" size="40" value="%h(zOldRoot)"></td>
  @ </tr>
  @ <tr>
  @   <td align="right">Module prefix:</td>
  @   <td><input type="text" name="m" size="40" value="%h(zOldModule)"></td>
  @ </tr>
  @ </table><br>
  @ <input type="submit" value="Submit">
  @ </p>
  @
  @ <p>
  @ After changing the CVS repository above, you will generally want to
  @ press the following button to cause the CVSROOT/history file to be
  @ reread from the new repository.  You can also use this button to
  @ resynchornize the database if a prior read of the CVSROOT/history
  @ file failed or if you have manually edited the CVSROOT/history file.</p>
  @ <p><input type="submit" name="rrh" value="Reread CVSROOT/history"></p>
  @ </form>
  @ </p>
  @ <hr>
  common_footer();
}

/*
** WEBPAGE: /setup_user
*/
void setup_user_page(void){
  const char *zWPswd, *zOldWPswd;

  /* The user must be at least the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  /*
  ** The "wpw" query parameter is "yes" if the CVSROOT/passwd file is
  ** writable and "no" if not.  
  ** Change it if it has changed.
  */
  zOldWPswd = db_config("write_cvs_passwd","yes");
  zWPswd = P("wpw");
  if( zWPswd && strcmp(zOldWPswd,zWPswd)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('write_cvs_passwd','%q');",
      zWPswd
    );
    zOldWPswd = zWPswd;
    db_config(0,0);
  }

  /*
  ** Import users out of the CVSROOT/passwd file if the user pressed
  ** the Import Users button.  Only setup can do this.
  */
  if( P("import_users") ){
    user_cvs_read();
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure User Database Linkage");
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ CVSTrac can update the CVSROOT/passwd file with the usernames and
  @ passwords of all CVSTrac users.  Enable or disable this feature
  @ below.</p>
  @ <p>Write User Changes to CVSROOT/passwd?
  cgi_optionmenu(0, "wpw", zOldWPswd, "Yes", "yes", "No", "no", 0);
  @ <input type="submit" value="Submit">
  @ </p>
  @
  @ <p>
  @ Use the following button to automatically create a CVSTrac user ID
  @ for every user currently named in CVSROOT/passwd.  The new users will
  @ be given the same access permissions as user "anonymous" plus check-out
  @ permission and check-in permission if CVS allows the user to write.</p>
  @ <p><input type="submit" name="import_users" value="Import CVS Users"></p>
  @ </form>
  @ </p>
  common_footer();
}

/*
** WEBPAGE: /setup_log
*/
void setup_logfile_page(void){
  const char *zLog, *zOldLog;

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  /* 
  ** The "log" query parameter specifies a log file into which a record
  ** of all HTTP hits is written.  Write this value if this has changed.
  ** Only setup can make this change.
  */
  zOldLog = db_config("logfile","");
  zLog = P("log");
  if( zLog && strcmp(zOldLog,zLog)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('logfile','%q');",
      zLog
    );
    zOldLog = zLog;
    db_config(0,0);
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure Log File");
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ Enter the name of file into which is written a log of all accesses
  @ to this CVSTrac server.  Leave the entry blank to disable logging:
  @ </p>
  @ <p>Log File: <input type="text" name="log" size="40" value="%h(zOldLog)">
  @ <input type="submit" value="Submit">
  @ </form>
  @ </p>
  common_footer();
}

/*
** WEBPAGE: /setup_newtkt
*/
void setup_newticket_page(void){
  char **azResult;
  const char *zState, *zOldState;
  const char *zAsgnto, *zOldAsgnto;
  const char *zType, *zOldType;
  const char *zPri, *zOldPri;
  const char *zSev, *zOldSev;

  /* The user must be at least the administrator in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okAdmin ){
    login_needed();
    return;
  }

  /*
  ** The "asgnto" query parameter specifies a userid who is assigned to
  ** all new tickets.  Record this value in the configuration table if
  ** it has changed.
  */
  zOldAsgnto = db_config("assignto","");
  zAsgnto = P("asgnto");
  if( zAsgnto && strcmp(zOldAsgnto,zAsgnto)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('assignto','%q');", zAsgnto
    );
    zOldAsgnto = zAsgnto;
    db_config(0,0);
  }

  /*
  ** The "istate" query parameter specifies the initial state for new
  ** tickets.  Record any changes to this value.
  */
  zOldState = db_config("initial_state","");
  zState = P("istate");
  if( zState && strcmp(zOldState,zState)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('initial_state','%q');",
      zState
    );
    zOldState = zState;
    db_config(0,0);
  }

  /*
  ** The "type" query parameter specifies the initial type for new
  ** tickets.  Record any changes to this value.
  */
  zOldType = db_config("dflt_tkt_type","code");
  zType = P("type");
  if( zType && strcmp(zOldType,zType)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('dflt_tkt_type','%q');",
      zType
    );
    zOldType = zType;
    db_config(0,0);
  }

  /*
  ** The "pri" query parameter specifies the initial priority for new
  ** tickets.  Record any changes to this value.
  */
  zOldPri = db_config("dflt_priority","1");
  zPri = P("pri");
  if( zPri && strcmp(zOldPri,zPri)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('dflt_priority','%q');",
      zPri
    );
    zOldPri = zPri;
    db_config(0,0);
  }

  /*
  ** The "sev" query parameter specifies the initial severity for new
  ** tickets.  Record any changes to this value.
  */
  zOldSev = db_config("dflt_severity","1");
  zSev = P("sev");
  if( zSev && strcmp(zOldSev,zSev)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('dflt_severity','%q');",
      zSev
    );
    zOldSev = zSev;
    db_config(0,0);
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure New Ticket Defaults");
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ Select a user to whom new tickets will be assigned by default:</p><p>
  @ Assigned To:
  azResult = db_query("SELECT id FROM user UNION SELECT '' ORDER BY id");
  cgi_v_optionmenu(0, "asgnto", zOldAsgnto, (const char**)azResult);
  @ </p>
  @
  @ <p>
  @ Select the initial state that new tickets are created in:</p><p>
  @ Initial State:
  cgi_v_optionmenu2(0, "istate", zOldState, (const char**)db_query(
     "SELECT name, value FROM enums WHERE type='status'"));
  @ </p>
  @
  @ <p>
  @ Select the default type for new tickets:</p><p>
  @ Default Type:
  cgi_v_optionmenu2(0, "type", zOldType, (const char**)db_query(
     "SELECT name, value FROM enums WHERE type='type'"));
  @ </p>
  @
  @ <p>
  @ Select the default priority for new tickets:</p><p>
  @ Default Priority:
  cgi_optionmenu(0, "pri", zOldPri, "1", "1", "2", "2", "3", "3", "4", "4",
      "5", "5", (char*)0);
  @ </p>
  @
  @ <p>
  @ Select the default severity for new tickets:</p><p>
  @ Default Severity:
  cgi_optionmenu(0, "sev", zOldSev, "1", "1", "2", "2", "3", "3", "4", "4",
      "5", "5", (char*)0);
  @ </p>
  @
  @ <p>
  @ <input type="submit" value="Submit">
  @ </form>
  @ </p>
  common_footer();
}

/*
** Generate a string suitable for inserting into a <TEXTAREA> that
** describes all allowed values for a particular enumeration.
*/
static char *enum_to_string(const char *zEnum){
  char **az;
  char *zResult;
  int i, j, nByte;
  int len1, len2, len3;
  int mx1, mx2, mx3;
  int rowCnt;
  az = db_query("SELECT name, value, color FROM enums "
                "WHERE type='%s' ORDER BY idx", zEnum);
  rowCnt = mx1 = mx2 = mx3 = 0;
  for(i=0; az[i]; i+=3){
    len1 = strlen(az[i]);
    len2 = strlen(az[i+1]);
    len3 = strlen(az[i+2]);
    if( len1>mx1 ) mx1 = len1;
    if( len2>mx2 ) mx2 = len2;
    if( len3>mx3 ) mx3 = len3;
    rowCnt++;
  }
  if( mx2<mx1 ) mx2 = mx1;
  nByte = (mx1 + mx2 + mx3 + 11)*rowCnt + 1;
  zResult = malloc( nByte );
  if( zResult==0 ) exit(1);
  for(i=j=0; az[i]; i+=3){
    const char *z1 = az[i];
    const char *z2 = az[i+1];
    const char *z3 = az[i+2];
    if( z1[0]==0 ){ z1 = "?"; }
    if( z2[0]==0 ){ z2 = z1; }
    if( z3[0] ){
      sprintf(&zResult[j], "%*s    %*s   (%s)\n", -mx1, z1, -mx2, z2, z3);
    }else{
      sprintf(&zResult[j], "%*s    %s\n", -mx1, z1, z2);
    }
    j += strlen(&zResult[j]);
  }
  db_query_free(az);
  zResult[j] = 0;
  return zResult;
}

/*
** Given text that describes an enumeration, fill the ENUMS table with
** coresponding entries.
**
** The text line oriented.  Each line represents a single value for
** the enum.  The first token on the line is the internal name.
** subsequent tokens are the human-readable description.  If the last
** token is in parentheses, then it is a color for the entry.
*/
static void string_to_enum(const char *zEnum, const char *z){
  int i, j, n;
  int cnt = 1;
  char *zColor;
  char zName[50];
  char zDesc[200];

  db_execute("DELETE FROM enums WHERE type='%s'", zEnum);
  while( isspace(*z) ){ z++; }
  while( *z ){
    assert( !isspace(*z) );
    for(i=1; z[i] && !isspace(z[i]); i++){}
    n = i>49 ? 49 : i;
    memcpy(zName, z, n);
    zName[n] = 0;
    z += i;
    while( *z!='\n' && isspace(z[1]) ){ z++; }
    if( *z=='\n' || *z==0 ){
      strcpy(zDesc, zName);
      zColor = "";
    }else{
      int lastP1 = -1;
      int lastP2 = -1;
      z++;
      for(j=i=0; *z && *z!='\n'; z++){
        if( j<199 && (j==0 || !isspace(*z) || !isspace(zDesc[j-1])) ){
          zDesc[j++] = *z;
        }
        if( *z=='(' ){ lastP1 = j-1; }
        else if( *z==')' ){ lastP2 = j-1; }
        else if( !isspace(*z) ){ lastP2 = -1; }
      }
      zDesc[j] = 0;
      if( lastP2>lastP1 && lastP1>1 ){
        zColor = &zDesc[lastP1+1];
        zDesc[lastP2] = 0;
        j = lastP1;
        while( j>0 && isspace(zDesc[j-1]) ){ j--; }
        zDesc[j] = 0;
      }else{
        zColor = "";
      }
    }
    db_execute(
       "INSERT INTO enums(type,idx,name,value,color) "
       "VALUES('%s',%d,'%q','%q','%q')",
       zEnum, cnt++, zName, zDesc, zColor
    );
    while( isspace(*z) ) z++;
  }
}

/*
** WEBPAGE: /setup_enum
*/
void setup_enum_page(void){
  char *zText;
  const char *zEnum;
  int nRow;
  const char *zTitle;
  const char *zName;

  /* The user must be at least the administrator in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okAdmin ){
    login_needed();
    return;
  }

  /* What type of enumeration are we entering.
  */
  zEnum = P("e");
  if( zEnum==0 ){ zEnum = "subsys"; }
  if( strcmp(zEnum,"subsys")==0 ){
    zTitle = "Configure Subsystem Names";
    zName = "subsystem";
    nRow = 20;
  }else
  if( strcmp(zEnum,"type")==0 ){
    zTitle = "Configure Ticket Types";
    zName = "type";
    nRow = 6;
  }else
  if( strcmp(zEnum,"status")==0 ){
    zTitle = "Configure Ticket States";
    zName = "status";
    nRow = 10;
  }else
  {
    common_add_menu_item("setup", "Back");
    common_header("Unknown Enumeration");
    @ <p>URL error:  The "e" query parameter specifies an unknown
    @ enumeration type: "%h(zEnum)".</p>
    @
    @ <p>Press the "Back" link above to return to the setup menu.</p>
    common_footer();
    return;
  }

  /*
  ** The "s" query parameter is a long text string that specifies
  ** the names of all subsystems.  If any subsystem names have been
  ** added or removed, then make appropriate changes to the subsyst
  ** table in the database.
  */
  if( P("x") ){
    db_execute("BEGIN");
    string_to_enum(zEnum, P("x"));
    db_execute("COMMIT");
  }

  /* Genenerate the page.
  */
  common_add_menu_item("setup", "Main Setup Menu");
  common_header(zTitle);
  zText = enum_to_string(zEnum);
  @ <p>
  @ The allowed values of the "%s(zName)" attribute of tickets
  @ are listed below.
  @ You may edit this text and press apply to change the set of
  @ allowed values.
  @ </p>
  @
  @ <p>
  @ The token on the left is the value as it is stored in the database.
  @ The text that follows is a human-readable description for the meaning
  @ of the token.  A color name for use in reports may optionally appear
  @ in parentheses after the description.
  @ </p>
  @
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ <input type="hidden" name="e" value="%s(zEnum)">
  @ <textarea cols=60 rows=%d(nRow) name="x">%h(zText)</textarea>
  @ <p><input type="submit" value="Submit"></p>
  @ </form>
  common_footer();
}

/*
** WEBPAGE: /setup_udef
*/
void setup_udef_page(void){
  int idx, i;
  const char *zName;
  const char *zText;

  /* The user must be at least the administrator in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okAdmin ){
    login_needed();
    return;
  }

  /* Write back results if requested.
  */
  idx = atoi(PD("idx","0"));
  zName = P("n");
  zText = P("x");
  if( idx>=1 && idx<=5 && zName && zText ){
    char zEnum[20];
    char *zName2 = trim_string(zName);
    char *zDesc2 = trim_string(PD("d",""));
    sprintf(zEnum,"extra%d", idx);
    db_execute("BEGIN");
    if( zName2[0] ){
      string_to_enum(zEnum, zText);
      db_execute(
        "REPLACE INTO config(name,value) VALUES('%s_name','%q');",
        zEnum, zName2
      );
      db_execute(
        "REPLACE INTO config(name,value) VALUES('%s_desc','%q');",
        zEnum, zDesc2
      );
    }else{
      db_execute("DELETE FROM config WHERE name='%s_name'", zEnum);
      db_execute("DELETE FROM config WHERE name='%s_desc'", zEnum);
      db_execute("DELETE FROM enums WHERE type='%s'", zEnum);
    }
    db_execute("COMMIT");
    db_config(0,0);
  }

  /* Genenerate the page.
  */
  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure User-Defined Fields");
  @ <p>
  @ Five extra columns named "extra1" through "extra5" exist in the 
  @ TICKET table of the database.  These columns can be defined for
  @ application specific use using this configuration page.
  @ </p>
  @
  @ <p>
  @ Each column is controlled by a separate form below.  The column will
  @ be displayed on ticket reports if and only if its has a non-blank
  @ display name.  User's see the column as its display name, not as
  @ "extra1".
  @ </p>
  @
  @ <p>
  @ Allowed values for the column can be specified in the text box.
  @ The same format is used here
  @ as when specifying <a href="setup_enum?e=type">ticket types</a>,
  @ <a href="setup_enum?e=status">ticket states</a> and
  @ <a href="setup_enum?e=subsys">subsystem names</a>.
  @ There is one allowed value per line.  
  @ The token on the left is the value as it is stored in the database.
  @ The text that follows is a human-readable description for the meaning
  @ of the token.  A color name for use in reports may optionally appear
  @ in parentheses after the description.
  @ </p>
  @
  @ <p>
  @ The Allowed Values box may be left blank.
  @ If allowed values are defined for the column, then users will restricted
  @ to the values specified when changing the value of the column.
  @ If no allowed values are defined, then the column can be set to
  @ arbitrary text.
  @ </p>
  @
  @ <p>
  @ The Description box may be left blank. 
  @ If a description is provided, then this field may be entered on the
  @ new ticket page.  If no description is given, this field can be modified 
  @ on the edit ticket page but will not appear on the new ticket page.
  @ </p>
  for(i=0; i<5; i++){
    const char *zOld;
    char *zAllowed;
    const char *zDesc;
    char zEnumName[30];
    sprintf(zEnumName,"extra%d_name",i+1);
    zOld = db_config(zEnumName,"");
    zEnumName[6] = 0;
    zAllowed = enum_to_string(zEnumName);
    sprintf(zEnumName,"extra%d_desc",i+1);
    zDesc = db_config(zEnumName,"");
    @ <hr>
    @ <h3>Database column "extra%d(i+1)":</h3>
    @ <form action="%s(g.zPath)" method="GET">
    @ <input type="hidden" name="idx" value="%d(i+1)">
    @ Display Name:
    @ <input type="text" name="n" value="%h(zOld)"><br>
    @ Allowed Values: (<i>Name Desc Color</i> - omit for free text)<br>
    @ <textarea cols=60 rows=15 name="x">%h(zAllowed)</textarea><br>
    @ Description: (HTML - Leave blank to omit from new-ticket page)<br>
    @ <textarea cols=60 rows=5 name="d">%h(zDesc)</textarea><br>
    @ <input type="submit" value="Submit">
    @ </form>
  }
  common_footer();
}

/*
** WEBPAGE: /setup_chng
*/
void setup_chng_page(void){
  const char *zNotify, *zOldNotify;

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  /*
  ** The "notify" query parameter is the name of a program or script that
  ** is run whenever a ticket is created or modified.  Modify the notify
  ** value if it has changed.  Only setup can do this.
  */
  zOldNotify = db_config("notify","");
  zNotify = P("notify");
  if( zNotify && strcmp(zOldNotify,zNotify)!=0 ){
    db_execute(
      "REPLACE INTO config(name,value) VALUES('notify','%q');",
      zNotify
    );
    zOldNotify = zNotify;
    db_config(0,0);
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure Ticket Change Notification");
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ Enter a shell command to run whenever a ticket is
  @ created or modified.  The following substitutions are made
  @ on the string before it is passed to /bin/sh:
  @
  @ <table border=1 cellspacing=0 cellpadding=5 align="right" width="45%%">
  @ <tr><td bgcolor="#e0c0c0">
  @ <big><b>Important Security Note</b></big>
  @
  @ <p>Be sure to enclose all text substitutions in single-quotes.
  @ (ex <tt>'%%d'</tt>)  Otherwise, a user could cause arbitrary shell
  @ commands to be run on your system.</p>
  @  
  @ <p>Text is stripped of all single-quotes and backslashs before it is
  @ substituted, so if the substitution is itself enclosed in single-quotes,
  @ it will always be treated as a single token by the shell.</p>
  @
  @ <p>For best security, use only the <b>%%n</b> substitution and have
  @ a Tcl or Perl script extract other fields directly from the database.</p>
  @ </td></tr></table>
  @
  @ <blockquote>
  @ <table cellspacing="5" cellpadding="0">
  @ <tr><td width="40"><b>%%a</b></td>
  @     <td>UserID of the person the ticket is assigned to</td></tr>
  @ <tr><td><b>%%A</b></td><td>E-mail address of person assigned to</td></tr>
  @ <tr><td><b>%%c</b></td><td>Contact information for the originator</td></tr>
  @ <tr><td><b>%%d</b></td><td>The description</td></tr>
  @ <tr><td><b>%%n</b></td><td>The ticket number</td></tr>
  @ <tr><td><b>%%p</b></td><td>The project name</td></tr>
  @ <tr><td><b>%%r</b></td><td>The remarks section</td></tr>
  @ <tr><td><b>%%s</b></td><td>The status of the ticket</td></tr>
  @ <tr><td><b>%%t</b></td><td>The title of the ticket</td></tr>
  @ <tr><td><b>%%u</b></td>
  @     <td>UserID of the person who made this change</td></tr>
  @ <tr><td><b>%%w</b></td><td>UserID of the originator of the ticket</td></tr>
  @ <tr><td><b>%%y</b></td><td>Type of ticket</td></tr>
  @ <tr><td><b>%%%%</b></td><td>The literal character "<b>%%</b>"</td></tr>
  @ </table>
  @ </blockquote>
  @
  @ <input type="text" name="notify" size="70" value="%h(zOldNotify)">
  @ <input type="submit" value="Submit">
  @ </form>
  @ </p>
  common_footer();
}

/*
** WEBPAGE: /setup_diff
*/
void setup_diff_page(void){
  const char *zDiff, *zOldDiff;
  const char *zList, *zOldList;

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  /*
  ** The "diff" query parameter is the name of a program or script that
  ** is run whenever a ticket is created or modified.  Modify the filediff
  ** value if it has changed.  Only setup can do this.
  */
  zOldDiff = db_config("filediff","");
  zDiff = P("diff");
  if( zDiff && strcmp(zOldDiff,zDiff)!=0 ){
    if( zDiff[0] ){
      db_execute(
        "REPLACE INTO config(name,value) VALUES('filediff','%q');",
        zDiff
      );
    }else{
      db_execute("DELETE FROM config WHERE name='filediff'");
    }
    zOldDiff = zDiff;
    db_config(0,0);
  }

  /*
  ** The "list" query parameter is the name of a program or script that
  ** is run whenever a ticket is created or modified.  Modify the filelist
  ** value if it has changed.  Only setup can do this.
  */
  zOldList = db_config("filelist","");
  zList = P("list");
  if( zList && strcmp(zOldList,zList)!=0 ){
    if( zList[0] ){
      db_execute(
        "REPLACE INTO config(name,value) VALUES('filelist','%q');",
        zList
      );
    }else{
      db_execute("DELETE FROM config WHERE name='filelist'");
    }
    zOldList = zList;
    db_config(0,0);
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure Source Code Diff Program");
  @ <form action="%s(g.zPath)" method="POST">
  @ <p>Enter a shell command to run in order to compute the difference between
  @ two versions of the same RCS file.  The output can be either plain text
  @ or HTML.  If HTML, then the first non-whitespace character of output
  @ should be a "<".  Otherwise the output will be assumed to be plain text.</p>
  @
  @ <table border=1 cellspacing=0 cellpadding=5 align="right" width="33%%">
  @ <tr><td bgcolor="#e0c0c0">
  @ <big><b>Important Security Note</b></big>
  @
  @ <p>Be sure to enclose the substitutions in single-quotes.
  @ (examples: <tt>'%%F'</tt> or <tt>'%%V2'</tt>)
  @ Otherwise, a user who can check in new files
  @ (with unusual names) can cause arbitrary shell
  @ commands to be run on your system.</p>
  @  
  @ <p>CVSTrac will not attempt to diff a file whose name contains a
  @ single-quote or backslash
  @ so if the substitution is itself enclosed in single-quotes, it will always
  @ be treated as a single token by the shell.</p>
  @ </td></tr></table>
  @
  @ <p>The following substitutions are made prior to executing the program:</p>
  @
  @ <blockquote>
  @ <table cellspacing="5" cellpadding="0">
  @ <tr><td width="40" valign="top"><b>%%F</b></td>
  @     <td>The name of the RCS file to be diffed.  This is a full
  @         pathname including the "<b>,v</b>" suffix.</td></tr>
  @ <tr><td><b>%%V1</b></td><td>The first version of be diffed</td></tr>
  @ <tr><td><b>%%V2</b></td><td>The second version to be diffed</td></tr>
  @ <tr><td><b>%%%%</b></td><td>The literal character "<b>%%</b>"</td></tr>
  @ </table>
  @ </blockquote>
  @
  @ <input type="text" name="diff" size="70" value="%h(zOldDiff)">
  @ <input type="submit" value="Submit">
  @
  @ <p>If you leave the above entry blank, the following command is used:</p>
  @
  @ <blockquote><pre>
  @ rcsdiff -q -r'%%V1' -r'%%V2' -u '%%F'
  @ </pre></blockquote>
  @ </form>
  @ </p>
  @ <hr>
  @ <form action="%s(g.zPath)" method="POST">
  @ <p>Enter below a shell command to run in order to list the content
  @ of a single version of an RCS file.  The output can be either plain text
  @ or HTML.  If HTML, then the first non-whitespace character of output
  @ should be a "<".  Otherwise the output will be assumed to be plain text.</p>
  @
  @ <p>This command is used on the "chngview" page to show the content 
  @ of files that are newly added to the CVS repository.</p>
  @
  @ <p>The following substitutions are made prior to executing the program:</p>
  @
  @ <blockquote>
  @ <table cellspacing="5" cellpadding="0">
  @ <tr><td width="40" valign="top"><b>%%F</b></td>
  @     <td>The name of the RCS file to be diffed.  This is a full
  @         pathname including the "<b>,v</b>" suffix.</td></tr>
  @ <tr><td><b>%%V</b></td><td>The version to be listed</td></tr>
  @ <tr><td><b>%%%%</b></td><td>The literal character "<b>%%</b>"</td></tr>
  @ </table>
  @ </blockquote>
  @
  @ <input type="text" name="list" size="70" value="%h(zOldList)">
  @ <input type="submit" value="Submit">
  @
  @ <p>If you leave the above entry blank, the following command is used:</p>
  @
  @ <blockquote><pre>
  @ co -q -p'%%V' '%%F' | diff -c /dev/null -
  @ </pre></blockquote>
  @ </form>
  @ </p>
  common_footer();
}

/*
** WEBPAGE: /setup_header
*/
void setup_header_page(void){
  const char *zHeader, *zFooter;

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  /*
  ** If both "header" and "footer" query parameters are present, then
  ** change the header and footer to the values of those parameters.
  ** Only the setup user can do this.
  */
  if( P("header") && P("footer") ){
    db_execute(
      "REPLACE INTO config VALUES('header','%q');"
      "REPLACE INTO config VALUES('footer','%q');",
       trim_string(P("header")),
       trim_string(P("footer"))
    );
    db_config(0,0);
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Configure Headers & Footers");
  @ <p>
  @ Enter HTML used for the header and footer of every page.
  @ If you leave these entries blank, default headers and/or footers
  @ are used.  If you enter a filename (beginning with a "/" character)
  @ instead of HTML text, then the
  @ file is read at runtime and used for the header or footer.</p>
  @
  @ <p>Substitutions are made within the header and footer text.  These
  @ substitutions are made to the HTML regardless of whether the HTML
  @ is entered directly below or is read from a file.</p>
  @
  @ <blockquote>
  @ <table>
  @ <tr><td width="40"><b>%%N</b></td><td>The name of the project</td></tr>
  @ <tr><td><b>%%T</b></td><td>The title of the current page</td></tr>
  @ <tr><td><b>%%V</b></td><td>The version number of CVSTrac</td></tr>
  @ <tr><td><b>%%%%</b></td><td>The literal character "<b>%%</b>"</td></tr>
  @ </table>
  @ </blockquote>
  @
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  zHeader = db_config("header","");
  zFooter = db_config("footer","");
  @ Header:<br>
  @ <textarea cols=80 rows=8 name="header">%h(zHeader)</textarea><br>
  @ Footer:<br>
  @ <textarea cols=80 rows=8 name="footer">%h(zFooter)</textarea><br>
  @ <input type="submit" value="Submit">
  @ </form>
  @ </p>
  common_footer();
}

/*
** Make a copy of file zFrom into file zTo.  If any errors occur,
** return a pointer to a string describing the error.
*/
static const char *file_copy(const char *zFrom, const char *zTo){
  FILE *pIn, *pOut;
  int n;
  long long int total = 0;
  char zBuf[10240];
  pIn = fopen(zFrom, "r");
  if( pIn==0 ){
    return mprintf(
      "Unable to copy files - cannot open \"%h\" for reading.", zFrom
    );
  }
  unlink(zTo);
  pOut = fopen(zTo, "w");
  if( pOut==0 ){
    fclose(pIn);
    return mprintf(
      "Unable to copy files - cannot open \"%h\" for writing.", zTo
    );
  }
  while( (n = fread(zBuf, 1, sizeof(zBuf), pIn))>0 ){
    if( fwrite(zBuf, 1, n, pOut)<n ){
      fclose(pIn);
      fclose(pOut);
      return mprintf(
        "Copy operation failed after %lld bytes.  Is the disk full?", total
      );
    }
    total += n;
  }
  fclose(pIn);
  fclose(pOut);
  return 0;
}

/*
** WEBPAGE: /setup_attach
*/
void setup_attachment_page(void){
  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  if( P("sz") ){
    int sz = atoi(P("sz"))*1024;
    db_execute("REPLACE INTO config VALUES('max_attach_size',%d)", sz);
    db_config(0, 0);
    cgi_redirect("setup");
  }
 
  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Set Maximum Attachment Size");
  @ <p>
  @ Enter the maximum attachment size below.  If you enter a size of
  @ zero, attachments are disallowed.  The SQLite database is unable
  @ to handle attachments larger than about 1000KB so do not enter a
  @ maximum size larger than that amount.
  @ </p>
  @
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ Maximum attachment size in kilobytes: 
  @ <input type="text" name="sz" value="%d(attachment_max()/1024)" size=5>
  @ <input type="submit" value="Set">
  @ </form>
  @ </p>
  common_footer();
}

/*
** WEBPAGE: /setup_throttle
*/
void setup_throttle_page(void){
  int mxHit = atoi(db_config("throttle","0"));

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }
  

  if( P("sz") ){
    int sz = atoi(P("sz"));
    db_execute("REPLACE INTO config VALUES('throttle',%d)", sz);
    db_config(0, 0);
    cgi_redirect("setup");
  }
 
  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Set Maximum Anonymous Hits Per Hour");
  @ <p>
  @ Enter the limit on the number of anonymous accesses from the same
  @ IP address that can occur within one hour.  Enter zero to disable
  @ the limiter.
  @ </p>
  @
  @ <p>
  @ <form action="%s(g.zPath)" method="POST">
  @ Maximum hits per hour: 
  @ <input type="text" name="sz" value="%d(mxHit)" size=5>
  @ <input type="submit" value="Set">
  @ </form>
  @ </p>
  @
  @ <p>
  @ The limiter works by maintain a table in the database (the ACCESS_LOAD
  @ table) that records the time of last access and a "load" for each
  @ IP address.  The load reduces exponentially with a half-life of
  @ one hour.  Each new access increases the load by 1.0.  When the
  @ load exceeds the threshold above, the load automatically doubles and
  @ access is denied.
  @ </p>
  @
  @ <p>
  @ Any attempt to access the page named "honeypot" automatically increases
  @ the load to twice the threshold.  There a hidden hyperlinks to the
  @ honeypot on every page.  The idea is to trick spiders into visiting
  @ this honeypot so that they can have their access terminated quickly.
  @ </p>
  @
  @ <p>
  @ The limiter the honeypot only work for users that are not
  @ logged in - anonymous users.  Users who are logged in can visit
  @ any page (including the honeypot) as often as they want and will
  @ never be denied access.
  @ </p>
  @
  @ <p>A summary of the <a href="info_throttle">Access Log</a> is available
  @ separately.</p>
  common_footer();
}

/*
** WEBPAGE: /setup_backup
*/
void setup_backup_page(void){
  char *zDbName = mprintf("%s.db", g.zName);
  char *zBuName = mprintf("%s.db.bu", g.zName);
  const char *zMsg = 0;

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  if( P("bkup") ){
    db_execute("BEGIN");
    zMsg = file_copy(zDbName, zBuName);
    db_execute("COMMIT");
  }else if( P("rstr") ){
    db_execute("BEGIN");
    zMsg = file_copy(zBuName, zDbName);
    db_execute("COMMIT");
  }
 
  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Backup The Database");
  if( zMsg ){
    @ <p><font color="#ff0000">%s(zMsg)</font></p>
  }
  @ <p>
  @ Use the buttons below to make a safe (atomic) backup or restore
  @ of the database file.   The original database is in the file
  @ named <b>%h(zDbName)</b> and the backup is in 
  @ <b>%h(zBuName)</b>.
  @ </p>
  @
  @ <p>
  @ It is always safe to do a backup.  The worst that can happen is that
  @ you can overwrite a prior backup.  But a restore can destroy your
  @ database if the backup copy you are restoring from is incorrect.
  @ Use caution when doing a restore.
  @ </p>
  @
  @ <form action="%s(g.zPath)" method="POST">
  @ <p><input type="submit" name="bkup" value="Backup"></p>
  @ <p><input type="submit" name="rstr" value="Restore"></p>
  @ </form>
  common_footer();
}

#if 0  /* TO DO */
/*
** WEB-PAGE: /setup_repair
*/
void setup_repair_page(void){

  /* The user must be the setup user in order to see
  ** this screen.
  */
  login_check_credentials();
  if( !g.okSetup ){
    cgi_redirect("setup");
    return;
  }

  /* 
  ** Change a check-in number.
  */
  cnfrom = atoi(PD("cnfrom"),"0");
  cnto = atoi(PD("cnto"),"0");
  if( cnfrom>0 && cnto>0 && cnfrom!=cnto ){
    const char *zOld;
    zOld = db_short_query(
       "SELECT rowid FROM chng WHERE cn=%d", cnfrom
    );
    if( zOld || zOld[0] ){
      db_execute(
        "BEGIN;"
        "DELETE FROM chng WHERE cn=%d;"
        "UPDATE chng SET cn=%d WHERE cn=%d;"
        "UPDATE filechng SET cn=%d WHERE cn=%d;"
        "UPDATE xref SET cn=%d WHERE cn=%d;"
        "COMMIT;",
        cnto,
        cnto, cnfrom,
        cnto, cnfrom,
        cnto, cnfrom
      );
    }
  }

  /*
  ** Remove duplicate entries in the FILECHNG table.  Remove check-ins
  ** from the CHNG table that have no corresponding FILECHNG entries.
  */
  if( P("rmdup") ){
    db_execute(
      "BEGIN;"
      "DELETE FROM filechng WHERE rowid NOT IN ("
         "SELECT min(rowid) FROM filechng "
         "GROUP BY filename, vers||'x'"
      ");"
      "DELETE FROM chng WHERE milestone=0 AND cn NOT IN ("
         "SELECT cn FROM filechng"
      ");"
      "COMMIT;"
    );
  }

  common_add_menu_item("setup", "Main Setup Menu");
  common_header("Repair The Database");
  @ <p>
  @ You can use this page to repair damage to a database that was caused
  @ when the CVSROOT/history file was read incorrectly.  The problem may
  @ have resulted from corruption in the CVS repository or a system 
  @ malfunction or from a bug in CVSTrac.  (All known bugs of this kind have
  @ been fixed but you never know when a new one might appear.)
  @ </p>
  @
  @ 
  @ </p>
  common_footer();
}
#endif
