;;;
;;; GLTEST0.LSP
;;;
;;; Translation of "gltest0.c" by Brian Paul
;;;

(defun redraw (display window)
 (format t "Redrawing~%")
 (gl:glClear gl:GL_COLOR_BUFFER_BIT)
 (gl:glIndexi 1) ; does what?
 (gl:glBegin gl:GL_LINES)
 (gl:glVertex3f 0.0 0.0 0.0)
 (gl:glVertex3f 1.0 1.0 0.0)
 (gl:glEnd)
 (gl:glBegin gl:GL_LINES)
 (gl:glVertex3f 0.0 0.0 0.0)
 (gl:glVertex3f 1.0 0.0 0.0)
 (gl:glEnd)
 (gl:glBegin gl:GL_LINES)
 (gl:glVertex3f 0.0 0.0 0.0)
 (gl:glVertex3f 0.0 0.5 0.0)
 (gl:glEnd)
 (gl:glFlush)
 (gl:glxswapbuffers display window))

;;; always keeps 2d geometry fullsize in window
(defun resize (width height)
 (format t "Resizing~%")
 (gl:glviewport 0 0 width height)
 (gl:glmatrixmode gl:GL_PROJECTION)
 (gl:glloadidentity)
 (gl:glortho -1.0 1.0 -1.0 1.0 -1.0 1.0)
 (gl:glmatrixmode gl:GL_MODELVIEW))

;;;
;;; Event loop:
;;; 1. "gobble" redraw events to prevent lag/queue overflow.
;;; 2. would have to check other events if there were more than one window
;;;    active...
;;; It seems that a single Resize occurs only once resize done.
;;; For some reason, several Expose occur during resize.  (Need to gobble).
;;; Also, since redraw may be slow, need to gobble.
;;;
(defun event-loop (display)
 ;; Event loop
 (let ((debug t)
       (event (xlib:make-xevent)))
  (when debug (format t "Event-loop.~%"))
  (loop
   ;; Wait for next event
   (when debug (format t "Wait..."))
   (xlib:xnextevent display event)
   (let ((event-type (xlib:xanyevent-type event))
	 (window (xlib:xanyevent-window event)))
    (when debug (format t "event-type:~a~%" event-type))
    (cond
      ;; Return on buttonpress event.
      ((eq event-type xlib:buttonpress) (return))
      ;; Expose
      ((eq event-type xlib:expose)
       ;; Gobble all other expose events
       (loop
	(unless (> (xlib:xeventsqueued display xlib:queuedalready) 0)
	 (return))
	(xlib:xnextevent display event)
	(let ((event-type (xlib:xanyevent-type event)))
	 (unless (eq event-type xlib:expose)
	  (xlib:xputbackevent display event)
	  (return))
	 (when debug (format t "Gobble event-type:~a~%" event-type))))
       (redraw display window))
      ;; Resize
      ((eq event-type xlib:configurenotify)
       (resize (xlib:xconfigureevent-width event)
	       (xlib:xconfigureevent-height event))))))))


;;;
;;; This call binds GL to an existing X window.
;;;

(defun bind-gl-to-window (display screen window)
 (let ((debug t))
  (when debug (format t "Bind-gl-to-current-window.~%"))
  ;;
  (when debug (format t "XGetWindowAttributes..."))
  (let* ((attr (xlib:make-xwindowattributes))
	 (foo (xlib:xgetwindowattributes display window attr))
	 (class (xlib:xwindowattributes-class attr))
	 (depth (xlib:xwindowattributes-depth attr))
	 (visual (xlib:xwindowattributes-visual attr))
	 (visual-class (xlib:visual-class visual)))
   (when debug
    (format t "screen:~a, " screen)
    (format t "class:~a, depth:~a, " class depth)
    (format t "visual-class:~a~%" visual-class))
   ;;
   (when debug (format t "XMatchVisualInfo..."))
   (let* ((visualinfo (xlib:make-xvisualinfo))
	  (num-visuals (xlib:xmatchvisualinfo display screen depth
					       visual-class visualinfo)))
    (unless (> num-visuals 0)
     (error "BIND-GL-TO-WINDOW: Could not get visual of class:~a, depth~a!"
	    visual-class depth))
    (when debug (format t "~a visuals found.~%" num-visuals))
    ;;
    (when debug (format t "glXCreateContext..."))
    (let ((glx-context (gl:glxcreatecontext display visualinfo
					    xlib:NULL gl:GL_TRUE)))
     (when debug (format t "~%glXMakeCurrent..."))
     (gl:glxmakecurrent display window glx-context))))
  (when debug (format t "~%Done.~%"))))

 
(defun create-gl-simple-window (display width height)
 (let* ((screen (xlib:xdefaultscreen display))
        (root (xlib:xrootwindow display screen))
        (black-pixel (xlib:xblackpixel display screen))
        (white-pixel (xlib:xwhitepixel display screen))
        (window (xlib:xcreatesimplewindow display root 0 0 width height
					   1 black-pixel white-pixel)))
  ;; Enable events
  (xlib:xselectinput display window
		      (+ xlib:structurenotifymask
			 xlib:exposuremask
			 xlib:buttonpressmask))
  ;; Bind to GL
  (bind-gl-to-window display screen window)
  ;; Map window
  (xlib:xmapwindow display window)
  ;; Return window
  window))

(defun main ()
  (format t "GLTEST0.~%")
  (let* ((display (xlib:xopendisplay ""))
	 (window (create-gl-simple-window display 400 300)))
    ;;
    ;; The following show two different ways to return values from C routines:
    ;;
    ;; The first way is to allocate values on C side.
    ;; (The asterisk at the end of the variable names indicates they are
    ;;  pointers.)
    (let ((root* (make-int))
	  (x* (make-int)) (y* (make-int))
	  (width* (make-int)) (height* (make-int))
	  (border-width* (make-int)) (depth* (make-int)))
      (xlib::xgetgeometry display window root* x* y* width* height*
			  border-width* depth*)
      (format t "XGetGeometry: x=~a, y=~a, width=~a, height=~a, "
	      (int-ref x*) (int-ref y*) (int-ref width*) (int-ref height*))
      (format t "border-width:~a, depth:~a~%"
	      (int-ref border-width*) (int-ref depth*)))
    ;;
    ;; The second way is to allocate values on LISP side.
    ;; Note: This does not seem to work if we pass pointers to LISP scalars;
    ;; The LISP objects must be arrays!
    (let ((root (make-array '1 :element-type 'fixnum))
          (x (make-array '1 :element-type 'fixnum))
          (y (make-array '1 :element-type 'fixnum))
          (width (make-array '1 :element-type 'fixnum))
          (height (make-array '1 :element-type 'fixnum))
          (border-width (make-array '1 :element-type 'fixnum))
          (depth (make-array '1 :element-type 'fixnum)))
      (xlib::xgetgeometry display window (inta-ptr root)
			  (inta-ptr x) (inta-ptr y)
			  (inta-ptr width) (inta-ptr height)
			  (inta-ptr border-width) (inta-ptr depth))
      (format t "XGetGeometry: x=~a, y=~a, width=~a, height=~a, "
	      (aref x 0) (aref y 0) (aref width 0) (aref height 0))
      (format t "border-width:~a, depth:~a~%"
              (aref border-width 0) (aref depth 0)))
    ;;
    ;; Continue with demo.
    (gl:glClearIndex 0.0)
    (gl:glShadeModel gl:GL_FLAT)
    (gl:glclearcolor 0.0 0.0 0.0 1.0)
    (event-loop display)))
