(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

open Options

module O = Options
module R = Omom_rc
module A = Omom_actions
module G = Omom_gen
module V = Omom_variables
module T = Omom_templates

let rec map_once_if predicat action = function
  | h :: t ->
      if predicat h then
	action h :: t
      else
	t
  | [] -> []

class data file =
  let opt_file = O.create_options_file file in
  let opt_ocaml =
    define_option opt_file ["ocaml"] ""
      (list_option R.variable_option) R.default_ocaml
  and opt_common =
    define_option opt_file ["common"] ""
      (list_option R.variable_option)  R.default_common
  and opt_templates = 
    define_option opt_file ["templates"] ""
      (list_option R.template_option) [] in
object(self)
  val file = file

  val mutable ocaml = !!opt_ocaml
  val mutable common =  !!opt_common
  val mutable templates = !!opt_templates

  method get_ocaml = ocaml

  method get_common = common

  method get_templates = templates

  method get_template name = 
    List.find (fun x -> T.name_of_template x = name) templates

  method set_ocaml v =
    ocaml <- v

  method set_common v =
    common <- v
      
  method set_templates v =
    templates <- v

  method changed =
    ocaml <> !!opt_ocaml || common <> !!opt_common || templates <> !!opt_templates

  method load = 
    O.load opt_file

  method set_ocaml_action var value action =
    ocaml <- map_once_if
      (fun x -> V.name_of_variable x = var)
      (fun _ -> V.new_variable var value action)
      ocaml

  method set_common_action var value action =
    common <- map_once_if
      (fun x -> V.name_of_variable x = var)
      (fun _ -> V.new_variable var value action)
      common
      
  method set_template_action tpl_name var value action =
    templates <- map_once_if
      (fun x -> T.name_of_template x = tpl_name)
      (fun t -> 
	 let var_list = map_once_if
			  (fun x -> V.name_of_variable x = var)
			  (fun _ -> V.new_variable var value action)
			  (T.var_list_of_template t) in
	 T.new_template
	   (T.name_of_template t)
	   (T.filename_of_template t)
	   var_list)
      templates

  method add_template tpl_name tpl_file =
    templates <- 
    templates @ [T.new_template tpl_name tpl_file []]

  method remove_template tpl_name =
    templates <-
    List.filter (fun (name, _, _) ->
		   if name = tpl_name then false else true) templates

  method generate template =
    let tpl = List.find (fun t -> (T.name_of_template t) = template)
		templates in
    let eval_function =
      let env = Hashtbl.create 32 in
      let add_or_not_to_env = function
	| (n, V.V_String (s, true), a) -> 
	    Hashtbl.add env n (V.V_String (s, true), a)
	| (n, V.V_List l, a) -> 
	    (match List.filter (fun (v,b) -> b) l with
	      | [] -> ()
	      | purged_list ->
		  Hashtbl.add env n (V.V_List purged_list, a))
	| _ -> () in
	List.iter add_or_not_to_env ocaml;
	List.iter add_or_not_to_env common;
	List.iter add_or_not_to_env (T.var_list_of_template tpl);
	function var ->
	  try
	    let (value, action) = Hashtbl.find env var in
	      (A.action_of_string action) value
	  with
	      Not_found -> var in
      G.gen_make (T.filename_of_template tpl) eval_function
	
  method init =
    self#load;
    self#set_ocaml !!opt_ocaml;
    self#set_common !!opt_common;
    self#set_templates !!opt_templates

  method save =
    opt_ocaml =:= ocaml;
    opt_common =:= common;
    opt_templates =:= templates;
    O.save_with_help opt_file

  initializer 
    self#init
end
