/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Implements the base class CD writer classes.
 *	CDwriter is a bad name... should be process
 *	manager, these classes control the order of
 *	process execution.
 *
 *	by Tony Sideris	(02:41AM Aug 11, 2001)
 *================================================*/
#include "arson.h"

#include <qcombobox.h>
#include <qfileinfo.h>
#include <qlabel.h>
#include <qlayout.h>

#include <klocale.h>
#include <kapp.h>

#include "progressdlg.h"
#include "processmgr.h"
#include "process.h"
#include "layout.h"

#define ARSON_ISO_MAX_LABEL		32

/*========================================================*/

ArsonProcessUI::ArsonProcessUI (void)
	: m_pWriter(NULL),
	m_nMaxProgress(0),
	m_nProgress(0)
{
	//	Nothing...
}

/*========================================================*/
/*	Base class process manager
 *========================================================*/

ArsonProcessMgr::ArsonProcessMgr (ArsonProcessUI *pUI)
	: m_pUI(pUI),
	m_pProcess(NULL)
{
	ARSON_INSTANCE_INCR("procmgr");
}

ArsonProcessMgr::~ArsonProcessMgr (void)
{
	ARSON_INSTANCE_DECR("procmgr");

	delete m_pProcess;
	delete m_pUI;
}

/*========================================================*/

void ArsonProcessMgr::setProcess (ArsonProcess *proc)
{
	if ((m_pProcess = proc))
	{		
		if (!proc->execute())
		{
			DBGOUT("FAILED TO EXECUTE PROCESS!!!\n%s\n",
				proc->commandString().latin1());
		}
	}
	else
	{
		Trace("Job Complete, process == NULL\n");
		jobComplete();
	}
}

/*========================================================*/

void ArsonProcessMgr::abort (void)
{
	if (m_pProcess)
	{
		if (m_pProcess->isRunning())
		{
			DBGOUT("Killing process...\n");
			m_pProcess->kill();
		}

		delete m_pProcess;
		m_pProcess = NULL;

		jobComplete();
	}
}

/*========================================================*/

void ArsonProcessMgr::begin (const ArsonProcessOpts &opt)
{
	m_opts = opt;

	Assert(m_pUI != NULL);
	m_pUI->begin();
}

void ArsonProcessMgr::taskComplete (ArsonProcess *ptr)
{
	//	Nothing...
}

void ArsonProcessMgr::jobComplete (void)
{
	m_pUI->end();
}

/*========================================================*/
/*	A cd writer specific implementation of the
 *	progress dialog
 *========================================================*/

ArsonCdWriterProgress::ArsonCdWriterProgress (QWidget *parent, const char *name)
	: ArsonSimpleProgress(
		i18n("Please insert a blank CD, and press Start to continue..."),
		i18n("Burn Progress"),
		parent,
		name ? name : "cdwriter")
{
	QLabel *pl = new QLabel(i18n("Burn &Method: "), ctrlParent());
	QLabel *ps = new QLabel(i18n("&Speed:"), ctrlParent());
	const QSizePolicy sp (QSizePolicy::Maximum, QSizePolicy::Fixed);
	static QString text[] = {
		i18n("Simulation Only"),
		i18n("Write Only"),
		i18n("Simulate, then Write"),
	};

	m_pHow = new ArsonProgressCombobox(ctrlParent(), "burnhow");

	for (int index = 0; index < ARRSIZE(text); ++index)
		m_pHow->insertItem(text[index]);

	m_pHow->setCurrentItem(0);

	m_pSpeed = new ArsonProgressCombobox(ctrlParent(), "speed");

	m_pSpeed->insertItem(QString::number(1));
	for (int index = 2; index <= 12; index = index + 2)
		m_pSpeed->insertItem(QString::number(index));

	for (int index = 16; index <= 48; index = index + 4)
		m_pSpeed->insertItem(QString::number(index));

	m_pSpeed->setEditable(true);
	m_pSpeed->setEditText(QString::number(1));

	pl->setSizePolicy(sp);
	ps->setSizePolicy(sp);
	ps->setBuddy(m_pSpeed);
	pl->setBuddy(m_pHow);

	layoutRow() << pl << m_pHow << spacer(12) << ps << m_pSpeed;

	m_pEject = new ArsonProgressCheckbox(i18n("&Eject disk when complete"), ctrlParent(), optEject);
	m_pOverburn = new ArsonProgressCheckbox(i18n("Allow &overburns"), ctrlParent(), optOverBurn);
	m_pNoFix = new ArsonProgressCheckbox(i18n("Leave session &open"), ctrlParent(), optNoFix);

	layoutRow() << m_pEject << m_pOverburn << m_pNoFix;
}

void ArsonCdWriterProgress::processOpts (ArsonProcessOpts &opts)
{
	ArsonSimpleProgress::processOpts(opts);

	opts.addLong(optSpeed, m_pSpeed->currentText().toLong());
	opts.addLong(optSeq, m_pHow->currentItem());
	opts.addBool(optEject, m_pEject->isChecked());
	opts.addBool(optOverBurn, m_pOverburn->isChecked());
	opts.addBool(optNoFix, m_pNoFix->isChecked());
}

/*========================================================*/
/*	A progress dialog containing a label edit field.
 *========================================================*/

ArsonImgCreateProgress::ArsonImgCreateProgress (QWidget *parent, const char *name)
	: ArsonCdWriterProgress(parent, name), m_pIsoLabel(NULL)
{
	//	Nothing...
}

/*========================================================*/

void ArsonImgCreateProgress::setAutoIsoLabel (bool automatic)
{
	if (m_pIsoLabel)
		m_pIsoLabel->setName(automatic ? "isolabel" : NULL);
}

void ArsonImgCreateProgress::setIsoLabel (const QString &str)
{
	if (m_pIsoLabel)
		m_pIsoLabel->setText(str);
}

/*========================================================*/

void ArsonImgCreateProgress::showLabelEdit (QBoxLayout *layout)
{
	QLabel *pl = new QLabel(i18n("Volume &Label:"), ctrlParent());
	m_pIsoLabel = new ArsonProgressLineedit(ctrlParent(), "isolabel");

	m_pIsoLabel->setMaxLength(ARSON_ISO_MAX_LABEL);
	m_pIsoLabel->setText(i18n("ArsonImage"));
	pl->setBuddy(m_pIsoLabel);

	if (layout)
		ArsonLayout(layout) << pl << m_pIsoLabel;
	else
		layoutRow() << pl << m_pIsoLabel;
}

#ifndef InBetween
#define InBetween(c,m,x)	((c)>=(m)&&(c)<=(x))
#endif

QString ArsonImgCreateProgress::label (void) const
{
	if (m_pIsoLabel)
	{
		QString str (m_pIsoLabel->text()), dest;

		for (int index = 0; index < str.length(); ++index)
			if (InBetween(str[index], 'A', 'Z') ||
				InBetween(str[index], 'a', 'z') ||
				InBetween(str[index], '0', '9') ||
				str[index] == '_')
				dest += str[index];

		return dest.left(ARSON_ISO_MAX_LABEL);
	}

	return QString();
}

/*========================================================*/

void ArsonImgCreateProgress::processOpts (ArsonProcessOpts &opts)
{
	ArsonCdWriterProgress::processOpts(opts);
	opts.addString(optIsoLabel, label());
}

/*========================================================*/
/*	Base for cd writer process managers
 *========================================================*/

ArsonCdWriter::ArsonCdWriter (ArsonProcessUI *pUI)
	: ArsonProcessMgr(pUI), m_state(State_None)
{
	//	Nothing...
}

/*========================================================*/
/*
uint ArsonCdWriter::fixationTime (uint total)
{
	const double PERCENT = 0.10;
	return (uint) ((double) total * PERCENT);
}
*/
/*========================================================*/

bool ArsonCdWriter::writeCd (void)
{
	ArsonWriterProcess *ptr = createWriterProcess();

	m_state = State_Burn;
	
	if (opts().getLong(optSeq) != Burn)
	{
		ptr->setSimulate(true);
		m_state = State_Test;
	}

	try { setProcess(ptr); }
	catch (ArsonError &err) {
		err.report();
		return false;
	}

	return true;
}

void ArsonCdWriter::taskComplete (ArsonProcess *proc)
{
	if (proc == process() &&	//	Ignore secondary processes
		proc->successful() &&	//	And the test was successful
		opts().getLong(optSeq) == Both && m_state == State_Test)
	{
		ArsonWriterProcess *ptr = createWriterProcess();
		m_state = State_Burn;

		try { setProcess(ptr); }
		catch (ArsonError &err) {
			err.report();
		}
	}

	ArsonProcessMgr::taskComplete(proc);
}

/*========================================================*/
/*	Writer implementation that writes a cuefile
 *========================================================*/

#define TraceCwd()	Trace("CWD is now %s\n", (const char *) QDir::current().canonicalPath())

ArsonCueWriter::ArsonCueWriter (ArsonProcessUI *pUI, const char *cuefile)
	: ArsonCdWriter(pUI), m_cuefile(QString::null)
{
	if (cuefile && !setCueFile(cuefile))
		throw ArsonError(i18n("CUE file not readable"));
}

/*========================================================*/

bool ArsonCueWriter::setCueFile (const char *filename)
{
	QFileInfo fi (QFile::encodeName((m_cuefile = filename)));

	if (!fi.isReadable())
		return false;

	/*	We must be in the directory of the CUE file when
	 *	we write the CD beacuse most CUE files do not
	 *	contain absolute paths to image files.
	 */
	m_currentDir = QDir::current();
	chdir(fi.dir().canonicalPath());
	TraceCwd();
	return true;
}

/*========================================================*/

void ArsonCueWriter::begin (const ArsonProcessOpts &opts)
{
	ArsonCdWriter::begin(opts);

	if (m_cuefile != QString::null)
		writeCd();
}

/*========================================================*/

ArsonWriterProcess *ArsonCueWriter::createWriterProcess (void)
{
	return new ArsonCdrdaoProcess(this, m_cuefile);
}

/*========================================================*/

ArsonCueWriter::~ArsonCueWriter (void)
{
	/*	Restore the current directory to the old
	 *	directory before the write.
	 */
	chdir(m_currentDir.canonicalPath());
	TraceCwd();
}

/*========================================================*/
/*	Writer implementation that writes an ISO file
 *========================================================*/

ArsonIsoWriter::ArsonIsoWriter (ArsonProcessUI *pUI, const char *isofile)
	: ArsonCdWriter(pUI),
	m_isoFile(isofile)
{
	//	Nothing
}

/*========================================================*/

ArsonWriterProcess *ArsonIsoWriter::createWriterProcess (void)
{
	if (opts().programPref(PROGGRP_DATACD) == "cdrecord")
		return new ArsonCdrecordDataProcess(this, m_isoFile);

	return new ArsonCdrdaoIsoProcess(this, m_isoFile);
}

/*========================================================*/

void ArsonIsoWriter::begin (const ArsonProcessOpts &opts)
{
	ArsonCdWriter::begin(opts);

	if (m_isoFile != QString::null)
		writeCd();
}

/*========================================================*/
/*	Process Options - This class stores the options in
 *	the progress dialog.
 *========================================================*/

ArsonProcessOpts::ArsonProcessOpts (void)
	: ArsonConfig(ACONFIG)	//	Copy of global options
{
	//	Nothing else...
}

/*========================================================*/

#ifdef ARSONDBG
void dumpMap (const QMap<QCString,QVariant> &opts)
{
	QMap<QCString,QVariant>::ConstIterator it, end;

	for (it = opts.begin(), end = opts.end(); it != end; ++it)
	{
		switch (it.data().type())
		{
		case QVariant::Bool:
			Trace("[%14s]: Bool: %d\n", it.key().data(), it.data().toBool());
			break;

		case QVariant::String:
			Trace("[%14s]: String: %s\n", it.key().data(), it.data().toString().ascii());
			break;

		case QVariant::Int:
			Trace("[%14s]: Long: %u\n", it.key().data(), it.data().toInt());
			break;
		}
	}
}
#	define DUMPMAP(mp)		dumpMap(mp)
#else
#	define DUMPMAP(mp)
#endif

void ArsonProcessOpts::addBool (const char *name, bool val)
{
	m_opts[name] = QVariant(val, 0);
	DUMPMAP(m_opts);
}

void ArsonProcessOpts::addString (const char *name, const QString &str)
{
	m_opts[name] = QVariant(str);
	DUMPMAP(m_opts);
}

void ArsonProcessOpts::addLong (const char *name, long val)
{
	m_opts[name] = QVariant(int(val));
	DUMPMAP(m_opts);
}

/*========================================================*/

#ifdef ARSONDBG
#	define WARNNOOPT(name)		Trace("WARNING: process opt %s not found!\n", (name))
#else
#	define WARNNOOPT(name)
#endif	//	ARSONDBG

bool ArsonProcessOpts::getBool (const char *name, bool deft) const
{
	PROCESSOPTS::ConstIterator it = m_opts.find(name);

	if (it != m_opts.end())
		return it.data().toBool();

	WARNNOOPT(name);
	return deft;
}

QString ArsonProcessOpts::getString (const char *name, const QString &deft) const
{
	PROCESSOPTS::ConstIterator it = m_opts.find(name);

	if (it != m_opts.end())
		return it.data().toString();

	WARNNOOPT(name);
	return deft;
}

long ArsonProcessOpts::getLong (const char *name, long deft) const
{
	PROCESSOPTS::ConstIterator it = m_opts.find(name);

	if (it != m_opts.end())
		return long(it.data().toInt());

	WARNNOOPT(name);
	return deft;
}

/*========================================================*/

bool ArsonProcessOpts::hasOpt (const char *name) const
{
	return m_opts.contains(name);
}

/*========================================================*/
