/* $Id: ArkModelState.cpp,v 1.43 2003/03/20 17:23:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkCache.h>
#include <Ark/ArkModelState.h>
#include <Ark/ArkNetwork.h>

namespace Ark
{

   ModelState::ModelState (Model *model) :
      m_Model(),
      m_Skin(),
      m_BoneMatrices(NULL),
      m_Sequence(NULL)
   {
       SetModel(model);
   }

   ModelState::ModelState () :
      m_Model(),
      m_Skin(),
      m_BoneMatrices(NULL),
      m_Sequence()
   {
      Clear();
   }


   ModelState::~ModelState()
   {
      if (m_BoneMatrices)
	 delete[] m_BoneMatrices;
   }
   
   void
   ModelState::Clear()
   {
      m_Time = 0.0;
      m_SequenceName = "";
      m_SequenceChanged = true;
      m_Sequence = SequencePtr();
      
      SetController (0, 0.0f);
      SetController (1, 0.0f);
      SetController (2, 0.0f);
      SetController (3, 0.0f);
      SetMouth (0.0f);

      //Play (LOOP, "idle");
   }

   void
   ModelState::SetSkin (Skin *skin)
   {
       m_Skin = SkinPtr(skin);
   }

   bool
   ModelState::SetModel (Model *mdl)
   {
      if (!mdl) 
	  return false;

      m_Model = ModelPtr(mdl);

      SetSkin(NULL);

      // Create bone matrices.
      if (m_BoneMatrices)
	 delete[] m_BoneMatrices;

      if (m_Model->m_Skeleton &&
	  m_Model->m_Skeleton->m_Bones.size() != 0)
      {
	 m_BoneMatrices = new Matrix44 [m_Model->m_Skeleton->m_Bones.size()];
      }

      Clear();
      return true;
   }
   
   /**
    * Set the model with the given name and skin. The 'name' should be
    * in the form skin@model where 'skin' is the name of the skin file
    * and 'model' the name of the model.
    */
   bool
   ModelState::SetModel (Cache *cache, const String &name)
   {
       ModelPtr mdl;
       SkinPtr skin;

       m_Sequence = SequencePtr();

       // "@" not found
       String::size_type pos = name.find("@", 0);
       if (pos != String::npos)
       {
	  String skinname(name, 0, pos);
	  String modelname(name, pos + 1, String::npos);

	  cache->Get (V_SKIN,  skinname,  skin);
	  cache->Get (V_MODEL, modelname, mdl);
       }
       else
	  cache->Get (V_MODEL, name, mdl);

       const bool res = SetModel(&*mdl);

       SetSkin (skin ? &*skin : &mdl->m_Skin);
       
       return res;
   }

   bool
   ModelState::Play (PlayMode pm, const String &name)
   {
      if (!m_Model || !m_Model->m_Skeleton)
	 return false;

      if (m_SequenceName == name)
	 return true;
      
      m_Sequence = m_Model->m_Skeleton->GetSequence (name);

      m_SequenceChanged = true;
      m_Time = 0.0;
      m_Mode = pm;
      
      return true;
   }

   void
   ModelState::ComputeMatrix ()
   {
      //Matrix44 m; m.Translate (m_Position);
      m_Matrix = m_Rotation.GetMatrix();
      m_Matrix.PostTranslate(m_Position);

      if (m_Model && m_Model->m_Skeleton)
      {
	 // Compute bone matrices.
	 m_Model->m_Skeleton->SetupBones (*this, m_BoneMatrices,
					  &m_Matrix,
					  m_Model->m_Scale);
      }
   }

   void
   ModelState::Update (float dt)
   {
      if (!m_Model || !m_Model->m_Skeleton)
	 return;

      m_Model->m_Skeleton->Animate (*this, dt);
   }
   
   
   BBox ModelState::ExtractBbox () const
   {
      BBox bb;

     
      if (m_Model)
      {
	 if (m_Sequence && m_Model->m_Skeleton
	    && !m_Sequence->m_BBox.Empty())
	 {
	    //FIXME: can't we avoid the modelstate having to access skel data ?
	    bb = m_Model->m_Skeleton->GetBBox (m_Sequence,
			  		       m_Model->m_Scale); 
	 }
	 else
	    bb = m_Model->m_BBox;
      }
      else
	 bb.AddPoint (Vector3());

      return bb;
   }
   
   scalar
   ModelState::SetController (int iController, scalar value)
   {
      if (!m_Model || !m_Model->m_Skeleton)
	 return 0.0;
      
      scalar result;
      BoneController *cont = m_Model->GetController (iController);
      
      if (cont == NULL)
	 return 0.0;
      
      cont->ComputeSetting (value, &result);
      m_Controller[cont->m_Index] = result;
      
      // Return the approximation we did
      return result;
   }
   
   
#define MOUTHCONT 4
   scalar
   ModelState::SetMouth (scalar value)
   {
      if (!m_Model || !m_Model->m_Skeleton)
	 return false;

      BoneController *cont = m_Model->GetController (MOUTHCONT);

      if (cont == NULL)
	 return 0.0;
      
      cont->ComputeSetting (value, &m_Mouth);
 
      // Return the approximation we did
      return m_Mouth;
   }
   
   // Reading/Writing a model state (ie. across a network).
   bool ModelState::Write (WriteStream &file)
   {
      NetWriteString(file, m_SequenceName);
      NetWriteInt(file, m_Mode);
      NetWriteScalar(file, m_Time);
      return !file.fail();
   }
   
   bool ModelState::Read (Stream &file)
   {
      String sequence;
      scalar time = 0.0f;
      
      int mode;
      NetReadString (file, sequence);
      NetReadInt(file, mode);
      NetReadScalar(file, time);
      
      Play (static_cast<PlayMode>(mode), sequence);
      m_Time = time;
      
      return true;
   }

}
