/* $Id: ArkFactory.cpp,v 1.11 2003/04/06 14:22:15 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/ArkFactory.h>
#include <Ark/ArkFileSys.h>
#include <Ark/ArkStream.h>
#include <Ark/ArkSystem.h>

#include "my_dlsym.h"

namespace Ark
{

   class LibraryMap : public std::map<String,lt_dlhandle> {}; 
   typedef LibraryMap::iterator LibraryMapIt;

   typedef bool (*LibRegister)(FactoryList *factor);

   FactoryList::FactoryList()
   {
      static bool ltinited = false;
      if (!ltinited)
      {
	 ltinited = true;
	 lt_dlinit();
      }

      m_Libraries = new LibraryMap;
      LoadConfig();

      std::vector<RegisterList::Entry>::iterator it;
      for (it = RegisterList::s_Entries.begin();
	   it != RegisterList::s_Entries.end(); ++it)
	 RegisterFactory(it->name, it->fact);
   }

   FactoryList::~FactoryList()
   {
      // FIXME: Close libs, release mem, etc.
   }

   bool
   FactoryList::LoadLibrary(const String &library)
   {
      if (library != "static") 
      {
	 // It's not a static factory, so try to load the corresponding lib, 
         // if it isn't already.
	 lt_dlhandle &libh = (*m_Libraries)[library];

	 if (libh == 0) // Load it
	 {
	    const char* libraryName = library.c_str();
	    libh = lt_dlopenext(libraryName);

	    if (libh == 0)
	    {
		   const char* error_message = lt_dlerror();

	       Ark::Sys()->Error("Cannot load dynamic library '%s': %s",
				 libraryName, error_message);
	       return false;
	    }

	    function_pointer fp = my_dlsym (libh, "ArkRegister");
	    LibRegister reg = (LibRegister) (fp);

	    if (reg == 0)
	    {
	       Ark::Sys()->Error("Dynamic library '%s' isn't an Ark factory "
                                 "library: it doesn't have an 'ArkRegister' "
                                 "symbol.", libraryName);
	       return false;
	    }

	    if (!(*reg)(this))
	    {
	       Ark::Sys()->Warning("There was an error while running the "
				   "ArkRegister function of %s'.",
				   libraryName);
	    }
	 }
      }

      return true;
   }

   Factory *
   FactoryList::GetFactory(const String &name, const VersionInfo &version)
   {
      // Find the real name (for instance, "ark.renderer" usually maps to 
      // ark.renderer.opengl).
      String rname = m_Config.GetStr(name, name);
	  const char* cname = name.c_str();

      String library = m_Config.GetStr(rname + "::Lib", "static");
	  const char* clibrary = library.c_str();

      Sys()->Log("Loading extension library '%s'\n", library.c_str());

      // Find the corresponding library.
      if (!LoadLibrary(library))
	  {
		  Sys()->Warning("Factory '%s' could not be loaded from arkfactories.cfg", cname);
		  return NULL;
	  }

      // Now the factory should have been added to the list.
      Factory *fac = m_Factories[rname];

      if (fac == NULL)
	  {
		  if (library == "static")
		  {
			  Sys()->Warning("Could not find the factory in the 'static' library. Check the factory configuration file.");
		  }
		  else
		  {
			  Sys()->Warning("Factory '%s' cannot be found in library '%s'", cname, clibrary);
		  }
		  return NULL;
	  }

      if (fac->m_Version.m_Major != version.m_Major || 
	  fac->m_Version.m_Minor != version.m_Minor || 
	  fac->m_Version.m_Micro != version.m_Micro ||
	  strcmp(fac->m_Version.m_Interface, version.m_Interface)) 
      {
	 Sys()->Warning("Cannot get factory '%s', the library  version "
                        "is different as the requested one. (the library '%s',"
			" provides %s %d.%d.%d, while %s %d.%d.%d is "
			"requested)\n",
			cname,
			clibrary,
			fac->m_Version.m_Interface,
			fac->m_Version.m_Major,
			fac->m_Version.m_Minor,
			fac->m_Version.m_Micro,
			version.m_Interface,
			version.m_Major,
			version.m_Minor,
			version.m_Micro);			
	 return NULL;
      }

      return fac;
   }

   bool
   FactoryList::RegisterFactory(const String &name, Factory *fact)
   {
      Ark::Sys()->Log("Registering factory '%s'\n", name.c_str());
      m_Factories[name] = fact;
      return true;
   }

   static bool cfgFactoryAddPath (void *, Config *self, int nargs,
				  String *args)
   {
      for (size_t i = 0; i < (size_t)nargs; i++)
	 lt_dladdsearchdir(Sys()->FS()->GetFileName(args[i]).c_str());

      return true;
   }

   static bool cfgFactoryPreload (FactoryList *factory, Config *self,int nargs,
				  String *args)
   {
      Sys()->Log("Pre-loading extension library '%s'\n", args[0].c_str());
      factory->LoadLibrary(args[0]);
      return true;
   }

   bool FactoryList::LoadConfig()
   {
      m_Config.AddFunction
              ("factoryList::AddSearchDir", -1,
               (Config::CfgCb) &cfgFactoryAddPath, this);

      m_Config.AddFunction
              ("factoryList::Preload", 1,
               (Config::CfgCb) &cfgFactoryPreload, this);

      m_Config.LoadSystem("arkfactories.cfg");
      return true;
   }
   ///////////////////////////////////////////////////////////////
   Factory::Factory (VersionInfo version)
   {
      m_Version = version;
   }


   ///////////////////////////////////////////////////////////////
   std::vector<RegisterList::Entry> RegisterList::s_Entries;

   RegisterList::RegisterList(const String &name, Factory *fact)
   {
      Entry e(name, fact);
      s_Entries.push_back(e);
   }
}
