// ---------------------------------------------------------------------------
// - String.hpp                                                              -
// - standard object library - string class definition                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_STRING_HPP
#define  ALEPH_STRING_HPP

#ifndef  ALEPH_LITERAL_HPP
#include "Literal.hpp"
#endif

namespace aleph {

  /// The String class is a literal class used to manipulate c-strings.
  /// The basic constructors are provided with a c-string or a regular string.
  /// Internally, the string is a reference counted structure which simplify
  /// the operation during the copy construction or assignment. Note that this
  /// reference count has nothing to do with the object reference count.
  /// Standard methods to retreive the length, a character, or to manipulate
  /// the string are provided.
  /// @author amaury darsch

  class String : public Literal {
  private:
    // the string record is a structure
    struct s_string* p_string;

  public:
    /// create an empty string
    String (void);

    /// create a string from a c-string
    /// @param s the c-string to copy
    String (const char* s);

    /// create a string from a single character
    /// @param c the character to map to a string
    String (const char c);

    /// copy constructor for this string class
    /// @param that the string to copy
    String (const String& that);

    /// destroy this string
    ~String (void);

    /// @return the class name
    String repr (void) const;

    /// @return a literal representation of this string
    String toliteral (void) const;

    /// @return a printable representation of this string
    String tostring (void) const;

    /// @return a clone of this string
    Object* clone (void) const;

    /// @return the string serial code
    t_byte serialid (void) const;

    /// serialize this string to an output stream
    /// @param os the output stream to write
    void wrstream (class Output& os) const;

    /// deserialize a string from an input stream
    /// @param is the input steam to read in
    void rdstream (class Input& is);

    /// @return a quark by interning the string
    long toquark (void) const;

    /// intern a cstring and return a quark
    /// @param name the name to intern
    static long intern (const char* name);

    /// map a string with a quark
    /// @param quark the quark to map
    static const String& qmap (const long quark);

    /// @return true if the string is nil
    bool isnil (void) const;

    /// assign a c-string to this string
    /// @param s the string to assign
    /// @return the current string
    String& operator = (const char* s);

    /// assign a character to this string
    /// @param c the character to assign
    /// @return the current string
    String& operator = (const char c);

    /// assign a string to this string
    /// @param s the string to assign
    /// @return the current string
    String& operator = (const String& s);

    /// get a character at a certain position
    /// @param pos the character position
    /// @return the character at this position
    char operator [] (const long index) const;

    /// concatenate the string with another one
    /// @param the string to add to the current one
    /// @return a newly created string
    String operator + (const String& s) const;

    /// concatenate the string with a character
    /// @param the character to add to this string
    /// @return a newly created string
    String operator + (const char c) const;

    /// concatenate the string with an integer
    /// @param the integer to add to the string
    /// @return a newly created string
    String operator + (const long value) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator == (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator == (const char* s) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator != (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string are equal
    bool operator != (const char* s) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string is less than the other
    bool operator < (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string is less than the other
    bool operator < (const char* s) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string is less equal than the other
    bool operator <= (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string is less equal than the other
    bool operator <= (const char* s) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string is greater than the other
    bool operator > (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string is greater than the other
    bool operator > (const char* s) const;

    /// compare this string with another one
    /// @param s the string to compare with
    /// @return true if the string is greater equal than the other
    bool operator >= (const String& s) const;

    /// compare this string with a c-string
    /// @param s the string to compare with
    /// @return true if the string is greater equal than the other
    bool operator >= (const char* s) const;

    /// @return the length of this string
    long length (void) const;

    /// @return the length of a c-string
    static long length (const char* s);

    /// @return a character representation of this string or nil.
    char* tochar (void) const;

    /// remove leading blanks and tabs from this string
    /// @return a new clean string
    String stripl (void) const;

    /// remove trailing blanks and tabs from this string
    /// @return a new clean string
    String stripr (void) const;

    /// Remove leading and trailing blanks and tabs from this string
    /// @return a new clean string
    String strip (void) const;

    /// @return a string converter to upper case
    String toupper (void) const;

    /// @return a string converter to lower case
    String tolower (void) const;

    /// return the right substring starting at index
    /// @param index the start index
    /// @return the extracted substring
    String rsubstr (const long index) const;

    /// return the left substring upto (but not including) the index
    /// @param index the end index
    /// @return the extracted substring
    String lsubstr (const long index) const;

    /// extract a sub string by index
    /// @param lidx the left index
    /// @param ridx the right index
    String substr (const long lidx, const long ridx) const;

    /// @return the hashid for this string
    long hashid (void) const;

    /// fill with string on the left with a character for a given size
    /// @param the fill character
    /// @param the target size
    String lfill (const char c, const long size) const;

    /// fill with string on the rigth with a character for a given size
    /// @param the fill character
    /// @param the target size
    String rfill (const char c, const long size) const;

    /// split a string with blank and tab
    Vector* split (void) const;

    /// split a string based on a split argument
    /// @param sbrk the split string
    Vector* split (const String& sbrk) const;

    /// extract a stirng between a control character
    Vector* extract (const char cbrk) const;

    /// generate a new string
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// operate this string with another object
    /// @param robj   the current runnable
    /// @param type   the operator type
    /// @param object the operand object
    Object* oper (Runnable* robj, t_oper type, Object* object);

    /// set an object to this string
    /// @param robj   the current runnable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Runnable* robj, Nameset* nset, Object* object);

    /// apply this string with a set of arguments by quark
    /// @param robj   the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
