#include "CalendarEventModel.h"

@implementation CalendarEventModel

- (id) init
{
	return [self initWithStartDate:[NSCalendarDate calendarDate]
					  timeInterval:0];
}

- (id) initWithStartDate:(NSCalendarDate *)date
			timeInterval:(NSTimeInterval)stopTime
{
	_observers = [[NSMutableSet alloc] init];

	ASSIGN(_createDate, [NSCalendarDate calendarDate]);
	ASSIGN(_modifyDate, _createDate);
	ASSIGN(_string, @"");

	[self setStartDate:date
		  timeInterval:stopTime];
	return self;
}

- (void) dealloc
{
	id en;
	id <CalendarEventModel> observer;

	en = [_observers objectEnumerator];
	while ((observer = [en nextObject]))
	{
		[observer calendarEventModelWillBeFreed:self];
	}

	NSLog(@"free %@",self);

	RELEASE(_observers);
	RELEASE(_startDate);
	RELEASE(_createDate);
	RELEASE(_modifyDate);
	[super dealloc];
}

- (id) description
{
	return [NSString stringWithFormat:@"%@ (%d mins)",[_startDate description],(int)(_stopTime/60)];
}


- (void) addObserver:(id <CalendarEventModel>)observer
{
	[_observers addObject:observer];

	[observer calendarEventModel:self
			  didUpdateStartDate:_startDate
					timeInterval:_stopTime];

}

- (void) removeObserver:(id)observer
{
	[_observers removeObject:observer];
}

- (void) setStartDate:(NSCalendarDate *)startDate
		 timeInterval:(NSTimeInterval)stopTime
{
	BOOL modify = NO;
	id en;
	id <CalendarEventModel>  observer;

	if (stopTime <= 0)
	{
		stopTime = 0;
	}

	if (![startDate isEqualToDate:_startDate])
	{
		ASSIGN(_startDate, startDate);
		modify = YES;
	}
	if (stopTime != _stopTime)
	{
		_stopTime = stopTime;
		modify = YES;
	}
	
	if (modify == YES)
	{
		ASSIGN(_modifyDate, [NSCalendarDate calendarDate]);
		en = [_observers objectEnumerator];
		while ((observer = [en nextObject]))
		{
			[observer calendarEventModel:self
					  didUpdateStartDate:_startDate
							timeInterval:_stopTime];
		}
	}
}


- (NSCalendarDate *) startDate
{
	return _startDate;
}

- (void) setTimeInterval:(NSTimeInterval)stopTime
{
	[self setStartDate:_startDate
		  timeInterval:stopTime];
}

- (NSTimeInterval) timeInterval
{
	return _stopTime;
}

- (NSCalendarDate *) stopDate
{
	return [_startDate addTimeInterval:_stopTime];
}

- (NSCalendarDate *) createDate
{
	return _createDate;
}

- (NSCalendarDate *) modifyDate
{
	return _modifyDate;
}

- (NSComparisonResult) compare: (CalendarEventModel *)otherModel
{
	NSComparisonResult result = [_startDate compare:[otherModel startDate]];
	if (result == NSOrderedSame)
	{
		return [_createDate compare:[otherModel createDate]];
	}

	return result;
}

- (NSString *) stringValue
{
	return _string;
}

- (void) setStringValue:(NSString *)string
{
	id en;
	id observer;
	if (![string isEqualToString:_string])
	{
		ASSIGN(_string, string);
		en = [_observers objectEnumerator];
		while ((observer = [en nextObject]))
		{
			[observer calendarEventModel:self
					didUpdateStringValue:_string];
		}
	}
}

- (BOOL) isDuringDate:(NSDate *)startDate
			  andDate:(NSDate *)stopDate
{
	id d;

	if ([startDate compare:stopDate] == NSOrderedDescending)
	{
		d = startDate;
		startDate = stopDate;
		stopDate = d;
	}

	if ([[_startDate addTimeInterval:_stopTime] compare:startDate] == NSOrderedAscending)
	{
		return NO;
	}

	if ([_startDate compare:stopDate] == NSOrderedAscending)
	{
		return YES;
	}

	return NO;
	
}

@end

@implementation CalendarEventModelManager
- (id) init
{
	_observers = [[NSMutableSet alloc] init];
	_events = [[NSMutableArray alloc] init];

	[self _loadDataFromDisk];

	return self;
}

- (void) _loadDataFromDisk
{
	NSString *calendarFilePath = [CalendarUserLibraryPath() stringByAppendingString:@"/Calendar.plist"];
	NSArray *data;
	id en;
	id entry;
	CalendarEventModel *model;

	if (![[NSFileManager defaultManager] fileExistsAtPath:calendarFilePath])
	{
		/* calendar not exist, create one */
		/* TODO assert this */
		[[NSFileManager defaultManager] createDirectoryAtPath:CalendarUserLibraryPath()
												   attributes:nil];

		data = [[NSMutableArray alloc] init];
		[data writeToFile:calendarFilePath
			   atomically:NO];
		NSRunAlertPanel(@"Create Database",
				@"File '%@' has been created.",
				@"OK",nil,nil,calendarFilePath);
		AUTORELEASE(data);
	}
	else
	{
		data = [[NSMutableArray alloc] initWithContentsOfFile:calendarFilePath];
		NSAssert(data != nil, @"Failed to read data");
		AUTORELEASE(data);

		en = [data objectEnumerator];
		while ((entry = [en nextObject]))
		{
			model = [self createModelWithStartDate:[entry objectForKey:@"StartDate"]
									  timeInterval:[[entry objectForKey:@"TimeInterval"] floatValue]];
			[model setStringValue:[entry objectForKey:@"StringValue"]];
		}
	}
}

- (void) dealloc
{
	RELEASE(_observers);
	RELEASE(_events);
	[super dealloc];
}

- (CalendarEventModel *) createModelWithStartDate:(NSCalendarDate *)startDate
									 timeInterval:(NSTimeInterval)stopTime
{
	CalendarEventModel *newModel;
	newModel = [[CalendarEventModel alloc] initWithStartDate:startDate
												timeInterval:stopTime];
	[self addModel:newModel];

	return AUTORELEASE(newModel);
}

- (void) addObserver:(id <CalendarEventModelManager>)observer
{
	[_observers addObject:observer];
}

- (void) removeObserver:(id)observer
{
	[_observers removeObject:observer];
}


- (void) addModel:(CalendarEventModel *)model
{
	id en;
	id observer;

	if (![_events containsObject:model])
	{
		[_events addObject:model];
		[model addObserver:self];
		en = [_observers objectEnumerator];
		while ((observer = [en nextObject]))
		{
			[observer calendarEventModelManager:self
									didAddModel:model];
		}
	}
}

- (void) removeModel:(id)model
{
	id en;
	id observer;

	if ([_events containsObject:model])
	{
		[model removeObserver:self];
		[_events removeObject:model];
		en = [_observers objectEnumerator];
		while ((observer = [en nextObject]))
		{
			[observer calendarEventModelManager:self
								didAddModel:nil];
		}
	}

}


- (NSArray *) allEvents
{
	return _events;
}

- (NSArray *) allEventsWithString:(NSString *)str;
{
	NSMutableArray *array;
	id en,model;
	array = [NSMutableArray array];
	en = [_events objectEnumerator];

	while ((model = [en nextObject]))
	{
		if ([model stringValue] == nil)
		{
			continue;
		}

		if ([[model stringValue] rangeOfString:str
									   options:NSCaseInsensitiveSearch].length > 0)
		{
			[array addObject:model];
		}
	}
	return array;
}

- (NSArray *) allEventsDuringDate:(NSCalendarDate *)startDate
						  andDate:(NSCalendarDate *)stopDate
{
	NSMutableArray *array;
	id en;
	CalendarEventModel *model;

	array = [NSMutableArray array];
	en = [_events objectEnumerator];

	while ((model = [en nextObject]))
	{
		if ([model isDuringDate:startDate
						andDate:stopDate])
		{
			[array addObject:model];
		}
	}

	return array;
}

/** <CalendarEventModel> **/
- (void) calendarEventModelWillBeFreed:(id)model
{
	id en;
	id observer;

	[model removeObserver:self];

	en = [_observers objectEnumerator];
	while ((observer = [en nextObject]))
	{
		[observer calendarEventModelManager:self
							 didUpdateModel:nil];
	}
}

- (void) calendarEventModel:(CalendarEventModel *)model
		 didUpdateStartDate:(NSCalendarDate *)startDate
			   timeInterval:(NSTimeInterval)time
{
	id en;
	id observer;

	en = [_observers objectEnumerator];
	while ((observer = [en nextObject]))
	{
		[observer calendarEventModelManager:self
							 didUpdateModel:model];
	}

}

/* FIXME this would be slow, set dirty flag or something */
- (void) calendarEventModel:(id)model
	   didUpdateStringValue:(NSString *)string
{
	[self saveEventsToDisk];
}

- (void) saveEventsToDisk
{
	NSString *calendarFilePath = [CalendarUserLibraryPath() stringByAppendingString:@"/Calendar.plist"];
	NSArray *data;
	id en;
	CalendarEventModel *model;
	id dict;


	data = [[NSMutableArray alloc] init];
	AUTORELEASE(data);

	en = [_events objectEnumerator];
	while ((model = [en nextObject]))
	{
		dict = [NSMutableDictionary dictionary];

		[dict setObject:[model createDate] forKey:@"CreateDate"];
		[dict setObject:[model modifyDate] forKey:@"ModifyDate"];
		[dict setObject:[model startDate] forKey:@"StartDate"];
		[dict setObject:[model stringValue] forKey:@"StringValue"];
		[dict setObject:[NSString stringWithFormat:@"%g",[model timeInterval]] forKey:@"TimeInterval"];
		[data addObject:dict];
	}

	[data writeToFile:calendarFilePath
		   atomically:NO];
}

@end
