/*
** Copyright (C) 2001-2005 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#ifndef _GNUVD_H_
#define _GNUVD_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /*HAVE_CONFIG*/

#include "buffer.h"


/*
 * enumeration of VD errors
 */
enum VDErrorEnum {
	VD_ERR_OK        =  0,
	VD_ERR_NOT_FOUND,
	VD_ERR_UNAVAILABLE,
        VD_ERR_CONNECT,
	VD_ERR_SOCKET,
	VD_ERR_PROXY,
        VD_ERR_DNS,
	VD_ERR_READ,
	VD_ERR_HTTP,
	VD_ERR_MEM,
	VD_ERR_POINTER,
        VD_ERR_PARAM 
};
typedef int VDError;


/*
 * VD query structure
 */
struct _VDQuery {
	char*	    _search_term;
	char*	    _error;
	Buffer*     _result_buffer;
};


typedef struct _VDQuery VDQuery;


/* VD callback function for progressive loading */
typedef void (*VDCallback) (int bytes_read);


/* VDQuery functions */

/*
 * vd_query_new:
 * create a new vd query
 *
 * word:      the word you are looking for (\0 terminated)
 * exception: out parameter receiving exception info
 *
 * returns: ptr to a VDQuery, or NULL if it fails
 *
 * preconditions: word != NULL &&
 *                ex   != NULL 
 */
VDQuery*       vd_query_new     (const char* word, VDError *ex);



/*
 * vd_query_destroy:
 * destroy a vd_query (freeing allocated memory)
 *
 */
void           vd_query_destroy (VDQuery *vd_query);

/*
 * vd_query_perform:
 * perform the vd query over the internet
 * 
 * vd_query: a vd query
 * ex      : a vd exception, receiving error information
 *
 * return an error code (VDError)
 * 
 * preconditions: vd_query != NULL &&
 *                ex       != NULL
 */
void   vd_query_perform (const VDQuery *vd_query, VDError *ex);


/*
 * vd_query_perform:
 * perform the vd query over the internet progressively, ie. call
 * callback function after reading a certain # of bytes or WOULDBLOCK
 *
 * 
 * vd_query: a vd query
 * callback:    callback function to call after reading @chunk_size bytes
 *		or WOULDBLOCK
 * chunk_size: number of bytes to read at a time before calling @callback
 *
 * ex      : a vd exception, receiving error information
 *
 * return an error code (VDError)
 * 
 * preconditions: vd_query != NULL &&
 *                ex       != NULL
 */
void     vd_query_perform_progressive (const VDQuery *vd_query, 
				       VDCallback callback,
				       int chunk_size,
				       VDError *ex);




/* vd_query_results:
 * return the results of the query
 *
 * vd_query: a query object
 * ex:       an exception object receiving error information
 *
 * returns: a Buffer object with the results
 *
 * preconditions: vd_query != NULL &&
 *                vd_exception != NULL
 */
const Buffer*  vd_query_results (const VDQuery *vd_query);



/* vd_query_search_term:
 * return the searchterm for the query
 *
 * vd_query: a query object
 *
 * returns: a char ptr with the search term
 *
 * preconditions: vd_query != NULL &&
 */
const char *vd_query_search_term (const VDQuery *vd_query);

/* vd_error_string:
 * get message for error code
 *
 * err: an error codee
 *
 * returns: a char ptr with the error msg
 */
const char* vd_error_string (int err); 

#endif /*_GNUVD_H_*/
