/* tool.c --- Command line interface to libykclient.
 *
 * Copyright (c) 2006, 2007, 2008, 2009, 2011 Yubico AB
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *    * Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *
 *    * Redistributions in binary form must reproduce the above
 *      copyright notice, this list of conditions and the following
 *      disclaimer in the documentation and/or other materials provided
 *      with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "ykclient.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include <getopt.h>

const char *usage =
  "Usage: ykclient [options] <client_id> <yubikey_otp>\n"
  "  client_id: your client id integer\n"
  "  yubikey_otp: One-time password generated by yubikey\n"
  "\n"
  "  Options :\n"
  "    --url URL		Validation service URL (eg: \"http://api.yubico.com/wsapi/verify?id=%%d&otp=%%s\")\n"
  "    --apikey Key		API key for HMAC validation of request/response\n";

static struct option long_options[] = {
  {"url", 1, 0, 'u'},
  {"apikey", 1, 0, 'a'},
  {0, 0, 0, 0}
};

/* Parse command line parameters. */
void
parse_args (int argc, char *argv[],
	    int *client_id, char **token, char **url, char **api_key)
{
  while (1)
    {
      int option_index = 0;

      int c = getopt_long (argc, argv, "",
			   long_options, &option_index);
      if (c == -1)
	break;

      switch (c)
	{
	case 'a':
	  if (strlen (optarg) < 16)
	    {
	      fprintf (stderr,
		       "error: API key must be at least 16 characters");
	      exit (EXIT_FAILURE);
	    }
	  *api_key = optarg;
	  break;
	case 'u':
	  if (strncmp ("http://", optarg, 7) != 0
	      && strncmp ("https://", optarg, 8) != 0)
	    {
	      fprintf (stderr, "error: validation url must be http or https");
	      exit (EXIT_FAILURE);
	    }
	  *url = optarg;
	  break;
	}
    }

  if (argc - optind != 2)
    {
      fprintf (stderr, "%s", usage);
      exit (EXIT_FAILURE);
    }

  /* Now get mandatory numeric client_id */
  *client_id = atoi (argv[optind++]);

  if (*client_id <= 0)
    {
      fprintf (stderr, "error: client identity must be a non-zero integer.");
      exit (EXIT_FAILURE);
    }

  /* Likewise mandatory OTP token */
  *token = argv[optind++];
  if (strlen (*token) < 32)
    {
      fprintf (stderr,
	       "error: modhex encoded token must be at least 32 characters");
      exit (EXIT_FAILURE);
    }
}

int
main (int argc, char *argv[])
{
  int client_id;
  char *token, *url = NULL, *api_key = NULL;
  int ret, optind;

  parse_args (argc, argv, &client_id, &token, &url, &api_key);

  /* Debug. */
  fprintf (stderr, "Input:\n");
  if (url)
    fprintf (stderr, "  validation URL: %s\n", url);
  fprintf (stderr, "  client id: %d\n", client_id);
  fprintf (stderr, "  token: %s\n", token);
  if (api_key != NULL)
    fprintf (stderr, "  api key: %s\n", api_key);

  ret =
    ykclient_verify_otp_v2 (NULL, token, client_id, NULL, 1,
			    (const char **) &url, api_key);

  printf ("Verification output (%d): %s\n", ret, ykclient_strerror (ret));

  if (ret != YKCLIENT_OK)
    return EXIT_FAILURE;

  return EXIT_SUCCESS;
}
