/**
 * Map.java
 *
 * Created on Jul 10, 2006
 */
package maths;

/**
 * Cette interface represente une application du plan complexe dans lui meme.
 *
 * @author Nicolae
 */
public interface Map {

	/**
	 * Renvoie la partie reele de <code>f(re, im)</code>.
	 *
	 * @param re partie reele.
	 * @param im partie imaginaire.
	 * @return la partie reele de la valeur de la fonction.
	 */
	public double re(double re, double im);
	
	/**
	 * Renvoie la partie imaginaire de <code>f(re, im)</code>.
	 *
	 * @param re partie reele.
	 * @param im partie imaginaire.
	 * @return la partie reele de la valeur de la fonction.
	 */
	public double im(double re, double im);		
			
	/**
	 * Calcule la valeur de l'application et l'ecrit dans <code>val</code>,
	 * un vecteur de longueur au moins 2. 
	 *
	 * @param re partie reele.
	 * @param im partie imaginaire.
	 * @param val le buffer pour le resultat.
	 */
	public void value(double re, double im, double val[]);

	/**
	 * Somme de cette application avec <code>g</code>.
	 * 
	 * @param g l'application a aditionner.
	 * @return la somme.
	 */
	public Map add(Map g);

	/**
	 * Difference de cette application avec <code>g</code>.
	 * 
	 * @param g l'application a soustraire.
	 * @return la difference.
	 */
	public Map sub(Map g);

	/**
	 * Produit de cette application par <code>g</code>.
	 * 
	 * @param g l'application a multiplier.
	 * @return le produit.
	 */
	public Map mult(Map g);

	/**
	 * Rapport de cette application par <code>g</code>.
	 * 
	 * @param g l'application.
	 * @return le rapport.
	 */
	public Map divide(Map g);

	/**
	 * Composition de cette application avec <code>g</code>.
	 * 
	 * @param g l'application.
	 * @return la composition.
	 */
	public Map comp(Map g);
		
	/**
	 * Renvoie un vecteur avec les branches inverses de l'application.
	 *
	 * @return les branches inverses ou <code>null</code> si cette aplication ne peut pas
	 * les calculer.
	 */
	public Map[] inverse();
	
	/**
	 * Renvoie la preimage de l'ensemble <code>A</code>.
	 *
	 * @param A l'ensemble.
	 * @return la preimage.
	 */
	public Set preImage(Set A);

	/**
	 * Renvoie l'image de l'ensemble <code>A</code>. Attention : certaines application ne peuvent 
	 * pas calculer les images de tout ensemble. L'exponentielle a par example une infinite
	 * des branches inverses et en ne peut pas calculer l'image d'un ensemble non born.
	 * Si <code>inverse()</code> fonctionne (nombre fini des branches inverses, calculables)
	 * on peut calculer l'ensemble image.
	 *
	 * @param A l'ensemble.
	 * @return l'image du <code>A</code> ou <code>null</code> si l'application ne peut pas
	 *  la calculer.
	 */
	public Set image(Set A);
		
	/**
	 * Renvoie la derivee si cela existe.
	 * 
	 * @return la derivee ou <code>null</code> si cela n'existe pas.
	 */
	public Map differential();
}
