/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef CAMERA_H
#define CAMERA_H

#include <glib.h>
#include <coreTools/toolPhysic.h>

/**
 * PI180:
 *
 * Value of pi / 180.
 */
#define PI180 0.017453292522

/**
 * VisuOpenGLView:
 * @camera: a #VisuOpenGLCamera structure;
 * @window: a #OpenGLWindow structure;
 *
 * A container structure to deal with OpenGL observer position, size of rendering
 * viewport...
 */
typedef struct _VisuOpenGLView VisuOpenGLView;
/**
 * VisuOpenGLCamera:
 * @d_red: a factor for perspective from 1. to inifnity. With one, the nose of
 *         the observer is completly set on the rendered object, and the size
 *         of the observer is neglectible compared to the size of the object.
 * @theta: the theta angle in spherical coordinates of the position of the observer ;
 * @phi: the phi angle in spherical coordinates of the position of the observer ;
 * @omega: rotation of the observer on itself ;
 * @xs: a value for translation of the viewport on x axis ;
 * @ys: a value for translation of the viewport on y axis ;
 * @gross: a value of zoom ;
 * @length0: a length reference to adimension all values, by default,
 * this is the longest diagonal of the current box (without
 * duplication) ;
 * @up: (in) (array fixed-size=3): the current up vector.
 * @upAxis: which axis define the north pole.
 * @centre: (in) (array fixed-size=3): position of the eye look at ;
 * @eye: (in) (array fixed-size=3): position of the eye.
 * @unit: the unit of @length0.
 * 
 * Values to define the position of the observer.
 */
typedef struct _VisuOpenGLCamera VisuOpenGLCamera;

/**
 * ViewAxis:
 * @VIEW_X: the up axis is X ;
 * @VIEW_Y: the up axis is Y ;
 * @VIEW_Z: the up axis is Z.
 *
 * Define the up axis.
 */
typedef enum
  {
    VIEW_X,
    VIEW_Y,
    VIEW_Z
  } ViewAxis;

struct _VisuOpenGLCamera
{
  /* Perspective. */
  double d_red;
  /* Orientation. */
  double theta, phi, omega;
  /* Position. */
  double xs, ys;
  /* Zoom. */
  double gross;
  /* A length reference and its unit. */
  double length0;
  ToolUnits unit;

  /* Up vector. */
  double up[3];
  /* Up axis. */
  ViewAxis upAxis;
  /* Eye target and eye position. */
  double centre[3], eye[3];
};

/**
 * VISU_CAMERA_THETA:
 *
 * Value used in the openGLCameraSet_thetaPhiOmega() method to store the tetha angle.
 */
#define VISU_CAMERA_THETA (1 << 1)
/**
 * VISU_CAMERA_PHI:
 *
 * Value used in the openGLCameraSet_thetaPhiOmega() method to store the phi angle.
 */
#define VISU_CAMERA_PHI   (1 << 2)
/**
 * VISU_CAMERA_OMEGA:
 *
 * Value used in the openGLCameraSet_thetaPhiOmega() method to store the omega angle.
 */
#define VISU_CAMERA_OMEGA   (1 << 3)
gboolean openGLCameraSet_thetaPhiOmega(VisuOpenGLCamera *camera, float valueTheta,
				       float valuePhi, float valueOmega, int mask);
/**
 * MASK_XS:
 *
 * Value used in the openGLCameraSet_XsYs() method to store the horizontal offset.
 */
#define MASK_XS (1 << 1)
/**
 * MASK_YS:
 *
 * Value used in the openGLCameraSet_XsYs() method to store the vertical offset.
 */
#define MASK_YS   (1 << 2)
gboolean openGLCameraSet_XsYs(VisuOpenGLCamera *camera,
			      float valueX, float valueY, int mask);
gboolean openGLCameraSet_gross(VisuOpenGLCamera *camera, float value);
gboolean openGLCameraSet_persp(VisuOpenGLCamera *camera, float value);

gboolean openGLCameraSet_refLength(VisuOpenGLCamera *camera, float value, ToolUnits unit);
float openGLCameraGet_refLength(VisuOpenGLCamera *camera, ToolUnits *unit);

/*
 * _OpenGLWindow:
 */
struct _OpenGLWindow
{
  guint width, height;
  double near, far;
  double left, right, bottom, top;
};
/**
 * OpenGLWindow:
 * @width : the width of the window ;
 * @height : the height of the window ;
 * @near : the beginning of the viewport on z axis (z for observer) ;
 * @far : the end of the viewport on z axis (z for observer) ;
 * @left : the left of the viewport on x axis ;
 * @right : the right of the viewport on x axis ;
 * @bottom : the bottom of the viewport on y axis ;
 * @top : the top of the viewport on y axis ;
 *
 * Values to describe the window where the render is done.
 */
typedef struct _OpenGLWindow OpenGLWindow;
gboolean openGLWindowSet_viewport(OpenGLWindow *window, guint width, guint height);
/**
 * VisuOpenGLViewGet_fileUnitPerPixel:
 * @view: a valid #VisuOpenGLView object.
 *
 * This method is used to know the ratio of a pixel with the unit of the file.
 * WARNING : this method is valid only when the camera is position at infinity.
 * 
 * Returns: how much of a unit of file is in a pixel.
 */
float VisuOpenGLViewGet_fileUnitPerPixel(VisuOpenGLView *view);
/**
 * VisuOpenGLViewGet_screenAxes:
 * @view: a valid #VisuOpenGLView.
 * @xAxis: (in) (array fixed-size=3): three float values representing x axis ;
 * @yAxis: (in) (array fixed-size=3): three float values representing y axis.
 *
 * This method is used to get the coordinates in box frame of x axis and y axis
 * of the current camera view.
 */
void VisuOpenGLViewGet_screenAxes(VisuOpenGLView *view, float xAxis[3], float yAxis[3]);


float openGLViewGet_zCoordinate(VisuOpenGLView *view, float xyz[3]);


void openGLViewGet_realCoordinates(VisuOpenGLView *view, float xyz[3],
				   float winx, float winy, float winz);


void openGLViewSet_upAxis(VisuOpenGLView *view, ViewAxis upAxis);



/*
 * _VisuOpenGLView:
 */
struct _VisuOpenGLView
{
  VisuOpenGLCamera *camera;
  OpenGLWindow *window;
};

/**
 * VisuOpenGLViewNew:
 *
 * Create a new #VisuOpenGLView object with default values.
 *
 * Returns: (transfer none): the newly created object (to be free with VisuOpenGLViewFree()).
 */
VisuOpenGLView* VisuOpenGLViewNew(void);
/**
 * VisuOpenGLViewFree:
 * @view: a valid #VisuOpenGLView object.
 *
 * Free the given @view.
 */
void VisuOpenGLViewFree(VisuOpenGLView *view);
/**
 * VisuOpenGLViewCopy:
 * @view: a valid #VisuOpenGLView object.
 *
 * Copy operator (newly #VisuOpenGLView created with g_malloc() to be freed be g_free()).
 *
 * Returns: (transfer none): a newly allocated #VisuOpenGLView with same values than @view.
 */
VisuOpenGLView* VisuOpenGLViewCopy(VisuOpenGLView *view);
/**
 * VisuOpenGLCameraGet_numberOfFacettes:
 * @camera: a valid #VisuOpenGLCamera object ;
 * @winSize: the size of the rendering area.
 * @dimension: the size of the object which asks for its number of facettes.
 *
 * This is a function to get the number of "facettes" advised
 * by the server (according to its policy on rendering)
 * to draw an object according to a given dimension.
 *
 * Returns: the number of facettes the object should used.
 */
gint VisuOpenGLCameraGet_numberOfFacettes(VisuOpenGLCamera *camera, guint winSize,
                                          float dimension);

void openGLProject(OpenGLWindow *window, const VisuOpenGLCamera *camera, float extens);
void openGLModelize(VisuOpenGLCamera *camera);

/**
 * VisuOpenGLViewSet_precision:
 * @value: a positive value (100 is normal precision).
 *
 * This function change the value of the parameter precisionOfRendering. It
 * changes the number of facettes advised for every objects. It allows to
 * increase or decrease the number of polygons drawn and thus acts on the
 * speed of rendering.
 *
 * Returns: TRUE if the signals OpenGLFacetteChanged and
 *          OpenGLAskForReDraw should be emitted.
 */
int VisuOpenGLViewSet_precision(float value);
/**
 * VisuOpenGLViewGet_precision:
 *
 * This function retrieve the value of the parameter precisionOfRendering.
 *
 * Returns: the actual precision.
 */
float VisuOpenGLViewGet_precision(void);
/**
 * openGLViewRotate_box:
 * @view: a valid #VisuOpenGLView object ;
 * @dTheta: a float value ;
 * @dPhi: a float value ;
 * @angles: (in) (array fixed-size=2): a storing area two floats.
 *
 * This methods rotates the camera of the given @view of (@dTheta, @dPhi) and
 * put new theta and phi angles in @angles, first being theta and second phi.
 */
void openGLViewRotate_box(VisuOpenGLView *view, float dTheta, float dPhi, float angles[2]);
/**
 * openGLViewRotate_camera:
 * @view: a valid #VisuOpenGLView object ;
 * @dTheta: a float value ;
 * @dPhi: a float value ;
 * @angles: (in) (array fixed-size=3): a storing area three floats.
 *
 * This methods rotates the camera of the given @view of (@dTheta, @dPhi).
 * @dTheta is taken as displacement along camera x axis and dPhi along camera y axis.
 * Then, computations are done to obtain new theta, phi and omega values. They are
 * put in @angles, first being theta, second phi and third omega.
 */
void openGLViewRotate_camera(VisuOpenGLView *view, float dTheta, float dPhi, float angles[3]);


/**
 * VisuOpenGLViewInit:
 *
 * This method is used by V_Sim during initilization process and should
 * not be called.
 */
void VisuOpenGLViewInit(void);

#endif
