/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2012
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

/*
 * Render n lines of a stream (opened text file) into an image (cairo surface)
 * of one long single line of text.
 * - One single stream is likely to be splitted into several text parts.
 * - n is limited because surface width must be <= (XPIXMAP_MAXWIDTH = 32 K - 1)
 * (DO THIS STILL APPLY TO CAIRO IMAGE SURFACES ?)
 * - Return addr of newly created surface.
 * If returned surface is NULL then render_exit_status is set accordingly and
 * should be checked to know which error occured
 * (otherwise render_exit_status = OK or RENDER_NO_RESOURCE or
 * RENDER_LINE_TOO_LONG).
 * LINE_MAXLEN should be set to ensure that we won't have too-long lines.
 */
cairo_surface_t *render_stream_to_surface(FILE *fp, FeedLinkAndOffset *link_and_offset, const Params *prm, \
		int *render_exit_status)
{
	TickerEnv		*env = get_ticker_env();
	char			*txt_str;
	PangoLayout		*p_layout;
	PangoFontDescription	*f_des;
	int			layout_width, layout_height;
	cairo_t			*cr;
	float			shadow_k;
	char			tmp[4];
	int			i, j;

	*render_exit_status = OK;
	env->stream_fully_read = FALSE;
	/*
	 * Create layout
	 */
	p_layout = pango_layout_new(gtk_widget_get_pango_context(env->win));
	pango_layout_set_attributes(p_layout, NULL);
	pango_layout_set_single_paragraph_mode(p_layout, TRUE);

	f_des = pango_font_description_from_string((const char *)prm->font_n_s);
	pango_layout_set_font_description(p_layout, f_des);
	pango_font_description_free(f_des);
	pango_layout_context_changed(p_layout);
	/*
	 * Get text string from stream
	 */
	txt_str = l_str_new(NULL);

	if (fp != NULL) {
		txt_str = stream_to_htext(fp, txt_str, p_layout, prm, render_exit_status);
		if (*render_exit_status != OK && *render_exit_status != RENDER_LINE_TOO_LONG) {
			l_str_free(txt_str);
			return NULL;
		}
	} else {
		txt_str = void_stream_to_htext(txt_str, p_layout);
		*render_exit_status = RENDER_NO_RESOURCE;
	}
	/*
	 * Translate link ranks inside (eventually splitted) text into offsets in surface.
	 *
	 * Scan text for "<LINK_TAG_CHAR>00n" and get
	 * offset = layout_width(< text from start to tag >),
	 * for every rss item found
	 * then remove tags and fill an array of FeedLinkAndOffset's
	 * with offsets (rank and url variables are already filled).
	 */
	for (i = 0; i < NFEEDLINKANDOFFSETMAX; i++)	/* We reset all offsets to 0 */
		(link_and_offset + i)->offset_in_surface = 0;
	for (i = 0; txt_str[i] != '\0'; i++) {
		if (txt_str[i] == LINK_TAG_CHAR) {
			str_n_cpy(tmp ,txt_str + i + 1, 3);
			j = atoi(tmp);
			txt_str[i] = '\0';
			if (j > 0 && j < NFEEDLINKANDOFFSETMAX)
				(link_and_offset + j)->offset_in_surface = get_layout_width(p_layout, txt_str);
			txt_str = l_str_cat(txt_str, txt_str + i + 4);
		}
	}
	/*
	 * Fill layout
	 */
	pango_layout_set_text(p_layout, txt_str, -1);
	l_str_free(txt_str);

	pango_layout_context_changed(p_layout);
	pango_layout_get_pixel_size2(p_layout, &layout_width, &layout_height);
	env->surf_width = (gint)layout_width;
	env->surf_height = (gint)layout_height;
	if (env->surf_width > XPIXMAP_MAXWIDTH) {
		*render_exit_status = RENDER_CAIRO_IMAGE_SURFACE_TOO_WIDE;
		return NULL;
	}
	/*
	 * Create cairo image surface onto which layout will be rendered
	 */
	env->c_surf = cairo_image_surface_create(CAIRO_FORMAT_ARGB32,
		env->surf_width, env->surf_height);
	if (cairo_surface_status(env->c_surf) != CAIRO_STATUS_SUCCESS) {
		cairo_surface_destroy(env->c_surf);
		*render_exit_status = RENDER_CREATE_CAIRO_IMAGE_SURFACE_ERROR;
		return (env->c_surf = NULL);
	}
	cr = cairo_create(env->c_surf);
	/*
	 * Render layout
	 * (are used cairo operators OK ?)
	 */
	/* Draw background */
	cairo_set_source_rgba(cr,
		(float)prm->bg_color.red / G_MAXUINT16,
		(float)prm->bg_color.green / G_MAXUINT16,
		(float)prm->bg_color.blue / G_MAXUINT16,
		(float)prm->bg_color_alpha / G_MAXUINT16
		);
	cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
	cairo_paint(cr);
	/* Draw foreground */
	if (prm->shadow == 'y') {
		/* Draw shadow */
		if (prm->shadow_fx < 0)
			shadow_k = 1.0;
		else if (prm->shadow_fx > 10)
			shadow_k = 0.0;
		else
			shadow_k = 1.0 - (float)prm->shadow_fx / 10.0;
		cairo_set_source_rgba(cr,
			(float)prm->bg_color.red * shadow_k / G_MAXUINT16,
			(float)prm->bg_color.green * shadow_k / G_MAXUINT16,
			(float)prm->bg_color.blue * shadow_k / G_MAXUINT16,
			(float)prm->bg_color_alpha * shadow_k / G_MAXUINT16);
		pango_cairo_update_layout(cr, p_layout);
		cairo_move_to(cr, prm->shadow_offset_x, prm->shadow_offset_y);
		cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
		pango_cairo_show_layout(cr, p_layout);
	}
	/* Draw text */
	cairo_set_source_rgba(cr,
		(float)prm->fg_color.red / G_MAXUINT16,
		(float)prm->fg_color.green / G_MAXUINT16,
		(float)prm->fg_color.blue / G_MAXUINT16,
		(float)prm->fg_color_alpha / G_MAXUINT16);
	pango_cairo_update_layout(cr, p_layout);
	cairo_move_to(cr, 0, 0);
	cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
	pango_cairo_show_layout(cr, p_layout);
	/* Drawing done - return surface */
	if (p_layout != NULL)
		g_object_unref(p_layout);
	cairo_destroy(cr);
	return env->c_surf;
}

/*
 * Split stream into a a set of long lines so that, after rendering of each one,
 * we get a cairo image surface whose width is <= (XPIXMAP_MAXWIDTH = 32 K - 1).
 * If one line is longer than tmp2_size, it will get truncated and CUTLINE_DELIMITER
 * will be appended but we should make sure that this will not happen.
 */
char *stream_to_htext(FILE *fp, char *txt_str, PangoLayout *p_layout, const Params *prm, int *render_exit_status)
{
	TickerEnv	*env = get_ticker_env();
	static int	text_splitted = FALSE;
	static char	*ending_str = NULL;
	long int	file_pos;
	char		*tmp1, *tmp2;
	size_t		tmp1_size = LINE_MAXLEN * 4;	/* Anyways, this is adjusted by getline() */
	size_t		tmp2_size =  64 * 1024;		/* This should always be enough (hmmm...) */
	int		c = 0;
	char		*empty_line;
	int		empty_line_length, empty_line_width, txt_str_width;/* Length in chars, width in pixels */
	int		i, j, k;
#ifndef G_OS_WIN32
	ssize_t		n;
#endif

	*render_exit_status = OK;
	/*
	 * Creating an "empty line"
	 */
	empty_line = l_str_new(NULL);
	do
		empty_line = l_str_cat(empty_line, " ");
	while ((empty_line_width = get_layout_width(p_layout, empty_line)) < env->drwa_width);
	empty_line_length = strlen(empty_line);
	if (env->reload_rq)
		text_splitted = FALSE;
	/*
	 * We start with an "empty line" but if text has been splitted.
	 */
	if (!text_splitted) {
		txt_str = l_str_cat(txt_str, empty_line);
		txt_str_width = empty_line_width;
	} else {
		if (ending_str != NULL) {
			txt_str = l_str_cat(txt_str, ending_str);
			txt_str_width = get_layout_width(p_layout, txt_str);
		} else {
			*render_exit_status = RENDER_NULL_ENDINGSTR;
			return txt_str;
		}
	}
	if (ending_str != NULL) {
		l_str_free(ending_str);
		ending_str = NULL;
	}
	text_splitted = FALSE;
	/*
	 * Then we read stream (text file) into txt_str, one line at a time.
	 * Text file has been previously formatted with extra newline's so that
	 * no line is longer than LINE_MAXLEN.
	 * We must make sure that no line needs to be cut (width > (XPIXMAP_MAXWIDTH = 32 K - 1)
	 * after rendering) but how do we set LINE_MAXLEN properly ?
	 * LINE_MAXLEN < (XPIXMAP_MAXWIDTH = 32 K - 1) / FONT_MAXSIZE = about 163 if maxsize = 200
	 * so 128 should always be OK (see tickr.h)
	 *
	 * (It could be better, for performance reasons, to replace LINE_MAXLEN
	 * by a variable like: line_maxlen = k * XPIXMAP_MAXWIDTH / font_size,
	 * especially when font_size is big.)
	 */
	tmp1 = malloc2((tmp1_size + 1) * sizeof(char));	/* tmp1 will get 1 line */
	tmp2 = malloc2((tmp2_size + 1) * sizeof(char));	/* tmp2 will get 1 "expanded" line */
	/*
	 * txt_str will get all n lines
	 */
	while (1) {
		file_pos = ftell(fp);
#ifndef G_OS_WIN32
		if ((n = getline(&tmp1, &tmp1_size, fp)) == -1) {
#else
		if (fgets(tmp1, tmp1_size, fp) == NULL) {
#endif
			/*
			 * eof or error ?
			 */
			if (feof(fp) == 0) {
				*render_exit_status = RENDER_GETLINE_ERROR;
				return txt_str;
			}
			/*
			 * Text file fully read
			 * -> reset to beginning in single selection mode
			 * or
			 * -> set flag so that to load next feed in multiple selection mode
			 */
			if (env->selection_mode == SINGLE)
				fseek(fp, 0, SEEK_SET);
			else
				env->stream_fully_read = TRUE;
			break;
		} else {
#ifndef G_OS_WIN32
			for (i = 0, j = 0; i < n; i++) {
#else
			for (i = 0, j = 0; i < (int)strlen(tmp1); i++) {
#endif
				/*
				 * This is not supposed to happen but if tmp2_size was wrongly set
				 */
				if ((size_t)j >= tmp2_size - strlen(prm->cutline_delimiter) - 1) {
					str_n_cpy(&tmp2[j], prm->cutline_delimiter, OPTION_VALUE_MAXLEN);
					j += strlen(prm->cutline_delimiter);
					warning(FALSE, 1, "stream_to_htext(): Line is too long");
					*render_exit_status = RENDER_LINE_TOO_LONG;
					break;
				}
				c = tmp1[i];
				k = 1;
				if (c == '\n') {
					str_n_cpy(&tmp2[j], prm->line_delimiter, OPTION_VALUE_MAXLEN);
					j += strlen(prm->line_delimiter);
					break;
				}
				if (prm->special_chars == 'y') {
					if (c == prm->new_page_char) {
						c = ' ';
						k = empty_line_length;
					} else if (c == prm->tab_char) {
						c = ' ';
						k = TAB_SIZE;
					}
				}
				while (k-- > 0) {
					if (prm->upper_case_text != 'y')
						tmp2[j++] = c;
					else
						tmp2[j++] = toupper(c);
				}
			}
			tmp2[j] = '\0';
		}
		txt_str_width += get_layout_width(p_layout, tmp2);
		if (txt_str_width > XPIXMAP_MAXWIDTH - 2 * empty_line_width - prm->shadow_offset_x) {
			/*
			 * Need to split text into several parts.
			 * We don't add this line but will use it in next surface.
			 */
			fseek(fp, file_pos, SEEK_SET);
			text_splitted = TRUE;
			break;
		} else
			txt_str = l_str_cat(txt_str, tmp2);
	}
	/*
	 * And we end with an "empty line" but if text has been splitted,
	 */
	if (!text_splitted) {
		txt_str = l_str_cat(txt_str, empty_line);
		ending_str = NULL;
	} else {
		/*
		 * in which case we keep "tail" of txt_str whose width is drwa_width.
		 */
		i = strlen(txt_str) - 1;
		while (get_layout_width(p_layout, txt_str + i--) <= env->drwa_width);
		ending_str = l_str_new(txt_str + i + 1);
	}
	free2(tmp1);
	free2(tmp2);
	l_str_free(empty_line);
	return txt_str;
}

char *void_stream_to_htext(char *txt_str, PangoLayout *p_layout)
{
	TickerEnv	*env = get_ticker_env();
	char		*empty_line;

	empty_line = l_str_new(NULL);
	do
		empty_line = l_str_cat(empty_line, " ");
	while (get_layout_width(p_layout, empty_line) < env->drwa_width);

	txt_str = l_str_cat(txt_str, empty_line);
	txt_str = l_str_cat(txt_str, "No resource");
	txt_str = l_str_cat(txt_str, empty_line);
	l_str_free(empty_line);

	return txt_str;
}

int get_fsize_from_layout_height(gint height, const char *fontname)
{
	PangoLayout		*p_layout;
	PangoFontDescription	*f_des;
	char			fnamesize[FONT_MAXLEN + 1];
	char			fname[FONT_NAME_MAXLEN + 1], fsize[FONT_SIZE_MAXLEN + 1];
	char			check_str[] = "gjpqyÀÄÅÇÉÖÜ€£$!&({[?|";	/* Should be enough - Is it? */
	gint			height2;
	int			i = 1;

	p_layout = pango_layout_new(gtk_widget_get_pango_context(get_ticker_env()->win));
	pango_layout_set_attributes(p_layout, NULL);
	pango_layout_set_single_paragraph_mode(p_layout, TRUE);

	str_n_cpy(fname, fontname, FONT_NAME_MAXLEN);

	do {
		snprintf(fsize, FONT_SIZE_MAXLEN + 1, "%3d", i++);
		compact_font(fnamesize, fname, fsize);
		f_des = pango_font_description_from_string((const char *)fnamesize);
		pango_layout_set_font_description(p_layout, f_des);
		pango_font_description_free(f_des);
		height2 = get_layout_height(p_layout, check_str);
	} while (height2 < height && i < 1000);

	if (p_layout != NULL)
		g_object_unref(p_layout);

	return (i - 1);
}

gint get_layout_height_from_fnamesize(const char *fnamesize)
{
	PangoLayout		*p_layout;
	PangoFontDescription	*f_des;
	char			check_str[] = "gjpqyÀÄÅÇÉÖÜ€£$!&({[?|";
	gint			height;

	p_layout = pango_layout_new(gtk_widget_get_pango_context(get_ticker_env()->win));
	pango_layout_set_attributes(p_layout, NULL);
	pango_layout_set_single_paragraph_mode(p_layout, TRUE);

	f_des = pango_font_description_from_string(fnamesize);
	pango_layout_set_font_description(p_layout, f_des);
	pango_font_description_free(f_des);

	height = get_layout_height(p_layout, check_str);
	if (p_layout != NULL)
		g_object_unref(p_layout);

	return height;
}

gint get_layout_width(PangoLayout *p_layout, const char *str)
{
	int	layout_width, layout_height;

	pango_layout_set_text(p_layout, str, -1);
	pango_layout_context_changed(p_layout);
	pango_layout_get_pixel_size2(p_layout, &layout_width, &layout_height);

	return (gint)layout_width;
}

gint get_layout_height(PangoLayout *p_layout, const char *str)
{
	int	layout_width, layout_height;

	pango_layout_set_text(p_layout, str, -1);
	pango_layout_context_changed(p_layout);
	pango_layout_get_pixel_size2(p_layout, &layout_width, &layout_height);

	return (gint)layout_height;
}

/* To avoid troubles in case of an empty string inside layout */
void pango_layout_get_pixel_size2(PangoLayout *p_layout, int *layout_width, int *layout_height)
{
	if (strlen(pango_layout_get_text(p_layout)) > 0)
		pango_layout_get_pixel_size(p_layout, layout_width, layout_height);
	else {
		*layout_width = 0;
		*layout_height = 0;
		fprintf(STD_ERR,
			"pango_layout_get_pixel_size2(): Text length = 0 - Troubles ahead ?\n");
	}
	return;
}
