//dmx-strobe.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2011-2019
 *
 *  This file is part of roard a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

#include <roard/include/roard.h>

struct state {
 // config:
 ssize_t channel_brightness;
 ssize_t channel_frequency;
 ssize_t channel_trigger;
 uint8_t brightness;
 uint8_t frequency;

 // runtime:
 uint8_t tigger_value;
 struct roar_subscriber * subscription_event;
};

static struct state * g_state;
static struct state   g_state_init = {
 .channel_brightness = -1,
 .channel_frequency  = -1,
 .channel_trigger    = -1,
 .brightness         = 255,
 .frequency          = 64,
 .tigger_value       = 0
};

#define __set_XXX(type) \
static void __set_##type (uint8_t val) { \
 ROAR_DBG("__set_%s(val=%i): channel=%li", #type, val, (long int)g_state->channel_##type); \
 if ( g_state->channel_##type != -1 ) \
  light_dmxchannel_set(g_state->channel_##type, val); \
}

__set_XXX(brightness)
__set_XXX(frequency)
__set_XXX(trigger)

static void __handle_event(uint8_t event) {
 switch (event) {
  case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_BEAT:
    g_state->tigger_value += 7; // prime value > 3 for maximum 'random' effect.
    __set_trigger(g_state->tigger_value);
   break;
  case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_OFF:
    __set_frequency(0);
   break;
  case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_ON:
  case ROAR_ROARDMX_EVENT_STROBE|ROAR_ROARDMX_ETYPE_HOLD:
    __set_brightness(g_state->brightness);
    __set_frequency(g_state->frequency);
   break;
 }
}

static int _init  (struct roar_dl_librarypara * para) {
 struct roar_keyval * p;

 p = roar_keyval_lookup(para->argv, "channel-brightness", para->argc, 1);
 if ( p != NULL && p->value != NULL )
  g_state->channel_brightness = atoi(p->value);

 p = roar_keyval_lookup(para->argv, "channel-frequency", para->argc, 1);
 if ( p != NULL && p->value != NULL )
  g_state->channel_frequency = atoi(p->value);

 p = roar_keyval_lookup(para->argv, "channel-trigger", para->argc, 1);
 if ( p != NULL && p->value != NULL )
  g_state->channel_trigger = atoi(p->value);

 p = roar_keyval_lookup(para->argv, "brightness", para->argc, 1);
 if ( p != NULL && p->value != NULL )
  g_state->brightness = atoi(p->value);

 p = roar_keyval_lookup(para->argv, "frequency", para->argc, 1);
 if ( p != NULL && p->value != NULL )
  g_state->frequency = atoi(p->value);

 return 0;
}

static struct roar_dl_appsched sched = {
 .init   = _init,
 .free   = NULL,
 .update = NULL,
 .tick   = NULL,
 .wait   = NULL
};

static void cb_event(struct roar_notify_core * core, struct roar_event * event, void * userdata) {
 struct state * old = g_state;

 (void)core, (void)userdata;

 g_state = userdata;

 __handle_event(event->arg0);

 g_state = old;
}

static int init(struct roar_dl_librarypara * para, struct roar_dl_libraryinst * lib) {
 struct roar_event event;

 (void)para, (void)lib;

 memset(&event, 0, sizeof(event));

 event.event = ROAR_DATA_DMX512_EVENT;
 event.emitter = -1;
 event.target = -1;
 event.target_type = ROAR_OT_STREAM;

 g_state->subscription_event = roar_notify_core_subscribe(NULL, &event, cb_event, g_state);

 return 0;
}

static int unload(struct roar_dl_librarypara * para, struct roar_dl_libraryinst * lib) {
 (void)para, (void)lib;

 roar_notify_core_unsubscribe(NULL, g_state->subscription_event);

 return 0;
}

ROAR_DL_PLUGIN_START(dmx_strobe) {
 ROARD_DL_CHECK_VERSIONS();

 ROAR_DL_PLUGIN_META_PRODUCT_NIV("dmx-strobe", ROAR_VID_ROARAUDIO, ROAR_VNAME_ROARAUDIO);
 ROAR_DL_PLUGIN_META_VERSION(ROAR_VERSION_STRING);
 ROAR_DL_PLUGIN_META_LICENSE_TAG(GPLv3_0);
 ROAR_DL_PLUGIN_META_CONTACT_FLNE("Philipp", "Schafft", "ph3-der-loewe", "lion@lion.leolix.org");
 ROAR_DL_PLUGIN_META_DESC("Controls strobes using roard's DMX interface.");

 ROAR_DL_PLUGIN_REG_GLOBAL_DATA(g_state, g_state_init);
 ROAR_DL_PLUGIN_REG_APPSCHED(&sched); 
 ROAR_DL_PLUGIN_REG(ROAR_DL_FN_INIT, init);
 ROAR_DL_PLUGIN_REG_UNLOAD(unload);
} ROAR_DL_PLUGIN_END

//ll
