#ifndef _RHEO_BASIS_H
#define _RHEO_BASIS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:basis
NAME: @code{basis} - polynomial basis
@cindex  polynomial basis
@clindex basis
@cindex  reference element
@clindex reference_element
SYNOPSIS:
  @noindent
  The @code{basis} class defines functions that evaluates a polynomial
  basis and its derivatives on a point. The polynomial basis
  is designated by a string, e.g. "P0", "P1", "P2", "bubble",...
  indicating the basis family and the polynomial degree.
  The basis depends also of the reference element:
  triangle, square, tetrahedron, etc (@pxref{reference_element iclass}).
  For instance, on the square reference element, the "P1"
  string designates the common Q1 four-nodes basis on the reference square.
  The basis is evaluated by using the @code{eval}
  member function.
  The nodes associated to the Lagrange polynomial basis
  are also available by the member function @code{hat_node}.
  The Lagrange basis family is designated by @code{Pk},
  where @code{k} is any polynomial order.

  @noindent
  In addition to the usual nodal Lagrange family,
  two non-nodal basis are provided: the Bernstein family,
  designated by @code{Bk} and the spectral Dubiner family,
  designated by @code{Sk} (see Dubiner, 1991 J. Sci. Comput.).
  By contrast, these two non-nodal basis are called modal.
  For these modal basis, the nodes provided by the member function
  @code{hat_node} represent the nodes used by the interpolation
  operator onto this basis.

  @noindent
  The vector-valued Raviart-Thomas family is designated by @code{RTk}.

OPTIONS:
  The Lagrange basis recognize some options,
  transmitted to the constructor of the basis class:
  see @ref{basis_option class}.

NOTES:
  @noindent
  The @code{basis} class 
  is a @ref{smart_pointer iclass} class on a @code{basis_rep} class that is a pure virtual base class 
  for effective bases, e.g. @code{Pk_lagrange}, @code{Pk_bernstein}, @code{RTk}, etc.

LIMITATIONS:
  The @code{Sk} and @code{RTk} basis are still under development.

AUTHOR: Pierre.Saramito@imag.fr
DATE:   7 january 2004, 11 september 2017
End:
*/
#include "rheolef/reference_element.h"
#include "rheolef/reference_element_face_transformation.h"
#include "rheolef/diststream.h"
#include "rheolef/point.h"
#include "rheolef/tensor.h"
#include "rheolef/smart_pointer.h"
#include "rheolef/space_constant.h"
#include "rheolef/ublas-io.h"
#include "rheolef/basis_raw.h"
#include "rheolef/basis_option.h"
#include "rheolef/armadillo_util.h"

namespace rheolef {

template<class T>
class basis_rep: public basis_raw_rep<T> {
public:

// typedefs:

  typedef basis_raw_rep<T>            base;
  typedef typename base::size_type    size_type;
  typedef typename base::valued_type  valued_type;

// allocators:

  basis_rep (std::string basename, const basis_option& sopt);
  virtual ~basis_rep() {}

// accessors:

  virtual std::string family_name() const = 0;
  virtual size_type degree() const = 0;
  virtual std::string name() const;

  virtual valued_type valued_tag() const { return space_constant::scalar; }
  const std::string&  valued()     const { return space_constant::valued_name (valued_tag()); }
  virtual bool is_continuous() const    { return option().is_continuous(); }
          bool is_discontinuous() const { return ! is_continuous(); }
  virtual bool is_nodal() const = 0;
  virtual bool have_degree_parameter()   const { return true; }
  virtual bool have_continuous_feature() const { return true; }
  const basis_option& option() const;

  size_type first_inod (reference_element hat_K, size_type dim) const {
 	_initialize_guard (hat_K);
	return _first_inod [hat_K.variant()][dim]; }
  size_type first_idof (reference_element hat_K, size_type dim) const {
 	_initialize_guard (hat_K);
	return _first_idof [hat_K.variant()][dim]; }
  size_type nnod (reference_element hat_K, size_type dim) const {
  	return first_inod (hat_K, dim+1) - first_inod (hat_K, dim); }
  size_type ndof (reference_element hat_K, size_type dim) const {
  	return first_idof (hat_K, dim+1) - first_idof (hat_K, dim); }
  size_type ndof (reference_element hat_K) const {
  	return first_idof (hat_K, hat_K.dimension()+1); }
  size_type size (reference_element hat_K) const {
  	return ndof (hat_K); }
  size_type nnod (reference_element hat_K) const {
  	return first_inod (hat_K, hat_K.dimension()+1); }

  // used by DG:
  virtual void get_local_idof_on_side (
        reference_element            hat_K,
        const side_information_type& sid,
        arma::Col<size_type>&        loc_idof) const;

  const std::vector<point_basic<T> >& hat_node (reference_element hat_K) const;
  const arma::Mat<T>& vdm     (reference_element hat_K) const;
  const arma::Mat<T>& inv_vdm (reference_element hat_K) const;

// evaluate:

  // implementation note: valued type is run-time dependent,
  // so, provide a too rich class interface:
  // only one of them is implemented

  virtual void eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    arma::Col<T>&               value) const;	// scalar-valued
  virtual void eval(
    reference_element             hat_K,
    const point_basic<T>&         hat_x,
    std::vector<point_basic<T> >& value) const;	// vector-valued

  // evaluate all the basis on a full node set: (psi_j(xi))_{i,j}
  void eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    arma::Mat<T>&                        values) const; // scalar-valued
  void eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    ublas::matrix<point_basic<T> >&      values) const; // vector-valued

  // same, but matrix flatened in a linear arrow of size nrow*ncol
  void eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    arma::Col<T>&                        values) const; // scalar-valued

// evaluate the gradient:

  virtual void grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<point_basic<T> >& value) const;	// scalar-valued
  virtual void grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<tensor_basic<T> >& value) const; // vector-valued

  // evaluate all the grad-basis on a full node set: (grad_psi_j(xi))_{i,j}
  void grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    ublas::matrix<point_basic<T> >&      values) const; // scalar-valued
  void grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    ublas::matrix<tensor_basic<T> >&     values) const; // vector-valued

  // same, but matrix flatened in a linear arrow of size nrow*ncol
  void grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    std::vector<point_basic<T> >&        values) const; // scalar-valued

// internals:

  void put               (std::ostream& os, reference_element hat_K) const;
  void put_vector_valued (std::ostream& os, reference_element hat_K) const;
  void put_hat_node      (std::ostream& os, reference_element hat_K) const;
  void put_hat_node_on_side (
    std::ostream&                os,
    reference_element            hat_K,
    const side_information_type& sid) const;

  // helper for interface class:
  static basis_rep* make_ptr (std::string name, const basis_option& sopt);
  static std::string standard_naming (std::string family_name, size_t degree, const basis_option& sopt);
  static bool have_degree_parameter   (std::string family_name);
  static bool have_continuous_feature (std::string family_name);

  void         _initialize_guard (reference_element hat_K) const;
  virtual void _initialize       (reference_element hat_K) const = 0;

  // compute dofs, given f(xnod) for all nodes (needed by template compute_dof(f,&dof))
  virtual void _compute_dofs (
    reference_element     hat_K, 
    const arma::Col<T>&   f_xnod, 	             // scalar-valued case:
          arma::Col<T>&   dof) const;
  virtual void _compute_dofs (
    reference_element                     hat_K, 
    const std::vector<point_basic<T> >&   f_xnod,    // vector-valued case:
          arma::Col<T>&                   dof) const;

// internals:

  void _clear() const;

protected:
// data:
  basis_option                                 _sopt;

  mutable std::array<bool,
             reference_element::max_variant>        _have_initialize;

  // nodes and dofs are organized by dimension=0..3 of subgeos, for each hat_K:
  mutable std::array<std::array<size_type,5>,
             reference_element::max_variant>        _first_inod, _first_idof;

  mutable std::array<std::vector<point_basic<T> >,
             reference_element::max_variant>        _hat_node;

  // for interpolation (is it relevant for Dubiner ? for Bernstein ?)
  mutable std::array<arma::Mat<T>,
             reference_element::max_variant>        _vdm, _inv_vdm;

  mutable std::array<arma::Col<T>,
             reference_element::max_variant>       _phi;
  mutable std::array<std::vector<point_basic<T> >,
             reference_element::max_variant>       _v_phi;
public:
  // working array, used by compute_dof in base class, so public:
  mutable std::array<arma::Col<T>,
             reference_element::max_variant>        _raw_value;
  mutable std::array<std::vector<point_basic<T> >,
             reference_element::max_variant>        _raw_v_value;
};
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template<class T>
inline
basis_rep<T>::basis_rep (std::string basename, const basis_option& sopt)
: base(basename), 
  _sopt(sopt),
  _have_initialize(),
  _first_inod(),
  _first_idof(),
  _hat_node(),
  _vdm(),
  _inv_vdm(),
  _phi(),
  _v_phi(),
  _raw_value(),
  _raw_v_value()
{
  _clear();
}
template<class T>
inline
void
basis_rep<T>::_clear() const
{
  _have_initialize.fill (false);
}
template <class T>
inline
void
basis_rep<T>::_initialize_guard (reference_element hat_K) const
{
  if (_have_initialize [hat_K.variant()]) return;
  _have_initialize [hat_K.variant()] = true;
  _initialize (hat_K);
}
template<class T>
inline
const basis_option&
basis_rep<T>::option() const
{
  return _sopt;
}
template<class T>
inline
const std::vector<point_basic<T> >&
basis_rep<T>::hat_node (reference_element hat_K) const
{
  _initialize_guard (hat_K);
  return _hat_node [hat_K.variant()];
}
template<class T>
inline
const arma::Mat<T>&
basis_rep<T>::vdm (reference_element hat_K) const
{
  _initialize_guard (hat_K);
  return _vdm [hat_K.variant()];
}
template<class T>
inline
const arma::Mat<T>&
basis_rep<T>::inv_vdm (reference_element hat_K) const
{
  _initialize_guard (hat_K);
  return _inv_vdm [hat_K.variant()];
}
template<class T>
inline
void
basis_rep<T>::eval(
    reference_element                 hat_K,
    const point_basic<T>&             hat_x,
    arma::Col<T>&                     value) const
{
  error_macro ("basis \""<<base::name()<<"\": scalar-valued eval() member not implemented");
}
template<class T>
inline
void
basis_rep<T>::eval(
    reference_element                   hat_K,
    const point_basic<T>&               hat_x,
    std::vector<point_basic<T> >&       value) const
{
  error_macro ("basis \""<<base::name()<<"\": vector-valued eval() member not implemented");
}
template<class T>
inline
void
basis_rep<T>::grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<point_basic<T> >& value) const
{
  error_macro ("basis \""<<base::name()<<"\": scalar-valued grad_eval() member not implemented");
}
template<class T>
inline
void
basis_rep<T>::grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<tensor_basic<T> >& value) const
{
  error_macro ("basis \""<<base::name()<<"\": vector-valued grad_eval() member not implemented");
}
template<class T>
inline
void
basis_rep<T>::_compute_dofs (
    reference_element     hat_K, 
    const arma::Col<T>&   f_xnod, 	             // scalar-valued case:
          arma::Col<T>&   dof) const
{
  error_macro ("basis \""<<base::name()<<"\": scalar-valued _compute_dofs() member not implemented");
}
template<class T>
inline
void
basis_rep<T>::_compute_dofs (
    reference_element                     hat_K, 
    const std::vector<point_basic<T> >&   f_xnod,    // vector-valued case:
          arma::Col<T>&                   dof) const
{
  error_macro ("basis \""<<base::name()<<"\": vector-valued _compute_dofs() member not implemented");
}
// extract local dof-indexes on a side
template<class T>
inline
void
basis_rep<T>::get_local_idof_on_side (
  reference_element            hat_K,
  const side_information_type& sid,
  arma::Col<size_type>&        loc_idof) const
{
  error_macro ("basis \""<<base::name()<<"\": get_local_idof_on_side() member not implemented");
}
// ---------------------------------------------------------
// compute_dof
// ---------------------------------------------------------
// implementation note:
//   b.compute_dof   (f,&dof)
// is impossible with base_rep
// because template functions cannot be virtual
// cannot call a "b.compute_dof(f,&dof)" template-virtual member
// so we have to write at top level basis_rep a general
//   compute_dof   (b,f,&dof)
// and switch on all nodal/modal and valued variants
//
// for vector-valued RTk, it is problematic, because we
// only require f.n on the boundary and not all f components
//
namespace details {

template <class T, class Function>
typename
std::enable_if<
  is_scalar<typename function_traits<Function>::result_type>::value
 ,void
>::type
compute_dof (
  const basis_rep<T>& b,
  reference_element      hat_K,
  const Function&        f,
  arma::Col<T>&          dof)
{
  typedef reference_element::size_type  size_type;
  check_macro (b.valued_tag() == space_constant::scalar,
    "interpolate: incompatible scalar-valued function and "<<b.valued()<<"-valued basis");
  const std::vector<point_basic<T> >& hat_node = b.hat_node (hat_K);
  if (b.is_nodal()) {
    dof.resize (hat_node.size());
    for (size_type loc_inod = 0, loc_nnod = hat_node.size(); loc_inod < loc_nnod; ++loc_inod) {
      dof [loc_inod] = f (hat_node [loc_inod]);
    }
    return;
  }
  // modal basis:
  arma::Col<T> f_xnod (hat_node.size());
  for (size_type loc_inod = 0, loc_nnod = hat_node.size(); loc_inod < loc_nnod; ++loc_inod) {
    f_xnod [loc_inod] = f (hat_node [loc_inod]);
  }
  b._compute_dofs (hat_K, f_xnod, dof);
}
template <class T, class Function>
typename
std::enable_if<
  is_point<typename function_traits<Function>::result_type>::value
 ,void
>::type
compute_dof (
  const basis_rep<T>& b,
  reference_element      hat_K,
  const Function&        f,
  arma::Col<T>&          dof)
{
  typedef reference_element::size_type  size_type;
  check_macro (b.valued_tag() == space_constant::vector,
    "interpolate: incompatible vector-valued function and "<<b.valued()<<"-valued basis");
  const std::vector<point_basic<T> >& hat_node = b.hat_node (hat_K);
  std::vector<point_basic<T> > f_xnod (hat_node.size());
  for (size_type loc_inod = 0, loc_nnod = hat_node.size(); loc_inod < loc_nnod; ++loc_inod) {
    f_xnod [loc_inod] = f (hat_node [loc_inod]);
  }
  b._compute_dofs (hat_K, f_xnod, dof);
}

} // namespace details

//<basis:
template<class T>
class basis_basic : public smart_pointer_nocopy<basis_rep<T> > {
public:

// typedefs:

  typedef basis_rep<T>              rep;
  typedef smart_pointer_nocopy<rep> base;
  typedef typename rep::size_type   size_type;
  typedef typename rep::valued_type valued_type;

// allocators:

  basis_basic (std::string name = "");
  void reset (std::string& name);

// accessors:

  std::string family_name() const;
  size_type   degree() const;
  std::string name() const;
  size_type   size (reference_element hat_K) const;
  bool is_continuous() const;
  bool is_discontinuous() const;
  const basis_option& option() const;
  valued_type        valued_tag() const;
  const std::string& valued()     const;

  void get_local_idof_on_side (
        reference_element            hat_K,
        const side_information_type& sid,
        arma::Col<size_type>&        loc_idof) const;

// evaluate the basis:

  void eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    arma::Col<T>&               value) const; // scalar-valued
  void eval(
    reference_element             hat_K,
    const point_basic<T>&         hat_x,
    std::vector<point_basic<T> >& value) const; // vector-valued

  // evaluate it on a node set:
  void eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    arma::Mat<T>&                        values) const; // scalar-valued
  void eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    ublas::matrix<point_basic<T> >&      values) const; // vector-valued

  // same, but matrix flatened in a linear arrow of size nrow*ncol
  void eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    arma::Col<T>&                        values) const; // scalar-valued

// evaluate the gradient of the basis:

  void grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<point_basic<T> >& value) const;	// scalar-valued
  void grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<tensor_basic<T> >& value) const; // vector-valued

  // evaluate it on a node set:
  void grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    ublas::matrix<point_basic<T> >&      values) const; // scalar-valued
  void grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    ublas::matrix<tensor_basic<T> >&     values) const; // vector-valued

  // same, but matrix flatened in a linear arrow of size nrow*ncol
  void grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  nodes,
    std::vector<point_basic<T> >&        values) const; // scalar-valued

// compute_dof:

  template <class Function>
  void compute_dof (
    reference_element           hat_K,
    const Function&             f,
    arma::Col<T>&               dofs) const;

  const arma::Mat<T>& vdm     (reference_element hat_K) const;
  const arma::Mat<T>& inv_vdm (reference_element hat_K) const;

// output:

  void put          (std::ostream& os, reference_element hat_K) const;
  void put_hat_node (std::ostream& os, reference_element hat_K) const;
  void put_hat_node_on_side (
    std::ostream&                os,
    reference_element            hat_K,
    const side_information_type& sid) const;

  const std::vector<point_basic<T> >& hat_node (reference_element hat_K) const;

protected:
// internals:

  void _clear() const;
};
typedef basis_basic<Float> basis;
//>basis:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template<class T>
inline
basis_basic<T>::basis_basic(std::string name)
 : base()
{
  reset (name);
}
template<class T>
inline
void
basis_basic<T>::_clear() const
{
  return base::data()._clear();
}
template<class T>
inline
std::string
basis_basic<T>::family_name() const
{
  return base::data().family_name();
}
template<class T>
inline
std::string
basis_basic<T>::name() const
{
  return base::data().name();
}
template<class T>
inline
typename basis_basic<T>::size_type 
basis_basic<T>::degree() const
{
  return base::data().degree();
}
template<class T>
inline
typename basis_basic<T>::size_type
basis_basic<T>::size (reference_element hat_K) const
{
  return base::data().size (hat_K);
}
template<class T>
inline
const basis_option&
basis_basic<T>::option() const
{
  return base::data().option();
}
template<class T>
inline
bool
basis_basic<T>::is_continuous() const
{
  return base::data().is_continuous();
}
template<class T>
inline
bool
basis_basic<T>::is_discontinuous() const
{
  return base::data().is_discontinuous();
}
template<class T>
inline
typename basis_basic<T>::valued_type
basis_basic<T>::valued_tag() const
{
  return base::data().valued_tag();
}
template<class T>
inline
const std::string&
basis_basic<T>::valued() const
{
  return base::data().valued();
}
template<class T>
inline
const arma::Mat<T>&
basis_basic<T>::vdm (reference_element hat_K) const
{
  return base::data().vdm(hat_K);
}
template<class T>
inline
const arma::Mat<T>&
basis_basic<T>::inv_vdm (reference_element hat_K) const
{
  return base::data().inv_vdm(hat_K);
}
template<class T>
inline
void
basis_basic<T>::eval(
  reference_element      hat_K,
  const point_basic<T>&  hat_x,
  arma::Col<T>&          value) const
{
  base::data().eval (hat_K, hat_x, value);
}
template<class T>
inline
void
basis_basic<T>::eval(
  reference_element             hat_K,
  const point_basic<T>&         hat_x,
  std::vector<point_basic<T> >& value) const
{
  base::data().eval (hat_K, hat_x, value);
}
template<class T>
inline
void
basis_basic<T>::eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    arma::Mat<T>&                        value) const
{
  base::data().eval (hat_K, node, value);
}
template<class T>
inline
void
basis_basic<T>::eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    arma::Col<T>&                        value) const
{
  base::data().eval (hat_K, node, value);
}
template<class T>
inline
void
basis_basic<T>::eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    ublas::matrix<point_basic<T> >&      value) const
{
  base::data().eval (hat_K, node, value);
}
template<class T>
inline
void
basis_basic<T>::grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<point_basic<T> >& value) const
{
  base::data().grad_eval (hat_K, hat_x, value);
}
template<class T>
inline
void
basis_basic<T>::grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    ublas::matrix<point_basic<T> >&      value) const
{
  base::data().grad_eval (hat_K, node, value);
}
template<class T>
inline
void
basis_basic<T>::grad_eval(
    reference_element           hat_K,
    const point_basic<T>&       hat_x,
    std::vector<tensor_basic<T> >& value) const
{
  base::data().grad_eval (hat_K, hat_x, value);
}
template<class T>
inline
void
basis_basic<T>::grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    ublas::matrix<tensor_basic<T> >&     value) const
{
  base::data().grad_eval (hat_K, node, value);
}
template<class T>
inline
void
basis_basic<T>::grad_eval(
    reference_element                    hat_K,
    const std::vector<point_basic<T> >&  node,
    std::vector<point_basic<T> >&        value) const
{
  base::data().grad_eval (hat_K, node, value);
}
template<class T>
inline
const std::vector<point_basic<T> >&
basis_basic<T>::hat_node (reference_element hat_K) const
{
  return base::data().hat_node (hat_K);
}
template<class T>
template <class Function>
inline
void
basis_basic<T>::compute_dof (
  reference_element  hat_K,
  const Function&    f,
  arma::Col<T>&      dof) const
{
  return details::compute_dof (base::data(), hat_K, f, dof);
}
template<class T>
inline
void
basis_basic<T>::put (std::ostream& os, reference_element hat_K) const
{
  base::data().put (os, hat_K);
}
template<class T>
inline
void
basis_basic<T>::put_hat_node (std::ostream& os, reference_element hat_K) const
{
  base::data().put_hat_node (os, hat_K);
}
template<class T>
inline
void
basis_basic<T>::get_local_idof_on_side (
        reference_element            hat_K,
        const side_information_type& sid,
	arma::Col<size_type>&        loc_idof) const
{
  base::data().get_local_idof_on_side (hat_K, sid, loc_idof);
}
template<class T>
inline
void
basis_basic<T>::put_hat_node_on_side (
    std::ostream&                os,
    reference_element            hat_K,
    const side_information_type& sid) const
{
  base::data().put_hat_node_on_side (os, hat_K, sid);
}

}// namespace rheolef
#endif // _RHEO_BASIS_H
