% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\docType{methods}
\name{estimateSizeFactors}
\alias{estimateSizeFactors}
\alias{estimateSizeFactors,DESeqDataSet-method}
\title{Estimate the size factors for a \code{\link{DESeqDataSet}}}
\usage{
\S4method{estimateSizeFactors}{DESeqDataSet}(object, type = c("ratio",
  "iterate"), locfunc = stats::median, geoMeans, controlGenes, normMatrix)
}
\arguments{
\item{object}{a DESeqDataSet}

\item{type}{either "ratio" or "iterate". "ratio" uses the standard
median ratio method introduced in DESeq. The size factor is the
median ratio of the sample over a pseudosample: for each gene, the geometric mean
of all samples. "iterate" offers an alternative estimator, which can be
used even when all genes contain a sample with a zero. This estimator
iterates between estimating the dispersion with a design of ~1, and
finding a size factor vector by numerically optimizing the likelihood
of the ~1 model.}

\item{locfunc}{a function to compute a location for a sample. By default, the
median is used. However, especially for low counts, the
\code{\link[genefilter]{shorth}} function from the genefilter package may give better results.}

\item{geoMeans}{by default this is not provided and the
geometric means of the counts are calculated within the function.
A vector of geometric means from another count matrix can be provided
for a "frozen" size factor calculation}

\item{controlGenes}{optional, numeric or logical index vector specifying those genes to
use for size factor estimation (e.g. housekeeping or spike-in genes)}

\item{normMatrix}{optional, a matrix of normalization factors which do not yet
control for library size. Note that this argument should not be used (and
will be ignored) if the \code{dds} object was created using \code{tximport}.
In this case, the information in \code{assays(dds)[["avgTxLength"]]}
is automatically used to create appropriate normalization factors.
Providing \code{normMatrix} will estimate size factors on the
count matrix divided by \code{normMatrix} and store the product of the
size factors and \code{normMatrix} as \code{\link{normalizationFactors}}.
It is recommended to divide out the row-wise geometric mean of
\code{normMatrix} so the rows roughly are centered on 1.}
}
\value{
The DESeqDataSet passed as parameters, with the size factors filled
in.
}
\description{
This function estimates the size factors using the
"median ratio method" described by Equation 5 in Anders and Huber (2010).
The estimated size factors can be accessed using the accessor function \code{\link{sizeFactors}}.
Alternative library size estimators can also be supplied
using the assignment function \code{\link{sizeFactors<-}}.
}
\details{
Typically, the function is called with the idiom:

\code{dds <- estimateSizeFactors(dds)}

See \code{\link{DESeq}} for a description of the use of size factors in the GLM.
One should call this function after \code{\link{DESeqDataSet}}
unless size factors are manually specified with \code{\link{sizeFactors}}.
Alternatively, gene-specific normalization factors for each sample can be provided using
\code{\link{normalizationFactors}} which will always preempt \code{\link{sizeFactors}}
in calculations.

Internally, the function calls \code{\link{estimateSizeFactorsForMatrix}}, 
which provides more details on the calculation.
}
\examples{

dds <- makeExampleDESeqDataSet(n=1000, m=4)
dds <- estimateSizeFactors(dds)
sizeFactors(dds)

dds <- estimateSizeFactors(dds, controlGenes=1:200)

m <- matrix(runif(1000 * 4, .5, 1.5), ncol=4)
dds <- estimateSizeFactors(dds, normMatrix=m)
normalizationFactors(dds)[1:3,]

geoMeans <- exp(rowMeans(log(counts(dds))))
dds <- estimateSizeFactors(dds,geoMeans=geoMeans)
sizeFactors(dds)

}
\author{
Simon Anders
}
\references{
Reference for the median ratio method:

Simon Anders, Wolfgang Huber: Differential expression analysis for sequence count data. Genome Biology 2010, 11:106. \url{http://dx.doi.org/10.1186/gb-2010-11-10-r106}
}
\seealso{
\code{\link{estimateSizeFactorsForMatrix}}
}

