/***************************************************************************
 *   Copyright (C) 2007 by Matvey Kozhev                                   *
 *   sikon@lucidfox.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
 
#include "qink.h"
#include "device.h"

#include <QTextStream>

const Device& Device::invalid = Device(-1, -1, "");

Device::Device()
{
    *this = Device::invalid;
}

Device::Device(int type, int port, const QString& model)
    : type(type), port(port), model(model)
{
}

Device::~Device()
{
}

bool Device::operator==(const Device& dev) const
{
    return (type == dev.type) && (port == dev.port);
}

void Device::enumerate(QVector<Device> &list)
{
    list.clear();

    int callback;
    inklevel::ink_level level;

    // Detect printers (if any)
    for(int type = MIN_DEVICE_TYPE; type <= MAX_DEVICE_TYPE; type++)
    {
        for (int port = 0; port < LAST_PORT; port++)
        {
            callback = inklevel::get_ink_level(type, "", port, &level);
            Device *dev = 0;
            
            switch (callback)
            {
                case INKLEVEL_OK:
                    list.append(Device(type, port, QString::fromUtf8(level.model)));
                    dev = &list.last();
                    break;
                case PRINTER_NOT_SUPPORTED:
                    list.append(Device(type, port, QApplication::tr("Unsupported device")));
                    dev = &list.last();
                    break;
            }
            
            if(dev)
                dev->setInfo(level);
        }
    }
}

int Device::getInkLevel(unsigned short type) const
{
    if(levels.contains(type))
        return levels[type];
    
    return -1;
}

// convenience function
bool Device::isMissing(unsigned short type) const
{
    return getInkLevel(type) == -1;
}

void Device::setInfo(const inklevel::ink_level& info)
{
    status = info.status;
    levels.clear();
                
    for(int i = 0; i < MAX_CARTRIDGE_TYPES; i++)
    {
        if(info.levels[i][0] != CARTRIDGE_NOT_PRESENT)
            levels.insert(info.levels[i][0], info.levels[i][1]);
    }
}

int Device::update()
{
    inklevel::ink_level level;
    int callback = inklevel::get_ink_level(type, "", port, &level);
    
    if(callback == INKLEVEL_OK)
    {
        setInfo(level);
    }
    
    return callback;
}

QString Device::serialize() const
{
    return QString(type) + " " + QString(port);
}

void Device::deserialize(const QString& value)
{
    QString str = value;
    QTextStream(&str) >> type >> port;
}
