//                                               -*- C++ -*-
/**
 *  @file  HermitianMatrix.hxx
 *  @brief HermitianMatrix implements the classical mathematical hermitian matrix
 *
 *  Copyright (C) 2005-2014 Airbus-EDF-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author schueller
 *  @date   2012-04-18 17:56:46 +0200 (Wed, 18 Apr 2012)
 */
#ifndef OPENTURNS_HERMITIANMATRIX_HXX
#define OPENTURNS_HERMITIANMATRIX_HXX

#include "SquareComplexMatrix.hxx"
#include "Collection.hxx"


BEGIN_NAMESPACE_OPENTURNS

class SymmetricMatrix;
class IdentityMatrix;
class SquareMatrix;
class Matrix;
class TriangularComplexMatrix;

/**
 * @class HermitianMatrix
 *
 * HermitianMatrix implements the classical mathematical hermitian matrix
 */

class HermitianMatrix :
  public SquareComplexMatrix
{
  CLASSNAME;

#ifndef SWIG
  friend HermitianMatrix operator * (const NumericalComplex s,
                                     const HermitianMatrix & m);
#endif

public:

  typedef Collection<NumericalComplex>                       NumericalComplexCollection;
  typedef Collection<NumericalScalar>                        NumericalScalarCollection;
  typedef TypedInterfaceObject<ComplexMatrixImplementation>::Implementation     Implementation ;

  /** Default constructor */
  HermitianMatrix();

  /** Constructor with size (dim, which is the same for nbRows_ and nbColumns_) */
  explicit HermitianMatrix(const UnsignedLong dimension);

  /** Constructor with implementation */
  HermitianMatrix(const Implementation & i);

  /** String converter */
  String __repr__() const;
  String __str__(const String & offset = "") const;


  /** Get the dimension of the matrix */
  const UnsignedLong getDimension() const;

  /** HermitianMatrix transpose */
  HermitianMatrix transpose () const;

  /** HermitianMatrix conjugate */
  HermitianMatrix conjugate () const;

  /** HermitianMatrix conjugate and transpose */
  HermitianMatrix conjugateTranspose () const;

  /** Get the real part of the matrix */
  SymmetricMatrix real() const;

  /** Get the imaginary part of the matrix */
  SquareMatrix imag() const;

#ifndef SWIG
  /** Operator () gives access to the elements of the matrix (to modify these elements) */
  /** The element of the matrix is designated by its row number i and its column number j */
  NumericalComplex & operator () (const UnsignedLong i,
                                  const UnsignedLong j) ;

  /** Operator () gives access to the elements of the matrix (read only) */
  /** The element of the matrix is designated by its row number i and its column number j */
  const NumericalComplex operator () (const UnsignedLong i,
                                      const UnsignedLong j) const;
#endif

  /** Check if the internal representation is really hermitian */
  void checkHermitian() const;

  /** HermitianMatrix additions */
  HermitianMatrix operator + (const HermitianMatrix & m) const;

  /** ComplexMatrix additions */
  SquareComplexMatrix operator + (const SquareComplexMatrix & m) const;

  /** HermitianMatrix substractions */
  HermitianMatrix operator - (const HermitianMatrix & m) const;

  /** ComplexMatrix substractions */
  SquareComplexMatrix operator - (const SquareComplexMatrix & m) const;

  /** Multiplication with a NumericalComplex */
  HermitianMatrix operator * (const NumericalComplex s) const;

  /** ComplexMatrix multiplications (must have consistent dimensions) */
  ComplexMatrix operator * (const ComplexMatrix & m) const;

  /** ComplexMatrix multiplications (must have consistent dimensions) */
  SquareComplexMatrix operator * (const SquareComplexMatrix & m) const;

  /** HermitianMatrix multiplications  */
  SquareComplexMatrix operator * (const HermitianMatrix & m) const;

  /** TriangularComplexMatrix multiplications  */
  SquareComplexMatrix operator * (const TriangularComplexMatrix & m) const;

  /** Real SquareMatrix multiplications */
  SquareComplexMatrix operator * (const SquareMatrix & m) const;

  /** Real Matrix multiplications (must have consistent dimensions) */
  ComplexMatrix operator * (const Matrix & m) const;

  /** Real SymmetricMatrix multiplications  */
  SquareComplexMatrix operator * (const SymmetricMatrix & m) const;

  /** Real IdentityMatrix multiplications  */
  HermitianMatrix operator * (const IdentityMatrix & m) const;

  /** Multiplication with a NumericaComplexCollection (must have consistent dimensions) */
  NumericalComplexCollection operator * (const NumericalComplexCollection & p) const;

  /** Multiplication with a NumericaScalarCollection (must have consistent dimensions) */
  NumericalComplexCollection operator * (const NumericalScalarCollection & p) const;

  /** Multiplication with a NumericalPoint (must have consistent dimensions) */
  NumericalComplexCollection operator * (const NumericalPoint & p) const;

  using SquareComplexMatrix::operator *;

  /** HermitianMatrix integer power */
  HermitianMatrix power(const UnsignedLong n) const;

  /** Division by a NumericalComplex*/
  HermitianMatrix operator / (const NumericalComplex s) const;

  /** Compute the Cholesky factor  */
  TriangularComplexMatrix computeCholesky(const Bool keepIntact = true);

private:

  /** Check if one needs to symmetrized the internal representation of the tensor */
  mutable Bool hasBeenHermitianized_;

}

; /* class HermitianMatrix */


inline HermitianMatrix operator * (const NumericalComplex s,
                                   const HermitianMatrix & m)
{
  return m.operator * (s);
}


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_HERMITIANMATRIX_HXX */
