#! /usr/bin/env python
# -*- coding: utf-8 -*-

#   OpenTeacher
#   depends on: python-qt4
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.

#Version, used at the moment in the about dialog
__version__ = "2.3"

#This are the contributors shown in the AboutDialog. Add your name if you contributed
#to on OpenTeacher.
CODE_CONTRIBUTORS = (
	u"Milan Boers",
	u"Marten de Vries",
	u"Cas Widdershoven",
)

TRANSLATION_CONTRIBUTORS = (
	u"Australian English: Jared Norris, Joel Pickett",
	u"Bahasa Indonesia: Prayudi Satriyo N",
	u"Dansk: Aputsiaq Niels Janussen, Cromag",
	u"Deutsch: Achmed Decker, Dennis Baudys, HarzG, Peter Widdershoven,\nStartlett, Tim O., Wolfgang Schweer, ubuntuuser87, zerwas",
	u"Español: Adolfo Jayme Barrientos, Iván, Morgus, Nelson Álvarez Sáez",
	u"Français: Achraf FOUWAD, BriceF, Cyril Lavier, Guus, Léo POUGHON,\nRedGuff",
	u"Frysk: Dooitze de Jong",
	u"Galego: xesusmosquera",
	u"Hrvatski: Saša Teković, Tomislav Marčinković",
	u"Italiano: Gianluca Piana, Guybrush88, enrico di paolantonio",
	u"Magyar: Gabor Kelemen, Papp Bence, Richard Somlói, blaselinux,\nkilo aka Gabor Kmetyko",
	u"Nederlands: Guus, Ivo, Marten de Vries, Rachid",
	u"Norsk bokmål: Jo-Erlend Schinstad, Mathias Bynke",
	u"Occitan: Cédric VALMARY (Tot en òc)",
	u"Polski: Magdalena Gazda, Vorbis^, soee",
	u"Português brasileiro: Otto Robba",
	u"Română: Ciprian Panaite",
	u"Slovenščina: Andrej Znidarsic, Mojca Ograjšek",
	u"Svenska: Daniel Nylander, Jonas Zetterholm, Martin Lundberg",
	u"Türkçe: Tolga, kosti",
	u"Valencià: Xangel",
	u"čeština: Jakub Šnapka, Jan Havran",
	u"Ўзбек тили: Akmal Xushvaqov",
	u"Русский: Pasha.P.Komar, Ruiz Aw, Skai Falkorr, Vyacheslav Sharmanov",
	u"Українська мова: Alex Krynko",
	u"српски језик: Мирослав Николић",
	u"עברית: Omer Omer",
	u"العربية: Abdelmonam Kouka, Ahmed Osama, Ahmed Sghaier, Ahmed Shams,\nAhmed Toulan, El Achèche ANIS, Islam Hassan, Islam Wazery,\nMohamed El Sharnoby, TheNightPhoenix, computeristgeek",
	u"日本語: Emmanuel Chanel",
	u"正體字: BlueT - Matthew Lien - 練喆明, Louie Chen",
	u"简化字: Xu Bin, ZhangCheng, jack",
	u"한국어: Jinkyu Yi",
)

#This are the characters shown in the enter_table if the user didn't change them.
TABLE_CHARS = (
	(u"à", u"á", u"â", u"ä", u"ã", u"å"),
	(u"À", u"Á", u"Â", u"Ä", u"Ã", u"Å"),
	(u"è", u"é", u"ê", u"ë", u"Ç", u"ç"),
	(u"È", u"É", u"Ê", u"Ë", u"Ñ", u"ñ"),
	(u"ì", u"ì", u"î", u"ï", u"Û", u"û"),
	(u"Ì", u"Í", u"Î", u"Ï", u"Ú", u"ú"),
	(u"ò", u"ó", u"ô", u"ö", u"Ü", u"ü"),
	(u"Ò", u"Ó", u"Ô", u"Ö", u"Ù", u"ß"),
	(u"\,", u"\;", u"\=", u"", u"", u""),
)

#The same for the greek chars. The difference is they're hardcoded.
GREEK_CHARS = (
	(u"α", u"Α", u"β", u"Β", u"γ", u"Γ"),
	(u"δ", u"Δ", u"ε", u"Ε", u"ζ", u"Ζ"),
	(u"η", u"Η", u"θ", u"Θ", u"ι", u"Ι"),
	(u"κ", u"Κ", u"λ", u"Λ", u"μ", u"Μ"),
	(u"ν", u"Ν", u"ξ", u"Ξ", u"ο", u"Ο"),
	(u"π", u"Π", u"ρ", u"Ρ", u"σ", u"Σ"),
	(u"ς", u"τ", u"Τ", u"υ", u"Υ", u"φ"),
	(u"Φ", u"χ", u"Χ", u"ψ", u"Ψ", u"ω"),
	(u"Ω", u"", u"", u"", u"", u""),
)

#Same as greek chars.
CYRILLIC_CHARS = (
	(u"А", u"а", u"Б", u"б", u"В", u"в"),
	(u"Г", u"г", u"Д", u"д", u"Е", u"е"),
	(u"Ё", u"ё", u"Ж", u"ж", u"З", u"з"),
	(u"И", u"и", u"Й", u"й", u"К", u"к"),
	(u"Л", u"л", u"М", u"м", u"Н", u"н"),
	(u"О", u"о", u"П", u"п", u"Р", u"р"),
	(u"С", u"с", u"Т", u"т", u"У", u"у"),
	(u"Ф", u"ф", u"Х", u"х", u"Ц", u"ц"),
	(u"Ч", u"ч", u"Ш", u"ш", u"Щ", u"щ"),
	(u"Ъ", u"ъ", u"Ы", u"ы", u"Ь", u"ь"),
	(u"Э", u"э", u"Ю", u"ю", u"Я", u"я"),
)

LANGUAGES = {
	u"de": u"Deutsch",
	u"en": u"English",
	u"es": u"Español",
	u"hu": u"Magyar",
	u"nl": u"Nederlands",
	u"ru": u"Русский",
	u"tr": u"Türkçe",
	u"ar": u"العربية",
	u"da": u"Dansk",
	u"en_AU": u"Australian English",
	u"fr": u"Français",
	u"fy": u"Frysk",
	u"he": u"עברית",
	u"hr": u"Hrvatski",
	u"id": u"Bahasa Indonesia",
	u"it": u"Italiano",
	u"ja": u"日本語",
	u"ko": u"한국어",
	u"nb": u"Norsk bokmål",
	u"pl": u"Polski",
	u"sl": u"Slovenščina",
	u"sr": u"српски језик",
	u"sv": u"Svenska",
	u"zh_CN": u"简化字",
	u"zh_TW": u"正體字",
	u"ro": u"Română",
	u"oc": u"Occitan",
	u"pt_BR": u"Português brasileiro",
	u"uz": u"Ўзбек тили",
	u"ca@valencia": u"Valencià",
	u"uk": u"Українська мова",
	u"cs": u"čeština",
	u"gl": u"Galego",
}

DEFAULT_FADE_DURATION_WHEN_WRONG = 2000
DEFAULT_REPEAT_MODE_HINT_DURATION = 3000

DOCUMENTATION_URL = "http://openteacher.org/documentation.html"

#import PyQt4 and dependancies
import sys
from PyQt4 import QtGui, QtCore, QtWebKit, QtNetwork

#import mainwindow
import gui.openteacher

#import dialogs
import gui.wrtsLogin
import gui.wrtsListChoice
import gui.about
import gui.settings

#Import file types
import fileTypes

#Import errors
import errors

#Import external API's
import api.wrts

#Import internal data types
import words

#Resources
import gui.icons_rc
import resources

#import python modules used in this script
import difflib
import math
import os
import random
import re

#We only translate the Qt-classes we need

#QDialogButtonBox
QtCore.QCoreApplication.translate("QDialogButtonBox", "OK")
QtCore.QCoreApplication.translate("QDialogButtonBox", "&OK")
QtCore.QCoreApplication.translate("QDialogButtonBox", "Cancel")
QtCore.QCoreApplication.translate("QDialogButtonBox", "&Cancel")

QtCore.QCoreApplication.translate("QDialogButtonBox", "&Yes")
QtCore.QCoreApplication.translate("QDialogButtonBox", "&No")

QtCore.QCoreApplication.translate("QDialogButtonBox", "Save")
QtCore.QCoreApplication.translate("QDialogButtonBox", "&Save")
QtCore.QCoreApplication.translate("QDialogButtonBox", "Open")
QtCore.QCoreApplication.translate("QDialogButtonBox", "Close")
QtCore.QCoreApplication.translate("QDialogButtonBox", "&Close")

QtCore.QCoreApplication.translate("QDialogButtonBox", "Close without Saving")
QtCore.QCoreApplication.translate("QDialogButtonBox", "Don't save")
QtCore.QCoreApplication.translate("QDialogButtonBox", "Discard")

#QShortcut
QtCore.QCoreApplication.translate("QShortcut", "Ctrl")
QtCore.QCoreApplication.translate("QShortcut", "Return")
QtCore.QCoreApplication.translate("QShortcut", "Del")
QtCore.QCoreApplication.translate("QShortcut", "Shift")

class AboutDialog(QtGui.QDialog):
	def __init__(self, parent=None):
		QtGui.QDialog.__init__(self, parent)

		self.ui = gui.about.Ui_Dialog()
		self.ui.setupUi(self)

		self.connect(self.ui.fullLicenseButton, QtCore.SIGNAL("clicked()"), self.showFullLicense)

		codeItem = QtGui.QTreeWidgetItem(self.ui.treeWidget, [QtCore.QCoreApplication.translate("OpenTeacher", "Code")])
		codeItem.setIcon(0, QtGui.QIcon(":/award.png"))
		codeItem.setExpanded(True)
		self.ui.treeWidget.addTopLevelItem(codeItem)

		l10nItem = QtGui.QTreeWidgetItem(self.ui.treeWidget, [QtCore.QCoreApplication.translate("OpenTeacher", "Localisation")])
		l10nItem.setIcon(0, QtGui.QIcon(":/award.png"))
		l10nItem.setExpanded(True)
		self.ui.treeWidget.addTopLevelItem(l10nItem)

		for coder in CODE_CONTRIBUTORS:
			treeItem = QtGui.QTreeWidgetItem(codeItem, [coder])
			treeItem.setIcon(0, QtGui.QIcon(":/humanbeing.png"))
			self.ui.treeWidget.addTopLevelItem(treeItem)

		for translator in TRANSLATION_CONTRIBUTORS:
			treeItem = QtGui.QTreeWidgetItem(l10nItem, [translator])
			treeItem.setIcon(0, QtGui.QIcon(":/humanbeing.png"))
			self.ui.treeWidget.addTopLevelItem(treeItem)

		self.ui.treeWidget.sortItems(0, QtCore.Qt.AscendingOrder)
		self.ui.treeWidget.setSelectionMode(QtGui.QAbstractItemView.NoSelection)

		#Set the version info (the text on the first tab)
		versionInfo = u"""
			<!doctype html>
			<html>
				<head></head>
				<body>
					<p style='font-size:12pt; font-weight:600;'>
						OpenTeacher {version}
					</p>
					<p style='font-size:11pt'>
						{first_line}<br />
						{second_line}
					</p>
					<p style='font-size:9pt'>
						© {copyright_years}, {openteacher_authors}<br />
						<a href='http://openteacher.org/'>{project_website}</a>
					</p>
				</body>
			</html>
		"""

		slogan = QtCore.QCoreApplication.translate("OpenTeacher", "OpenTeacher helps you learn a foreign language vocabulary.")
		firstLine, secondLine = self._splitLineCloseToMiddle(unicode(slogan))
 
		versionInfo = versionInfo.format(
			version = __version__,
			first_line = firstLine,
			second_line = secondLine,
			copyright_years = "2008-2010",
			openteacher_authors = QtCore.QCoreApplication.translate("OpenTeacher", "OpenTeacher authors"),
			project_website = QtCore.QCoreApplication.translate("OpenTeacher", "Project website")
		)
		
		self.ui.versionLabel.setText(versionInfo)

	def _splitLineCloseToMiddle(self, line):
		"""Used to split the slogan at the space closest to the middle of it."""

		#determine the middle and find the closest spaces
		middle = len(line) /2
		distanceFromLeft = line[:middle].find(" ")
		distanceFromRight = line[middle:].find(" ")

		#Check if spaces were found
		if distanceFromLeft == -1 and distanceFromRight == -1:
			return (line, "")
		elif distanceFromLeft == -1:
			pos = middle + distanceFromRight
			return (line[:pos], line[pos:])
		elif distanceFromRight == -1:
			pos = middle - distanceFromLeft
			return (line[:pos], line[pos:])
		else:
			#left is closest
			if distanceFromLeft < distanceFromRight:
				pos = middle - distanceFromLeft
				return (line[:pos], line[pos:])
			#right is closest
			else:
				pos = middle + distanceFromRight
				return (line[:pos], line[pos:])

	def showFullLicense(self):
		widget = QtGui.QWidget(self.ui.tabWidget)
		layout = QtGui.QVBoxLayout()
		textEdit = QtGui.QTextEdit()
		textEdit.setReadOnly(True)

		licenseFile = QtCore.QFile(":/COPYING")
		licenseFile.open(QtCore.QIODevice.ReadOnly)
		text = QtCore.QString(licenseFile.readAll())
		textEdit.setPlainText(text)

		layout.addWidget(textEdit)
		widget.setLayout(layout)
		self.ui.tabWidget.removeTab(1)
		self.ui.tabWidget.insertTab(1, widget, QtCore.QCoreApplication.translate("OpenTeacher", "License"))
		self.ui.tabWidget.setCurrentWidget(widget)

class OpenTeacherWebPage(QtWebKit.QWebPage):
	noConnection = QtCore.pyqtSignal()

	def __init__(self, parent=None):
		QtWebKit.QWebPage.__init__(self, parent)

		self.connect(self, QtCore.SIGNAL("loadFinished(bool)"), self.updateStatus)

	def userAgentForUrl(self, url):
		return "OpenTeacher/%s (+http://openteacher.org/)" % __version__

	def updateStatus(self, ok):
		if not ok:
			self.noConnection.emit()

class DocumentationDialog(QtGui.QDialog):
	def __init__(self, language, parent=None):
		QtGui.QDialog.__init__(self, parent)

		self.language = language

		self.webView = QtWebKit.QWebView()
		page = OpenTeacherWebPage(self.webView)
		self.webView.setPage(page)

		request = QtNetwork.QNetworkRequest(QtCore.QUrl(DOCUMENTATION_URL))
		request.setRawHeader("Accept-Language", self.language)
		self.webView.load(request)

		vbox = QtGui.QVBoxLayout()
		vbox.setContentsMargins(0, 0, 0, 0)
		vbox.addWidget(self.webView)
		self.setLayout(vbox)

		self.setWindowTitle(QtCore.QCoreApplication.translate("DocumentationDialog", "Documentation"))

		self.connect(page, QtCore.SIGNAL("noConnection()"), self.enableFallback)

	def enableFallback(self):
		indexFile = QtCore.QFile(":/docs/index.html")
		indexFile.open(QtCore.QIODevice.ReadOnly)
		html = QtCore.QString(indexFile.readAll())
		self.webView.setHtml(html, QtCore.QUrl("qrc:/docs/"))

class SettingsDialog(QtGui.QDialog):
	def __init__(self, settings, charsModel, parent=None):
		#Initialize GUI
		QtGui.QDialog.__init__(self, parent)
		self.ui = gui.settings.Ui_Dialog()
		self.ui.setupUi(self)

		#Set warning text (can't be done in Qt Designer because it messes up translations)
		warning = u"""
			<!doctype html>
			<html>
				<head></head>
				<body>
					<strong>{warning}:</strong> {first_line}<br />
					{second_line}
				</body>
			</html>
		"""
		warning = warning.format(
			warning = QtCore.QCoreApplication.translate("OpenTeacher", "Warning"),
			first_line = QtCore.QCoreApplication.translate("OpenTeacher", "This credentials are saved in plain text on your harddrive."),
			second_line = QtCore.QCoreApplication.translate("OpenTeacher", "Don't use this function on a public computer!")
		)
		self.ui.warningLabel.setText(warning)

		#Load current settings
		self.settings = settings

		#Get avaible languages
		resourceContent = QtCore.QDir(":/translations").entryList()
		languages = []

		for name in resourceContent:
			name = unicode(name)
			if name.endswith(".qm"):
				name = name[:name.rfind(".qm")]
				try:
					language = LANGUAGES[name]
				except KeyError:
					#fallback to avoid crashes.
					language = name
				languages.append(language)
		languages.sort()

		#Put them into the languageComboBox
		self.ui.languageComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "Make your choice"))
		self.ui.languageComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "System default"))
		self.ui.languageComboBox.addItem("English")
		self.ui.languageComboBox.insertSeparator(self.ui.languageComboBox.count())
		self.ui.languageComboBox.addItems(languages)

		#Show 'overwrite'-dialogs
		if self.settings.value("ask_overwrite").isNull():
			askOverwrite = True
		else:
			askOverwrite = bool(self.settings.value("ask_overwrite").toBool())
		self.ui.askOverwriteCheckBox.setChecked(askOverwrite)

		#Show 'do you want to save'-dialogs
		if self.settings.value("ask_saving").isNull():
			askSaving = True
		else:
			askSaving = bool(self.settings.value("ask_saving").toBool())
		self.ui.askSavingCheckBox.setChecked(askSaving)

		#Show 'stop lesson'-dialogs
		if self.settings.value("ask_stop_lesson").isNull():
			askStopLesson = True
		else:
			askStopLesson = bool(self.settings.value("ask_stop_lesson").toBool())
		self.ui.askStopLessonCheckBox.setChecked(askStopLesson)

		#Show 'lesson completed'-dialogs
		if self.settings.value("view_results").isNull():
			viewResults = True
		else:
			viewResults = bool(self.settings.value("view_results").toBool())
		self.ui.viewResultsCheckBox.setChecked(viewResults)

		#Fade_duration_when_wrong
		if self.settings.value("fade_duration_when_wrong").isNull():
			fadeDurationWhenWrong = DEFAULT_FADE_DURATION_WHEN_WRONG
		else:
			fadeDurationWhenWrong = int(self.settings.value("fade_duration_when_wrong").toInt()[0])
		self.ui.fadeDurationWhenWrongSpinBox.setValue(fadeDurationWhenWrong)

		#Repeat_mode_hint_duration
		if self.settings.value("repeat_mode_hint_duration").isNull():
			repeatModeHintDuration = DEFAULT_REPEAT_MODE_HINT_DURATION
		else:
			repeatModeHintDuration = int(self.settings.value("repeat_mode_hint_duration").toInt()[0])
		self.ui.repeatModeHintDurationSpinBox.setValue(repeatModeHintDuration)

		#Note notation
		self.ui.noteNotationComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "Make your choice (all examples are good to bad)"))
		self.ui.noteNotationComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "Percents") + " (100-0)")
		self.ui.noteNotationComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "Dutch") + " (10-1)")
		self.ui.noteNotationComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "American") + " (A-F)")
		self.ui.noteNotationComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "German") + " (1-6)")
		self.ui.noteNotationComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "French") + " (20-0)")
		self.ui.noteNotationComboBox.addItem(QtCore.QCoreApplication.translate("SettingsDialog", "ECTS") + " (A-F)")

		##Hardly typable chars
		self.charsModel = charsModel
		#Allow editing while settings-dialog is shown.
		self.charsModel.readOnly = False
		#Link the model to the view
		self.ui.hardlyTypableCharsTableView.setModel(self.charsModel)
		#Resize columns to contents
		self.ui.hardlyTypableCharsTableView.resizeColumnsToContents()

		##WRTS username + password
		#fill boxes
		if not self.settings.value("wrts_username").isNull():
			username = unicode(self.settings.value("wrts_username").toString())
			self.ui.wrtsUsernameLineEdit.setText(username)
		if not self.settings.value("wrts_password").isNull():
			password = unicode(self.settings.value("wrts_password").toString())
			self.ui.wrtsPasswordLineEdit.setText(password)
		#Connect wrts test button
		self.connect(self.ui.wrtsTestButton, QtCore.SIGNAL("clicked()"), self.checkWrts)

	def checkWrts(self):
		username = unicode(self.ui.wrtsUsernameLineEdit.text())
		password = unicode(self.ui.wrtsPasswordLineEdit.text())

		connection = api.wrts.WrtsConnection()
		try:
			connection.logIn(username, password)
		except Exception, e:
			text = unicode(e)
			title = QtCore.QCoreApplication.translate("Settings", "Connection to WRTS failed")
		else:
			text = QtCore.QCoreApplication.translate("Settings", "Connection to WRTS succeeded")
			title = QtCore.QCoreApplication.translate("Settings", "Connection to WRTS succeeded")
		QtGui.QMessageBox.information(self, title, text)

	def saveSettings(self):
		#Language
		currentIndex = self.ui.languageComboBox.currentIndex()
		#If the user changed the language (language != the default 'make a choice')
		if currentIndex != 0:
			if currentIndex == 1:
				#System default, == don't specify one in the settings-file
				self.settings.remove("language")
			else:
				#A real language
				language = self.ui.languageComboBox.currentText()
				for name in LANGUAGES:
					if language == LANGUAGES[name]:
						code = name
				try:
					code
				except NameError:
					#the language is the code, because it was used as a
					#fallback earlier.
					code = language
				self.settings.setValue("language", code)

			#Tell the user the language changed.
			QtGui.QMessageBox.information(self,
										  QtCore.QCoreApplication.translate("Settings", "Restart please"),
										  QtCore.QCoreApplication.translate("Settings", "You changed the language. In order to use the newly selected language, you need to restart OpenTeacher."))

		#Show 'overwrite'-dialogs
		askOverwrite = self.ui.askOverwriteCheckBox.isChecked()
		self.settings.setValue("ask_overwrite", askOverwrite)

		#Show 'do you want to save'-dialogs
		askSaving = self.ui.askSavingCheckBox.isChecked()
		self.settings.setValue("ask_saving", askSaving)

		#Show 'stop lesson'-dialogs
		askStopLesson = self.ui.askStopLessonCheckBox.isChecked()
		self.settings.setValue("ask_stop_lesson", askStopLesson)

		#Show 'lesson completed'-dialogs
		viewResults = self.ui.viewResultsCheckBox.isChecked()
		self.settings.setValue("view_results", viewResults)
		
		#Fade_duration_when_wrong
		fadeDurationWhenWrong = self.ui.fadeDurationWhenWrongSpinBox.value()
		self.settings.setValue("fade_duration_when_wrong", fadeDurationWhenWrong)

		#Repeat_mode_hint_duration
		repeatModeHintDuration = self.ui.repeatModeHintDurationSpinBox.value()
		self.settings.setValue("repeat_mode_hint_duration", repeatModeHintDuration)

		#NoteNotation
		currentIndex = self.ui.noteNotationComboBox.currentIndex()
		#If the user changed the language (language != the default 'make a choice')
		if currentIndex != 0:
			if currentIndex == 1:
				self.settings.setValue("note_notation", "percents")
			elif currentIndex == 2:
				self.settings.setValue("note_notation", "Dutch")
			elif currentIndex == 3:
				self.settings.setValue("note_notation", "American")
			elif currentIndex == 4:
				self.settings.setValue("note_notation", "German")
			elif currentIndex == 5:
				self.settings.setValue("note_notation", "French")
			elif currentIndex == 6:
				self.settings.setValue("note_notation", "ECTS")

		##Hardly typable characters
		#Set the charsModel back to read-only.
		self.charsModel.readOnly = True
		#Save the chars to the configuration, if they aren't the defaults.
		if self.charsModel.characters != map(list, TABLE_CHARS):
			self.settings.setValue("chars", self.charsModel.characters)
		else:
			self.settings.remove("chars")

		#WRTS
		wrtsUsername = self.ui.wrtsUsernameLineEdit.text()
		wrtsPassword = self.ui.wrtsPasswordLineEdit.text()
		if wrtsUsername == "":
			self.settings.remove("wrts_username")
		else:
			self.settings.setValue("wrts_username", wrtsUsername)
		if wrtsPassword == "":
			self.settings.remove("wrts_password")
		else:
			self.settings.setValue("wrts_password", wrtsPassword)

class CharactersTableModel(QtCore.QAbstractTableModel):
	"""This class is a tableModel, which means it can be used in a QTableView. (What happens in OpenTeacher.)
	   It is used by the user to enter characters that are hard to type, and also in the settings-window,
	   where the user can change the hard characters itself.
	
	   Useful properties:
	   - characters - a list of lists of characters(strings)
	   - readOnly - if true, the model doesn't accept edits."""
	def __init__(self, characters, readOnly, parent=None):
		QtCore.QAbstractTableModel.__init__(self, parent)
		
		#Set the characters inside the model.
		self.characters = map(list, characters)
		self.readOnly = readOnly
	
	def headerData(self, column, orientation, role):
		"""Returns the headers to the tableView"""
		if role == QtCore.Qt.DisplayRole:
			return QtCore.QVariant(unicode(column +1))
		else:
			return QtCore.QVariant()

	def rowCount(self, parent=None):
		"""Returns the amount of character rows"""
		return len(self.characters)

	def columnCount(self, parent=None):
		"""Returns the length of the first character row. (and should be equal to the length of all rows!)"""
		try:
			return len(self.characters[0])
		except IndexError:
			return 0

	def data(self, index, role=QtCore.Qt.DisplayRole):
		"""Returns the character for a specified position (index)."""
		if index.isValid() and (role == QtCore.Qt.DisplayRole or role == QtCore.Qt.EditRole):
			return self.characters[index.row()][index.column()]
	
	def setData(self, index, value, role):
		"""Does the same as data(), but now it saves instead of returns."""
		letter = unicode(value.toString())
		#Only save when not readOnly, valid index, and when a character (len=1)
		if index.isValid() and role == QtCore.Qt.EditRole and len(letter) == 1 and not self.readOnly:
			self.characters[index.row()][index.column()] = letter
			return True
		return False

	def flags(self, index):
		"""Return flags for the specified index. Returns baseflags, and in addition the
		ItemIsEditable-flag when the model's read-only property isn't true."""
		baseflags = QtCore.QAbstractTableModel.flags(self, index)
		if self.readOnly:
			return baseflags
		else:
			return baseflags | QtCore.Qt.ItemIsEditable

class WordListTableModel(QtCore.QAbstractTableModel):
	"""This class is a tableModel, which means it can be used in a QTableView. (What happens in OpenTeacher.)
	   It allows to view and change the data in exactly the way OpenTeacher needs it. It's also one of the in
	   OpenTeacher possible parents for the master WordList. (And pretty much the default.)"""

	def __init__(self, wordList, parent=None):
		QtCore.QAbstractTableModel.__init__(self, parent)

		#Set the wordList inside the model. This doesn't use _wordList, because that would damage the
		#tableModel.
		self.wordList = wordList

		#Sets the horizontal headers. (Vertical headers (numbers) are generated dynamically)
		self._horizontalHeaders = [QtCore.QCoreApplication.translate("WordListTableModel", "Question"),
								  QtCore.QCoreApplication.translate("WordListTableModel", "Answer"),
								  QtCore.QCoreApplication.translate("WordListTableModel", "Second Answer (optional)"),
								  QtCore.QCoreApplication.translate("WordListTableModel", "Errors")]

	def headerData(self, column, orientation, role):
		"""Returns the headers to the tableView"""
		if orientation == QtCore.Qt.Horizontal and role == QtCore.Qt.DisplayRole:
			#Horizontal; just looks up the column in the 'self._horizontalHeaders'-list.
			return QtCore.QVariant(self._horizontalHeaders[column])
		elif orientation == QtCore.Qt.Vertical and role == QtCore.Qt.DisplayRole:
			#Returns the number of the column +1, which creates a human-readable (with one -not zero
			#starting-) rowNumber.character
			return QtCore.QVariant(unicode(column +1))
		else:
			#If role is different I suppose... Much examples do this.
			return QtCore.QVariant()
 
	def rowCount(self, parent=None):
		"""Return the length of the wordList. Should only be used by the tableView, users can also call the
		   more pythonic len() on the wordList."""
		return len(self._wordList)

	def columnCount(self, parent=None):
		"""Returns the length of the horizontal header. (Every column has a header, so that's just fine.)"""
		return len(self._horizontalHeaders)

	def data(self, index, role=QtCore.Qt.DisplayRole):
		"""Returns the appropiate data for the appropiate column. Used by the tableView. Unuseful complex
		   for humans..."""
		if index.isValid() and (role == QtCore.Qt.DisplayRole or role == QtCore.Qt.EditRole):
			column = index.column()
			row = index.row()
			if column == 0:
				return QtCore.QVariant(self._wordList[self.indexes[row]].question)
			elif column == 1:
				return QtCore.QVariant(self._wordList[self.indexes[row]].answer)
			elif column == 2:
				return QtCore.QVariant(self._wordList[self.indexes[row]].secondAnswer)
			elif column == 3:
				return QtCore.QVariant(self._wordList[self.indexes[row]].results)
		return QtCore.QVariant()

	def setData(self, index, value, role):
		"""Same as data(), but now it sets data instead of returning it."""
		#do the strip() here already so an empty string never can become the result.
		value = unicode(value.toString()).strip()
		if index.isValid() and role == QtCore.Qt.EditRole:
			column = index.column()
			if column == 0:
				if value != "":
					self._wordList[self.indexes[index.row()]].question = value
				else:
					return False
			elif column == 1:
				if value != "":
					self._wordList[self.indexes[index.row()]].answer = value
				else:
					return False
			elif column == 2:
				self._wordList[self.indexes[index.row()]].secondAnswer = value
			self.emit(QtCore.SIGNAL("dataChanged(QModelIndex, QModelIndex)"), index, index)
			return True
		else:
			return False

	def flags(self, index):
		"""Return flags for the specified index. Only used in practise by the tableView again.
		   Returns baseflags, only some position get the ItemIsEditable-flag."""
		baseflags = QtCore.QAbstractTableModel.flags(self, index)
		if index.column() in (0, 1, 2, 4):
			return baseflags | QtCore.Qt.ItemIsEditable
		else:
			return baseflags

	def sort(self, sortColumn, order=QtCore.Qt.AscendingOrder):
		"""Sorts the wordList, is handy if you want to sort it manually, but OpenTeacher doesn't
		   use it itself. sortColumn is the column which is the key, and the order tells if sorting
		   should happen ascending or descending. It's only used by the tableView when a user clicks
		   one of the headers to sort the wordListTable."""
		self.emit(QtCore.SIGNAL("layoutAboutToBeChanged()"))
		if sortColumn == 0:
			sortedList = sorted(self._wordList, key=lambda word:word.question)
		elif sortColumn == 1:
			sortedList = sorted(self._wordList, key=lambda word:word.answer)
		elif sortColumn == 2:
			sortedList = sorted(self._wordList, key=lambda word:word.secondAnswer)
		elif sortColumn == 3:
			sortedList = sorted(self._wordList, key=lambda word:word.results)
		if order == QtCore.Qt.DescendingOrder:
			sortedList.reverse()
		self.indexes = []
		for item in sortedList:
			self.indexes.append(self._wordList.index(item))
		self.emit(QtCore.SIGNAL("layoutChanged()"))

	def addWord(self, word):
		"""Adds a word, and informs the TableView of the new word. Use this, and not the method
		   addWord of the WordList, if the wordList is viewed in the tableView, otherwise the
		   tableView may become corrupted!"""
		self.beginInsertRows(QtCore.QModelIndex(), self.rowCount(), self.rowCount())
		#Add the word
		self._wordList.addWord(word)
		self.indexes.append(self._wordList.index(word))
		#Tell Qt that the row is inserted
		self.endInsertRows()

	def _setWordList(self, wordList):
		"""Sets the wordList into the tableModel. It also resets the model afterwards, so the
		   view doesn't break after this action."""
		self.indexes = range(len(wordList))
		self._wordList = wordList
		self.reset()

	wordList = property(lambda self: self._wordList, fset=_setWordList)

	def removeWord(self, row):
		"""Remove a word/row from the wordsTable. The only parameter is a rowIndex (int)"""
		#Tells Qt we're going to remove a row
		self.beginRemoveRows(QtCore.QModelIndex(), row, row)
		#Removes the selected word from the actual wordList.
		del self._wordList[self.indexes[row]]
		#Remove the index which pointed to the word, and save it in deletedIndex
		deletedIndex = self.indexes.pop(row)
		#This decrements every index which was larger than the deletedIndex. This is needed to fill the deleted word its place.
		for i in xrange(len(self.indexes)):
			if self.indexes[i] > deletedIndex:
				self.indexes[i] -= 1
		#Tell Qt the removing is done, and that it can update attached views
		self.endRemoveRows()

class Fader(QtCore.QObject):
	"""This class is used to let one QPalette.Role ('role') color of a 'widget'
	   fade for 'duration'. Before the fade starts, the color is set to
	   startColor. When done callback is called.
	
	   Methods:
		- start()
		- stop()
		
	   Properties:
		- active
		- widget
		- role
		- duration
		- startColor
		- callback
	   """
	def __init__(self, widget, role, duration, callback=None, startColor=None, parent=None):
		QtCore.QObject.__init__(self, parent)
		self.widget = widget
		self.role = role
		self.duration = duration
		self.startColor = startColor
		self.callback = callback
		self.active = False

		self._timer = QtCore.QTimer(self)
		self.connect(self._timer, QtCore.SIGNAL("timeout()"), self._updateColor)

	def start(self):
		#Set the fader to 'active' for users
		self.active = True
		#Get the current used palette from the 
		self._palette = self.widget.palette()
		#Change the text in the palette to the start color if one
		if self.startColor:
			self._palette.setColor(self.role, self.startColor)
		#Start the animation by starting the timer
		self._timer.start(self.duration/255)

	def stop(self):
		if not self.active:
			return
		#Stop the timer
		self._timer.stop()
		#Clear the text in the widget to prevent it from being shown
		#when the old color is reset.
		self.widget.clear()
		#Reset the color
		color = QtGui.QPalette().color(self.role)
		self._palette.setColor(self.role, color)
		self.widget.setPalette(self._palette)
		#Set the fader to 'inactive' for users
		self.active = False

	def _updateColor(self):
		"""This method is called by a timer, it updates the font color"""
		#Get the current color
		color = self._palette.color(self.role)
		#Check if we're done yet
		if color.alpha() -1 > 0:
			#No, update the alpha of the color
			color.setAlpha(color.alpha() - 1)
			#Update the color in the widget
			self._palette.setColor(self.role, color)
			self.widget.setPalette(self._palette)
		else:
			#We are, let's stop.
			self.stop()
			#And call the callback, if one
			if self.callback:
				self.callback()

	def __del__(self):
		"""Stop timer before deleting, prevents Qt from shouting
		   warnings (keeps the terminal clean)"""
		self._timer.stop()

class OpenTeacher(QtGui.QMainWindow):
	"""The main class of OpenTeacher; controls GUI and all other classes."""

	class LessonMode:
		"""This is an "enum", to tell checkTyping which LineEdit it has to read and in which Label it has to write"""
		TYPING, THINK, SHUFFLE, REPEAT = xrange(4)

	class Tabs:
		ENTER, TEACH = xrange(2)

	def __init__(self, parent=None):
		#Load mainwindow
		QtGui.QWidget.__init__(self, parent)
		self.ui = gui.openteacher.Ui_MainWindow()
		self.ui.setupUi(self)
		self.splitters = {
			"wordlistAndEnter_splitter": self.ui.wordlistAndEnter_splitter,
			"charAndInput_splitter": self.ui.charAndInput_splitter,
			"metadataAndInput_splitter": self.ui.metadataAndInput_splitter,
			"teachSplitter": self.ui.teachSplitter,
			"innerTeachSplitter": self.ui.innerTeachSplitter,
		}

		#Set the current filename to nothing, needed for the try of opening a file after this.
		self.saveFilename = unicode()

		#No lesson active at the start of the program, it opens with the entertab.
		self.lessonActive = False

		#Create an empty wordlist and link it to the tableview (self.ui.wordsTable)
		wordList = words.WordList()

		#Get the "Repeat"-mode-GUI ready
		self.enableAnsweringRepeatMode(False)

		self.wordListTableModel = WordListTableModel(wordList)
		self.ui.wordsTable.setModel(self.wordListTableModel)

		#Load the settings; if there aren't a new settingsfile is created.
		self.settings = QtCore.QSettings("OpenTeacher", "OpenTeacher")

		#Try opening the first argument; this gives the possibility of starting the program with
		#a loaded file automatically, and is useful for 'open with'-actions from the OS. (Users
		#can bind .ot, .t2k and .wrts to OpenTeacher)
		try:
			self.openFile(unicode(sys.argv[1], sys.getfilesystemencoding()))
		#If there isn an error because there's no file in sys.argv, or an invalid one,
		#pass it silently.
		except (IndexError, IOError, UnicodeDecodeError, errors.UnsupportedFileImportError):
			pass

		if self.settings.value("current_directory").isNull():
			self.currentDir = unicode(os.path.expanduser("~"))
		else:
			self.currentDir = unicode(self.settings.value("current_directory").toString())

		#create instances of CharactersTableModel; one for some (mostly) non-ascii chars and
		#one for greek chars. (both read-only) The hard chars are loaded from the settings,
		#if they're there (the user ever changed them.)
		if self.settings.value("chars").isNull():
			self.charModel = CharactersTableModel(TABLE_CHARS, True) # We use a copy, the original has to stay intact! (for comparing later on)
		else:
			#Parse the setting into a normal list...
			chars = []
			for innerList in self.settings.value("chars").toList():
				newList = []
				for letter in innerList.toList():
					newList.append(unicode(letter.toString()))
				chars.append(newList)
			self.charModel = CharactersTableModel(chars, True)
		self.greekModel = CharactersTableModel(GREEK_CHARS, True)
		self.cyrillicModel = CharactersTableModel(CYRILLIC_CHARS, True)

		#Set the models in the views
		self.ui.charTable_enter.setModel(self.charModel)
		self.ui.charTable_teach.setModel(self.charModel)

		self.ui.greekTable_enter.setModel(self.greekModel)
		self.ui.greekTable_teach.setModel(self.greekModel)
		
		self.ui.cyrillicTable_enter.setModel(self.cyrillicModel)
		self.ui.cyrillicTable_teach.setModel(self.cyrillicModel)

		#Size columns to contentsize in the charTables of the GUI
		self.ui.charTable_enter.resizeColumnsToContents()
		self.ui.charTable_enter.resizeRowsToContents()
		self.ui.greekTable_enter.resizeColumnsToContents()
		self.ui.greekTable_enter.resizeRowsToContents()
		self.ui.cyrillicTable_enter.resizeColumnsToContents()
		self.ui.cyrillicTable_enter.resizeRowsToContents()		
		self.ui.charTable_teach.resizeColumnsToContents()
		self.ui.charTable_teach.resizeRowsToContents()
		self.ui.greekTable_teach.resizeColumnsToContents()
		self.ui.greekTable_teach.resizeRowsToContents()
		self.ui.cyrillicTable_teach.resizeColumnsToContents()
		self.ui.cyrillicTable_teach.resizeRowsToContents()

		#Get the size of the charview like there aren't any scrollbars needed
		charVerticalHeader = self.ui.charTable_enter.verticalHeader()
		charHorizontalHeader = self.ui.charTable_enter.horizontalHeader()

		charWidth = charVerticalHeader.sizeHint().width() + charHorizontalHeader.length()

		#Get the size of the greekview like there aren't any scrollbars needed
		greekVerticalHeader = self.ui.greekTable_enter.verticalHeader()
		greekHorizontalHeader = self.ui.greekTable_enter.horizontalHeader()

		greekWidth = greekVerticalHeader.sizeHint().width() + greekHorizontalHeader.length()

		#Get the size of the cyrillicview like there aren't any scrollbars needed
		cyrillicVerticalHeader = self.ui.cyrillicTable_enter.verticalHeader()
		cyrillicHorizontalHeader = self.ui.cyrillicTable_enter.horizontalHeader()

		cyrillicWidth = cyrillicVerticalHeader.sizeHint().width() + cyrillicHorizontalHeader.length()
		
		#Set that as the minimumsize (to prevent from scrollbars popping in.)
		self.ui.charTable_enter.setMinimumWidth(charWidth)
		self.ui.charTable_teach.setMinimumWidth(charWidth)
		self.ui.greekTable_enter.setMinimumWidth(greekWidth)
		self.ui.greekTable_teach.setMinimumWidth(greekWidth)
		self.ui.cyrillicTable_enter.setMinimumWidth(cyrillicWidth)
		self.ui.cyrillicTable_teach.setMinimumWidth(cyrillicWidth)
		
		#Set a default text in the enterTextField
		enteredWords = u"""{first_line}\n{second_line}\n{third_line}"""

		enteredWords = enteredWords.format(
			first_line = QtCore.QCoreApplication.translate("OpenTeacher", "een = one"),
			second_line = QtCore.QCoreApplication.translate("OpenTeacher", "twee = two"),
			third_line = QtCore.QCoreApplication.translate("OpenTeacher", "drie = three")
		)
		self.ui.enterTextField.setText(enteredWords)

		self.backToTheLesson()

		#Load the positions from splitters from the settings, because the user's choice
		#is (almost :P) always the best.
		for splitter_name in self.splitters:
			splitter = self.splitters[splitter_name]
			splitter.restoreState(self.settings.value(splitter_name).toByteArray())

		#resize word list view header
		self.ui.wordsTable.horizontalHeader().resizeSections(
			QtGui.QHeaderView.ResizeToContents
		)

		#Create a connection to WRTS; not useable until the user logged in.
		self.wrtsConnection = api.wrts.WrtsConnection()

		#The current file is empty or loaded unchanged from a file, so it is
		#already saved. Adjusts the GUI, and makes it possible to track the
		#state of the loaded file later.
		self.setSaved()
		self.checkEmpty()

		#Connect menu- and toolbaractions to their methods.
		self.connect(self.ui.actionAbout, QtCore.SIGNAL("triggered()"), self.showAboutDialog)
		self.connect(self.ui.actionDocumentation, QtCore.SIGNAL("triggered()"), self.showDocumentationDialog)
		self.connect(self.ui.actionSettings, QtCore.SIGNAL("triggered()"), self.showSettingsDialog)

		self.connect(self.ui.actionNew, QtCore.SIGNAL("triggered()"), self.newFile)
		self.connect(self.ui.actionSave, QtCore.SIGNAL("triggered()"), self.saveFunc)
		self.connect(self.ui.actionSave_As, QtCore.SIGNAL("triggered()"), self.saveWithDialog)
		self.connect(self.ui.actionOpen, QtCore.SIGNAL("triggered()"), self.openFunc)
		self.connect(self.ui.actionWRTS_Import, QtCore.SIGNAL("triggered()"), self.importWRTS)
		self.connect(self.ui.actionWRTS_Export, QtCore.SIGNAL("triggered()"), self.exportWRTS)
		self.connect(self.ui.actionPrint, QtCore.SIGNAL("triggered()"), self.printFunc)
		self.connect(self.ui.actionExit, QtCore.SIGNAL("triggered()"), self, QtCore.SLOT("close()"))

		#Connect all buttons to their methods
		self.connect(self.ui.enterButton, QtCore.SIGNAL("clicked()"), self.enter)
		self.connect(self.ui.removeButton, QtCore.SIGNAL("clicked()"), self.remove)
		self.connect(self.ui.changeLessonSettingsButton, QtCore.SIGNAL("clicked()"), self.showLessonSettings)
		self.connect(self.ui.backToTheLessonButton, QtCore.SIGNAL("clicked()"), self.backToTheLesson)
		self.connect(self.ui.checkButton, QtCore.SIGNAL("clicked()"), self.checkTyping)
		self.connect(self.ui.viewAnswerButton, QtCore.SIGNAL("clicked()"), self.viewAnswer)
		self.connect(self.ui.rightButton, QtCore.SIGNAL("clicked()"), self.answerRight)
		self.connect(self.ui.wrongButton, QtCore.SIGNAL("clicked()"), self.answerWrong)
		self.connect(self.ui.correctButton, QtCore.SIGNAL("clicked()"), self.correctLast)
		self.connect(self.ui.checkButtonShuffle, QtCore.SIGNAL("clicked()"), self.checkTyping)
		self.connect(self.ui.correctButtonShuffle, QtCore.SIGNAL("clicked()"), self.correctLast)
		self.connect(self.ui.startRepeat_button, QtCore.SIGNAL("clicked()"), self.startRepeatMode)
		self.connect(self.ui.checkButtonRepeat, QtCore.SIGNAL("clicked()"), self.checkTyping)
		self.connect(self.ui.correctButtonRepeat, QtCore.SIGNAL("clicked()"), self.correctLast)

		#Connect the special symbol-buttons to their methods.
		self.connect(self.ui.charTable_enter, QtCore.SIGNAL("clicked(QModelIndex)"), self.addSymbol_charTable_enter)
		self.connect(self.ui.greekTable_enter, QtCore.SIGNAL("clicked(QModelIndex)"), self.addSymbol_greekTable_enter)
		self.connect(self.ui.cyrillicTable_enter, QtCore.SIGNAL("clicked(QModelIndex)"), self.addSymbol_cyrillicTable_enter)
		self.connect(self.ui.charTable_teach, QtCore.SIGNAL("clicked(QModelIndex)"), self.addSymbol_charTable_teach)
		self.connect(self.ui.greekTable_teach, QtCore.SIGNAL("clicked(QModelIndex)"), self.addSymbol_greekTable_teach)
		self.connect(self.ui.cyrillicTable_teach, QtCore.SIGNAL("clicked(QModelIndex)"), self.addSymbol_cyrillicTable_teach)

		#Connect the metadata-boxes to their methods.
		self.connect(self.ui.titleTextField, QtCore.SIGNAL("textChanged(QString)"), self.setTitle)
		self.connect(self.ui.questionLanguageTextField, QtCore.SIGNAL("textChanged(QString)"), self.setQuestionLanguage)
		self.connect(self.ui.answerLanguageTextField, QtCore.SIGNAL("textChanged(QString)"), self.setAnswerLanguage)

		#Connect the lessonproperty-optionboxes to a method which refilters the wordList with the new properties.
		self.connect(self.ui.questionorderBox, QtCore.SIGNAL("currentIndexChanged(int)"), self.typeChange)
		self.connect(self.ui.lessontypeBox, QtCore.SIGNAL("currentIndexChanged(int)"), self.typeChange)
		self.connect(self.ui.lessonorderBox, QtCore.SIGNAL("currentIndexChanged(int)"), self.typeChange)
		self.connect(self.ui.askwordsBox, QtCore.SIGNAL("currentIndexChanged(int)"), self.typeChange)

		#If the user changes the wordListTable, it isn't anymore the same as the saved version. (So the GUI is set to unsaved.)
		self.connect(self.wordListTableModel, QtCore.SIGNAL("dataChanged(QModelIndex, QModelIndex)"), self.setUnsaved)

		#If the master-tabwidget changes, a method which handles starting/stopping of lessons is called.
		self.connect(self.ui.tabWidget, QtCore.SIGNAL("currentChanged(int)"), self.tabChange)

		#If the lesson tabwidget changes, all question-modes are reset.
		self.connect(self.ui.questionType, QtCore.SIGNAL("currentChanged(int)"), self.resetQuestionMode)

################### Methods about entering words
	#Methods to enter symbols into the enterTextField.

	def addSymbol_charTable_enter(self, index):
		self.ui.enterTextField.insertPlainText(self.charModel.characters[index.row()][index.column()])
		self.ui.enterTextField.setFocus()

	def addSymbol_greekTable_enter(self, index):
		self.ui.enterTextField.insertPlainText(self.greekModel.characters[index.row()][index.column()])
		self.ui.enterTextField.setFocus()
		
	def addSymbol_cyrillicTable_enter(self, index):
		self.ui.enterTextField.insertPlainText(self.cyrillicModel.characters[index.row()][index.column()])
		self.ui.enterTextField.setFocus()

	def addSymbol_charTable_teach(self, index):
		char = self.charModel.characters[index.row()][index.column()]
		if self.LessonMode.TYPING == self.ui.questionType.currentIndex():
			self.ui.transEdit.insert(char)
			self.ui.transEdit.setFocus()
		elif self.LessonMode.SHUFFLE == self.ui.questionType.currentIndex():
			self.ui.transEditShuffle.insert(char)
			self.ui.transEditShuffle.setFocus()
		elif self.LessonMode.REPEAT == self.ui.questionType.currentIndex():
			self.ui.transEditRepeat.insert(char)
			self.ui.transEditRepeat.setFocus()

	def addSymbol_greekTable_teach(self, index):
		char = self.greekModel.characters[index.row()][index.column()]
		if self.LessonMode.TYPING == self.ui.questionType.currentIndex():
			self.ui.transEdit.insert(char)
			self.ui.transEdit.setFocus()
		elif self.LessonMode.SHUFFLE == self.ui.questionType.currentIndex():
			self.ui.transEditShuffle.insert(char)
			self.ui.transEditShuffle.setFocus()
		elif self.LessonMode.REPEAT == self.ui.questionType.currentIndex():
			self.ui.transEditRepeat.insert(char)
			self.ui.transEditRepeat.setFocus()
			
	def addSymbol_cyrillicTable_teach(self, index):
		char = self.cyrillicModel.characters[index.row()][index.column()]
		if self.LessonMode.TYPING == self.ui.questionType.currentIndex():
			self.ui.transEdit.insert(char)
			self.ui.transEdit.setFocus()
		elif self.LessonMode.SHUFFLE == self.ui.questionType.currentIndex():
			self.ui.transEditShuffle.insert(char)
			self.ui.transEditShuffle.setFocus()
		elif self.LessonMode.REPEAT == self.ui.questionType.currentIndex():
			self.ui.transEditRepeat.insert(char)
			self.ui.transEditRepeat.setFocus()

	def enter(self):
		"""Method to enter the words from the enterTextField into the wordsTable"""

		#No line without equals sign passed yet.
		withoutEqualsSign = False
		#Get the user input from the GUI
		userInput = unicode(self.ui.enterTextField.toPlainText())

		if userInput.strip() == "":
			#There wasn't any input.
			self.showError(errors.InputEmptyError())
			return

		lines = userInput.split('\n')
		#Run over the input line by line.
		for line in lines[:]:
			if line.strip() == "":
				continue
			try:
				#try to split question and answer(s)
				question, answers = re.split(r"(?<!\\)[=\t]", line, maxsplit=1)
			except IndexError:
				#Input doesn't have a equals-sign or a tab. Set the GUI
				#to unsaved, and put the unprocessed lines back into the
				#textEdit. Then inform the user.
				text = u"\n".join(lines)
				self.ui.enterTextField.setText(text)
				
				self.setUnsaved()
				self.checkEmpty()

				self.showError(errors.NoEqualsSignInInputError())
				return

			question = question.strip()
			answers = answers.strip()
			if question == "" or answers == "":
				#Wrong formatted input (and not a problem with the equals sign/tab)
				self.showError(errors.InputFormatError())

				#Input doesn't have a equals-sign or a tab. Set the GUI
				#to unsaved, and put the unprocessed lines back into the
				#textEdit. Then inform the user.
				text = u"\n".join(lines)
				self.ui.enterTextField.setText(text)

				self.setUnsaved()
				self.checkEmpty()

				return

			#split answers
			answers = re.split(r"(?<!\\)[,;]", answers, maxsplit=1)
			#strip them
			answers = map(lambda a: a.strip(), answers)
			#and remove the empty ones
			answers = filter(lambda a: a != "", answers)

			#Instantiate a word and set the data
			word = words.Word()
			word.question = question

			word.answer = answers[0]
			if len(answers) == 2:
				word.secondAnswer = answers[1]
			#Add the word to the wordListTable, which also is at the moment the masterwordlist.
			self.wordListTableModel.addWord(word)
				
			#Remove the processed line
			del lines[0]

		#Everything went fine, clear the enterTextField and set the GUI to unsaved.
		self.ui.enterTextField.clear()
		self.setUnsaved()
		self.checkEmpty()

		#Tell the user
		self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "The word(s) were entered successfully."))

	def remove(self):
		"""This method removes all the selected rows."""
	#Check if something is selected
		#Get the selectionmodel of the tableView
		selectionModel = self.ui.wordsTable.selectionModel()
		if selectionModel.hasSelection():
			#Get the indexes of all the selected columns
			indexes = selectionModel.selectedIndexes()
			#Create list for the indexes of the selected rows.
			rows = []
			#Loop over all indexes
			for index in indexes:
				#Get the row out of every index, that's the only important part.
				row = index.row()
				#If that rownumber isn't already in the rows-list:
				if not row in rows:
					#Add it to it
					rows.append(row)

			#Now we have an unsorted list of the rownumbers which should be deleted. It are simple integers.

			#Sort the row-list (mostly it is already, but to be sure.)
			rows = sorted(rows)
			#Reverse the list, so the last row is deleted first. (Then the rownumbers don't shift)
			rows.reverse()
			#Delete every row, one by one
			for row in rows:
				self.wordListTableModel.removeWord(row)

			#Adjust the GUI
			self.setUnsaved()
			self.checkEmpty()

			#Tell the user everything went fine (well, it didn't crash :p)
			self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "The word(s) were removed successfully."))

		else:
			self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "No selection was detected."))
################### END Methods about entering words
################### Methods about change-checking
	def setUnsaved(self):
		#Set the application unsaved
		self.saved = False
		#Change the GUI to the new situation
		self.checkSaved()

	def setSaved(self):
		#Set the application saved
		self.saved = True
		#Change the GUI to the new situation
		self.checkSaved()

	def checkSaved(self):
		#If current WordList is empty, disable Save; if document is saved or not.
		if len(self.currentWordList) == 0:
			self.ui.actionSave.setEnabled(False)
		else:
			#If saved, the save-button isn't enabled. And vice versa.
			self.ui.actionSave.setEnabled(not self.saved)

	def checkEmpty(self):
		"""Checks if there aren words in the wordListTable. The GUI is adjusted to that check."""
		#true if rows in the current WordList, false if not.
		boolean = len(self.currentWordList) != 0
			
		#Update the gui
		self.ui.actionSave_As.setEnabled(boolean)
		self.ui.menu_Export.setEnabled(boolean)
		self.ui.actionPrint.setEnabled(boolean)
		self.ui.tabWidget.setTabEnabled(self.Tabs.TEACH, boolean)

################### END Methods about change-checking

################### Lesson methods

########## Start/stop lesson
	def startLesson(self):
		"""Prepares a lesson"""
		#Set some vars
		self.lessonActive = True
		self.lastMode = None

		#Clean GUI before starting the lesson (less confusing)
		self.resetScore()

		#Try to create a lessonwordlist from the wordlist stored into the wordListTableModel. - This is a masterwordlist-switch.
		#The lessonwordlist is now always most recent. For example, results are stored in the lessonwordlist, not in the tablemodel.
		try:
			self.lessonWordList = words.LessonWordList(self.wordListTableModel.wordList)
		except errors.NoWordsEnteredError, e:
			#It went wrong, tell the user, directly end the lesson, and return.
			self.showError(e)
			self.stopLesson()
			return

		#Set the default lessonproperties (the combobox-indexes match the number
		#in the 'enum' which is the input.)
		self.lessonWordList.questionOrder = self.ui.questionorderBox.currentIndex()
		self.lessonWordList.lessonType = self.ui.lessontypeBox.currentIndex()
		self.lessonWordList.lessonOrder = self.ui.lessonorderBox.currentIndex()
		self.lessonWordList.wordsToAsk = self.ui.askwordsBox.currentIndex()

		#Filter the wordlist with the lessonproperties.
		try:
			self.lessonWordList.filter()
		except errors.NoWordsLeftError, e:
			#Problem during filtering, set most GUI-components to non-active and tell the user. Then return.
			
			#TranslationEdits
			self.ui.transEdit.setReadOnly(True)
			self.ui.transEditShuffle.setReadOnly(True)
			self.ui.transEditRepeat.setReadOnly(True)

			#CheckButtons
			self.ui.checkButton.setEnabled(False)
			self.ui.checkButtonShuffle.setEnabled(False)
			self.ui.checkButtonRepeat.setEnabled(False)

			#CorrectButtons
			self.ui.correctButton.setEnabled(False)
			self.ui.correctButtonShuffle.setEnabled(False)
			self.ui.correctButtonRepeat.setEnabled(False)

			#In-mind widgets
			self.ui.viewAnswerButton.setEnabled(False)
			self.ui.rightButton.setEnabled(False)
			self.ui.wrongButton.setEnabled(False)

			#Clear wordLabel
			self.ui.wordLabel.clear()

			#Clear TranslationEdits
			self.ui.transEdit.clear()
			self.ui.transEditShuffle.clear()
			self.ui.transEditRepeat.clear()

			#Clear correctLabels
			self.ui.correctLabel.clear()
			self.ui.correctLabelShuffle.clear()
			self.ui.correctLabelRepeat.clear()

			self.showError(e)
			return

		#Disable the correctButton, because at the beginning of a lesson, you can't correct a word.
		self.ui.correctButton.setEnabled(False)
		self.ui.correctButtonShuffle.setEnabled(False)
		self.ui.correctButtonRepeat.setEnabled(False)

		##Create the wrongFader (The one used when an answer is wrong)
		#Load the fade duration
		if self.settings.value("fade_duration_when_wrong").isNull():
			fadeDuration = DEFAULT_FADE_DURATION_WHEN_WRONG
		else:
			fadeDuration = int(self.settings.value("fade_duration_when_wrong").toInt()[0])

		#Create the fader
		self.wrongFader = Fader(None,
								QtGui.QPalette.Text,
								fadeDuration,
								self.answerWrong,
								QtGui.QColor(218, 15, 15))
		##Create the repeatFader (The one used by the REPEAT mode)
		#Get the hinted fade duration
		if self.settings.value("repeat_mode_hint_duration").isNull():
			hintDuration = DEFAULT_REPEAT_MODE_HINT_DURATION
		else:
			hintDuration = int(self.settings.value("repeat_mode_hint_duration").toInt()[0])
		#Create the Fader
		self.repeatFader = Fader(self.ui.hintLabelRepeat,
								 QtGui.QPalette.WindowText,
								 hintDuration,
								 self.startRetypingRepeatMode)

		#Get the first word
		self.nextWord()

	def stopLesson(self):
		"""Stops the active lesson"""
		#If lesson is active, stop it
		if self.lessonActive:
			#set the lesson to inactive
			self.lessonActive = False
			#Check if the lessonWordList still exists.
			try:
				self.lessonWordList
			except AttributeError:
				#It doesn't exist, just go on with the next operation
				pass
			else:
				#It does exist, get the wordList from the lessonWordList, and put it into the wordListTable.
				#This is a master-wordlist change, the one in the table is now used for edits etc.
				self.wordListTableModel.wordList = self.lessonWordList.wordList
				#Remove the now useless lessonWordList
				del self.lessonWordList
			#Delete 'lastMode'
			del self.lastMode

			#Clean up the GUI
			self.ui.wordLabel.clear()
			self.ui.tabWidget.setCurrentIndex(self.Tabs.ENTER)

########## END start/stop lesson

########## Lesson settings show/hide
	def showLessonSettings(self):
		self.ui.changeLessonSettingsButton.hide()
		self.ui.lessonStackedWidget.setCurrentIndex(0) #settings

	def backToTheLesson(self):
		self.ui.changeLessonSettingsButton.show()
		self.ui.lessonStackedWidget.setCurrentIndex(1) #lesson

########## END Lesson settings show/hide

########## Shared lesson functions
	def nextWord(self):
		"""This method gets the next word of the lesson, and adjusts the GUI to it."""
		#Get the new word
		try:
			self.currentWord = self.lessonWordList.getNext()
		except words.LessonWordList.LessonDoneException:
			#View the results and clean up then return
			self.updateScore()
			self.viewResults()
			self.stopLesson()
			return

		#A word was passed by the LessonWordList, show it to the user
		self.ui.wordLabel.setText(self.currentWord.question)
		#And also calculate a hint and show it in the hint mode.
		self.setHint(self.currentWord.answer)

		#Make all practising-types ready for the next word.
		self.resetTyping()
		self.resetInMind()
		self.resetShuffle()
		#the repeat mode isn't here, because it needs a special approach.
		#It's resetted when it's used, by the resetQuestionMode. Otherwise
		#the fade animation would play, which isn't only unneeded (unvisible),
		#but also can give problems if the user is switching tabs within the
		#fade duration to the repeat-tab. (Then the animation and the start
		#button would be displayed -> wrong.)

		#Set focus to the right widget
		if self.LessonMode.TYPING == self.ui.questionType.currentIndex():
			self.ui.transEdit.setFocus()
		elif self.LessonMode.THINK == self.ui.questionType.currentIndex():
			self.ui.viewAnswerButton.setFocus()
		elif self.LessonMode.SHUFFLE == self.ui.questionType.currentIndex():
			self.ui.transEditShuffle.setFocus()
		elif self.LessonMode.REPEAT == self.ui.questionType.currentIndex():
			#Now look if the start-button has to be viewed. Focus happens
			#inside the resetRepeat function.
			if self.lastMode == self.LessonMode.REPEAT:
				#Not new in the Repeat Mode, don't show the start button
				self.resetRepeat(False)
			else:
				#New in the repeat; show the startButton
				self.resetRepeat(True)

	def checkTyping(self):
		"""This method is called by the user, it checks the answer, shows a nice animation if its wrong,
		   and then marks the word right/wrong."""
		##Get the answer of the user
		#Check typing mode to check, and set associated vars
		if self.LessonMode.TYPING == self.ui.questionType.currentIndex():
			userInput = self.ui.transEdit
			correctLabel = self.ui.correctLabel
		elif self.LessonMode.SHUFFLE == self.ui.questionType.currentIndex():
			userInput = self.ui.transEditShuffle
			correctLabel = self.ui.correctLabelShuffle
		elif self.LessonMode.REPEAT == self.ui.questionType.currentIndex():
			userInput = self.ui.transEditRepeat
			correctLabel = self.ui.correctLabelRepeat

		answers = re.split(r"(?<!\\)[,;]", unicode(userInput.text()), maxsplit=1)
		answers = map(lambda a: a.strip(), answers)

		#One answer, the first one.
		if len(answers) == 1 and answers[0] == self.currentWord.answer:
			self.answerRight()
		#One answer, the second one.
		elif len(answers) == 1 and self.currentWord.secondAnswerSet() and answers[0] == self.currentWord.secondAnswer:
			self.answerRight()
		#Two answers, both are given.
		elif len(answers) == 2 and self.currentWord.secondAnswerSet() and self.currentWord.answer in answers and self.currentWord.secondAnswer in answers:
			self.answerRight()
		#Wrong answer
		else:
			#Create a string with the first and, if set, the second answer
			rightAnswer = self.currentWord.answer
			if self.currentWord.secondAnswerSet():
				rightAnswer += "; " + self.currentWord.secondAnswer
			#Check if the input looks like the answer or the second answer.
			try:
				compareTo = difflib.get_close_matches(unicode(userInput.text()),
					[self.currentWord.answer, self.currentWord.secondAnswer])[0]
			except IndexError:
				#It doesn't, set compareTo to None
				compareTo = None

			#If they look like each other.
			if compareTo:
				#Show the differences graphical
				output = "%s: <b>%s</b> [" % (QtCore.QCoreApplication.translate("OpenTeacher", "Correct answer"), rightAnswer)
				lst = list(difflib.ndiff(unicode(userInput.text()), compareTo))
				for item in lst:
					if item.startswith('+ '):
						output += '<span style="color: #1da90b;"><u>%s</u></span>' % item[2:]
					elif item.startswith('- '):
						output += '<span style="color: #da0f0f;"><s>%s</s></span>' % item[2:]
					else:
						output += item[2:]
				correctLabel.setText(output + "]")
			#If they don't look like each other, just the answer is shown.
			else:
				correctLabel.setText("%s: <b>%s</b>" % (QtCore.QCoreApplication.translate("OpenTeacher", "Correct answer"), rightAnswer))

			#Disable GUI while the animation is running
			userInput.setReadOnly(True)
			self.ui.checkButton.setEnabled(False)
			self.ui.checkButtonShuffle.setEnabled(False)
			self.ui.checkButtonRepeat.setEnabled(False)
			self.ui.correctButton.setEnabled(False)
			self.ui.correctButtonShuffle.setEnabled(False)
			self.ui.correctButtonRepeat.setEnabled(False)

			#Set the widget, and then start the wrongFader
			self.wrongFader.widget = userInput
			self.wrongFader.start()

	def correctLast(self):
		"""Corrects the last checked word; all the hard work is done by the lessonWordList."""
		#Correct it
		try:
			#Remove the previous word from the list.
			self.lessonWordList.correctPreviousWord()
		except words.LessonWordList.LessonDoneException:
			#Deactivate the correctButton; it's useless after one use.
			self.ui.correctButton.setEnabled(False)
			self.ui.correctButtonShuffle.setEnabled(False)
			self.ui.correctButtonRepeat.setEnabled(False)
			#Update the score in the GUI
			self.updateScore()

			#View the results and clean up if the last word on the list was corrected
			self.viewResults()
			self.stopLesson()
		else:
			#Deactivate the correctButton; it's useless after one use.
			self.ui.correctButton.setEnabled(False)
			self.ui.correctButtonShuffle.setEnabled(False)
			self.ui.correctButtonRepeat.setEnabled(False)
			#Update the score in the GUI
			self.updateScore()
		
		#Check typing mode to check, and set associated vars
		if self.LessonMode.TYPING == self.ui.questionType.currentIndex():
			userInput = self.ui.transEdit
		elif self.LessonMode.SHUFFLE == self.ui.questionType.currentIndex():
			userInput = self.ui.transEditShuffle
		elif self.LessonMode.REPEAT == self.ui.questionType.currentIndex():
			userInput = self.ui.transEditRepeat
		userInput.setFocus()

	def answerRight(self):
		"""Sets the answer to right, called by program or user when in 'in mind'-mode"""
		self.ui.correctButton.setEnabled(False)
		self.ui.correctButtonShuffle.setEnabled(False)
		self.ui.correctButtonRepeat.setEnabled(False)
		self.lessonWordList.answerRight()
		self._wordAnswered()

	def answerWrong(self):
		"""Set the answer to wrong, called by program or user when in 'in mind'-mode"""
		self.ui.correctButton.setEnabled(True)
		self.ui.correctButtonShuffle.setEnabled(True)
		self.ui.correctButtonRepeat.setEnabled(True)
		self.lessonWordList.answerWrong()
		self._wordAnswered()

	def _wordAnswered(self):
		#Set GUI to unsaved
		self.setUnsaved()
		#Update the score in the GUI
		self.updateScore()
		#Set the 'lastMode'
		self.lastMode = self.ui.questionType.currentIndex()
		#Get the next word
		self.nextWord()

########## END Shared lesson functions

########## Typing mode functions
	def resetTyping(self):
		"""Makes the 'Typing'-part of the GUI ready for the next word"""
		self.ui.transEdit.setReadOnly(False)
		self.ui.checkButton.setEnabled(True)
		self.ui.correctLabel.clear()
		self.ui.transEdit.clear()

########## END Typing mode functions

########## In Mind mode functions
	def viewAnswer(self):
		"""Show the anser ('in mind'-mode, called by user)"""
		#Build the answer, a combination between first and second answer (if exists)
		answer = self.currentWord.answer
		if self.currentWord.secondAnswerSet():
			answer += "; " + self.currentWord.secondAnswer
		#Update the GUI
		self.ui.thinkLabel.setText(unicode(QtCore.QCoreApplication.translate("OpenTeacher", "Translation: ")) + answer)
		self.ui.viewAnswerButton.setVisible(False)
		self.ui.rightButton.setVisible(True)
		self.ui.wrongButton.setVisible(True)
		
		self.ui.rightButton.setFocus()

	def resetInMind(self):
		"""Makes the 'In mind'-part of the GUI ready for the next word"""
		self.ui.thinkLabel.setText(QtCore.QCoreApplication.translate("OpenTeacher", "Think about the answer, and press the 'View answer' button when you're done."))
		self.ui.viewAnswerButton.setVisible(True)
		self.ui.rightButton.setVisible(False)
		self.ui.wrongButton.setVisible(False)

		self.ui.viewAnswerButton.setEnabled(True)
		self.ui.rightButton.setEnabled(True)
		self.ui.wrongButton.setEnabled(True)

########## END In Mind functions

########## Shuffled hint mode functions
#
#	Two solutions for the OpenTeacher 2.2 bug at 24-10-11:
#
#	def setHint(self, answer):
#		hint = QtCore.QCoreApplication.translate("OpenTeacher", "Hint:") + u" "
#		if len(answer) > 1 and not answer.strip(answer[0]): #If len(answer) == 1, it is always the same shuffled#			for i in range(255): #Avoid "while True"
#				hintList = list(answer) #Make random.shuffle() possible
#				random.shuffle(hintList) #The actual shuffling of the word
#				if hintList != list(answer): #Check if the hint isn't the same as the answer
#					hint += u"".join(hintList) #Put the shuffled word in a string
#					self.ui.hintLabelShuffle.setText(hint) #Set the hint
#				else:
#					continue #It's the same; try again
#		else:
#			hint += u"." * len(answer) #It's only one character (0 should already be caught), so the hint string is only a dot
#			self.ui.hintLabelShuffle.setText(hint) #Set the hint

	def setHint(self, answer):
		hint = QtCore.QCoreApplication.translate("OpenTeacher", "Hint:") + u" "

		#Shuffle list making sure that no letter gets at the same
		#position if possible.
		oldAnswer = list(answer)
		answer = oldAnswer[:]
		for i in range(len(answer) -1):
			j = i + 1 + int(random.random() * (len(answer) -i -1))

			answer[i], answer[j] = answer[j], answer[i]
		#Check if the word has a minimum length and could be shuffled.
		#If not, the hint is a few dots, as much as there are letters
		#in the answer
		if len(answer) <= 2 or oldAnswer == answer:
			hint += u"." * len(answer)
		else:
			hint += u"".join(answer)

		#Show the hint in the GUI
		self.ui.hintLabelShuffle.setText(hint)

	def resetShuffle(self):
		"""Makes the 'Shuffle'-part of the GUI ready for the next word"""
		self.ui.transEditShuffle.setReadOnly(False)
		self.ui.checkButtonShuffle.setEnabled(True)
		self.ui.correctLabelShuffle.clear()
		self.ui.transEditShuffle.clear()

########## END Shuffled hint mode functions

########## Repeat mode methods

	def startRepeatMode(self):
		self.resetRepeat(False)

	def startRetypingRepeatMode(self):
		self.enableAnsweringRepeatMode(True)

	def enableAnsweringRepeatMode(self, boolean):
		self.ui.hintLabelRepeat.setVisible(not boolean)
		self.ui.transEditRepeat.setVisible(boolean)
		self.ui.checkButtonRepeat.setVisible(boolean)
		self.ui.correctButtonRepeat.setVisible(boolean)
		self.ui.correctLabelRepeat.setVisible(boolean)
		self.ui.translationLabelRepeat.setVisible(boolean)

	def resetRepeat(self, start):
		"""Loads the next word in the REPEAT mode"""
		self.ui.checkButtonRepeat.setEnabled(True)
		self.ui.transEditRepeat.clear()
		self.ui.transEditRepeat.setReadOnly(False)
		self.ui.correctLabelRepeat.clear()

		self.enableAnsweringRepeatMode(False)
		if self.repeatFader.active:
			self.repeatFader.stop()

		if start:
			#Show start button & give it the foucs
			self.ui.startRepeat_button.setVisible(True)
			self.ui.startRepeat_button.setFocus()
			#create hint
			hint = QtCore.QCoreApplication.translate("OpenTeacher", "Click the button to start.")
		else:
			#Don't show start button, give the translationEdit the focus
			self.ui.startRepeat_button.setVisible(False)
			self.ui.transEditRepeat.setFocus()

			#create hint
			hint = QtCore.QCoreApplication.translate("OpenTeacher", "Hint:") + " " + "".join(self.currentWord.answer)

			#Start the fade animation
			self.repeatFader.start()
		#Set hint into UI
		self.ui.hintLabelRepeat.setText(hint)

########## END Repeat mode methods

########## Score methods

	def updateScore(self):
		"""Gets the score from the lessonWordList, and puts it into the GUI at the right places."""
		self.ui.totalQuestionsLabel.setText(unicode(self.lessonWordList.numberOfAnswers))
		self.ui.rightAnswersLabel.setText(unicode(self.lessonWordList.numberOfRightAnswers))
		self.ui.noteLabel.setText(self.note)

		self.ui.progressBar.setMaximum(self.lessonWordList.numberOfQuestions)
		self.ui.progressBar.setValue(self.lessonWordList.numberOfAnswers)
	
	def resetScore(self):
		"""Resets the score in the GUI, so a new lessons starts clear"""
		self.ui.totalQuestionsLabel.setText(u"0")
		self.ui.rightAnswersLabel.setText(u"0")
		self.ui.noteLabel.setText(u"0")

		self.ui.progressBar.setMaximum(100)
		self.ui.progressBar.setValue(0)

	@property
	def note(self):
		"""Returns the note in the format described by the settings."""
		noteNotation = unicode(self.settings.value("note_notation").toString())
		if noteNotation == "Dutch":
			return self.lessonWordList.dutchNote
		elif noteNotation == "American":
			return self.lessonWordList.americanNote
		elif noteNotation == "German":
			return self.lessonWordList.germanNote
		elif noteNotation == "French":
			return self.lessonWordList.frenchNote
		elif noteNotation == "ECTS":
			return self.lessonWordList.ectsNote
		else:
			#percents and no choice
			return self.lessonWordList.percentsNote

	def viewResults(self):
		"""View the results after a lesson. Simply a QMessageBox with some text and the note, got from
		   the lessonWordList."""
		if self.settings.value("view_results").isNull():
			viewResults = True
		else:
			viewResults = bool(self.settings.value("view_results").toBool())
		if viewResults:
			QtGui.QMessageBox.information(self, QtCore.QCoreApplication.translate("OpenTeacher", "Test completed!"),
										   QtCore.QCoreApplication.translate("OpenTeacher","Test completed! Your note:")+ " " + self.note)
########## END Score methods

########## Other lesson methods
	def resetQuestionMode(self, index):
		"""Resets modes which need to be resetted when the lessonType
		   tabwidget changes it's currentIndex. (THINK & REPEAT)"""
		if index == self.LessonMode.THINK:
			self.resetInMind()
		elif index == self.LessonMode.REPEAT:
			self.resetRepeat(True)

########## END Other lesson methods

################### END Lesson methods
################### WRTS-api methods

	def wrtsLogin(self):
		"""Asks user for email/password and login. Raises possibly WrtsLoginError,
		   WRTSConnectionError or AbortedByUserError."""
		#Try to login with the settings
		if not self.settings.value("wrts_username").isNull():
			username = unicode(self.settings.value("wrts_username").toString())
			password = unicode(self.settings.value("wrts_password").toString())
			try:
				self.wrtsConnection.logIn(username, password)
				return
			except errors.WrtsLoginError:
				pass
		#Creates a dialog used for getting user and password, and executes it
		loginDialog = self.setupDesignedDialog(gui.wrtsLogin)
		loginDialog.exec_()
		if loginDialog.result():
			#User clicked OK, let the connection try to login. Raises possibly WrtsLoginError
			self.wrtsConnection.logIn(unicode(loginDialog.ui.email.text()),
									  unicode(loginDialog.ui.pw.text()))
		else:
			#User clicked cancel, raise an error, to tell higher level functions.
			raise errors.AbortedByUserError()

	def importWRTS(self):
		"""Import a list from WRTS - callable by user (menu)"""
		#Stop ongoing lesson - don't save file yet, it happens later on
		#if needed.
		if not self.handleLessonStopping():
			return
		#Handle login if not logged in
		if not self.wrtsConnection.loggedIn:
			try:
				self.wrtsLogin()
			#when something goes wrong, show and return
			except (errors.WrtsLoginError, errors.WrtsConnectionError), e:
				self.showError(e)
				return
			except errors.AbortedByUserError, e:
				print e
				return

		#Create a dialog for the wordlist names of WRTS
		listChoiceDialog = self.setupDesignedDialog(gui.wrtsListChoice)

		#Get the wordList names from the internet, if there was an error connection, the error is
		#shown to the user. Then the app returns, so the user can try again.
		try:
			parsedLists = self.wrtsConnection.lists
		except errors.WrtsConnectionError, e:
			self.showError(e)
			return

		#Get the parsed lists
		lists = parsedLists.lists

		#Add them to the dialog
		for list in lists:
			listChoiceDialog.ui.list.addItem(list)
		listChoiceDialog.exec_()

		#If the user clicked OK
		if listChoiceDialog.result():
			#Get the url of the wordList from the parsed xml
			url = parsedLists.getWordListUrl(listChoiceDialog.ui.list.currentRow())
			#Download the wordlist itself
			try:
				wordList = self.wrtsConnection.importWordList(url)
			except errors.WrtsConnectionError, e:
				#Show error and return, so the user can retry
				self.showError(e)
				return

			#Check for open wordlist
			if not self.handleSaving():
				return

			#Set the wordlist, got from wrts. (Master wordlist change!)
			self.setWordList(wordList)
			#Set unsaved, the wordlist isn't saved in a local file
			self.setUnsaved()
			self.checkEmpty()
			#Show the user that above all was a succes
			self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "The wordlist from WRTS was imported successfully."))
		else:
			#Record to the terminal that the user aborted (the user doesn't really need to know that, he/she already does.)
			print errors.AbortedByUserError()

	def exportWRTS(self):
		"""Exports the current list to WRTS, called by the user. (action)"""
		#Handle login if not logged in
		if not self.wrtsConnection.loggedIn:
			try:
				self.wrtsLogin()
			#when something goes wrong, show and return
			except (errors.WrtsLoginError, errors.WrtsConnectionError), e:
				self.showError(e)
				return
			except errors.AbortedByUserError, e:
				print e
				return

		#Get the current wordlist
		wordList = self.currentWordList
		try:
			#Try exporting the wordlist
			self.wrtsConnection.exportWordList(wordList)
		except (errors.NotEnoughMetadataError, errors.WrtsConnectionError), e:
			#Show an error if one occurs
			self.showError(e)
			return
		else:
			#No error occured, tell the user all went well.
			title = QtCore.QCoreApplication.translate("OpenTeacher", "Export successful")
			text = QtCore.QCoreApplication.translate("OpenTeacher", "Word list exported to WRTS.")
			QtGui.QMessageBox.information(self, title, text)
			self.statusBar().showMessage(text)

################### END WRTS-api methods
################### Methods handling situations based on user-input
	def handleSaving(self):
		"""This method is a shortcut for saving the current wordList, so it's ready for loading another wordList, quit the application, or
		   something similar. (If not already saved, and the wordList isn't empty.)
		   Returns true when a method can go on without doing anything on application-saving. Returns false if user clicked cancel/doesn't
		   think the wordList is important enough to save. (Or saving went wrong, but that's mostly not the case.)"""

		#Stop an ongoing lesson (if one)
		self.stopLesson()
		#Check if saving is worth the effort.
		if self.settings.value("ask_saving").isNull():
			askSaving = True
		else:
			askSaving = bool(self.settings.value("ask_saving").toBool())
		if not self.saved and len(self.currentWordList) != 0 and askSaving:
			#Ask ik the user wants to save
			reply = QtGui.QMessageBox().question(self, QtCore.QCoreApplication.translate("OpenTeacher", "Message"), QtCore.QCoreApplication.translate("OpenTeacher", "Do you want to save changes?"), QtGui.QMessageBox.Save | QtGui.QMessageBox.Discard | QtGui.QMessageBox.Cancel, QtGui.QMessageBox.Save)
			if reply == QtGui.QMessageBox.Save:
				return self.saveFunc()
			elif reply == QtGui.QMessageBox.Discard:
				return True
			elif reply == QtGui.QMessageBox.Cancel:
				return False
		else:
			#If already saved, don't make it more difficult then it is.
			return True

	def handleLessonStopping(self):
		"""This method is a shortcut for stopping the current lesson (if one) with asking the user if that's allowed.
		   Returns true if the lesson has stopped, and false if not."""
		if self.settings.value("ask_stop_lesson").isNull():
			askStopLesson = True
		else:
			askStopLesson = bool(self.settings.value("ask_stop_lesson").toBool())
		if self.lessonActive and askStopLesson:
			#Are you sure you want to stop the lesson? Yes/No
			warn = QtGui.QMessageBox.question(self, QtCore.QCoreApplication.translate("OpenTeacher","Exit lesson"), QtCore.QCoreApplication.translate("OpenTeacher", "Are you sure you want to stop the lesson?"), QtGui.QMessageBox.Yes, QtGui.QMessageBox.No)
			if warn == QtGui.QMessageBox.Yes:
				self.stopLesson()
				return True
			elif warn == QtGui.QMessageBox.No:
				return False
		else:
			self.stopLesson()
			return True

################### END Methods handling situations based on user-input
################### Filehandling methods

	def newFile(self):
		"""This happens when you click New"""
		#Stop the lesson if active, and save the wordList. Returns if the user doesn't want to create an new file suddenly
		if not self.handleLessonStopping():
			return
		if not self.handleSaving():
			return

		#Create a new, empty wordlist, and put it into the wordListTable (masterwordlist-switch!)
		self.setWordList(words.WordList())
		#Set the saveFileName to none, because a new file is saved nowhere (yet).
		self.saveFilename = unicode()

		#Set it saved (until something is added)
		self.setSaved()
		#Disable most GUI-elements, because yes, it's empty.
		self.checkEmpty()

		#Reset the windowtitle
		self.setWindowTitle("OpenTeacher")
		#Tell the user everything went smooth.
		self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "New file created."))

	def openFunc(self):
		"""Function to handle opening files (with dialog, user side of it)"""
		#Stop ongoing lesson and save current file.
		if not self.handleLessonStopping():
			return
		if not self.handleSaving():
			return
		#Ask for a file
		openFilename = unicode(QtGui.QFileDialog.getOpenFileName(
			self,
			QtCore.QCoreApplication.translate("OpenTeacher", "Open..."),
			self.currentDir,
			QtCore.QCoreApplication.translate("OpenTeacher", "Supported") + " (*.ot *.t2k *.wrts *.xml)")
		)
		#If not clicked cancel
		if openFilename:
			#Set the dir to the last opened directory
			self.currentDir = unicode(os.path.split(openFilename)[0])
			#Load file
			try:
				self.openFile(openFilename)
			except errors.UnsupportedFileImportError, e:
				#inform user when an error occured.
				self.showError(e)
			else:
				#Nothing weird happened, tell the user...
				self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "The selected file was opened successfully."))
		else:
			#clicked cancel, record that to terminal for debugging purposes.
			print errors.AbortedByUserError()

	def openFile(self, filename):
		#get the extension, to make sure what kind of file we have.
		extension = os.path.splitext(os.path.basename(filename))[1]
		if extension == ".ot":
			#Parse as OpenTeacher file
			parsedFile = fileTypes.OpenTeacherFile(filename)
		elif extension == ".t2k":
			#Parse as Teach2000 file
			parsedFile = fileTypes.Teach2000File(filename)
		elif extension == ".wrts":
			#Parse as WRTS file
			parsedFile = fileTypes.WrtsFile(filename)
		elif extension == ".xml":
			#Parse ABBY Lingvo Tutor file
			parsedFile = fileTypes.ABBYLingvoTutorFile(filename)
		else:
			#Unsupported file, tell user.
			raise errors.UnsupportedFileImportError()

		#Import file
		try:
			parsedFile.load()
		except IOError:
			self.showError(errors.OpenError())
			return
		except errors.ContentError, e:
			self.showError(e)
			return

		#Set the imported wordList into the wordListTable. Make sure there isn't a lesson active when opening a file,
		#because then there are two master-wordLists, which is wrong. (The new one is not shown to the user.) That
		#check is done in openFunc() at this moment.
		self.setWordList(parsedFile.wordList)

		#Update GUI
		self.setSaved()
		self.checkEmpty()

		#Set the current filename to the filename of the file.
		self.saveFilename = unicode(filename)
		#Update the windowtitle with the filename
		self.setWindowTitle("%s - OpenTeacher" % os.path.basename(self.saveFilename))

	def saveFunc(self):
		"""This happens when you click Save"""
		#If there isn't a filename known for the current file, ask one
		if self.saveFilename == unicode():
			return self.saveWithDialog()
		else:
			#Just save with the known filename
			return self.save(self.saveFilename)

	def saveWithDialog(self):
		"""Method to handle saving files -> Happens when you click Save As"""

		#Creates filters
		filters = (("OpenTeacher (*.ot)", ".ot"),
				   ("Teach2000 (*.t2k)", ".t2k"),
				   ("Wrts (*.wrts)", ".wrts"),
				   ("%s (*.txt)" % QtCore.QCoreApplication.translate("OpenTeacher", "Plain text"), ".txt"))

		filtersStr = ""
		for filter in filters:
			filtersStr += filter[0] + ";;"
		filtersStr = filtersStr[:-2]

		#change filters into a dict-type!
		filters = dict(filters)

		#Creates a filedialog
		filetype = QtCore.QString()
		saveLocation = unicode(QtGui.QFileDialog.getSaveFileName(self,
			QtCore.QCoreApplication.translate("OpenTeacher", "Save..."),
			self.currentDir,
			filtersStr,
			filetype
		))

		if not saveLocation:
			print errors.AbortedByUserError()
			return

		#Check extension, add from the selected filter when needed.
		name, ext = os.path.splitext(saveLocation)
		if ext == "":
			saveLocation = saveLocation + filters[unicode(filetype)]

		#Set the dir to the last opened directory
		self.currentDir = unicode(os.path.split(saveLocation)[0])

		#Save the file.
		return self.save(saveLocation)

	def save(self, location, checkOverwrite=False):
		#Get the extension
		extension = os.path.splitext(location)[1]

		#Checks what filetype should be used, and creates an instance of it, which can be used to save the file.
		if extension == ".ot":
			#OpenTeacher
			fileContent = fileTypes.OpenTeacherFile(location)
		elif extension == ".t2k":
			#Teach2000
			fileContent = fileTypes.Teach2000File(location)
		elif extension == ".wrts":
			#Wrts
			fileContent = fileTypes.WrtsFile(location)
		elif extension == ".txt":
			#Plain text
			fileContent = fileTypes.TextFile(location)
		else:
			#Show an error, which tells the filetype isn't supported to the user. Then return.
			self.showError(errors.UnsupportedFileExportError())
			return False

		wordList = self.currentWordList
		#Set an id (some exports require it).
		for word in wordList:
			word.id = wordList.index(word)

		#Set the wordlist
		fileContent.wordList = wordList

		#Export it to location
		try:
			fileContent.save()
		except IOError:
			#show nice message
			self.showError(errors.SaveError())
			return False

		#saving
		if extension not in ["txt"]:
			#Set variables, update GUI
			self.saveFilename = unicode(location)
			self.setWindowTitle("%s - OpenTeacher" % os.path.basename(self.saveFilename))
			self.setSaved()
			#Tell the user saving was a succes.
			self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "File saved."))
			return True
		#exporting
		else:
			return False

################### END Filehandling methods
################### Wordlist methods
	def setWordList(self, wordList):
		"""Shortcut for setting a wordList into the wordListTableModel."""
		#Set the metadata of the wordList into the metadatafields of the interface
		self.ui.titleTextField.setText(wordList.title)
		self.ui.questionLanguageTextField.setText(wordList.questionLanguage)
		self.ui.answerLanguageTextField.setText(wordList.answerLanguage)

		#Set the wordList
		self.wordListTableModel.wordList = wordList
		#Make sure that the user can directly start entering words.
		self.ui.enterTextField.setFocus()

	@property
	def currentWordList(self):
		"""Returns the current master-wordList."""
		if self.lessonActive:
			return self.lessonWordList.wordList
		else:
			return self.wordListTableModel.wordList

	def setTitle(self, text):
		"""Called by Qt when the title-field of the GUI was modified. Sets the new title in the model. (so it can be used for saving,
		   exporting, etc.) Also sets the GUI to unsaved."""
		self.currentWordList.title = text
		self.setUnsaved()

	def setQuestionLanguage(self, text):
		"""Also called by Qt"""
		self.currentWordList.questionLanguage = text
		self.setUnsaved()

	def setAnswerLanguage(self, text):
		"""And this one also."""
		self.currentWordList.answerLanguage = text
		self.setUnsaved()

################### END Wordlist methods
################### Other methods

	def printFunc(self):
		"""This method prints the current wordlist."""
		#Check if the wordList isn't empty (shouldn't be, the print-action is not enabled then.)
		if len(self.currentWordList) != 0:
			#Get the current wordList
			wordList = self.currentWordList
			#Create a plain text document, because that's what we're doing: printing == printing the wordList exported as .txt.
			textContent = fileTypes.TextFile(None)

			#Set the wordList
			textContent.wordList = wordList

			#Get the text as a string, and put it in a QTextDocument, which is printable by Qt.
			text = QtGui.QTextDocument(textContent.getText())
			#Set the document title to 'OpenTeacher' (with the wordList title if one), it's
			#shown in the printqueue.
			if wordList.titleSet():
				text.setMetaInformation(QtGui.QTextDocument.DocumentTitle, "OpenTeacher - %s" % wordList.title)
			else:
				text.setMetaInformation(QtGui.QTextDocument.DocumentTitle, "OpenTeacher")
			#Set to a font where every letter has the same width. (We're working with spaces.)
			text.setDefaultFont(QtGui.QFont("courier"))
			#Create a printer
			printer = QtGui.QPrinter()

			#Ask for settings with the default print-dialog.
			printdialog = QtGui.QPrintDialog(printer)
			printdialog.setModal(True)
			printdialog.setWindowTitle(QtCore.QCoreApplication.translate("OpenTeacher","Print Document"))

			# If printdialog is closed (and not canceled), print the page(s)
			if printdialog.exec_():
				text.print_(printer)
				#Tell the user that he/she has to run to his/her printer.
				self.statusBar().showMessage(QtCore.QCoreApplication.translate("OpenTeacher", "The file was printed successfully."))
			else:
				#Record to the terminal that the user canceled.
				print errors.AbortedByUserError()
		else:
			#Inform the user
			self.showError(errors.NoWordsEnteredError())

	def showSettingsDialog(self):
		"""Shows the setttings dialog. Called by user via action."""
		settingsDialog = SettingsDialog(self.settings, self.charModel, self)
		settingsDialog.exec_()
		if settingsDialog.result():
			settingsDialog.saveSettings()
		else:
			print errors.AbortedByUserError()

	def showAboutDialog(self):
		"""Shows the about dialog. Called by user via action."""
		aboutDialog = AboutDialog(self)
		aboutDialog.exec_()

	def showDocumentationDialog(self):
		"""Shows the documentation dialog. Called by the user via action."""
		#Get language
		#If the settings-dialog is never opened
		if self.settings.value("language").isNull():
				langAndCountry = unicode(QtCore.QLocale().name())
				lang = langAndCountry[:langAndCountry.find("_")]
		else:
				lang = unicode(self.settings.value("language").toString())

		docDialog = DocumentationDialog(lang, self)
		docDialog.show()

	def setupDesignedDialog(self, designedDialogClass):
		"""Method which does setup a dialog created by Qt Designer, and returns it to the user. All left is calling exec_()"""
		dialog = QtGui.QDialog()
		dialog.ui = designedDialogClass.Ui_Dialog()
		dialog.ui.setupUi(dialog)
		return dialog

	def showError(self, error):
		"""Shows an error to the user, by printing it to the terminal, viewing it in the statusbar, and showing it in a QMessageBox."""
		print error
		self.statusBar().showMessage(unicode(error))
		QtGui.QMessageBox.warning(self, "OpenTeacher: " + QtCore.QCoreApplication.translate("OpenTeacher", "Error") + " " + unicode(error.errorNumber), error.message)

	def tabChange(self, newTab):
		"""Method called by Qt when tab is being changed"""
		# if newTab == lessonTab and lesson isn't active
		if newTab == self.Tabs.TEACH and self.lessonActive == False:
			#start lesson
			self.startLesson()
		#if newTab == inputTab and lesson is active
		if newTab == self.Tabs.ENTER and self.lessonActive == True:
			#First undo the tabswitch
			self.ui.tabWidget.setCurrentIndex(self.Tabs.TEACH)
			#Ask the user if he/she wants to stop the running lesson.
			self.handleLessonStopping()
		#All other circumstances pass without problems.
		else:
			pass

	def typeChange(self, index):
		"""Function is called when you change a lesson setting. It's a pretty name for 'startLesson'."""
		self.startLesson()

	def closeEvent(self, event):
		"""This happens when you close the application"""
		#If the user doesn't want to stop the lesson, don't close.
		if not self.handleLessonStopping():
			event.ignore()
		#If the user cancels saving, saves as txt or something like that, don't close.
		elif not self.handleSaving():
			event.ignore()
		#Else, close.
		else:
			#Save currentDir
			self.settings.setValue("current_directory", self.currentDir)

			#Save the splitter-states...
			for splitter_name in self.splitters:
				splitter = self.splitters[splitter_name]
				self.settings.setValue(splitter_name, splitter.saveState())

			event.accept()
################### END Other methods

#Load app
def main():
	#Instantiate the mainloop
	app = QtGui.QApplication(sys.argv)

	#Load settings
	settings = QtCore.QSettings("OpenTeacher", "OpenTeacher")

	#Get language
	#If the settings-dialog is never opened
	if settings.value("language").isNull():
			langAndCountry = unicode(QtCore.QLocale().name())
			lang = langAndCountry[:langAndCountry.find("_")]
			if lang in ("en", "pt", "zh"):
				lang = unicode(QtCore.QLocale().name())
	else:
			lang = unicode(settings.value("language").toString())

	if lang != "en":
			#Load translation
			appTranslator = QtCore.QTranslator()
			appTranslator.load(":/translations/%s.qm" % lang)
			app.installTranslator(appTranslator)

	#Shouldn't be needed, but gives a huge speedup. (Probably because
	#only one QSettings tries to get to a settingsfile.)
	del(settings)

	#Show the mainwindow
	openTeacher = OpenTeacher()
	openTeacher.show()

	#Start the mainloop, and exit when it ends.
	sys.exit(app.exec_())

#If not run as a module:
if __name__ == "__main__":
    main()
