# -*- coding: utf-8 -*-
#
#  Copyright (C) 2003-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import logging
import os
import hashlib
import logging

import numpy
import gtk
import cairo


class BaseTransparentWindow(gtk.Window):
    __gsignals__ = {'screen-changed': 'override', }

    def __init__(self, type=gtk.WINDOW_TOPLEVEL):
        gtk.Window.__init__(self, type=type)
        self.set_decorated(False)
        self.set_resizable(False)
        self.screen_changed()

    def screen_changed(self, old_screen=None):
        screen = self.get_screen()
        if self.is_composited():
            colormap = screen.get_rgba_colormap()
            self.supports_alpha = True
        else:
            logging.debug('screen does NOT support alpha.\n')
            colormap = screen.get_rgb_colormap()
            self.supports_alpha = False
        self.set_colormap(colormap)


class TransparentWindow(BaseTransparentWindow):

    def __init__(self, type=gtk.WINDOW_TOPLEVEL):
        BaseTransparentWindow.__init__(self, type=type)
        self.set_app_paintable(True)
        self.set_focus_on_map(False)
        self.__position = (0, 0)
        self.__surface_position = (0, 0)
        self.__redraw = None
        self.connect_after('size_allocate', self.size_allocate)
        # create drawing area
        self.darea = gtk.DrawingArea()
        self.darea.show()
        self.darea.connect = self.wrap_connect
        self.add(self.darea)

    def wrap_connect(self, signal, user_function, user_data=None):
        if signal in ['expose_event', 'expose-vent']:
            self.__redraw = (user_function, user_data)
            gtk.DrawingArea.connect(self.darea, signal, self.wrap_draw)
        else:
            if user_data is not None:
                gtk.DrawingArea.connect(
                    self.darea, signal, user_function, user_data)
            else:
                gtk.DrawingArea.connect(
                    self.darea, signal, user_function)

    def wrap_draw(self, darea, event):
        if self.__redraw is None:
            return
        cr = darea.window.cairo_create()
        cr.translate(*self.get_draw_offset())
        user_function, user_data = self.__redraw
        if user_data is not None:
            user_function(darea, cr, user_data)
        else:
            user_function(darea, cr)

    def set_shape_surface(self, surface, x, y):
        w, h = self.get_child().get_size_request() # XXX
        bitmap = gtk.gdk.Pixmap(None, w, h, 1)
        cr = bitmap.cairo_create()
        cr.set_source_rgb(0.0, 0.0, 0.0)
        cr.set_operator(cairo.OPERATOR_CLEAR)
        cr.paint()
        cr.translate(*self.get_draw_offset())
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.set_source_surface(surface, x, y)
        cr.paint()
        del cr
        if self.is_composited():
            self.input_shape_combine_mask(bitmap, 0, 0)
        else:
            self.shape_combine_mask(bitmap, 0, 0)

    def update_size(self, w, h):
        self.get_child().set_size_request(w, h) # XXX
        self.queue_resize()

    def size_allocate(self, widget, event):
        new_x, new_y = self.__position
        gtk.Window.move(self, new_x, new_y)

    def move(self, x, y):
        left, top, scrn_w, scrn_h = get_workarea()
        w, h = self.get_child().get_size_request() # XXX
        new_x = min(max(left, x), left + scrn_w - w)
        new_y = min(max(top, y), top + scrn_h - h)
        gtk.Window.move(self, new_x, new_y)
        self.__position = (new_x, new_y)
        self.__surface_position = (x, y)
        self.darea.queue_draw()

    def get_draw_offset(self):
        window_x, window_y = self.__position
        surface_x, surface_y = self.__surface_position
        return surface_x - window_x, surface_y - window_y

    def winpos_to_surfacepos(self, x, y, scale):
        window_x, window_y = self.__position
        surface_x, surface_y = self.__surface_position
        new_x = int((x - (surface_x - window_x)) * 100 / scale)
        new_y = int((y - (surface_y - window_y)) * 100 / scale)
        return new_x, new_y


def get_png_size(path):
    if not path or not os.path.exists(path):
        return 0, 0
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = ext.lower()
    if ext == '.dgp':
        buf = get_DGP_IHDR(path)
    elif ext == '.ddp':
        buf = get_DDP_IHDR(path)
    else:
        buf = get_png_IHDR(path)
    assert buf[0:8] == '\x89PNG\r\n\x1a\n' # png format
    assert buf[12:16] == 'IHDR' # name of the first chunk in a PNG datastream
    w = buf[16:20]
    h = buf[20:24]
    width = (ord(w[0]) << 24) + (ord(w[1]) << 16) + (ord(w[2]) << 8) + ord(w[3])
    height = (ord(h[0]) << 24) + (ord(h[1]) << 16) + (ord(h[2]) << 8) + ord(h[3])
    return width, height

def get_DGP_IHDR(path):
    head, tail = os.path.split(path)
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) // 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        return get_png_IHDR(path)
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    buf = ''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            buff = ''.join((buf, chr(ord(c) ^ ord(key[key_pos]))))
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    return buf

def get_DDP_IHDR(path):
    size = os.path.getsize(path)
    key = size << 2    
    buf = ''
    with open(path, 'rb') as f:
        for i in range(24):
            c = f.read(1)
            key = (key * 0x08088405 + 1) & 0xffffffff
            buf = ''.join((buf, chr((ord(c) ^ key >> 24) & 0xff)))
    return buf

def get_png_IHDR(path):
    with open(path, 'rb') as f:
        buf = f.read(24)
    return buf

def __pixbuf_new_from_file(path):
    if os.name == 'nt': # XXX
        path = unicode(path, 'mbcs').encode('utf-8')
    return gtk.gdk.pixbuf_new_from_file(path)

def __surface_new_from_file(path):
    if os.name == 'nt': # XXX
        path = unicode(path, 'mbcs').encode('utf-8')
    return cairo.ImageSurface.create_from_png(path)

def create_icon_pixbuf(path):
    try:
        pixbuf = __pixbuf_new_from_file(path)
    except: # compressed icons are not supported. :-(
        pixbuf = None
    else:
        pixbuf = pixbuf.scale_simple(16, 16, gtk.gdk.INTERP_BILINEAR)
    return pixbuf

def create_blank_pixbuf(width, height):
    pixbuf = gtk.gdk.Pixbuf(gtk.gdk.COLORSPACE_RGB, True, 8, width, height)
    pixbuf.fill(0xffffffffL)
    return pixbuf

def create_blank_surface(width, height): ## FIXME
    surface = cairo.ImageSurface(cairo.FORMAT_ARGB32, width, height)
    return surface

def create_pixbuf_from_DGP_file(path):
    head, tail = os.path.split(path)
    filename = tail
    m_half = hashlib.md5(filename[:len(filename) // 2]).hexdigest()
    m_full = hashlib.md5(filename).hexdigest()
    tmp = ''.join((m_full, filename))
    key = ''
    j = 0
    for i in range(len(tmp)):
        value = ord(tmp[i]) ^ ord(m_half[j])
        if not value:
            break
        key = ''.join((key, chr(value)))
        j += 1
        if j >= len(m_half):
            j = 0
    key_length = len(key)
    if key_length == 0: # not encrypted
        logging.warning(''.join((filename, ' generates a null key.')))
        pixbuf = __pixbuf_new_from_file(filename)
        return pixbuf
    key = ''.join((key[1:], key[0]))
    key_pos = 0
    loader = gtk.gdk.PixbufLoader('png')
    with open(path, 'rb') as f:
        while 1:
            c = f.read(1)
            if c == '':
                break
            loader.write(chr(ord(c) ^ ord(key[key_pos])), 1)
            key_pos += 1
            if key_pos >= key_length:
                key_pos = 0
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_pixbuf_from_DDP_file(path):
    with open(path, 'rb') as f:
        buf = f.read()
    key = len(buf) << 2
    loader = gtk.gdk.PixbufLoader('png')
    for i in range(len(buf)):
        key = (key * 0x08088405 + 1) & 0xffffffff
        loader.write(chr((ord(buf[i]) ^ key >> 24) & 0xff), 1)
    pixbuf = loader.get_pixbuf()
    loader.close()
    return pixbuf

def create_pixbuf_from_file(path, is_pnr=True, use_pna=False):
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    ext = ext.lower()
    if ext == '.dgp':
        pixbuf = create_pixbuf_from_DGP_file(path)
    elif ext == '.ddp':
        pixbuf = create_pixbuf_from_DDP_file(path)
    else:
        pixbuf = __pixbuf_new_from_file(path)
    if is_pnr:
        array = pixbuf.get_pixels_array()
        if not pixbuf.get_has_alpha():
            r, g, b = array[0][0]
            pixbuf = pixbuf.add_alpha(True, chr(r), chr(g), chr(b))
        else:
            ar = numpy.frombuffer(array, numpy.uint32)
            rgba = ar[0]
            ar[ar == rgba] = 0x00000000
    if use_pna:
        path = os.path.join(head, ''.join((basename, '.pna')))
        if os.path.exists(path):
            assert pixbuf.get_has_alpha()
            pna_pixbuf = __pixbuf_new_from_file(path)
            pna_array = pna_pixbuf.get_pixels_array()
            assert pna_pixbuf.get_bits_per_sample() / 8 == 1
            pixbuf_array = pixbuf.get_pixels_array()
            pixbuf_array[:,:,3] = pna_array[:,:,0]
    return pixbuf

def create_surface_from_file(path, is_pnr=True, use_pna=False):
    head, tail = os.path.split(path)
    basename, ext = os.path.splitext(tail)
    pixbuf = create_pixbuf_from_file(path, is_pnr, use_pna)
    surface = cairo.ImageSurface(cairo.FORMAT_ARGB32,
                                 pixbuf.get_width(), pixbuf.get_height())
    cr = cairo.Context(surface)
    cr = gtk.gdk.CairoContext(cr)
    cr.set_source_pixbuf(pixbuf, 0, 0)
    cr.set_operator(cairo.OPERATOR_SOURCE)
    cr.paint()
    del cr
    return surface

def create_pixmap_from_file(path):
    pixbuf = create_pixbuf_from_file(path)
    pixmap, mask = pixbuf.render_pixmap_and_mask(1)
    return pixmap, mask

def get_workarea():
    scrn = gtk.gdk.screen_get_default()
    root = scrn.get_root_window()
    if not hasattr(scrn, 'supports_net_wm_hint') or \
            not scrn.supports_net_wm_hint('_NET_CURRENT_DESKTOP') or \
            not scrn.supports_net_wm_hint('_NET_WORKAREA'):
        left, top, width, height, depth = root.get_geometry()
    else:
        index = root.property_get('_NET_CURRENT_DESKTOP')[2][0] * 4
        left, top, width, height = root.property_get('_NET_WORKAREA')[2][index:index+4]
    return left, top, width, height
