/*
 * rtmsg.c - route message parser
 * Copyright (C) 2014 Tetsumune KISO <t2mune@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef _RTNETLINK_
#define _RTNETLINK_

/*
 * parse route attributes
 */
static inline void parse_rtattr(struct rtattr *tb[], int max, struct rtattr *rta, int len)
{
	memset(tb, 0, sizeof(struct rtattr *) * (max + 1));
	while (RTA_OK(rta, len)) {
		if (rta->rta_type <= max)
			tb[rta->rta_type] = rta;
		rta = RTA_NEXT(rta, len);
	}
}

/*
 * parse route attributes nested
 */
static inline void parse_nested_rtattr(struct rtattr *tb[], int max, struct rtattr *rta)
{
	parse_rtattr(tb, max, RTA_DATA(rta), RTA_PAYLOAD(rta));
}

#ifndef IFLA_RTA
#define IFLA_RTA(r) \
	((struct rtattr*)(((char*)(r)) + NLMSG_ALIGN(sizeof(struct ifinfomsg))))
#endif

#ifndef IFLA_PAYLOAD
#define IFLA_PAYLOAD(n)	NLMSG_PAYLOAD(n,sizeof(struct ifinfomsg))
#endif

/*
 * parse interface infromation attributes
 */
static inline void parse_ifinfo(struct rtattr *tb[], struct nlmsghdr *nlh)
{
	struct ifinfomsg *ifim = NLMSG_DATA(nlh);
	parse_rtattr(tb, IFLA_MAX, IFLA_RTA(ifim), IFLA_PAYLOAD(nlh));
}

#if HAVE_DECL_IFLA_LINKINFO
/*
 * parse link information attributes
 */
static inline void parse_linkinfo(struct rtattr *tb[], struct rtattr *linkinfo)
{
	parse_nested_rtattr(tb, IFLA_INFO_MAX, linkinfo);
}
#endif

#if HAVE_DECL_IFLA_VLAN_UNSPEC
/*
 * parse vlan information attributes
 */
static inline void parse_vlan(struct rtattr *tb[], struct rtattr *vlaninfo)
{
	parse_nested_rtattr(tb, IFLA_VLAN_MAX, vlaninfo);
}
#endif

#if HAVE_DECL_IFLA_MACVLAN_UNSPEC
/*
 * parse macvlan information attributes
 */
static inline void parse_macvlan(struct rtattr *tb[], struct rtattr *macvlan)
{
	parse_nested_rtattr(tb, IFLA_MACVLAN_MAX, macvlan);
}
#endif

#if HAVE_DECL_IFLA_VXLAN_UNSPEC
/*
 * parse vxlan information attributes
 */
static inline void parse_vxlan(struct rtattr *tb[], struct rtattr *vxlan)
{
	parse_nested_rtattr(tb, IFLA_VXLAN_MAX, vxlan);
}
#endif

#if HAVE_DECL_IFLA_GRE_UNSPEC
/*
 * parse gre information attributes
 */
static inline void parse_gre(struct rtattr *tb[], struct rtattr *greinfo)
{
	parse_nested_rtattr(tb, IFLA_GRE_MAX, greinfo);
}
#endif

#ifndef NDA_RTA
#define NDA_RTA(r) \
	((struct rtattr*)(((char*)(r)) + NLMSG_ALIGN(sizeof(struct ndmsg))))
#endif

#ifndef NDA_PAYLOAD
#define NDA_PAYLOAD(n)	NLMSG_PAYLOAD(n,sizeof(struct ndmsg))
#endif

/*
 * parse neighbor discovery attributes
 */
static inline void parse_ndisc(struct rtattr *tb[], struct nlmsghdr *nlh)
{
	struct ndmsg *ndm = NLMSG_DATA(nlh);
	parse_rtattr(tb, NDA_MAX, NDA_RTA(ndm), NDA_PAYLOAD(nlh));
}

#ifndef IFA_RTA
#define IFA_RTA(r) \
	((struct rtattr*)(((char*)(r)) + NLMSG_ALIGN(sizeof(struct ifaddrmsg))))
#endif

#ifndef IFA_PAYLOAD
#define IFA_PAYLOAD(n)	NLMSG_PAYLOAD(n,sizeof(struct ifaddrmsg))
#endif

/*
 * parse interface address attributes
 */
static inline void parse_ifaddr(struct rtattr *tb[], struct nlmsghdr *nlh)
{
	struct ifaddrmsg *ifam = NLMSG_DATA(nlh);
	parse_rtattr(tb, IFA_MAX, IFA_RTA(ifam), IFA_PAYLOAD(nlh));
}

#ifdef HAVE_LINUX_FIB_RULES_H
#ifndef FRA_RTA
#define FRA_RTA(r) \
	((struct rtattr *)(((char*)(r)) + NLMSG_ALIGN(sizeof(struct fib_rule_hdr))))
#endif

#ifndef FRA_PAYLOAD
#define FRA_PAYLOAD(n) NLMSG_PAYLOAD(n,sizeof(struct fib_rule_hdr))
#endif

/*
 * parse fib rule attributes
 */
static inline void parse_frule(struct rtattr *tb[], struct nlmsghdr *nlh)
{
	struct fib_rule_hdr *frh = NLMSG_DATA(nlh);
	parse_rtattr(tb, FRA_MAX, FRA_RTA(frh), FRA_PAYLOAD(nlh));
}
#endif

#ifndef TCA_RTA
#define TCA_RTA(r) \
	((struct rtattr*)(((char*)(r)) + NLMSG_ALIGN(sizeof(struct tcmsg))))
#endif

#ifndef TCA_PAYLOAD
#define TCA_PAYLOAD(n) NLMSG_PAYLOAD(n,sizeof(struct tcmsg))
#endif

/*
 * parse traffic control attributes
 */
static inline void parse_tc(struct rtattr *tb[], struct nlmsghdr *nlh)
{
	struct tcmsg *tcm = NLMSG_DATA(nlh);
	parse_rtattr(tb, TCA_MAX, TCA_RTA(tcm), TCA_PAYLOAD(nlh));
}

#ifndef TA_RTA
#define TA_RTA(r)  \
	((struct rtattr*)(((char*)(r)) + NLMSG_ALIGN(sizeof(struct tcamsg))))
#endif

#ifndef TA_PAYLOAD
#define TA_PAYLOAD(n) NLMSG_PAYLOAD(n,sizeof(struct tcamsg))
#endif

/*
 * parse traffic control attributes
 */
static inline void parse_tca(struct rtattr *tb[], struct nlmsghdr *nlh)
{
	struct tcamsg *tcam = NLMSG_DATA(nlh);
	parse_rtattr(tb, TCAA_MAX, TA_RTA(tcam), TA_PAYLOAD(nlh));
}

/*
 * parse traffic control qdisc tbf attributes
 */
static inline void parse_tbf(struct rtattr *tbf[], struct rtattr *tb)
{
	parse_nested_rtattr(tbf, TCA_TBF_MAX, tb);
}


/*
 * parse traffic control qdisc red attributes
 */
static inline void parse_red(struct rtattr *red[], struct rtattr *tb)
{
	parse_nested_rtattr(red, TCA_RED_MAX, tb);
}

/*
 * parse traffic control qdisc gred attributes
 */
static inline void parse_gred(struct rtattr *gred[], struct rtattr *tb)
{
	parse_nested_rtattr(gred, TCA_GRED_MAX, tb);
}

#if HAVE_DECL_TCA_CHOKE_UNSPEC
/*
 * parse traffic control qdisc choke attributes
 */
static inline void parse_choke(struct rtattr *choke[], struct rtattr *tb)
{
	parse_nested_rtattr(choke, TCA_CHOKE_MAX, tb);
}
#endif

/*
 * parse traffic control qdisc htb attributes
 */
static inline void parse_htb(struct rtattr *htb[], struct rtattr *tb)
{
	parse_nested_rtattr(htb, TCA_HTB_MAX, tb);
}

/*
 * parse traffic control qdisc htb attributes
 */
static inline void parse_hfsc(struct rtattr *hfsc[], struct rtattr *tb)
{
	parse_nested_rtattr(hfsc, TCA_HFSC_MAX, tb);
}

/*
 * parse traffic control qdisc cbq attributes
 */
static inline void parse_cbq(struct rtattr *cbq[], struct rtattr *tb)
{
	parse_nested_rtattr(cbq, TCA_CBQ_MAX, tb);
}


/*
 * parse traffic control qdisc dsmark attributes
 */
static inline void parse_dsmark(struct rtattr *dsmark[], struct rtattr *tb)
{
	parse_nested_rtattr(dsmark, TCA_DSMARK_MAX, tb);
}

/*
 * parse traffic control qdisc netem attributes
 */
static inline void parse_netem(struct rtattr *netem[], struct rtattr *tb)
{
	parse_rtattr(netem, TCA_NETEM_MAX, RTA_DATA(tb) + sizeof(struct tc_netem_qopt),
		RTA_PAYLOAD(tb) - sizeof(struct tc_netem_qopt));
}

#if HAVE_DECL_TCA_NETEM_LOSS
/*
 * parse traffic control qdisc netem loss attributes
 */
static inline void parse_netem_loss(struct rtattr *loss[], struct rtattr *tb)
{
	parse_nested_rtattr(loss, NETEM_LOSS_MAX, tb);
}
#endif

#if HAVE_DECL_TCA_DRR_UNSPEC
/*
 * parse traffic control qdisc drr attributes
 */
static inline void parse_drr(struct rtattr *drr[], struct rtattr *tb)
{
	parse_nested_rtattr(drr, TCA_DRR_MAX, tb);
}
#endif

#if HAVE_DECL_TCA_SFB_UNSPEC
/*
 * parse traffic control qdisc sfb attributes
 */
static inline void parse_sfb(struct rtattr *sfb[], struct rtattr *tb)
{
	parse_nested_rtattr(sfb, TCA_SFB_MAX, tb);
}
#endif

#if HAVE_DECL_TCA_QFQ_UNSPEC
/*
 * parse traffic control qdisc qfq attributes
 */
static inline void parse_qfq(struct rtattr *qfq[], struct rtattr *tb)
{
	parse_nested_rtattr(qfq, TCA_QFQ_MAX, tb);
}
#endif

#if HAVE_DECL_TCA_CODEL_UNSPEC
/*
 * parse traffic control qdisc codel attributes
 */
static inline void parse_codel(struct rtattr *codel[], struct rtattr *tb)
{
	parse_nested_rtattr(codel, TCA_CODEL_MAX, tb);
}
#endif

#if HAVE_DECL_TCA_FQ_CODEL_UNSPEC
/*
 * parse traffic control qdisc fq_codel attributes
 */
static inline void parse_fq_codel(struct rtattr *fq_codel[], struct rtattr *tb)
{
	parse_nested_rtattr(fq_codel, TCA_FQ_CODEL_MAX, tb);
}
#endif

/*
 * parse traffic control stats2 attributes
 */
static inline void parse_stats2(struct rtattr *stats2[], struct rtattr *tb)
{
	parse_nested_rtattr(stats2, TCA_STATS_MAX, tb);
}

#if HAVE_DECL_TCA_STAB_UNSPEC
/*
 * parse traffic control stab attributes
 */
static inline void parse_stab(struct rtattr *stab[], struct rtattr *tb)
{
	parse_nested_rtattr(stab, TCA_STAB_MAX, tb);
}
#endif

/*
 * parse traffic control filter u32 attributes
 */
static inline void parse_u32(struct rtattr *u32[], struct rtattr *tb)
{
	parse_nested_rtattr(u32, TCA_U32_MAX, tb);
}

/*
 * parse traffic control filter rsvp attributes
 */
static inline void parse_rsvp(struct rtattr *rsvp[], struct rtattr *tb)
{
	parse_nested_rtattr(rsvp, TCA_RSVP_MAX, tb);
}

/*
 * parse traffic control filter route attributes
 */
static inline void parse_route(struct rtattr *route[], struct rtattr *tb)
{
	parse_nested_rtattr(route, TCA_ROUTE4_MAX, tb);
}

/*
 * parse traffic control filter fw attributes
 */
static inline void parse_fw(struct rtattr *fw[], struct rtattr *tb)
{
	parse_nested_rtattr(fw, TCA_FW_MAX, tb);
}

/*
 * parse traffic control filter tcindex attributes
 */
static inline void parse_tcindex(struct rtattr *tcindex[], struct rtattr *tb)
{
	parse_nested_rtattr(tcindex, TCA_TCINDEX_MAX, tb);
}

#if HAVE_DECL_TCA_FLOW_UNSPEC
/*
 * parse traffic control filter flow attributes
 */
static inline void parse_flow(struct rtattr *flow[], struct rtattr *tb)
{
	parse_nested_rtattr(flow, TCA_FLOW_MAX, tb);
}
#endif

/*
 * parse traffic control filter basic attributes
 */
static inline void parse_basic(struct rtattr *basic[], struct rtattr *tb)
{
	parse_nested_rtattr(basic, TCA_BASIC_MAX, tb);
}

#if HAVE_DECL_TCA_CGROUP_UNSPEC
/*
 * parse traffic control filter cgroup attributes
 */
static inline void parse_cgroup(struct rtattr *cgroup[], struct rtattr *tb)
{
	parse_nested_rtattr(cgroup, TCA_CGROUP_MAX, tb);
}
#endif

/*
 * parse traffic control filter ematch tree attributes
 */
static inline void parse_ematch_tree(struct rtattr *em_tree[], struct rtattr *tb)
{
	parse_nested_rtattr(em_tree, TCA_EMATCH_TREE_MAX, tb);
}

/*
 * parse traffic control filter ematch list attributes
 */
static inline void parse_ematch_list(struct rtattr *em_list[], struct rtattr *em_tree, int num)
{
	parse_nested_rtattr(em_list, num, em_tree);
}

#ifdef HAVE_LINUX_TC_EMATCH_TC_EM_META_H
/*
 * parse traffic control filter ematch meta attributes
 */
static inline void parse_meta(struct rtattr *meta[], struct rtattr *em_list, int len)
{
	parse_rtattr(meta, TCA_EM_META_MAX, em_list, len);
}
#endif

/*
 * parse traffic control police attributes
 */
static inline void parse_police(struct rtattr *police[], struct rtattr *tb)
{
	parse_nested_rtattr(police, TCA_POLICE_MAX, tb);
}

/*
 * parse traffic control actions attributes
 */
static inline void parse_actions(struct rtattr *acts[], struct rtattr *tb)
{
	parse_nested_rtattr(acts, TCA_ACT_MAX_PRIO, tb);
}

/*
 * parse traffic control action attributes
 */
static inline void parse_action(struct rtattr *act[], struct rtattr *tb)
{
	parse_nested_rtattr(act, __TCA_ACT_MAX-1, tb);
}

/*
 * parse traffic control action mirred attributes
 */
static inline void parse_mirred(struct rtattr *mirred[], struct rtattr *tb)
{
	parse_nested_rtattr(mirred, TCA_MIRRED_MAX, tb);
}

/*
 * parse traffic control action gact attributes
 */
static inline void parse_gact(struct rtattr *gact[], struct rtattr *tb)
{
	parse_nested_rtattr(gact, TCA_GACT_MAX, tb);
}

#ifdef HAVE_LINUX_TC_ACT_TC_NAT_H
/*
 * parse traffic control action nat attributes
 */
static inline void parse_nat(struct rtattr *nat[], struct rtattr *tb)
{
	parse_nested_rtattr(nat, TCA_NAT_MAX, tb);
}
#endif

#ifdef HAVE_LINUX_TC_ACT_TC_SKBEDIT_H
/*
 * parse traffic control action skbedit attributes
 */
static inline void parse_skbedit(struct rtattr *skb[], struct rtattr *tb)
{
	parse_nested_rtattr(skb, TCA_SKBEDIT_MAX, tb);
}
#endif

/*
 * parse traffic control action pedit attributes
 */
static inline void parse_pedit(struct rtattr *pedit[], struct rtattr *tb)
{
	parse_nested_rtattr(pedit, TCA_PEDIT_MAX, tb);
}

#ifdef HAVE_LINUX_TC_ACT_TC_CSUM_H
/*
 * parse traffic control action csum attributes
 */
static inline void parse_csum(struct rtattr *csum[], struct rtattr *tb)
{
	parse_nested_rtattr(csum, TCA_CSUM_MAX, tb);
}
#endif

#endif /* _RTNETLINK_ */
