/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

// Methods for class CommonXSectView

#include <Assertions.hpp>

#include "CommonXSectView.h"
#include "DataObject.h"
#include "Decoder.h"
#include "ObjectList.h"
#include "PlotMod.h"
#include "PlotModTask.h"
#include "PmContext.h"
#include "SubPage.h"
#include "MvRequestUtil.hpp"

CommonXSectView::CommonXSectView ( Page& owner,
                                   const MvRequest& viewRequest, 
                                   const MvRequest& setupRequest ) :
                    PlotModView    ( owner, viewRequest, setupRequest ),
                    PlotModService ( setupRequest("service"), this ),
                    appViewReq_    ( viewRequest )
{
    yMin_ = viewRequest_("BOTTOM_LEVEL");
    yMax_ = viewRequest_("TOP_LEVEL");
}

CommonXSectView::CommonXSectView(const CommonXSectView &old) :
                 PlotModView(old), 
                 PlotModService  ( old ),
                 latMin_(old.latMin_),latMax_(old.latMax_),
                 lonMin_(old.lonMin_),lonMax_(old.lonMax_),
                 yMin_(old.yMin_), yMax_(old.yMax_),
                 applicationName_(old.applicationName_),
                 appViewReq_(old.appViewReq_)
{
    //empty
}

CommonXSectView::~CommonXSectView()
{
  // Empty
}

// Synchronous call service. Calls a service and wait its return.
bool CommonXSectView::CallService(const MvRequest &dataReq, PmContext & context, MvRequest& replyReq)
{
    if ( dataReq.getVerb() != GRIB && dataReq.getVerb() != NETCDF )
        return false;

    MvRequest appRequest(ApplicationName().c_str());
    appRequest("DATA")     = dataReq;
    appRequest("_CONTEXT") = viewRequest_;
    appRequest("_ACTION")  = (const char*)dataReq("_ACTION") ? (const char*)dataReq("_ACTION") : "prepare";
    appRequest("_VERB")    = (const char*)dataReq("_VERB") ? (const char*)dataReq("_VERB") : "";

    //( new PlotModTask (this, context, serviceName_.c_str(), appRequest ) )->run();
    PlotModTask* pmt = new PlotModTask (this, context, serviceName_.c_str(), appRequest );
    int err;
    replyReq = pmt->run(err);
    if( !replyReq || err != 0)
    {
       string message = serviceName_ + " application failed";
       PlotMod::Instance().MetviewError ( message.c_str(), "WARNING" );
       return false;
    }

    // Add a flag to indicate that there is a data attached to this View
    viewRequest_("_DATAATTACHED") = "YES";

    // This will be done by the destructor of the MvClient/MvTask
    //delete pmt; pmt = 0;  
    return true;
}

void CommonXSectView::Drop ( PmContext& context )
{
   // Process the drop
   MvIconDataBase& dataBase = Owner().IconDataBase();
   MvRequest dropRequest = context.InRequest();
   MvIcon dataUnit;
   while ( dropRequest )
   {
      // VisDefs are processed in one single goal. This is because
      // the processing of a single or a group of visdefs is different.
      // It is assumed that visdefs always come after a dataunit
      // (if it exists)
      MvRequest vdRequestList;
      if ( this->RetrieveVisdefList (dropRequest,vdRequestList) )
      {
         Owner().InsertVisDef (vdRequestList, dataUnit);

         if ( !dropRequest ) // no more drops
            break;
      }

      // Process a dataUnit
      MvRequest request = dropRequest.justOneRequest();
      string verb       =  request.getVerb();
      if ( ObjectList::IsDataUnit ( verb.c_str() ) == true )
      {
         // Call the service to process the request

         // After CallService is finished, function PlotModService::endOfTask
         // should be called and this function should perform the drawing.
         //Owner().HasDrawTask(false);
         //if ( ! CallService(request, context) )
         //   InsertDataRequest(request);

         // It calls an external service and waits for its return
         // If returns false, it means that this is just a dataUnit
         // that does not need to be processed by a external service
         MvRequest replyRequest;
         if ( !CallService(request, context, replyRequest) )
         {
            // Insert the dataunit in the data base at the Page level
            //  request("_PARENT_DATAUNIT") = 1;
            dataUnit = dataBase.InsertDataUnit ( request,Owner().Id() );
            // IMPORTANT: CHECK IF THIS COMMAND IS NEEDED IN THE InsertDataRequest
            // Save to be used when returning from service
            //  this->ParentDataUnitId ( dataUnit.Id() );

            // Add information to the SubPage
            DecodeDataUnit ( dataUnit );
         }
         else
         {
            // IMPORTANT: DO WE NEED TO CREATE THIS 'PARENT' DATA UNIT????
            //request("_PARENT_DATAUNIT") = 1;
            this->ParentDataUnitId ( 5 );
            this->InsertDataRequest ( replyRequest );
         }

         dropRequest.advance();
         continue;
      }

      // Process other ico
      if ( ObjectList::IsView ( verb.c_str() ) == true )
         this->UpdateView ( request );

      else if ( verb == PLOTSUPERPAGE )
      {
         context.AdvanceTo(PLOTSUPERPAGE);
         return;
      }

      else if ( verb == NEWPAGE )
      {
         context.AdvanceTo(NEWPAGE);
         return;
      }

      else if ( verb == "DRAWING_PRIORITY" )
      {
         Owner().SetDrawPriority ( request );

         // Redraw this page
         if ( strcmp(request ("_APPL"),"macro") )
            Owner().RedrawIfWindow();
      }
      else
         Owner().InsertCommonIcons(request);

      // This request is not a dataUnit. Clear dataUnit variable
      // to avoid a possible visdef in the next iteraction to be
      // associate to an old dataUnit.
      if ( dataUnit.Id() > 0 )
         dataUnit = MvIcon();

      dropRequest.advance();
   }

   context.AdvanceToEnd();
}

void
CommonXSectView::InsertDataRequest ( MvRequest& dropRequest )
{
   MvIconDataBase& dataBase = Owner().IconDataBase();

   // Save some specific values sent by the Data Applicaation
   ApplicationInfo(dropRequest);

   // Process the request
   MvIcon dataUnit;
   while ( dropRequest )
   {
      // VisDefs are processed in one single goal. This is because the processing
      // of a single or a group of visdefs is different. It is assumed that
      // visdefs always come after a dataunit (if it exists).
      MvRequest vdRequestList;
      if ( this->RetrieveVisdefList (dropRequest,vdRequestList) )
      {
         Owner().InsertVisDef (vdRequestList, dataUnit);

         if ( !dropRequest ) // no more drops
            break;
      }

      MvRequest req = dropRequest.justOneRequest();
      string verb = req.getVerb();
      if ( verb.find(NETCDF) != string::npos || verb.find("INPUT") != string::npos)
      {
         // Insert the dataunit in the data base at the Page level
         //  request("_PARENT_DATAUNIT") = 1;
         dataUnit = dataBase.InsertDataUnit ( req, Owner().Id() );
         //dataUnit = dataBase.InsertDataUnit ( req );

         //dataUnit.ParentId ( this->ParentDataUnitId () ); //Check if is needed

         // Add information to the SubPage
         DecodeDataUnit ( dataUnit );

         if ( this->ParentDataUnitId () )
         {
            dataUnit.ParentId ( this->ParentDataUnitId () ); //Check if is needed
            MvIcon parentDataUnit;
            dataBase.RetrieveIconFromList(DB_DATAUNIT,ParentDataUnitId (), parentDataUnit);
            parentDataUnit.SvcRequest ( req );
         }
      }
      else
         Owner().InsertCommonIcons(req);

      dropRequest.advance();
   }
}

// -- METHOD: DecodeDataUnit
//
// -- PURPOSE: 
//    . Extract the data and the metadata from a NetCDF request
//    . Provides the proper axis and graph definition according 
//      to the metada. 
//
void CommonXSectView::DecodeDataUnit ( MvIcon& dataUnit )
{
    Owner().InitMatching();

    // Build a new data decoder, which will provide information
    // about the data
    auto_ptr<Decoder> decoder ( DecoderFactory::Make ( dataUnit.Request() ) );
    ensure (decoder.get() != 0);

    int subpageId = 0;
    while ( decoder->ReadNextData() )
    {
        // Pass the data and metadata info to the page
        MatchingInfo dataInfo = decoder->CreateMatchingInfo();

        Owner().InsertDataUnit ( dataUnit.Id(), 0, 0, dataInfo, subpageId );
    }
}

#if 0
void 
CommonXSectView::DescribeAxis ( ObjectInfo& description, 
				MvRequest& axisRequest,
				const Cached& axisName )
{

  Cached macroName = ObjectInfo::SpaceToUnderscore ( axisName );

  description.ConvertRequestToMacro ( axisRequest, PUT_END,
				      macroName,"paxis");

}

void CommonXSectView::CheckAxis(double minValue, double maxValue,
				int axisType, MvRequest &axis,string title)
{
  int axisSpan = 0;

  axis.unsetParam("AXIS_TICK_POSITIONING");

  axisSpan = abs( (int)(maxValue - minValue) );

  if ( axisSpan > 0  )
    {
      axis("AXIS_TICK") = "ON";
      axis("AXIS_TICK_LABEL") = "ON";
      axis( "AXIS_MIN_VALUE" ) = minValue;
      axis( "AXIS_MAX_VALUE" ) = maxValue;
    }
  else
    {
      axis("AXIS_TICK_LABEL") = "OFF";
      axis("AXIS_TICK") = "OFF";
    }
  

  // Set any type dependent values
  if ( axisType == AXIS_LAT || axisType == AXIS_LON )
    {
      if ( ! ( (const char *)axis("AXIS_LINE") == Cached("OFF") ) )
	axis("AXIS_LINE") = "ON";

      if ( axisType == AXIS_LAT )
	{
	  if( minValue > 90.0 || minValue < -90.0 ||
	      maxValue > 90.0 || maxValue < -90.0 )
	    {
	      //-- xsect line via a pole, use plain numeric labels --
	      axis.unsetParam( "AXIS_TICK_LABEL_TYPE" );
	    }
	  else
	    {
	      //-- not via pole, use latitude labels
	      axis("AXIS_TICK_LABEL_TYPE") = "LATITUDE";
	    }
	  if ( ! ( (const char *)axis("AXIS_TIP_TITLE_TEXT") ) )
	    axis("AXIS_TIP_TITLE_TEXT")  = "LAT";
	}  
      else
	{
	  axis("AXIS_TICK_LABEL_TYPE") = "LONGITUDE";
	  if ( ! ( (const char *)axis("AXIS_TIP_TITLE_TEXT") ) )
	    axis("AXIS_TIP_TITLE_TEXT")  = "LON";
	}
      if ( axisSpan )
	{
	  axis("AXIS_TICK") = "ON";
	  axis("AXIS_TICK_LABEL") = "ON";

	}
      else
	{
	  // Generate a title for the point.
	  char buf[10];
	  sprintf(buf,"%01.1f",minValue);
	  if ( axisType == AXIS_LAT )
	    strcat(buf, (minValue > 0) ? "N" : "S");
	  else
	    strcat(buf, (minValue > 0) ? "E" : "W");
	  
	  axis("AXIS_TITLE_TEXT") = buf;
	}
    }
  else if ( axisType == AXIS_LOG )
    axis("AXIS_TICK_POSITIONING") = "LOGARITHMIC";
  
  // Set title if given and not set in current request.
  if ( title.length() > 0 )
    if ( ! ( (const char *)axis("AXIS_TITLE_TEXT") ) )
      axis("AXIS_TITLE_TEXT") = title.c_str(); 

}

void CommonXSectView::AddValues ( MvRequest& dataRequest, vector<double>& dataVector, 
				  const char* variableName )
{
  vector<double>::iterator itd = dataVector.begin();

  while ( itd != dataVector.end() )
    {
      double value = *itd;
      dataRequest.addValue ( variableName, value );
      ++itd;
    }


}
bool CommonXSectView::NetCDFtoMatrix(MvIcon & du,MvRequest& yAxis)
{
  int i;
  MvRequest Matrix = du.Request();

  MvRequest dcRequest =  Matrix.getSubrequest("_DC_REQUEST" );
  auto_ptr<Decoder> decoder ( DecoderFactory::Make ( dcRequest ) );

  int offset = Matrix("_OFFSET");
  const char *fullName = Matrix("_FILE_NAME");

  string currvar;
  MvRequest infoRequest;
  while ( decoder->ReadNextData() )
    {
      if ( decoder->CurrentOffset() == offset )
	break;
    }
  
  if ( offset != decoder->CurrentOffset() )
    return false;

  infoRequest = decoder->Request(); 

  FILE *fp = fopen(fullName,"w");
  if ( !fp ) 
    return false;
  
  string mvVerb = (const char *)infoRequest("MV_VERB");
  bool modlev = ( (const char *)infoRequest("LEVTYPE") && !strcmp(infoRequest("LEVTYPE"),"ML") );

  // We know that this is a netcdf decoder. Cast it and get the
  // MvNetCDF member enabling access to the variables.
  NetCDFDecoder *cdfDecoder = (NetCDFDecoder*)decoder.get();
  MvNetCDF &cdf = cdfDecoder->getCDF();

  //Get the values.
  MvNcVar *var_x = cdf.getVariable(offset );

  // Empty out the matrix to get rid of old values.
  Matrix = empty_request(NULL);
  Matrix.setVerb("MATRIX");
  
  // Check if we need to massage data. Only done if :
  // - NetCDF contains original data, with corresponding y levels AND
  // - Axis min or max is set, and are different from data min/max.
  //            OR
  // NetCDF contains original data and axis type is set to logarithmic
  bool changeData = false, maxSet = false, minSet = false;
  double yMin, yMax;
  bool logAxis = (  (const char*)yAxis("AXIS_TYPE") && 
		    strcmp(yAxis("AXIS_TYPE"),"LOGARITHMIC") == 0);
  bool dataLog = (  (const char*)infoRequest("AXIS_TICK_POSITIONING") && 
		    strcmp(infoRequest("AXIS_TICK_POSITIONING"),"LOGARITHMIC") == 0);

  if ( ( const char *)yAxis("AXIS_MIN_VALUE")) 
    { 
      yMin =yAxis("AXIS_MIN_VALUE");
      minSet = true;
    }
  if (  ( const char *)yAxis("AXIS_MAX_VALUE") ) 
    { 
      yMax =yAxis("AXIS_MAX_VALUE");
      maxSet = true;
    }

  long *edges = var_x->edges();
  long nrPoints = edges[var_x->getNumberOfDimensions() -1 ]; // Last dimensions is nr of points
  // Get the  y values.
  vector<double> yVector;
  double dataMax = yMax, dataMin = yMin;
  if ( var_x->getAttributeValues("_Y_VALUES",yVector) )
    {
      dataMax = yVector[0] ;
      dataMin = yVector[yVector.size() - 1 ];
    }
  // For contours, interpolate if axis max or min is set and different from data, and NetCDF file contains raw data
  // For wind, always interpolate if NetCDF file contains raw data.
  if ( (mvVerb != "PWIND" && ( ( minSet && dataMin != yMin) || (maxSet && dataMax != yMax ) || 
			       ( logAxis && logAxis != dataLog ) )
	&& yVector.size() == edges[0] ) || ( mvVerb == "PWIND" && yVector.size() == edges[1] ) )
    {
      changeData = true;
      if ( ! minSet ) { yMin = dataMin; yAxis("AXIS_MIN_VALUE") = dataMin; }
      if ( !maxSet ) { yMax = dataMax;yAxis("AXIS_MAX_VALUE") = dataMax; }
      if ( logAxis ) yAxis("AXIS_TYPE") = "LOGARITHMIC";
    }
  else  // Don't change values if derived data. This means the plot will be wrong. Only warn if data is still on PL.
    
    {
      if (  ( ( minSet  && yMin != dataMin ) || ( maxSet && yMax != dataMax ) || ( logAxis && dataLog != logAxis ) )
	    && ! modlev )
	PlotMod::MetviewError("Axis values ignored!\n\tCan't recreate data, so given axis will produce incorrect plot\n");
      
      yAxis("AXIS_MIN_VALUE") = dataMin;
      yAxis("AXIS_MAX_VALUE") = dataMax;
      if ( dataLog ) 
	yAxis("AXIS_TYPE") = "LOGARITHMIC";
      else
	yAxis.unsetParam("AXIS_TYPE");
    }
  // Write data.
  vector<double> xvector;
  var_x->get(xvector,edges);
  long offset2;

  int nrLevels = 85;

  if (  changeData )
    {
      if ( modlev ) nrLevels = int(yMin - yMax);
      if ( mvVerb == "PWIND" )
	{
	  GenerateData(fp,0,yMin,yMax,xvector,yVector,nrPoints,logAxis,modlev);
	  offset2 = ftell(fp);
	  GenerateData(fp,xvector.size()/2,yMin,yMax,xvector,yVector, nrPoints,logAxis,modlev);
	}
      else
	GenerateData(fp,0,yMin,yMax,xvector,yVector, nrPoints,logAxis,modlev);
    }
  else
    {
      if ( mvVerb == "PWIND" )
	{
	  nrLevels = edges[1];
	  for (i = 0; i < xvector.size()/2;i++)
	    fprintf(fp,"%e ",xvector[i]);
	  
	  offset2 = ftell(fp);
	  for (i = xvector.size()/2; i < xvector.size();i++)
	    fprintf(fp,"%e ",xvector[i]);
	}
      else
	{
	  nrLevels = edges[0];
	  for (i = 0; i < xvector.size();i++)
	    fprintf(fp,"%e ",xvector[i]);
	}
    }


  fclose(fp);

  // Now fill in the values that's used by PWIND or PCONT
  if ( mvVerb == "PCONT" )
    {
      Matrix("INPUT_FIELD") = "FILE";
      Matrix("_INPUT_FIELD_FILE") = fullName;
      Matrix("_INPUT_FIELD_FORMAT") = "ASCII";
      Matrix("_INPUT_FIELD_SIZE_1") = edges[1];
      Matrix("_INPUT_FIELD_SIZE_2") = nrLevels;

      // Set defaults
      if ( ! (const char *)Matrix("INPUT_FIELD_ORGANIZATION")  )
	Matrix("INPUT_FIELD_ORGANIZATION") = "FITTED";

      // These lines should maybe not be in, but are needed for the
      // plot to look OK.
      //if ( ! (const char *) Matrix("CONTOUR_HILO") )
      //Matrix("CONTOUR_HILO") = "OFF";
      if ( ! (const char *) Matrix("CONTOUR_LEVEL_COUNT") )
      	Matrix("CONTOUR_LEVEL_COUNT") = 20;
    }
  else if ( mvVerb == "PWIND") 
    {
      if ( ! (const char *) Matrix("INPUT_FIELD_ORGANIZATION") )
	Matrix("INPUT_FIELD_ORGANIZATION") = "FITTED";

      Matrix("INPUT_WIND_U_COMPONENT") = "FILE";
      Matrix("_INPUT_WIND_U_COMPONENT_FILE") = fullName;
      Matrix("_INPUT_WIND_U_COMPONENT_FORMAT") = "ASCII";

      Matrix("INPUT_WIND_V_COMPONENT") = "FILE";
      Matrix("_INPUT_WIND_V_COMPONENT_FILE") = fullName;
      Matrix("_INPUT_WIND_V_COMPONENT_FORMAT") = "ASCII";

      Matrix("_INPUT_WIND_U_COMPONENT_SIZE_1") = edges[2];
      Matrix("_INPUT_WIND_U_COMPONENT_SIZE_2") = nrLevels;

      Matrix("_INPUT_WIND_V_COMPONENT_SIZE_1") = edges[2];
      Matrix("_INPUT_WIND_V_COMPONENT_SIZE_2") = nrLevels;

      Matrix("_INPUT_WIND_U_COMPONENT_OFFSET") = 0;
      Matrix("_INPUT_WIND_V_COMPONENT_OFFSET") = offset2;
    }

  // Put the decoder related fields back
  Matrix("_OFFSET") = offset;
  Matrix("_DC_REQUEST" ) = dcRequest;
  Matrix("_FILE_NAME") = fullName;

  du.SaveRequest(Matrix);
  return true;
}
#endif

void CommonXSectView::Draw ( SubPage* subPage )
{
    MvIconDataBase& dataBase = Owner().IconDataBase();

    // Get the data units for this subpage
    list<DataObject*> doList;
    subPage->GetDataObjects(doList);
    list<DataObject*>::iterator ii;
    DataObject *dataObject = 0;
    DrawPriorMap drawPriorMap;
    DrawPriorMap::iterator j;

    // Loop over data objects
    MvIcon dataUnit;
    for ( ii = doList.begin(); ii != doList.end(); ii++ )
    {
        dataObject = (*ii);
        //if ( ! dataBase.RetrieveDataUnit (dataObject->DataUnitId(), dataUnit) )
        if ( ! dataBase.RetrieveIconFromList (DB_DATAUNIT, dataObject->DataUnitId(), dataUnit) )
            continue;

        // Retrieve visdef associated to this dataunit
        MvIconList vdList;
        dataObject->RetrieveMyVisDefList(dataUnit,vdList);

        DrawPriorMap tmpMap = dataObject->DrawPriority(vdList);
        for ( j = tmpMap.begin(); j != tmpMap.end(); ++j )
            drawPriorMap.insert ( *j );
    }

    // Do the drawing
    CommonDraw(subPage, drawPriorMap);
}

#if 0
void CommonXSectView::GenerateData(FILE *fp,int startIndex,double yMin,
				   double yMax,vector<double> &xvector,
				   vector<double> &yvector, int nrPoints, bool logax, bool modlev)
{

  int nrLevels = 85; // Hardcoded, same as in xsection application.

  int scale = 100;

  if ( modlev )
    {
      nrLevels = int(yMin - yMax);
      scale = 1;
    }
  int i;

  double PresBot = yMin * scale;
  double PresTop = yMax * scale;

  double lpa;
  double zdp = ( PresBot - PresTop)/ ( nrLevels - 1 );
  double deltalog, pk;
  int firstindex, nextindex;
 
  if ( logax ) 
    deltalog = (log10(PresBot) - log10(PresTop))/(nrLevels - 1);

  for ( int k = 0; k < nrLevels; k++ )
    {
      if ( logax )
	pk = exp((log10(PresTop) + (double)k*deltalog)*log(10.)); 
      else
	pk = modlev == 0 ? PresTop + k * zdp : PresTop + k;

      if ( FindPressure(yvector,pk,firstindex,nextindex,scale) )
	{ 
	  if ( modlev )
	    lpa = ( pk - (yvector[firstindex] * scale)) /
	      ((yvector[nextindex]*scale) - (yvector[firstindex]*scale));
	  else
	    if ( logax )
	      lpa = ( pk - (yvector[firstindex]*scale)) /
		((yvector[nextindex]*scale) - (yvector[firstindex]*scale));
	    else
	      lpa = log( pk/(yvector[firstindex]*scale)) /
		log((yvector[nextindex]*scale)/ (yvector[firstindex]*scale));
	  
	  double xfirst, xnext;
	  for (i = 0; i < nrPoints; i++ )
	    {
	      xfirst = xvector[startIndex + (firstindex * nrPoints) + i ];
	      xnext =  xvector[startIndex + (nextindex * nrPoints) + i ];
	      
	      if ( xfirst < BIGDOUBLE && xnext < BIGDOUBLE )
		fprintf(fp,"%e ",( xfirst + ( xnext - xfirst) *lpa ));
	      else
		fprintf(fp,"%e ", DONTCARE);
	    }
	  
	}
      else
	for (i=0;i<nrPoints;i++)
	  fprintf(fp,"%e ", DONTCARE);
    }
}
bool CommonXSectView::CheckLatLon(const string& name,double minVal, double maxVal,
				  MvRequest& axisRequest, const char *param)

{
  bool retVal = false;
  const char *tickType = axisRequest("AXIS_TICK_LABEL_TYPE");
  if ( axisRequest( "AXIS_ORIENTATION" ) != Cached ("VERTICAL" ) &&
      ( tickType && name == tickType ) )
    {
      if ( axisRequest.countValues("AXIS_MIN_VALUE") && axisRequest.countValues("AXIS_MAX_VALUE") )
	{
	  double minY = axisRequest("AXIS_MIN_VALUE");
	  double maxY = axisRequest("AXIS_MAX_VALUE");
	  if ( minY != minVal || maxY != maxVal )
	    {
	      minVal = minY;
	      maxVal = maxY;
	      double lat1 = viewRequest_(param,0);
	      double lat2 = viewRequest_(param,2);
	      double lon1 = viewRequest_(param,1);
	      double lon2 = viewRequest_(param,3);
	      viewRequest_.unsetParam(param);

	      if ( name == "LONGITUDE" )
		{
		  lon1 = minVal; lon2 = maxVal;
		}
	      else
		{
		  lat1 = minVal; lat2 = maxVal;
		}

	      viewRequest_.addValue(param,lat1);
	      viewRequest_.addValue(param,lon1);
	      viewRequest_.addValue(param,lat2);
	      viewRequest_.addValue(param,lon2);
	    }
	  retVal = true;
	}
    }
  return retVal;
}

bool CommonXSectView::FindPressure(vector<double> &yvector,double pk,int &firstindex,int &nextindex,int scale)
{
  int i  = 0;

  pk /= scale;

  while ( i < yvector.size() - 1 )
    {
      if ( (yvector[i] <= pk && pk <= yvector[i+1]) ||  ( yvector[i+1] <= pk && pk <= yvector[i] ) )
	{
	  firstindex = i; nextindex = i+1;
	  return true;
	}
      i++;
    }
  return false;
}
void
CommonXSectView::ReplaceAxis ( MvRequest& axisRequest )
{
  MvRequest oldAxis;
  bool changeData = false;

  if ( axisRequest( "AXIS_ORIENTATION" ) == Cached ("VERTICAL" ) )
    {
      // Check min/max values
      if ( axisRequest.countValues("AXIS_MIN_VALUE") && axisRequest.countValues("AXIS_MAX_VALUE") )
	{
	  double minY = axisRequest("AXIS_MIN_VALUE");
	  double maxY = axisRequest("AXIS_MAX_VALUE");
	  
	  if ( minY != (double)viewRequest_("BOTTOM_PRESSURE") || 
	       maxY != (double)viewRequest_("TOP_PRESSURE") )
	    {
	      viewRequest_("BOTTOM_PRESSURE") = minY;
	      viewRequest_("TOP_PRESSURE") = maxY;
	      changeData = true;
	    }
	}
      // Check if axis type is the same ( log or not).
      bool axisLog = (  (const char*)axisRequest("AXIS_TYPE") && 
			strcmp(axisRequest("AXIS_TYPE"),"LOGARITHMIC") == 0);
      bool viewLog = (  (const char*)viewRequest_("PRESSURE_LEVEL_AXIS") && 
			strcmp(viewRequest_("PRESSURE_LEVEL_AXIS"),"LOG") == 0);
      if ( axisLog != viewLog )
	{
	  if ( axisLog )
	    viewRequest_("PRESSURE_LEVEL_AXIS") = "LOG";
	  else
	   viewRequest_.unsetParam("PRESSURE_LEVEL_AXIS"); 
	  
	  changeData = true;
	}
    } // end vertical axis
  else if ( CheckHorizontal(axisRequest) )
    changeData = true;

  // Recreate data if needed.
  if ( changeData )
    RecreateData();
  else
    {
      Owner().NeedsRedrawing(true);
      Owner().EraseBackDraw();
    }

  Owner().RedrawIfWindow();

}

bool CommonXSectView::Reset(const MvRequest &req)
{
  if ( strcmp(req.getVerb(),"PAXIS") == 0 )
    {
      if ( req("AXIS_ORIENTATION") == Cached ("VERTICAL" ) && 
	   req.countValues("AXIS_MIN_VALUE") && req.countValues("AXIS_MAX_VALUE") )
	{
	  viewRequest_("BOTTOM_PRESSURE") = yMin_;
	  viewRequest_("TOP_PRESSURE") = yMax_;
	  return true;
	}
    }
  
  return false;
}
#endif
