// Author(s): Muck van Weerdenburg, Bert Lisser
// Copyright: see the accompanying file COPYING or copy at
// https://svn.win.tue.nl/trac/MCRL2/browser/trunk/COPYING
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//
/// \file ltsmin.cpp

#define NAME "ltsmin"
#define AUTHOR "Muck van Weerdenburg"

#include <string>
#include "ltsmin.h"
#include "mcrl2/utilities/logger.h"
#include "mcrl2/exception.h"
#include "mcrl2/utilities/tool.h"
#include "mcrl2/atermpp/aterm_init.h"

using namespace mcrl2::utilities::tools;
using namespace mcrl2::utilities;
using namespace mcrl2::log;

int traceLevel = 0, optimal = 0, classes = 0, add_state_parameter = 0;

struct t_tool_options
{
  std::string inputname;
  std::string outputname;
  int         task;
};

void doReduce(t_tool_options const& options);
void doBranchReduce(t_tool_options const& options);

typedef tool ltsmin_base;
class ltsmin_tool : public ltsmin_base
{
  private:
    t_tool_options options;

  public:
    ltsmin_tool() :
      ltsmin_base(NAME,AUTHOR,
                  "minimise an LTS",
                  "Minimise the LTS in the SVC format from INFILE using bisimulation reduction and save the resulting LTS to OUTFILE.\n"
                  "\n"
                  "Note that the input SVC file must have been generated by lps2lts."
                 )
    {
    }

    bool run()
    {
      SVCbool indexed = SVCfalse;

      if (SVCopen(inFile, const_cast < char* >(options.inputname.c_str()), SVCread, readIndex))
      {
        throw mcrl2::runtime_error(options.inputname + ": " + std::string(SVCerror(SVCerrno)));
      }
      else
      {
        if (SVCopen(outFile, const_cast < char* >(options.outputname.c_str()), SVCwrite, &indexed))
        {
          throw mcrl2::runtime_error(options.outputname + ": " + std::string(SVCerror(SVCerrno)));
        }
      }

      if (options.task == CMD_BRANCH_REDUCE)
      {
        doBranchReduce(options);
      }
      else
      {
        doReduce(options);
      }

      return true;
    }

  private:
    std::string synopsis() const
    {
      return "[OPTION]... INFILE OUTFILE\n";
    }

    void check_positional_options(const command_line_parser& parser)
    {
      if (2 < parser.arguments.size())
      {
        parser.error("too many file arguments");
      }
      else if (2 > parser.arguments.size())
      {
        parser.error("too few file arguments");
      }
    }

    void add_options(interface_description& desc)
    {
      ltsmin_base::add_options(desc);

      desc.
      add_option("equivalence", make_mandatory_argument("NAME"),
                 "use equivalence NAME:\n"
                 "  'strong' for strong bisimulation (default), or\n"
                 "  'branching' for branching bisimulation\n"
                 , 'e').
      add_option("add",
                 "do not minimise but save a copy of the original LTS extended with a "
                 "state parameter indicating the bisimulation class a state belongs to ", 'a').
      add_option("tau", make_mandatory_argument("ACTNAMES"),
                 "consider actions with name in comma-separated list ACTNAMES to be internal (tau) actions");
    }

    void parse_options(const command_line_parser& parser)
    {
      ltsmin_base::parse_options(parser);

      options.task = CMD_REDUCE;

      if (0 < parser.options.count("verbose") + parser.options.count("debug"))
      {
        traceLevel=1;
      }

      if (parser.options.count("equivalence"))
      {
        std::string eq_name(parser.option_argument("equivalence"));
        if (eq_name == "strong")
        {
          options.task = CMD_REDUCE;
        }
        else if (eq_name == "branching")
        {
          options.task = CMD_BRANCH_REDUCE;
        }
        else
        {
          parser.error("option -e/--equivalence has illegal argument '" + eq_name + "'");
        }
      }

      if (0 < parser.options.count("add"))
      {
        add_state_parameter = 1;
      }
      if (0 < parser.options.count("tau"))
      {
        std::string s = parser.option_argument("tau")+',';
        std::string::size_type pos, lastpos = 0;
        while ((pos = s.find(',',lastpos)) != std::string::npos)
        {
          add_tau_action(s.substr(lastpos,pos-lastpos));
          lastpos = pos+1;
        }
      }

      options.inputname = parser.arguments[0];
      options.outputname = parser.arguments[1];
      if (options.inputname == options.outputname)
      {
        parser.error("input file and output file are not allowed to be the same");
      }
    }

};

int main(int argc, char* argv[])
{
  MCRL2_ATERMPP_INIT(argc, argv)

  mCRL2log(warning) << "the use of this tool is deprecated; use ltsconvert instead" << std::endl;

  return ltsmin_tool().execute(argc,argv);
}

void doReduce(t_tool_options const& options)
{
  SVCstateIndex initState = ReadData();
  Reduce();
  if (add_state_parameter)
  {
    SVCbool b;
    SVCfile inf;
    SVCopen(&inf, const_cast < char* >(options.inputname.c_str()), SVCread, &b);
    WriteDataAddParam(&inf,initState,WITH_TAULOOPS);
  }
  else
  {
    WriteData(initState,WITH_TAULOOPS);
  }
}

void doBranchReduce(t_tool_options const& options)
{
  SVCstateIndex initState = ReadData();
  SCC();
//  initState = ReturnEquivalenceClasses(initState, DELETE_TAULOOPS);
  /* ATwarning("Number of states after deletion of tau loops: %d\n", nstate); */
  ReduceBranching();
  if (add_state_parameter)
  {
    SVCbool b;
    SVCfile inf;
    SVCopen(&inf, const_cast < char* >(options.inputname.c_str()), SVCread, &b);
    WriteDataAddParam(&inf,initState,DELETE_TAULOOPS);
  }
  else
  {
    WriteData(initState, DELETE_TAULOOPS);
  }
}
