/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iscriptkit.h"


#include "ierror.h"


namespace iScriptKit
{
	//
	//  Class for parameters.
	//
	template<class T>
	Constant<T>::Constant(iScript *script, const iString &name, T v) : Value(script,name,1,Constant<T>::UsagePattern())
	{
		this->mValue->Data()[0] = this->ConvertFromNative(v);
	}


	template<class T>
	Constant<T>::Constant(iScript *script, const iString &name, int dim, T *v) : Value(script,name,dim,Constant<T>::UsagePattern())
	{
		int i, n = this->mValue->Dim();
		Calculator::number_t *data = this->mValue->Data();
		if(v != 0)
		{
			for(i=0; i<n; i++) data[i] = this->ConvertFromNative(v[i]);
		}
		else
		{
			for(i=0; i<n; i++) data[i] = this->ConvertFromNative(0);
		}
	}

	
	template<class T>
	Constant<T>::Constant(iScript *script, const Calculator::result_t *val) : Value(script,val->Name(),val->Dim(),UsagePattern())
	{
		this->mValue->Copy(*val);
	}


	template<class T>
	inline T Constant<T>::ConvertToNative(Calculator::number_t v)
	{
		return T(v);
	}


	template<>
	inline bool Constant<bool>::ConvertToNative(Calculator::number_t v)
	{
		return (iCalculatorKit::ConvertToBoolRepresentation(v) > 0.5);
	}


	template<class T>
	inline Calculator::number_t Constant<T>::ConvertFromNative(T v)
	{
		return Calculator::number_t(v);
	}


	template<>
	inline Calculator::number_t Constant<bool>::ConvertFromNative(bool v)
	{
		return (v ? 1.0 : 0.0);
	}


	template<class T>
	iString Constant<T>::GetValueAsText() const
	{
		int n = this->mValue->Dim();
		Calculator::number_t *data = this->mValue->Data();

		if(n == 1)
		{
			return iString::FromNumber(ConvertToNative(data[0]));
		}
		else
		{
			int i;
			iString s("(");
			for(i=0; i<n; i++)
			{
				if(i > 0) s += ",";
				s += iString::FromNumber(ConvertToNative(data[i]));
			}
			s += ")";
			return s;
		}
	}


	template<class T>
	iString Constant<T>::GetTypeAsText() const
	{
		static const iString s(" scalar");
		static const iString a(" array");
		if(this->Dim() == 1) return this->GetTypeName()+s; else return this->GetTypeName()+a;
	}


	//
	//  Class for variables.
	//
	template<class T>
	Variable<T>::Variable(iScript *script, const iString &name) : Constant<T>(script,name,1,0)
	{
	}


	template<class T>
	Variable<T>::Variable(iScript *script, const iString &name, int dim) : Constant<T>(script,name,dim,0)
	{
	}


	template<class T>
	bool Variable<T>::Assign(const Calculator::result_t *v, short at, int index)
	{
		if(v == 0)
		{
			this->ReportError("Internal error: assigning a null pointer to the variable {"+this->Name()+"}.");  //  shouldn't ever happen
			return false;
		}

		int n = this->mValue->Dim();
		Calculator::number_t *data = this->mValue->Data();
		Calculator::number_t *src = v->Data();
		if(index == -1)  //  complete assignment
		{
			if(this->mValue->Dim()!=v->Dim() || this->mValue->Use()!=v->Use())
			{
				this->ReportError("Internal error: incompatible assignment of {"+v->Name()+"} to {"+this->mValue->Name()+"} has not been detected at compile-time.");
				return false;
			}
			int i;
			for(i=0; i<n; i++) data[i] = this->ConvertFromNative(this->Combine(this->ConvertToNative(data[i]),this->ConvertToNative(src[i]),at));
		}
		else
		{
			if(index<0 || index>=n)
			{
				this->ReportError("Accessing array "+this->mValue->Name()+" outside of bounds (with index="+iString::FromNumber(index+1)+").");
				return false;
			}
			data[index] = this->ConvertFromNative(this->Combine(this->ConvertToNative(data[index]),this->ConvertToNative(src[0]),at));
		}
#ifdef I_DEBUG
		iConsole::Display(iConsole::_Info,(this->Name()+"{"+this->GetTypeAsText()+"}: "+this->GetValueAsText()+"\n").ToCharPointer());
#endif
		return true;
	}


	template<class T>
	bool Variable<T>::Assign(T v, int index)
	{
		if(index<0 || index>=this->mValue->Dim())
		{
			this->ReportError("Accessing array "+this->mValue->Name()+" outside of bounds (with index="+iString::FromNumber(index+1)+").");
			return false;
		}
		this->mValue->Data()[index] = this->ConvertFromNative(v);
#ifdef I_DEBUG
		iConsole::Display(iConsole::_Info,(this->Name()+"{"+this->GetTypeAsText()+"}: "+this->GetValueAsText()+"\n").ToCharPointer());
#endif
		return true;
	}


	//
	//  Specific prototypes
	//
	template<class T>
	Statement* Prototype0<T>::CreateWorker() const
	{
		return new T(mScript,mCommand);
	}


	template<class T, class A>
	Statement* Prototype1<T,A>::CreateWorker() const
	{
		return new T(mScript,mCommand,mArgument);
	}


	template<class T, class A1, class A2>
	Statement* Prototype2<T,A1,A2>::CreateWorker() const
	{
		return new T(mScript,mCommand,mArgument1,mArgument2);
	}


	//
	//  Variable assignment statement
	//
	template<class T>
	VariableAssignmentStatement<T>::VariableAssignmentStatement(iScript *script, Variable<T> &var) : AssignmentStatement(script,var.Name(),Variable<T>::NaturalMask(),var.Dim(),var.Use()), mVar(var)
	{
		this->mAcceptsArrays = (var.Dim() > 1);
	}


	template<class T>
	bool VariableAssignmentStatement<T>::SetValue(const Calculator::result_t *result, int index)
	{
		if(result == 0)
		{
			this->ReportError("Internal error: null pointer for the calculation result.");  //  shouldn't ever happen
			return false;
		}
		
		return this->mVar.Assign(result,this->mAssignmentType,index);
	}


	//
	//  Function call assignment statement
	//
	template<class T>
	FunctionCallAssignmentStatement<T>::FunctionCallAssignmentStatement(iScript *script, const iString &command, FunctionType1 f, short mask) : AssignmentStatement(script,command,mask,1,Constant<T>::UsagePattern()), mFun1(f), mFun2(0)
	{
		this->mAcceptsArrays = false;
	}


	template<class T>
	FunctionCallAssignmentStatement<T>::FunctionCallAssignmentStatement(iScript *script, const iString &command, FunctionType2 f, short mask) : AssignmentStatement(script,command,mask,-1,Constant<T>::UsagePattern()), mFun1(0), mFun2(f)
	{
		this->mAcceptsArrays = true;
	}


	template<class T>
	bool FunctionCallAssignmentStatement<T>::SetValue(const Calculator::result_t *result, int index)
	{
		if(result == 0)
		{
			this->ReportError("Internal error: null pointer for the calculation result.");  //  shouldn't ever happen
			return false;
		}

		if(index == -1)
		{
			if(this->mFun1 != 0)
			{
				return this->mFun1(this->mScript,this->mAssignmentType,Constant<T>::ConvertToNative(result->Data()[0]));
			}
			else if(this->mFun2 != 0)
			{
				T *buffer = new T[result->Dim()];
				if(buffer == 0)
				{
					this->ReportError("There is not enough memory to proceed.");
					return false;
				}
				int i;
				for(i=0; i<result->Dim(); i++) buffer[i] = Constant<T>::ConvertToNative(result->Data()[i]);
				bool ret = this->mFun2(this->mScript,this->mAssignmentType,result->Dim(),buffer,-1);
				delete [] buffer;
				return ret;
			}
			else
			{
				this->ReportError("iScriptKit::FunctionCallAssignmentStatement is configured incorrectly.");
				return false;
			}
		}
		else
		{
			if(this->mFun2 != 0)
			{
				T tmp = Constant<T>::ConvertToNative(result->Data()[0]);
				return this->mFun2(this->mScript,this->mAssignmentType,1,&tmp,index);
			}
			else
			{
				this->ReportError("iScriptKit::FunctionCallAssignmentStatement is configured incorrectly.");
				return false;
			}
		}
	}


	template<class T>
	Prototype* VariableDeclarationStatement::NewPrototype(const iString &name, int dim) const
	{
		Variable<T> *v = new Variable<T>(this->mScript,name,dim);
		if(v != 0)
		{
			Prototype *p = new PrototypeHelper<T>(mScript,*v);
			if(p != 0)
			{
				this->mScript->AddVariable(v);
				return p;
			}
			delete v;
		}
		return 0;
	}
};


template<class T>
void iScript::AddPrototype(const iString &command)
{
	iScriptKit::Prototype *prototype = new iScriptKit::Prototype0<T>(this,command);
	this->RegisterPrototype(prototype);
}



template<class T, class A>
void iScript::AddPrototype(const iString &command, A arg)
{
	iScriptKit::Prototype *prototype = new iScriptKit::Prototype1<T,A>(this,command,arg);
	this->RegisterPrototype(prototype);
}


template<class T, class A1, class A2>
void iScript::AddPrototype(const iString &command, A1 arg1, A2 arg2)
{
	iScriptKit::Prototype *prototype = new iScriptKit::Prototype2<T,A1,A2>(this,command,arg1,arg2);
	this->RegisterPrototype(prototype);
}

