/*  This file is a part of gtkboard, a board games system.
    Copyright (C) 2003, Arvind Narayanan <arvindn@users.sourceforge.net>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA

*/
#include <stdio.h>
#include <assert.h>

#include "game.h"

#define TOWERS_CELL_SIZE 16
#define TOWERS_NUM_PIECES 3

//#define TOWERS_NUM_RINGS 7
#define TOWERS_BOARD_HEIT_REAL(num_rings) (2 * num_rings + 4)
#define TOWERS_TOWER_WID_REAL(num_rings) (2 * num_rings + 3)
#define TOWERS_BOARD_WID_REAL(num_rings) (3 * TOWERS_TOWER_WID_REAL(num_rings))
#define TOWERS_TOWER_WID TOWERS_TOWER_WID_REAL(num_rings)
#define TOWERS_BOARD_WID TOWERS_BOARD_WID_REAL(num_rings)
#define TOWERS_BOARD_HEIT TOWERS_BOARD_HEIT_REAL(num_rings)

#define TOWERS_EMPTY 0
#define TOWERS_POLE 1
#define TOWERS_RING_TOP  2
#define TOWERS_RING_BOT  3
#define TOWERS_RING_MID  3
#define TOWERS_RING_LEFT 4
#define TOWERS_RING_RIGHT 5

char towers_colors[6] = {150, 150, 150, 150, 150, 150};

static int num_rings = -1;

static char *pole_16_16_xpm[] = {
"16 16 5 1",
"1 c #AAAA66",
"2 c #999955",
"3 c #888844",
"4 c #80803b",
"5 c #777733",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
"5433221111223345",
};


static char *ring_top_16_16_xpm[] = {
"16 16 2 1",
"  c #cccc99",
". c #aaaa77",
"................",
"................",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
};

static char *ring_bot_16_16_xpm[] = {
"16 16 2 1",
"  c #cccc99",
". c #aaaa77",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"                ",
"................",
"................",
};


static char **towers_pixmaps[] =
{
	pole_16_16_xpm,
	ring_top_16_16_xpm,
	ring_bot_16_16_xpm,
};

static int towers_init_pos4[] = 
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 0, 0, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 0, 0, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 0, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 0, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
	0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
};

static int towers_init_pos5[] = 
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
	0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
};

static int towers_init_pos6[] = 
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
	0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
};

static int towers_init_pos7[] = 
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 0, 0, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 0, 0, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 0, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 0, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 0, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
	0, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,  
};


void towers_init ();

Game Towers = { TOWERS_CELL_SIZE, TOWERS_BOARD_WID_REAL(4), TOWERS_BOARD_HEIT_REAL(4),
	TOWERS_NUM_PIECES, 
	towers_colors, towers_init_pos4, towers_pixmaps, 
	"Tower of Hanoi", "Logic puzzles", 
	towers_init};

Game Towers5 = { TOWERS_CELL_SIZE, TOWERS_BOARD_WID_REAL(5), TOWERS_BOARD_HEIT_REAL(5),
	TOWERS_NUM_PIECES, 
	towers_colors, towers_init_pos5, towers_pixmaps, 
	"Tower of Hanoi 5", NULL, 
	towers_init};

Game Towers6 = { TOWERS_CELL_SIZE, TOWERS_BOARD_WID_REAL(6), TOWERS_BOARD_HEIT_REAL(6),
	TOWERS_NUM_PIECES, 
	towers_colors, towers_init_pos6, towers_pixmaps, 
	"Tower of Hanoi 6", NULL, 
	towers_init};

Game Towers7 = { TOWERS_CELL_SIZE, TOWERS_BOARD_WID_REAL(7), TOWERS_BOARD_HEIT_REAL(7),
	TOWERS_NUM_PIECES, 
	towers_colors, towers_init_pos7, towers_pixmaps, 
	"Tower of Hanoi 7", NULL,
	towers_init};

SCORE_FIELD towers_score_fields[] = {SCORE_FIELD_RANK, SCORE_FIELD_USER, SCORE_FIELD_SCORE, SCORE_FIELD_TIME, SCORE_FIELD_DATE, SCORE_FIELD_NONE};
char *towers_score_field_names[] = {"Rank", "User", "Moves", "Time", "Date", NULL};

static int towers_getmove (Pos *, int, int, GtkboardEventType, Player, byte **, int **);
static ResultType towers_who_won (Pos *, Player, char **);
void towers_reset_uistate ();
void towers_get_render (Pos *pos, byte *move, int **rmovp);

GameLevel towers_levels[] =
{
	{"4 rings", &Towers},
	{"5 rings", &Towers5},
	{"6 rings", &Towers6},
	{"7 rings", &Towers7},
	{ NULL, NULL }
};

void towers_init (Game *game)
{
	game_single_player = TRUE;
	game_getmove = towers_getmove;
	game_who_won = towers_who_won;
	game_scorecmp = game_scorecmp_def_iscore;
	game_score_fields =  towers_score_fields;
	game_score_field_names = towers_score_field_names;
	game_reset_uistate = towers_reset_uistate;
	game_get_render = towers_get_render;
	game_levels = towers_levels;
	game_doc_about_status = STATUS_COMPLETE;
	game_doc_about = 
		"Towers\n"
		"Single player game\n"
		"Status: Fully implemented\n"
		"URL: "GAME_DEFAULT_URL("towers");
	game_doc_rules = 
		" Transfer all the rings from the first pole to the last. A ring can never lie on a smaller one.";
		;

	if (game == &Towers)
		num_rings = 4;
	else if (game == &Towers5)
		num_rings = 5;
	else if (game == &Towers6)
		num_rings = 6;
	else if (game == &Towers7)
		num_rings = 7;
}

static gboolean towers_over (byte *board)
{
	return board [TOWERS_TOWER_WID/2] != TOWERS_RING_BOT && 
		board [TOWERS_TOWER_WID/2+TOWERS_TOWER_WID] != TOWERS_RING_BOT;
}

ResultType towers_who_won (Pos *pos, Player to_play, char **commp)
{
	static char comment[32];
	gboolean over = towers_over (pos->board);
	char *scorestr = over ? "You won. Moves:" : "Moves:";
	snprintf (comment, 32, "%s %d", scorestr, pos->num_moves);
	*commp = comment;
	return over ? RESULT_WON : RESULT_NOTYET;
}


static int cur_row = -1;

#define POLE_ROW(x) (((x)/TOWERS_TOWER_WID)*TOWERS_TOWER_WID+TOWERS_TOWER_WID/2)

// Find the heit and len of the top ring in the tower corresp to row
static void get_top_ring (byte *board, int row, int *heit, int *len)
{
	int pole_row = POLE_ROW (row);
	int i, j;
	for (j = board_heit - 1; j >= 0; j--)
		if (board [j * board_wid + pole_row] == TOWERS_RING_TOP)
			break;
	if (j < 0)
	{
		if (heit) *heit = -1;
		if (len) *len = -1;
		return; 
	}
	for (i=0; ; i++)
		if (board [j * board_wid + pole_row + i] != TOWERS_RING_TOP)
			break;
	if (heit) *heit = j;
	if (len) *len = i;
}

int towers_getmove 
	(Pos *pos, int x, int y, GtkboardEventType type, Player to_play, byte **movp, int ** rmovep)
{
	static byte move[512];
	static int rmove[512];
	byte *mp = move;
	int *rp = rmove;
	int heit1, heit2, len1, len2;
	int i, j, pole_row;
	
	if (type != GTKBOARD_BUTTON_RELEASE)
		return 0;
	if (cur_row < 0)
	{
		get_top_ring (pos->board, x, &heit1, &len1);
		if (heit1 < 0)
			return -1;
		cur_row = x;
		pole_row = POLE_ROW(x);
		for (j=0; j<=1; j++)
		for (i = pole_row - len1 + 1; i <= pole_row + len1 - 1; i++)
		{
			*rp++ = i;
			*rp++ = heit1 - j;
			*rp++ = RENDER_SHADE1;
		}
		*rp++ = -1;
		*rmovep = rmove;
		return 0;
	}
	
	get_top_ring (pos->board, cur_row, &heit1, &len1);
	get_top_ring (pos->board, x, &heit2, &len2);

	pole_row = POLE_ROW(cur_row);
	for (j=0; j<=1; j++)
	for (i = pole_row - len1 + 1; i <= pole_row + len1 - 1; i++)
	{
		*mp++ = i;
		*mp++ = heit1 - j;
		*mp++ = i == pole_row ? TOWERS_POLE : TOWERS_EMPTY;
		*rp++ = i;
		*rp++ = heit1 - j;
		*rp++ = 0;
	}
	*rp++ = -1;
	*rmovep = rmove;

	if (POLE_ROW(cur_row) == POLE_ROW(x))
	{
		cur_row = -1;
		return 0;
	}
	
	if (heit2 >= 0 && len1 > len2)
	{
		cur_row = -1;
		return -1;
	}

	heit2 += 2;

	pole_row = POLE_ROW(x);
	for (j=0; j<=1; j++)
	for (i = pole_row - len1 + 1; i <= pole_row + len1 - 1; i++)
	{
		*mp++ = i;
		*mp++ = heit2 - j;
		*mp++ = j == 0 ? TOWERS_RING_TOP : TOWERS_RING_BOT;
	}
	*mp++ = -1;
	*movp = move;
	cur_row = -1;
	return 1;
}

void towers_reset_uistate ()
{
	cur_row = -1;
}

void towers_get_render (Pos *pos, byte *move, int **rmovp)
{
	static int rmove[512];
	int *rp = rmove;
	int i, j;
	for (i=0; i<board_wid; i++)
	for (j=0; j<board_heit; j++)
		if (pos->render[j * board_wid + i] != RENDER_NONE)
		{
			*rp++ = i;
			*rp++ = j;
			*rp++ = RENDER_NONE;
		}
	*rp++ = -1;
	*rmovp = rmove;
}
