//**************************************************************************************************
//                                         PrcGNetList.cpp                                         *
//                                        -----------------                                        *
// Started     : 2004-01-28                                                                        *
// Last Update : 2015-02-16                                                                        *
// Copyright   : (C) 2004 by MSWaters                                                              *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "PrcGNetList.hpp"

//**************************************************************************************************
// Constructor.

PrcGNetList::PrcGNetList( void ) : PrcBase( wxPROCESS_REDIRECT )
{
  // Clear the object attributes
  bClear( );

  // Initialize the array of Guile procedure names and set the default
  InitGuileProcs( );

  // Attempt to set and find the gnetlist binary
  bSetBinary( BIN_GNETLIST );

  // Set the log file name
  bSetLogFile( GNETLST_LOG_FILE );
}

//**************************************************************************************************
// Destructor.

PrcGNetList::~PrcGNetList( )
{
}

//**************************************************************************************************
// Clear the object attributes.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcGNetList::bClear( void )
{
  bSetSchemFiles( wxT("") );
  bSetNetLstFile( wxT("") );
  bSetGuileProc ( wxT("") );

  return( true );
}

//**************************************************************************************************
// Initialize the array of possible Guile procedure names.

void  PrcGNetList::InitGuileProcs( void )
{
  m_osaGuileProcs.Clear( );

  m_osaGuileProcs.Add( wxT("")          ); // None
  m_osaGuileProcs.Add( wxT("allegro")   ); // Allegro netlist format
  m_osaGuileProcs.Add( wxT("bae")       ); // BAE netlist format
  m_osaGuileProcs.Add( wxT("bom")       ); // BOM - Bill of Materials
  m_osaGuileProcs.Add( wxT("bom2")      ); // BOM2 - Bill of Materials
  m_osaGuileProcs.Add( wxT("drc")       ); // DRC - Start of a design rule checker
  m_osaGuileProcs.Add( wxT("geda")      ); // gEDA - native format, mainly used for testing
  m_osaGuileProcs.Add( wxT("gossip")    ); // Gossip netlist format
  m_osaGuileProcs.Add( wxT("pads")      ); // PADS netlist format
  m_osaGuileProcs.Add( wxT("PCB")       ); // PCB
  m_osaGuileProcs.Add( wxT("PCBboard")  ); // PCBboard
  m_osaGuileProcs.Add( wxT("protelII")  ); // ProtelII netlist format
  m_osaGuileProcs.Add( wxT("spice")     ); // Spice compatible netlist format
  m_osaGuileProcs.Add( wxT("spice-sdb") ); // Enhanced spice compatible netlist format
  m_osaGuileProcs.Add( wxT("switcap")   ); // Switcap netlist format
  m_osaGuileProcs.Add( wxT("tango")     ); // Tango netlist format
  m_osaGuileProcs.Add( wxT("verilog")   ); // Verilog code
  m_osaGuileProcs.Add( wxT("vhdl")      ); // VHDL code
  m_osaGuileProcs.Add( wxT("vipec")     ); // VIPEC netlist format
}

//**************************************************************************************************
// Set the schematic file name/s.
//
// Argument List :
//   rosFNames - A string containing the full path and file name/s
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcGNetList::bSetSchemFiles( const wxString & rosFNames )
{
  wxStringTokenizer  ostk1;
  wxArrayString      osa1;

  ostk1.SetString( rosFNames );
  while( ostk1.HasMoreTokens( ) ) osa1.Add( ostk1.GetNextToken( ) );

  return( bSetSchemFiles( osa1 ) );
}

//**************************************************************************************************
// Set the schematic file name/s.
//
// Argument List :
//   rosaFNames - A string array containing the full path and file name/s
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcGNetList::bSetSchemFiles( const wxArrayString & rosaFNames )
{
  wxFileName  ofn1;
  size_t      sz1;

  // Clear the current list of schematic files
  m_ofnaSchemFiles.Clear( );

  // Check the argument is empty
  if( rosaFNames.IsEmpty( ) )                                  return( true );

  // Add the new schematic file name/s to the list
  for( sz1=0; sz1<rosaFNames.GetCount( ); sz1++ )
  {
    ofn1 = rosaFNames.Item( sz1 );
    if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxT(".") );

    if( ! ofn1.IsOk( )       ) continue;
    if( ! ofn1.FileExists( ) ) continue;

    m_ofnaSchemFiles.Add( ofn1 );
  }

  // Check that at least one schematic file name was accepted
  if( m_ofnaSchemFiles.IsEmpty( ) )                            return( false );

  // Set the log file path
  ofn1 = roGetLogFile( );
  ofn1.SetPath( m_ofnaSchemFiles.Item( 0 ).GetPath( ) );
  bSetLogFile( ofn1.GetFullPath( ) );

  // Check if any of the schematic files were invalid
  if( m_ofnaSchemFiles.GetCount( ) != rosaFNames.GetCount( ) ) return( false );

  return( true );
}

//**************************************************************************************************
// Set the full netlist file name.
//
// Argument List :
//   rosFName - A string containing the full path and file name
//              (If rosFName = "use-schem-name" the netlist name is to be generated from the
//               schematic file name)
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcGNetList::bSetNetLstFile( const wxString & rosFName )
{
  wxFileName  ofn1;

  // Set the netlist name
  if( ! rosFName.IsEmpty( ) )
  {
    if( rosFName == GNETLST_USE_SCHEM )
    { // Generate the netlist file name from the schematic file name
      ofn1 = rofnGetSchemFile( );
      ofn1.SetExt( wxT("ckt") );
    }
    else ofn1 = rosFName;

    // Set the netlist path if it hasn't been set
    if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxT(".") );

    // Check that the file name is OK
    if( ! ofn1.IsOk( ) ) return( false );
  }
  else m_ofnNetLstFile.Clear( );

  // Set the netlist file path and name
  m_ofnNetLstFile = ofn1;

  // Set the log file path
  ofn1 = roGetLogFile( );
  ofn1.SetPath( m_ofnNetLstFile.GetPath( ) );
  bSetLogFile( ofn1.GetFullPath( ) );

  return( true );
}

//**************************************************************************************************
// Set the Guile procedure to be used to import the schematic file.
//
// Argument List :
//   rosGProc - The Guile procedure name
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcGNetList::bSetGuileProc( const wxString & rosGProc )
{
  int  i1;

  i1 = m_osaGuileProcs.Index( rosGProc );
  if( i1 == wxNOT_FOUND ) return( false );

  m_szGuileProc = (size_t) i1;

  return( true );
}

//**************************************************************************************************
// Get a schematic file from the list of schematic files.
//
// Argument List :
//   szIndex - A zero based index to the file name
//
// Return Values :
//   Success - The procedure name
//   Failure - An empty file name

const wxFileName & PrcGNetList::rofnGetSchemFile( size_t szIndex )
{
  static  wxFileName  oFnEmpty;

  if( szIndex >= m_ofnaSchemFiles.GetCount( ) ) return( oFnEmpty );

  return( m_ofnaSchemFiles[ szIndex ] );
}

//**************************************************************************************************
// Get a Guile procedure from the list of procedure names.
//
// Argument List :
//   szIndex - A zero based index to the procedure name
//             (If szIndex = CUR_SEL_PROC return currently selected procedure)
//
// Return Values :
//   Success - The procedure name
//   Failure - An empty string

const wxString & PrcGNetList::rosGetGuileProc( size_t szIndex )
{
  static  wxString  osEmpty;

  if( szIndex == GNETLST_CUR_PROC ) szIndex = m_szGuileProc;

  if( szIndex >= m_osaGuileProcs.GetCount( ) ) return( osEmpty );

  return( m_osaGuileProcs[ szIndex ] );
}

//**************************************************************************************************
// Make a netlist file from a schematic file.
//
// (Eg. using the following: gnetlist -v -g spice-sdb -o test.ckt test.sch)
//
// Return Values :
//   true  - Success
//   false - Failure

bool  PrcGNetList::bExec( void )
{
  wxString  osArgLst;
  wxString  os1;
  size_t    sz1;

  // Test file names needed by this function
  if( ! bBinExists( ) )             return( false );
  if( m_ofnaSchemFiles.IsEmpty( ) ) return( false );
  if( ! m_ofnNetLstFile.IsOk( ) )
    if( ! bSetNetLstFile( ) )       return( false );

  // Enable GNetList verbose mode
  osArgLst = wxT("-v");

  // Specify the guile procedure name to be used
  osArgLst << wxT(" -g ");
  os1 = m_osaGuileProcs.Item( m_szGuileProc );
  if( os1.IsEmpty( ) ) os1 = GNETLST_GUILE_PROC;
  osArgLst << os1;

  // Append input and output file names
  osArgLst << wxT(" -o ") << m_ofnNetLstFile.GetFullPath( );
  for( sz1=0; sz1<m_ofnaSchemFiles.GetCount( ); sz1++ )
    osArgLst << wxT(" ") << m_ofnaSchemFiles.Item( sz1 ).GetFullPath( );
  bSetArgLst( osArgLst.c_str( ) );

  // Set the working directory
  os1 = rofnGetSchemFile( ).GetPath( );
  wxSetWorkingDirectory( os1 );

  // Execute the process
  if( ! PrcBase::bExecAsync( ) )    return( false );

  // Capture the process output
  if( ! bLogOutput( ) )             return( false );

  return( true );
}

//**************************************************************************************************
