/*
   Copyright (c) 2004, 2005 by AOSASA Shigeru and Red Hat, Inc.
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are
   met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above
   copyright notice, this list of conditions and the following
   disclaimer in the documentation and/or other materials provided
   with the distribution.  

   - Neither the name of the AOSASA Shigeru, Red Hat, Inc
   nor the names of its contributors may be used to endorse or
   promote products derived from this software without specific
   prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
   FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
   COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
   INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
   (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
   SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
   STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
   OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*
   This software development was supported by
   Information-technorogy Promotion Agency, Japan (IPA).
*/

#include "ginspector-builtin.h"

#include "ginspector-core.h"
#include "ginspector-gtk.h"
#include "ginspector-ui.h"

#include "glrr-gobject.h"
#include "glrr-gtk.h"

#include "misc-utils.h"



static void builtin_module_init (void);

static gpointer create_panel_child_list (gpointer data,
					 gpointer func_data,
					 gpointer invoke_data);



void
g_inspector_builtin_register_module_panel_child_list (void)
{
  {
    static GInspectorBuiltinModuleInfo builtin_module_info;

    builtin_module_info.name       = "builtin-panel-child-list";
    builtin_module_info.data       = NULL;
    builtin_module_info.init       = builtin_module_init;
    builtin_module_info.activate   = NULL;
    builtin_module_info.deactivate = NULL;

    g_inspector_builtin_register_module (&builtin_module_info);
  }
}

static void
builtin_module_init (void)
{
  g_inspector_register_procedure ("create-panel-child-list",
				  "",
				  G_INSPECTOR_PROCEDURE_TYPE_PANEL,
				  G_INSPECTOR_ELEMENT_TYPE_NONE,
				  G_TYPE_NONE,
				  create_panel_child_list,
				  NULL);
}



/***/

static void create_list (GtkListStore *list_store,
			 GtkContainer *container);

static void set_row (GtkListStore *list_store,
		     GtkWidget    *child,
		     GtkTreeIter  *tree_iter);
static void append_row (GtkListStore *list_store,
			GtkWidget    *child);
static void update_row (GtkListStore *list_store,
			GtkWidget    *child);
static void child_notify (GtkWidget  *widget,
			  GParamSpec *pspec,
			  gpointer    user_data);



enum
{
  COLUMN_0,

  COLUMN_CHILD,

  NUM_COMMON_COLUMNS
};

static GInspectorElementListColumnInfo common_columns[NUM_COMMON_COLUMNS] =
{
  G_INSPECTOR_ELEMENT_LIST_COLUMN_INFO_0,

  { COLUMN_CHILD,	"Child",	"gpointer",	"pointer" },
};


static void
child_notify (GtkWidget  *widget,
	      GParamSpec *pspec,
	      gpointer    user_data)
{
  GtkWidget *child;
  GtkListStore *list_store;

  child = widget;
  list_store = GTK_LIST_STORE (user_data);

  update_row (list_store,
	      child);
}

static void
update_row (GtkListStore *list_store,
	    GtkWidget    *child)
{
  gboolean r;
  GtkTreeIter tree_iter;

  r = g_inspector_element_list_get_iter (list_store,
					 &tree_iter,
					 G_INSPECTOR_ELEMENT_TYPE_GTYPEINSTANCE,
					 child);

  if (r == TRUE)
    {
      set_row (list_store, child, &tree_iter);
    }
}

static void
set_row (GtkListStore *list_store,
	 GtkWidget    *child,
	 GtkTreeIter  *tree_iter)
{
  GtkContainer *container;
  GParamSpec **properties;
  guint n_properties;

  container = GTK_CONTAINER (gtk_widget_get_parent (child));


  {
    GObjectClass *object_class;


    object_class = g_type_class_ref (G_OBJECT_TYPE (container));

    properties = gtk_container_class_list_child_properties (object_class,
							    &n_properties);
    
    g_type_class_unref (object_class);
  }


  {
    GParamSpec *spec;
    GValue value;
    gchar *text;
    guint i;

    for (i = 0; i < n_properties; ++i)
      {
	spec = properties[i];

	g_value_fill0 (&value);
	g_value_init (&value, spec->value_type);

	gtk_container_child_get_property (container, child,
					  spec->name, &value);

#if 0
	text = g_strdup_value_contents (&value);
#else
	text = g_value_to_string (&value);
#endif

	g_value_unset (&value);


	gtk_list_store_set (list_store, tree_iter,
			    (NUM_COMMON_COLUMNS + i), text,
			    -1);

	g_free (text);
      }
  }


  if (properties != NULL)
    {
      g_free (properties);
    }
}

static void
append_row (GtkListStore *list_store,
	    GtkWidget    *child)
{
  GtkTreeIter tree_iter;

  gtk_list_store_append (list_store, &tree_iter);

  g_inspector_element_list_set_row_element_gobject (list_store, &tree_iter,
						    G_OBJECT (child));

  gtk_list_store_set (list_store, &tree_iter,
		      COLUMN_CHILD,  child,
		      -1);

  set_row (list_store, child, &tree_iter);
}

static void
watch_child_property (GtkListStore *list_store,
		      GtkWidget    *child)
{
  {
    g_signal_handlers_disconnect_by_func (child,
					  child_notify,
					  list_store);
  }

  {
    GClosure *closure;

    closure = g_cclosure_new (G_CALLBACK (child_notify),
			      list_store,
			      NULL);

    g_object_watch_closure (G_OBJECT (list_store),
			    closure);

    g_signal_connect_closure (child,
			      "child_notify",
			      closure,
			      FALSE);
  }
}

static void
create_list (GtkListStore *list_store,
	     GtkContainer *container)
{
  GList *children;
  GList *tmp;
  GtkWidget *child;


  children = gtk_container_get_children (container);

  tmp = children;
  while (tmp != NULL)
    {
      child = GTK_WIDGET (tmp->data);
      tmp = g_list_next (tmp);

      append_row (list_store, child);
      watch_child_property (list_store, child);
    }

  g_list_free (children);
}

static void
update (GInspectorElementList *element_list,
	gpointer               user_data)
{
  GtkListStore *list_store;
  GtkContainer *container;

  list_store = GTK_LIST_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (element_list)));
  container = GTK_CONTAINER (user_data);

  gtk_list_store_clear (list_store);
  create_list (list_store,
	       container);
}

static GtkWidget *
create_element_list (GtkContainer *container)
{
  gint n_columns;
  GInspectorElementListColumnInfo *columns;
  GtkWidget *element_list;


  {
    GParamSpec **properties;
    guint n_properties;


    {
      GObjectClass *object_class;

	
      object_class = g_type_class_ref (G_OBJECT_TYPE (container));

      properties = gtk_container_class_list_child_properties (object_class,
							      &n_properties);
    
      g_type_class_unref (object_class);
    }


    {
      GInspectorElementListColumnInfo *column_info;
      GParamSpec *spec;
      guint i;

      n_columns = (NUM_COMMON_COLUMNS + n_properties);

      columns = g_new (GInspectorElementListColumnInfo, n_columns);

      for (i = 0; i < NUM_COMMON_COLUMNS; ++i)
	{
	  column_info = &(columns[i]);
	  
	  column_info->column     = common_columns[i].column;
	  column_info->title      = common_columns[i].title;
	  column_info->gtype_name = common_columns[i].gtype_name;
	  column_info->attribute  = common_columns[i].attribute;
	}

      for (i = 0; i < n_properties; ++i)
	{
	  spec = properties[i];
	  column_info = &(columns[NUM_COMMON_COLUMNS + i]);

	  column_info->column     = (NUM_COMMON_COLUMNS + i);
	  column_info->title      = spec->name;
	  column_info->gtype_name = "gchararray";
	  column_info->attribute  = "text";
	}
    }


    if (properties != NULL) {
      g_free (properties);
    }
  }

  element_list = g_inspector_element_list_new (columns,
					       n_columns);

  g_free (columns);


  gtk_widget_set_name (element_list, "element-list");


  return element_list;
}


static gpointer
create_panel_child_list (gpointer data,
			 gpointer func_data,
			 gpointer invoke_data)
{
  GtkContainer *container;

  GtkWidget *top;
  GtkWidget *panel;
  GtkWidget *frame;
  GtkWidget *scrolled_window;
  GtkWidget *element_list;


  container = GTK_CONTAINER (data);


  {
    panel = g_object_new (G_TYPE_INSPECTOR_PANEL,
                          NULL);

    G_INSPECTOR_PANEL (panel)->element_type = G_INSPECTOR_ELEMENT_TYPE_GTYPEINSTANCE;
    G_INSPECTOR_PANEL (panel)->element = container;

    top = panel;
  }

  {
    frame = g_inspector_panel_frame_new ();
    g_inspector_panel_frame_set_title (G_INSPECTOR_PANEL_FRAME (frame), "Children");

    gtk_container_add (GTK_CONTAINER (panel), frame);
  }

  {
    scrolled_window = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_container_add (GTK_CONTAINER (frame), scrolled_window);
  }

  {
    element_list = create_element_list (container);

    g_signal_connect (element_list,
		      "update",
		      G_CALLBACK (update),
		      container);

    gtk_container_add (GTK_CONTAINER (scrolled_window),
		       element_list);

    g_inspector_panel_attach_element_list (G_INSPECTOR_PANEL (panel),
					   G_INSPECTOR_ELEMENT_LIST (element_list));
  }


  {
    g_signal_connect_swapped (gtk_frame_get_label_widget (GTK_FRAME (frame)),
			      "clicked",
			      G_CALLBACK (g_inspector_panel_update),
			      panel);
  }




  g_inspector_panel_update (G_INSPECTOR_PANEL (panel));


  return top;
}
