#include "polyhedralcone.h"

#include "lp.h"
#include "subspace.h"
#include "symmetry.h"
#include "polymakefile.h"
#include <sstream>

//--------------------------------
// PolyhedralCone
//--------------------------------

static bool compareIntegerLists(IntegerVectorList const &a, IntegerVectorList const &b)
{
  assert(a.size()==b.size());

  IntegerVectorList::const_iterator B=b.begin();
  for(IntegerVectorList::const_iterator A=a.begin();A!=a.end();A++)
    {
      if(LexicographicTermOrder()(*A,*B))return true;
      if(LexicographicTermOrder()(*B,*A))return false;
      B++;
    }
  return false;
}

bool operator<(PolyhedralCone const &a, PolyhedralCone const &b)
{
  assert(a.state>=3);
  assert(b.state>=3);

  if(a.equations.size()<b.equations.size())return true;
  if(a.equations.size()>b.equations.size())return false;

  if(a.n<b.n)return true;
  if(a.n>b.n)return false;

  if(a.halfSpaces.size()<b.halfSpaces.size())return true;
  if(a.halfSpaces.size()>b.halfSpaces.size())return false;

  if(compareIntegerLists(a.equations,b.equations))return true;
  if(compareIntegerLists(b.equations,a.equations))return false;

  if(compareIntegerLists(a.halfSpaces,b.halfSpaces))return true;
  if(compareIntegerLists(b.halfSpaces,a.halfSpaces))return false;

  return false;
}

bool operator!=(PolyhedralCone const &a, PolyhedralCone const &b)
{
  return (a<b)||(b<a);
}


void PolyhedralCone::ensureStateAsMinimum(int s)
{
  if((state<1) && (s==1))
    {
      removeRedundantRows(&halfSpaces,&equations,false);
    }
  if((state<2) && (s>=2))
    {
      //      halfSpaces.sort();
      //      AsciiPrinter(Stderr).printVectorList(halfSpaces);

       if(halfSpaces.size()>25)
	 //	if(0)
	 {
	  IntegerVectorList h1;
	  IntegerVectorList h2;
	  bool a=false;
	  for(IntegerVectorList::const_iterator i=halfSpaces.begin();i!=halfSpaces.end();i++)
	    {
	      if(a)
		h1.push_back(*i);
	      else
		h2.push_back(*i);
	      a=!a;
	    }
	  PolyhedralCone c1(h1,equations);
	  PolyhedralCone c2(h2,equations);
	  c1.ensureStateAsMinimum(2);
	  c2.ensureStateAsMinimum(2);
	  halfSpaces=c1.halfSpaces;
	  for(IntegerVectorList::const_iterator i=c2.halfSpaces.begin();i!=c2.halfSpaces.end();i++)
	    halfSpaces.push_back(*i);
	}
	

       //      fprintf(Stderr,"Number half spaces: %i, number of equations: %i\n",halfSpaces.size(),equations.size());
      removeRedundantRows(&halfSpaces,&equations,true);
      //   fprintf(Stderr,"done\n");
    }
  if((state<3) && (s>=3))
    {
      //      fprintf(Stderr,"Number half spaces: %i, number of equations: %i\n",halfSpaces.size(),equations.size());
      Subspace v(equations,n);
      
      equations=v.getRepresentation();
      
      for(IntegerVectorList::iterator i=halfSpaces.begin();i!=halfSpaces.end();i++)
	{
	  *i=v.canonicalizeVector(*i);
	}
      halfSpaces.sort();
      //fprintf(Stderr,"done\n");
    }
  state=s;
}

void PolyhedralCone::canonicalize()
{
  ensureStateAsMinimum(3);
}


void PolyhedralCone::findFacets()
{
  ensureStateAsMinimum(2);
}


PolyhedralCone::PolyhedralCone(int ambientDimension):
  n(ambientDimension),
  state(1),
  multiplicity(-1)
{
}


PolyhedralCone::PolyhedralCone(IntegerVectorList const &halfSpaces_, IntegerVectorList const &equations_, int ambientDimension):
  halfSpaces(halfSpaces_),
  equations(equations_),
  state(0),
  multiplicity(-1)
{
  n=ambientDimension;
  if(n==-1)
    {
      if(!halfSpaces_.empty())
	n=halfSpaces_.begin()->size();
      else if(!equations_.empty())
	n=equations_.begin()->size();
      else
	{
	  assert(0);
	}
    }
  for(IntegerVectorList::const_iterator i=halfSpaces.begin();i!=halfSpaces.end();i++)
    {
      assert(i->size()==n);
    }
  for(IntegerVectorList::const_iterator i=equations.begin();i!=equations.end();i++)
    {
      assert(i->size()==n);
    }
  ensureStateAsMinimum(1);
  //  computeAndReduceLinearitySpace();
}


IntegerVector PolyhedralCone::getRelativeInteriorPoint()const
{
  //  ensureStateAsMinimum(1);
  assert(state>=1);
  IntegerVector ret;
  IntegerVectorList g=equations;
  int numberOfEqualities=g.size();
  g.insert(g.end(),halfSpaces.begin(),halfSpaces.end());
  IntegerVector equalitySet(g.size());
  for(int i=0;i<numberOfEqualities;i++)equalitySet[i]=1;

  if(!g.empty())
    ret=relativeInteriorPoint(g,&equalitySet);
  else
    ret=IntegerVector(n);//cone is the full space, lp code would fail, since the dimension is unknown

  for(IntegerVectorList::const_iterator i=equations.begin();i!=equations.end();i++)
    {
      assert(dotLong(*i,ret)==0);
    }

  for(IntegerVectorList::const_iterator i=halfSpaces.begin();i!=halfSpaces.end();i++)
    {
      if(!(dotLong(*i,ret)>0))
	{
	  fprintf(Stderr,"PolyhedralCone::relativeInteriorPoint() : halfSpaces not reduced or mistake in cdd interface!!!\n");
	}
    }

  return ret;
}


IntegerVectorList PolyhedralCone::getHalfSpaces()const
{
  return halfSpaces;
}


const IntegerVectorList &PolyhedralCone::getEquations()const
{
  assert(state>=1);
  return equations;
}


int PolyhedralCone::ambientDimension()const
{
  return n;
}


int PolyhedralCone::codimension()const
{
  return ambientDimension()-dimension();
  //  return getEquations().size();
}


int PolyhedralCone::dimension()const
{
  assert(state>=1);
  //  ensureStateAsMinimum(1);
  return ambientDimension()-equations.size();
}


bool PolyhedralCone::isZero()const
{
  return dimension()==0;
}


PolyhedralCone intersection(const PolyhedralCone &a, const PolyhedralCone &b)
{
  assert(a.ambientDimension()==b.ambientDimension());
  IntegerVectorList halfSpaces=a.halfSpaces;
  halfSpaces.insert(halfSpaces.end(),b.halfSpaces.begin(),b.halfSpaces.end());
  IntegerVectorList equations=a.equations;
  equations.insert(equations.end(),b.equations.begin(),b.equations.end());
  return PolyhedralCone(halfSpaces,equations,a.ambientDimension());
}


PolyhedralCone PolyhedralCone::positiveOrthant(int dimension)
{
  IntegerVectorList halfSpaces;

  for(int i=0;i<dimension;i++)halfSpaces.push_back(IntegerVector::standardVector(dimension,i));

  IntegerVectorList empty;
  return PolyhedralCone(halfSpaces,empty,dimension);
}


bool PolyhedralCone::isInStateMinimum(int s)const
{
  return state>=s;
}

int PolyhedralCone::getState()const
{
  return state;
}


void PolyhedralCone::print(class Printer *p)const
{
  if(0)
    {
      p->printString("Printing PolyhedralCone");
      p->printNewLine();
      p->printString("Ambient dimension: ");
      p->printInteger(n);
      p->printNewLine();
      if(isInStateMinimum(1))
	{
	  p->printString("Dimension: ");
	  p->printInteger(dimension());
	  p->printNewLine();
	}
      p->printString("Linearity space:");
      //  p->printNewLine();
      p->printVectorList(equations);
      p->printString("Inequalities:");
      p->printVectorList(halfSpaces);
      p->printString("Relative interior point:\n");
      p->printVector(getRelativeInteriorPoint());
      p->printNewLine();
      p->printString("Done printing PolyhedralCone.");
      p->printNewLine();
    }
  else
    {
      PolymakeFile polymakeFile;
      polymakeFile.create("NONAME","PolyhedralCone","PolyhedralCone");
      polymakeFile.writeCardinalProperty("AMBIENT_DIM",n);
      if(isInStateMinimum(1))
	{
	  polymakeFile.writeCardinalProperty("DIM",dimension());
	  //need to check that the following is done correctly
	  //       	  polymakeFile.writeCardinalProperty("LINEALITY_DIM",dimensionOfLinealitySpace());

	  polymakeFile.writeMatrixProperty("IMPLIED_EQUATIONS",rowsToIntegerMatrix(equations,n));
	}
      polymakeFile.writeCardinalProperty("LINEALITY_DIM",dimensionOfLinealitySpace());
      polymakeFile.writeMatrixProperty("LINEALITY_SPACE",rowsToIntegerMatrix(linealitySpace().dualCone().getEquations(),n));
      

      if(isInStateMinimum(2))
	polymakeFile.writeMatrixProperty("FACETS",rowsToIntegerMatrix(halfSpaces,n));
      else
	polymakeFile.writeMatrixProperty("INEQUALITIES",rowsToIntegerMatrix(halfSpaces,n));

      polymakeFile.writeCardinalVectorProperty("RELATIVE_INTERIOR_POINT",getRelativeInteriorPoint());


      stringstream s;
      polymakeFile.writeStream(s);
      string S=s.str();
      p->printString(S.c_str());
    }
}


static IntegerVector dehomogenize(IntegerVector const &v)
{
  assert(v.size()>0);

  IntegerVector ret(v.size()-1);

  for(int i=0;i<v.size()-1;i++)ret[i]=v[i];
  return ret;
}

static IntegerVectorList dehomogenize(IntegerVectorList const &l)
{
  IntegerVectorList ret;

  for(IntegerVectorList::const_iterator i=l.begin();i!=l.end();i++)
    {
      ret.push_back(dehomogenize(*i));
    }
  return ret;
}

PolyhedralCone PolyhedralCone::withLastCoordinateRemoved()const
{
  assert(n>0);

  return PolyhedralCone(dehomogenize(halfSpaces),dehomogenize(equations));
}

bool PolyhedralCone::containsPositiveVector()const
{
  PolyhedralCone temp=intersection(*this,PolyhedralCone::positiveOrthant(n));
  IntegerVector v=temp.getRelativeInteriorPoint();
  return v.isPositive();
}


int PolyhedralCone::dimensionOfLinealitySpace()const
{
  if(halfSpaces.empty())return dimension();
  IntegerVectorList a;
  PolyhedralCone temp(a,halfSpaces);
  temp=intersection(temp,*this);

  return temp.dimension();
}


bool PolyhedralCone::contains(IntegerVector const &v)const
{
  for(IntegerVectorList::const_iterator i=equations.begin();i!=equations.end();i++)
    {
      if(dotLong(*i,v)!=0)return false;
    }

  for(IntegerVectorList::const_iterator i=halfSpaces.begin();i!=halfSpaces.end();i++)
    {
      if(dotLong(*i,v)<0)return false;
    }

  return true;
}


bool PolyhedralCone::contains(PolyhedralCone const &c)const
{
  PolyhedralCone c2=intersection(*this,c);
  PolyhedralCone c3=c;
  c2.canonicalize();
  c3.canonicalize();
  return !(c2!=c3);
}


bool PolyhedralCone::containsRelatively(IntegerVector const &v)const
{
  assert(state>=1);
  for(IntegerVectorList::const_iterator i=equations.begin();i!=equations.end();i++)
    {
      if(dotLong(*i,v)!=0)return false;
    }

  for(IntegerVectorList::const_iterator i=halfSpaces.begin();i!=halfSpaces.end();i++)
    {
      if(dotLong(*i,v)<=0)return false;
    }

  return true;
}


PolyhedralCone PolyhedralCone::permuted(IntegerVector const &v)const
{
  PolyhedralCone ret(SymmetryGroup::permuteIntegerVectorList(halfSpaces,v),SymmetryGroup::permuteIntegerVectorList(equations,v),n);
  if(state>=1)ret.state=1;
  if(state>=2)ret.state=2;

  ret.ensureStateAsMinimum(state);
  return ret;
}


IntegerVectorList PolyhedralCone::extremeRays()const
{
  assert(dimension()==ambientDimension());

  IntegerVectorList ret;
  IntegerVectorList indices=extremeRaysInequalityIndices(halfSpaces);
  for(IntegerVectorList::const_iterator i=indices.begin();i!=indices.end();i++)
    {
      IntegerVectorList equations;
      for(int j=0;j<i->size();j++)
	{
	  IntegerVectorList::const_iterator a=halfSpaces.begin();
	  for(int k=0;k<(*i)[j];k++)
	    {
	      assert(a!=halfSpaces.end());
	      a++;
	    }
	  assert(a!=halfSpaces.end());
	  equations.push_back(*a);
	}
      ret.push_back(PolyhedralCone(halfSpaces,equations).getRelativeInteriorPoint());
    }
  return ret;
}


bool PolyhedralCone::isSimplicial()const
{
  assert(state>=2);
  
  //  ensureStateAsMinimum(2);
  //  AsciiPrinter P(Stderr);
  //  print(&P);
  return codimension()+getHalfSpaces().size()+dimensionOfLinealitySpace()==n;
}


bool PolyhedralCone::checkDual(PolyhedralCone const &c)const
{
  assert(dimensionOfLinealitySpace()+c.dimension()==ambientDimension());

  for(IntegerVectorList::const_iterator i=halfSpaces.begin();i!=halfSpaces.end();i++)
    {
      assert(c.contains(*i));
    }
  for(IntegerVectorList::const_iterator i=equations.begin();i!=equations.end();i++)
    {
      assert(c.contains(*i));
    }
  return true;
}


PolyhedralCone PolyhedralCone::dualCone()const
{
  assert(state>=1);

  IntegerVectorList dualInequalities,dualEquations;

  dual(ambientDimension(),halfSpaces,equations,&dualInequalities,&dualEquations);

  PolyhedralCone ret(dualInequalities,dualEquations);

  ret.ensureStateAsMinimum(state);
  //  ret.canonicalize();


  assert(checkDual(ret));
  assert(ret.checkDual(*this));

  return ret;
}


PolyhedralCone PolyhedralCone::linealitySpace()const
{
  IntegerVectorList l1=getEquations();
  IntegerVectorList l2=getHalfSpaces();

  l1.splice(l1.begin(),l2);
  
  IntegerVectorList temp;
  PolyhedralCone ret(temp,l1,n);
  ret.ensureStateAsMinimum(state);
  return ret;
}


int PolyhedralCone::getMultiplicity()const
{
  return multiplicity;
}

void PolyhedralCone::setMultiplicity(int m)
{
  multiplicity=m;
}
