/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_I2C_C 1

#include "gatos.h"
#include "i2c.h"
#include "ati.h"
#include "m64regs.h"
#include "r128regs.h"
#include "tvout.h"

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>

/* Desired I2C Bus Clock Frequency for the Rage128 Hardware I2C Driver */
/* For some reason, it only works for f <= ~80kHz /AA */
#define R128_CLOCK_FREQ	80000.0	/* 7.5kHz .. 100kHz */

/* Private global vars */

/* Values used by high level routines */
static int active=0 ; static u8 R128_M, R128_N, r128_time ;
static char *i2cident[128], *i2cname[128] ;
static char* drivertype[MAXI2CDRIVER+1] = {
  "R128", "A", "B", "C", "LG", "TB" } ;
static char* drivername[MAXI2CDRIVER+1] = {
  "Rage 128 Hardware", "DAC+GEN_TEST Register", "GP_IO Register",
  "Rage PRO", "GP_IO Register", "ImpacTV" } ;

/* Function pointers used by low level functions */
static struct {
  void (*scldir)(int) ; void (*setscl)(int) ; int (*getscl)(void) ;
  void (*sdadir)(int) ; void (*setsda)(int) ; int (*getsda)(void) ; }
driver[MAXI2CDRIVER+1] = {
  { NULL, NULL, NULL, NULL, NULL, NULL },	/* Rage128 H/W Driver */
  { &reg_scldir, &reg_setscl, &reg_getscl,	/* A (DAC+GEN_TEST) */
    &reg_sdadir, &reg_setsda, &reg_getsda },
  { &reg_scldir, &reg_setscl, &reg_getscl,	/* B (GP_IO Register) */
    &reg_sdadir, &reg_setsda, &reg_getsda },
  { &pro_scldir, &pro_setscl, &pro_getscl,	/* C (Rage PRO) */
    &pro_sdadir, &pro_setsda, &pro_getsda },
  { &reg_scldir, &reg_setscl, &reg_getscl,	/* LG (GP_IO Register) */
    &reg_sdadir, &reg_setsda, &reg_getsda },
  { &itv_scldir, &itv_setscl, &itv_getscl,	/* TB (ImpacTV) */
    &itv_sdadir, &itv_setsda, &itv_getsda }},
*i2c=&driver[0] ;

/* Register pointers, bitmasks and values used by lowlevel routines */
static volatile u32 *sclreg=NULL, *sdareg=NULL ;
static u32 scldir=0, sdadir=0, sclset=0, sdaset=0, sdaget=0 ;
static u32 i2c_cntl_0=0 ;
static u8 tv_i2c_cntl=0x00 ;

/* ------------------------------------------------------------------------ */
/* Initialization routine */

int i2c_init(void) {

  /* Mach64 trymodes: Try all known drivers (bits 1..MAXI2CDRIVER all set) */
  int i, found=0, trymodes=(1<<(MAXI2CDRIVER+1))-2 ;

#ifndef USE_DELAY_S
  if (gatos.bogomips) i2c_ddelay = gatos.bogomips/8 ; else RETURN(EINVAL) ;
#endif

  /* Rage128: Hardware I2C driver only (I2C bus lines not available to S/W),
   * Mach64:  Software I2C driver (I2C buslines controlled by driver) */
  if (ati.r128) {	/* Rage128; only one driver type available */

    gatos.i2c_mode = 0 ; if (i2c_init_hw()) RETURN(ENODEV) ; }

  else {		/* Some Mach64 variant; figure out which ... */

    /* If i2c_mode not set in gatos.conf, choose
     * default based on ATI chip ID and BIOS data. */
    if (!gatos.i2c_mode) {
      if ( ati.vt_a3n1 || ati.vt_a3s1 || ati.gt_a2s1
        || ati.vt_a4n1 || ati.vt_a4s1 )		   gatos.i2c_mode = 1 ;
      else if ( ati.lg_a1s1 || ati.lt )		   gatos.i2c_mode = 4 ;
      else if ( ati.doesvtb || ati.gt_c1u1 ||
                ati.gt_c1u2 || ati.gt2c )	   gatos.i2c_mode = 2 ;
      else if (ati.lt_c)			   gatos.i2c_mode = 3 ;
      else if ( ati.gt_c2u1 || ati.gt_c2u2 )
              if ( ati.gp_c2u1 || ati.gq_c2u1 ||
                   ati.gp_c2u2 || ati.gq_c2u2 )    gatos.i2c_mode = 2 ;
              else				   gatos.i2c_mode = 3 ;
           else if ( ati.gp_c2u1 || ati.gq_c2u1 ||
                     ati.gp_c2u2 || ati.gq_c2u2 )  gatos.i2c_mode = 2 ;
                else				   gatos.i2c_mode = 3 ; }

    if (gatos.i2c_mode == 2 &&
        gatos.aticard[gatos.cardidx].m64i2cinfo != 0xFFFF)
      gatos.i2c_mode = 5 ;

    /* Try all known Mach64 I2C bus drivers, start with *
     * driver chosen by user or by "logic" above ...    */
    while (trymodes && !found) {
      while ((gatos.i2c_mode<MAXI2CDRIVER) && !(trymodes&(1<<gatos.i2c_mode)))
        gatos.i2c_mode++ ;
      trymodes &= ~(1<<gatos.i2c_mode) ;
      if ( ! (found = !i2c_init_hw()) ) {
        if (VERBOSE) fprintf(stderr,
          "I2C Type %s (i2c_mode %d) driver init failed%s\n",
            drivertype[gatos.i2c_mode],gatos.i2c_mode,(trymodes)
              ?", trying another ..."
              :".\nAll known I2C Bus Drivers failed, bailing out ...") ;
        gatos.i2c_mode = 0 ; if (!trymodes) RETURN(ENODEV) ; } } }

  /* Found an I2C bus driver that works :-) */
  if (VERBOSE) printf("Using Type %s (%s) I2C Bus Driver\n",
    drivertype[gatos.i2c_mode],drivername[gatos.i2c_mode]) ;

  /* No I2C units detected yet */
  for ( i=0 ; i<128 ; i++ ) { i2cident[i] = NULL ; i2cname[i] = NULL ; }
  RETURN0 ; }

/* Setup control structures for an I2C Bus Driver,
 * initialize registers and test if the driver works.
 * Any given card will have at most one working I2C Bus Driver. */
static int i2c_init_hw(void) {
  int ok ; u32 save1=0, save2=0, save3=0, save4=0 ; double nm ;
  i2c = &driver[gatos.i2c_mode] ;
  switch (gatos.i2c_mode) {
    case 0:
      nm = gatos.refclock * 1000000.0 / (4.0*R128_CLOCK_FREQ) ;
      for ( R128_N=1 ; R128_N<255 ; R128_N++ )
        if (R128_N*(R128_N-1) > nm) break ;
      R128_M = R128_N - 1 ; r128_time = 2*R128_N ;
      R128_I2C_CNTL_1   = (r128_time<<24) | I2C_SEL | I2C_EN ;
      R128_I2C_CNTL_0_0 = I2C_DONE | I2C_NACK | I2C_HALT | I2C_SOFT_RST |
                          I2C_DRIVE_EN | I2C_DRIVE_SEL ; break ;
    case 1:
      sclreg = DAC_CNTL_PTR ; sdareg = GEN_TEST_CNTL_PTR ;
      sclset = 0x01000000 ; sdaset = 0x00000001 ; sdaget = 0x00000008 ;
      scldir = 0x08000000 ; sdadir = 0x00000020 ;
      save1 = DAC_CNTL ; save2 = GEN_TEST_CNTL ; save3 = GP_IO ;
      *sdareg |= 0x00000010 ; save4 = CRTC_H_TOTAL_DISP ;
      GP_IO &= 0x7FFFFFFF ; CRTC_H_TOTAL_DISP = save4 ;
      /*save4 = VMC_CONFIG ; VMC_CONFIG &= 0x7FFFFFFF ;*/ break ;
    case 2:
      sclreg = sdareg = GP_IO_PTR ; save1 = GP_IO ;
      sclset = 0x00000800 ; sdaset = sdaget = 0x00000010 ;
      scldir = 0x08000000 ; sdadir = 0x00100000 ; break ;
    case 3:
      save1 = I2C_CNTL_0 ; save2 = I2C_CNTL_1 ;
      I2C_CNTL_1 = 0x00400000 ; if (I2C_CNTL_1!=0x00400000) RETURN(ENODEV) ;
      i2c_cntl_0 = 0x0000C000 ; I2C_CNTL_0 = i2c_cntl_0|0x00040000 ; break ;
    case 4:
      sclreg = sdareg = GP_IO_PTR ; save1 = GP_IO ;
      sclset = 0x00000400 ; sdaset = sdaget = 0x00001000 ;
      scldir = 0x04000000 ; sdadir = 0x10000000 ; break ;
    case 5:
      if (!tvout_present()) RETURN(ENODEV) ;
      tvout_write32(TV_I2C_CNTL,0x00005500|tv_i2c_cntl) ; break ;
    default: RETURN(ENODEV) ; }
  /* Try driver by polling for tuner modules */
  ok = i2c_device(0xC0) + i2c_device(0xC2) +
       i2c_device(0xC4) + i2c_device(0xC6) ;
  /* Success if 1-3 tuners detected */
  if ((ok != 0) && (ok != 4)) RETURN0 ;
  /* Driver failed: SCL and/or SDA line(s) stuck (low or high), or *
   * manipulated ATI chip pins not connected to on-card I2C bus.   */
  i2c = &driver[0] ;
  switch (gatos.i2c_mode) {
    case 0: R128_I2C_CNTL_1 = 0x00000000L ; break ;
    case 1: DAC_CNTL = save1 ; GEN_TEST_CNTL = save2 ;
            GP_IO = save3 ; /*VMC_CONFIG = save4 ;*/ break ;
    case 2: GP_IO = save1 ; break ;
    case 3: I2C_CNTL_0 = save1 ; I2C_CNTL_1 = save2 ; break ;
    case 4: GP_IO = save1 ; break ;
    case 5: break ;
    default: RETURN(ENODEV) ; }
  RETURN(ENODEV) ; }

/* ------------------------------------------------------------------------ */
/* High-level (API) routines */

u8 i2c_read(u8 addr) {
  int nack=0 ; u8 data ;
  if (ati.r128) {
    r128_reset() ; R128_I2C_DATA = addr|1 ;
    nack = r128_go(1,1,I2C_RECEIVE) ; data = (nack) ? 0xFF : R128_I2C_DATA ;
    if (I2CTRAFFIC) fprintf(stderr,"I2C: < %02X%c =%02X- >\n",
      addr, nack?'-':'+', data) ; }
  else {
    i2c_start() ; i2c_sendbyte(addr|1) ;
    data = i2c_readbyte(1) ; i2c_stop() ; }
  return data ; }

int i2c_readn(u8 addr, u8 *data, int count) {
  int nack=0 ;
  if (ati.r128) {
    r128_reset() ; R128_I2C_DATA = addr|1 ;
    nack = r128_go(1,count,I2C_RECEIVE) ;
    if (I2CTRAFFIC) fprintf(stderr,"I2C: < %02X%c", addr, nack?'-':'+') ;
    while (count > 0) {
      --count ; *data = (nack) ? 0xFF : R128_I2C_DATA ; ++data ;
      if (I2CTRAFFIC) fprintf(stderr," =%02X%c", *data, count?'+':'-') ; }
    if (I2CTRAFFIC) fprintf(stderr," >\n") ; }
  else {
    i2c_start() ; i2c_sendbyte(addr|1) ;
    while (count > 0) {
      --count ; *data = i2c_readbyte((count!=0)) ; ++data ; }
    i2c_stop() ; }
  return 0 ; }

int i2c_write(u8 addr, u8 *data, int count) {
  int nack=-1, c=count ;
  if (ati.r128) {
    r128_reset() ; R128_I2C_DATA = addr ;
    while (c > 0) { --c ; R128_I2C_DATA = *data ; ++data ; }
    data -= count ; nack = r128_go(1,count,0) ;
    if (I2CTRAFFIC) {
      fprintf(stderr,"I2C: < %02X%c", addr, nack?'-':'+') ;
      while (count > 0) { --count ;
        fprintf(stderr," %02X%c", *data++, (count&&!nack)?'+':'-') ; }
      fprintf(stderr," >\n") ; } }
  else {
    i2c_start() ; i2c_sendbyte(addr) ;
    while (count > 0) { nack = i2c_sendbyte(*data) ; ++data ; --count ; }
    i2c_stop() ; }
  return nack ? -1 : 0 ; }

u8 i2c_readreg8(u8 addr, u8 reg) {
  u8 data ;
  if (ati.r128) { i2c_write(addr,&reg,1) ; data = i2c_read(addr) ; }
  else {
    i2c_start() ; i2c_sendbyte(addr) ; i2c_sendbyte(reg) ;
    i2c_start() ; i2c_sendbyte(addr|1) ; data = i2c_readbyte(1) ;
    i2c_stop() ; }
  return data ; }

int i2c_writereg8(u8 addr, u8 reg, u8 value) {
  u8 data[2] ; data[0] = reg ; data[1] = value ;
  i2c_write(addr,data,2) ; return 0 ; }

int i2c_device(u8 addr) {
  int nack=0 ;
  if (ati.r128) {
    r128_reset() ; R128_I2C_DATA = addr|1 ; nack = r128_go(0,1,0) ;
    if (I2CTRAFFIC) fprintf(stderr,"I2C: < %02X%c >\n", addr, nack?'-':'+') ; }
  else { i2c_start() ; nack = i2c_sendbyte(addr) ; i2c_stop() ; }
  return !nack ; }

/* ------------------------------------------------------------------------ */
/* Low-level Routines for Mach64 Software Drivers */

#ifndef USE_DELAY_S
/* delay n microseconds */
void i2c_delay(u32 n) {
  while (n > 0) { volatile int x = i2c_ddelay ; while (x > 0) --x ; --n ; } }
#endif

static void i2c_start(void) {
  if (I2CTRAFFIC) fprintf(stderr,(active) ? " <" : "I2C: <") ;
  active = 1 ; i2c->scldir(1) ; i2c->sdadir(1) ;
  i2c->setsda(1) ; i2c->setscl(1) ; i2c->setsda(0) ; i2c->setscl(0) ; }

static void i2c_stop(void) {
  active = 0 ; i2c->sdadir(1) ;
  i2c->setscl(0) ; i2c->setsda(0) ; i2c->setscl(1) ; i2c->setsda(1) ;
  i2c->scldir(0) ; i2c->sdadir(0) ; if (I2CTRAFFIC) fprintf(stderr," >\n") ; }

static u8 i2c_readbyte(int last) {
  int i ; u8 data=0 ;
  i2c->sdadir(0) ;
  for ( i=7 ; i>=0 ; i-- ) {
    i2c->setscl(1) ; if (i2c->getsda()) data |= (1<<i) ;
    I2C_SLEEP ; i2c->setscl(0) ; }
  i2c->sdadir(1) ; i2c->setsda(last) ; i2c->setscl(1) ;
  i2c->setscl(0) ; i2c->sdadir(0) ;
  if (I2CTRAFFIC) fprintf(stderr," =%02X%c", data, last?'-':'+') ;
  return data ; }

static int i2c_sendbyte(u8 data) {
  int nack, i ;
  i2c->sdadir(1) ;
  for ( i=7 ; i>=0 ; i-- ) {
    i2c->setscl(0) ; i2c->setsda(data&(1<<i)) ; i2c->setscl(1) ; }
  i2c->setscl(0) ; i2c->sdadir(0) ; i2c->setsda(1) ; i2c->setscl(1) ;
  nack = (i2c->getsda()) ? 1 : 0 ; i2c->setscl(0) ;
  if (I2CTRAFFIC) fprintf(stderr," %02X%c", data, nack?'-':'+') ;
  return nack ; }

/* ------------------------------------------------------------------------ */
/* Hardware routines for I2C mode R128 (Rage 128 Hardware Driver) */

static int r128_wait_ack(void) {
  int nack=0, n1=0, n2=0 ;
  while (n1++ < 10 && (R128_I2C_CNTL_0_1 & (I2C_GO>>8))) usleep(1) ;
  while (n2++ < 10 && !nack) {
    nack = R128_I2C_CNTL_0_0 & (I2C_DONE|I2C_NACK|I2C_HALT) ;
    if (!nack) usleep(1) ; }
  return (nack != I2C_DONE) ; }

static void r128_reset(void) {
  R128_I2C_CNTL_0 = I2C_SOFT_RST | I2C_HALT | I2C_NACK | I2C_DONE ; }

static int r128_go(int naddr, int ndata, u32 flags) {
  R128_I2C_CNTL_1 = (r128_time<<24) | I2C_EN | I2C_SEL | (naddr<<8) | ndata ;
  R128_I2C_CNTL_0 = (R128_N<<24) | (R128_M<<16) | I2C_GO |
                    I2C_START | I2C_STOP | I2C_DRIVE_EN | flags ;
  return r128_wait_ack() ; }

/* ------------------------------------------------------------------------ */
/* Hardware routines for I2C modes A, B and LG (DAC+GEN_TEST or GP_IO) */

static void reg_scldir(int set) {
  if (set) *sclreg |= scldir ; else *sclreg &= ~scldir ; }

static void reg_sdadir(int set) {
  if (set) *sdareg |= sdadir ; else *sdareg &= ~sdadir ; }

static void reg_setscl(int set) {
  if (set) *sclreg |= sclset ; else *sclreg &= ~sclset ; I2C_SLEEP ; }

static void reg_setsda(int set) {
  if (set) *sdareg |= sdaset ; else *sdareg &= ~sdaset ; I2C_SLEEP ; }

static int reg_getscl(void) { return (*sclreg & sclset) ; }
static int reg_getsda(void) { return (*sdareg & sdaget) ; }

/* ------------------------------------------------------------------------ */
/* Hardware routines for I2C mode C (Rage PRO) */

static void pro_scldir(int set) { }
static void pro_sdadir(int set) { }

static void pro_setscl(int set) {
  if (set) i2c_cntl_0 |= 0x00004000 ; else i2c_cntl_0 &= ~0x00004000 ;
  I2C_CNTL_0 = i2c_cntl_0 ; I2C_SLEEP ; }

static void pro_setsda(int set) {
  if (set) i2c_cntl_0 |= 0x00008000 ; else i2c_cntl_0 &= ~0x00008000 ;
  I2C_CNTL_0 = i2c_cntl_0 ; I2C_SLEEP ; }

static int pro_getscl(void) { return (I2C_CNTL_0 & 0x00004000) ; }
static int pro_getsda(void) { return (I2C_CNTL_0 & 0x00008000) ; }

/* ------------------------------------------------------------------------ */
/* Hardware routines for I2C mode TB (ImpacTV) */

static void itv_scldir(int set) {
  if (set) tv_i2c_cntl |= 0x01 ; else tv_i2c_cntl &= ~0x01 ;
  tvout_write_i2c_cntl8(tv_i2c_cntl) ; I2C_SLEEP ; }

static void itv_sdadir(int set) {
  if (set) tv_i2c_cntl |= 0x10 ; else tv_i2c_cntl &= ~0x10 ;
  tvout_write_i2c_cntl8(tv_i2c_cntl) ; I2C_SLEEP ; }

static void itv_setsda(int set) {
  if (set) tv_i2c_cntl |= 0x20 ; else tv_i2c_cntl &= ~0x20 ;
  tvout_write_i2c_cntl8(tv_i2c_cntl) ; I2C_SLEEP ; }

static void itv_setscl(int set) {
  if (set) tv_i2c_cntl |= 0x02 ; else tv_i2c_cntl &= ~0x02 ;
  tvout_write_i2c_cntl8(tv_i2c_cntl) ; I2C_SLEEP ; }

static int itv_getscl(void) { return (tvout_read_i2c_cntl8() & 0x04) ; }
static int itv_getsda(void) { return (tvout_read_i2c_cntl8() & 0x40) ; }

/* ------------------------------------------------------------------------ */
/* Debug and info routines */

static volatile int N=0, N1=0, N2=0, loop=0 ;
static char str[12] ;

#define SECS	2

static void i2c_alarm(int sig) {
  if (loop) ((N1) ? N2 : N1) = N ; if (N2) loop = 0 ; }

char* i2c_ident(u8 addr) {
  if (i2cident[addr>>1]) return i2cident[addr>>1] ;
  snprintf(str,sizeof(str),"UNKN @ 0x%02X",addr) ; return str ; }

char* i2c_name(u8 addr) {
  if (i2cname[addr>>1]) return i2cname[addr>>1] ;
  snprintf(str,sizeof(str),"UNKN @ 0x%02X",addr) ; return str ; }

void i2c_info(void) {
  int i, d=gatos.debug ; struct sigaction act, oldact ; struct itimerval it ;
  gatos.debug &= ~4 ;
  for ( i=0 ; i<256 ; i+=2 )
    if (i2c_device(i))
      if (!i2cident[i>>1]) i2c_register(i,"UNKN","Unclaimed Device") ;
      else fprintf(stderr,"I2C unit @ 0x%02X: %s\n",i,i2cname[i>>1]) ;
    else if (i2cident[i>>1]) fprintf(stderr,
      "I2C unit @ 0x%02X: %s DISAPPEARED\n",i,i2cname[i>>1]) ;
  /* The Rage128 driver uses usleep(), which uses the itimer ... /AA */
  if (ati.r128) return ;
  fprintf(stderr,"I2C clock frequency is ... ") ;
  it.it_value.tv_sec  = 0 ;     it.it_interval.tv_sec  = SECS ;
  it.it_value.tv_usec = 10000 ; it.it_interval.tv_usec = 0 ;
  act.sa_flags = 0 ; 
#if defined(__linux__)
  act.sa_restorer = NULL ;
#endif
  sigemptyset(&act.sa_mask) ; act.sa_handler = &i2c_alarm ;
  sigaction(SIGVTALRM,&act,&oldact) ; setitimer(ITIMER_VIRTUAL,&it,NULL) ;
  N1 = N2 = 0 ; loop = 1 ; if (!ati.r128) i2c_start() ;
  for ( N=0 ; loop ; N++ )
    if (ati.r128) i2c_device(gatos.bt829.addr) ; else i2c_sendbyte(0xFF) ;
  it.it_value.tv_sec  = it.it_interval.tv_sec  = 0 ;
  it.it_value.tv_usec = it.it_interval.tv_usec = 0 ;
  setitimer(ITIMER_VIRTUAL,&it,NULL) ; sigaction(SIGVTALRM,&oldact,NULL) ;
  if (!ati.r128) i2c_stop() ; N = N2-N1 ;
  fprintf(stderr,"%.1f kHz\n", 9.0*N/(1000.0*SECS)) ; gatos.debug = d ; }

void i2c_register(u8 addr, char *ident, char *name) {
  if (!name) name = ident ;
  i2cident[addr>>1] = (char*) malloc(8+strlen(ident)) ;
  i2cname[addr>>1] =  (char*) malloc(1+strlen(name)) ;
  if (i2cident[addr>>1])
    snprintf(i2cident[addr>>1],8+strlen(ident),"%s @ 0x%02X",ident,addr) ;
  if (i2cname[addr>>1]) strcpy(i2cname[addr>>1],name) ;
  if (VERBOSE) printf("I2C unit @ 0x%02X: %s\n",addr,name) ; }
