/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/test_data_source.hpp
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::Test_Data_Source
 * interface
 *
 * $Id: test_data_source.hpp,v 1.12 2005/06/23 09:54:26 esdentem Exp $
 * 
 * @author Christian Schallhart
 *
 * @test Interface Compliance Test-Suite: diagnostics/unittest/test_system/test_data_source.ts.hpp
 */

#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_DATA_SOURCE_HPP__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__TEST_DATA_SOURCE_HPP__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>
#include <diagnostics/unittest/namespace.hpp>

// used in the interface by reference
#include <string>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

/**
 * @class Test_Data_Source diagnostics/unittest/test_system/test_data_source.hpp
 *
 * @brief a Test_Data_Source allows to maintain a map
 * ::std::string->::std::string as underlying database for a @ref
 * Test_Data implementation. However, the implementation details are
 * hidden and access is limited. 
 *
 * There are only three access methods: @ref exists_entry checks
 * whether an entry exists, @ref get_entry returns an existing entry
 * and @ref set_entry sets a new entry or modifies an existing one.
 *
 * @nosubgrouping
 */
class Test_Data_Source 
{
public:
    /**
     * @throw never
     */
    virtual ~Test_Data_Source();

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Accessors
     * @{
     */
public:
    /**
     * @brief returns true if the contained set of entries is in a
     * defined state. 
     *
     * @throw never
     */
    virtual bool is_initialized() const=0;

    /**
     * @brief true iff an entry with @a id exists.
     *
     * @pre is_initialized()==true
     *
     * @throw Test_System_Exception if is_initialized()==false
     */
    virtual bool exists_entry(::std::string const & id) const =0;

    /**
     * @brief retuns the entry with @a id 
     *
     * @pre is_initialized()==true
     * @pre exists_entry(@a id)==true
     *
     * @throw Test_System_Exception if a precondition is not met OR
     * the implementation is not able to serve the request for
     * whatever implementation-dependent reason (such as disk full)
     */
    virtual ::std::string const & get_entry(::std::string const & id) const =0;
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Modifiers
     * @{
     */
public:
    /**
     * @brief sets the string with @a id to @a value
     *
     * @pre is_initialized()==true
     *
     * @throw Test_System_Exception if the precondition is not met OR
     * the implementation is not able to serve the request for whatever
     * implementation-dependent reason (such as disk full)
     */
    virtual void set_entry(::std::string const & id, ::std::string const & value) =0;
    //@}
};

UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


#endif
// vim:ts=4:sw=4
