# Copyright (C) 2006 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""A plugin for converting from CVS to bzr using a cvsps changeset listing.

This program directly accesses the RCS files using 'co' to avoid the overhead
of cvs.
"""

import os

from bzrlib import (
    branch,
    bzrdir,
    osutils,
    repository,
    transport,
    ui,
    )


class ConversionUpgrader(object):
    """Convert an older format conversion, into the newer layout."""

    def __init__(self, output):
        self._output = output

        bzr_repo_path = osutils.pathjoin(output, 'bzr_repo')

        self._old_repo_path = osutils.pathjoin(output, 'bzr_repo')
        self._target_bzr_base = osutils.pathjoin(output, 'bzr')

        self._open_repos = {}
        self._old_repo = None

    def upgrade(self):
        if not os.path.exists(self._old_repo_path):
            print 'The "bzr_repo" directory does not exist at:'
            print '  %s' % (self._old_repo_path,)
            print 'Nothing to do.'
            return

        try:
            if not os.path.exists(self._target_bzr_base):
                os.mkdir(self._target_bzr_base)
            self._old_repo = repository.Repository.open(self._old_repo_path)
            self._old_repo.lock_read()

            branches = self._find_all_old_branches()
            print 'found %d branches' % (len(branches),)
            self._copy_all_branches(branches)
        finally:
            self._close_open_repos()

    def _close_open_repos(self):
        if self._old_repo is not None:
            self._old_repo.unlock()

        for path, repo in self._open_repos.iteritems():
            repo.unlock()

    def _find_all_old_branches(self):
        """Traverse the tree, finding all old location branches."""
        branches = []

        pb = ui.ui_factory.nested_progress_bar()
        try:
            pb.update('finding branches', 0, 1)
            for (dir_relpath, dir_abspath), dirblock in \
                    osutils.walkdirs(self._old_repo_path):
                paths = dict((info[1], offset) for offset, info in enumerate(dirblock))
                if '.bzr' not in paths:
                    continue
                # Don't recurse into the .bzr directories
                del dirblock[paths['.bzr']]

                if dir_relpath == '':
                    continue

                # Not a branch directory
                if '.' not in dir_relpath:
                    continue
                 
                module, branch_name = dir_relpath.rsplit('.', 1)
                pb.update('found branch: %s %s' % (module, branch_name),
                          len(branches), len(branches))
                branches.append((module, branch_name, dir_abspath))
        finally:
            pb.finished()
        return branches

    def _copy_all_branches(self, branches):
        pb = ui.ui_factory.nested_progress_bar()
        try:
            n_branches = len(branches)
            for count, (module, branch_name, path) in enumerate(branches):
                pb.update('processing %-20.20s %-20.20s' % (module,branch_name), count, n_branches)

                source_branch = branch.Branch.open(path)
                source_branch.lock_read()
                try:
                    # Copy the data into the target repository
                    target_repo = self._get_target_repo(module)
                    last_rev = source_branch.last_revision()
                    if last_rev is not None:
                        target_repo.fetch(self._old_repo,
                                          revision_id=last_rev)

                    self._ensure_target_branch(module, branch_name,
                                               source_branch, target_repo)
                finally:
                    source_branch.unlock()
        finally:
            pb.finished()

    def _get_target_repo(self, module):
        # Have we opened it already?
        if module in self._open_repos:
            return self._open_repos[module]

        # Has it already been created? if so, open it, else create it
        target_module_path = osutils.pathjoin(self._target_bzr_base, module)
        if os.path.isdir(target_module_path):
            repo = repository.Repository.open(target_module_path)
        else:
            os.makedirs(target_module_path)
            os.mkdir(osutils.pathjoin(target_module_path, 'branches'))
            a_transport = transport.get_transport(target_module_path)

            format = bzrdir.BzrDirFormat.get_default_format()
            newdir = format.initialize_on_transport(a_transport)
            repo = newdir.create_repository(shared=True)
            repo.set_make_working_trees(False)

        repo.lock_write()
        self._open_repos[module] = repo
        return repo

    def _ensure_target_branch(self, module, branch_name, source_branch,
                              target_repo):
        """Make sure the target branch looks like the source branch."""
        target_branch_path = osutils.pathjoin(self._target_bzr_base, module,
                                              'branches', branch_name)
        if not os.path.isdir(target_branch_path):
            target_branch = bzrdir.BzrDir.create_branch_convenience(
                target_branch_path, force_new_tree=False)
        else:
            target_branch = branch.Branch.open(target_branch_path)

        # XXX: Quit peeking at the internals
        assert target_branch.repository.bzrdir.transport.base == target_repo.bzrdir.transport.base
        target_branch.repository = target_repo

        target_branch.lock_write()
        try:
            if target_branch.last_revision() == source_branch.last_revision():
                return

            source_branch.copy_content_into(target_branch)
            if branch_name == 'HEAD':
                if target_branch.get_parent() is not None:
                    target_branch.set_parent(None)
            else:
                head_branch_path = osutils.pathjoin(self._target_bzr_base, module,
                                                    'branches', 'HEAD')
                target_branch.set_parent(head_branch_path)
        finally:
            target_branch.unlock()
