# Copyright (C) 2006 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Error classes for cvsps."""

from bzrlib import errors


class InvalidMapFile(errors.BzrError):
    """Raised when there is a parsing error for the map file."""

    _fmt = "The map file %(fname)s is invalid: %(msg)s"

    def __init__(self, fname, msg=None):
        errors.BzrError.__init__(self)
        self.fname = fname
        self.msg = msg or ''


class InvalidMapModule(InvalidMapFile):
    """Raised when the module doesn't match the expected module."""

    _fmt = ("The map file %(fname)s points to a different cvs module."
            " Expected %(exp_module)s got %(act_module)s"
           )

    def __init__(self, fname, exp_module, act_module):
        InvalidMapFile.__init__(self, fname, msg='invalid module')
        self.exp_module = exp_module
        self.act_module = act_module


class CVSPSError(errors.BzrError):
    """Raised when cvsps indicates there was an error."""

    _fmt = ("cvsps did not return successfully. return code: %(returncode)s"
            " for command %(cmd)s"
           )

    def __init__(self, returncode, cmd):
        errors.BzrError.__init__(self)
        self.returncode = returncode
        self.cmd = ' '.join(cmd)


class COError(errors.BzrError):
    """Raised when 'co' indicates there was a problem.

    Pass in the return code, command that was run, and the output from stderr.
    """

    _fmt = ("co did not return successfully. return code: %(returncode)s"
            " for command %(cmd)s\n"
            "Error: %(error)s\n"
           )

    def __init__(self, returncode, cmd, error):
        errors.BzrError.__init__(self)
        self.returncode = returncode
        self.cmd = ' '.join(cmd)
        self.error = error


class CVSError(errors.BzrError):
    """Raised when 'cvs' indicates there was a problem.

    Pass in the return code, command that was run, and the output from stderr.
    """

    _fmt = ("cvs did not return successfully. return code: %(returncode)s"
            " for command %(cmd)s\n"
            "Error: %(error)s\n"
           )

    def __init__(self, returncode, cmd, error):
        errors.BzrError.__init__(self)
        self.returncode = returncode
        self.cmd = ' '.join(cmd)
        self.error = error


class NoRCSFile(errors.BzrError):
    """Raised when we can't find the corresponding ',v' file."""

    _fmt = ("Could not find the cvs versioned file for %(source)s."
            " Looking for it at %(versioned)s and %(versioned_attic)s."
           )

    def __init__(self, source, versioned, versioned_attic):
        errors.BzrError.__init__(self)
        self.source = source
        self.versioned = versioned
        self.versioned_attic = versioned_attic


class MissingProgram(errors.BzrError):
    """Raised when we can't run an external program."""

    _fmt = ('Could not run command: "%(command_str)s"'
            '\ndo you have %(program)s installed?'
            '\n%(error)s'
           )

    def __init__(self, command, error, program=None):
        errors.BzrError.__init__(self)
        self.command = command
        if not program:
            self.program = command[0]
        else:
            self.program = program
        self.command_str = ' '.join(command)
        self.error = error
