// Copyright (c) 2006, Rodrigo Braz Monteiro
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of the Aegisub Group nor the names of its contributors
//     may be used to endorse or promote products derived from this software
//     without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// Aegisub Project http://www.aegisub.org/

/// @file video_provider_manager.cpp
/// @brief Keep track of installed video providers
/// @ingroup video_input
///

#include "config.h"

#include "video_provider_manager.h"

#include "options.h"
#include "video_provider_avs.h"
#include "video_provider_cache.h"
#include "video_provider_dummy.h"
#include "video_provider_ffmpegsource.h"
#include "video_provider_yuv4mpeg.h"

#include <libaegisub/fs.h>
#include <libaegisub/log.h>
#include <libaegisub/util.h>

std::unique_ptr<VideoProvider> VideoProviderFactory::GetProvider(agi::fs::path const& video_file, std::string const& colormatrix) {
	std::vector<std::string> factories = GetClasses(OPT_GET("Video/Provider")->GetString());
	factories.insert(factories.begin(), "YUV4MPEG");
	factories.insert(factories.begin(), "Dummy");

	bool found = false;
	bool supported = false;
	std::string errors;
	errors.reserve(1024);

	for (auto const& factory : factories) {
		std::string err;
		try {
			auto provider = Create(factory, video_file, colormatrix);
			LOG_I("manager/video/provider") << factory << ": opened " << video_file;
			return provider->WantsCaching() ? agi::util::make_unique<VideoProviderCache>(std::move(provider)) : std::move(provider);
		}
		catch (agi::fs::FileNotFound const&) {
			err = "file not found.";
			// Keep trying other providers as this one may just not be able to
			// open a valid path
		}
		catch (VideoNotSupported const&) {
			found = true;
			err = "video is not in a supported format.";
		}
		catch (VideoOpenError const& ex) {
			supported = true;
			err = ex.GetMessage();
		}
		catch (agi::vfr::Error const& ex) {
			supported = true;
			err = ex.GetMessage();
		}

		errors += factory + ": " + err + "\n";
		LOG_D("manager/video/provider") << factory << ": " << err;
	}

	// No provider could open the file
	LOG_E("manager/video/provider") << "Could not open " << video_file;
	std::string msg = "Could not open " + video_file.string() + ":\n" + errors;

	if (!found) throw agi::fs::FileNotFound(video_file.string());
	if (!supported) throw VideoNotSupported(msg);
	throw VideoOpenError(msg);
}

void VideoProviderFactory::RegisterProviders() {
#ifdef WITH_AVISYNTH
	Register<AvisynthVideoProvider>("Avisynth");
#endif
#ifdef WITH_FFMS2
	Register<FFmpegSourceVideoProvider>("FFmpegSource");
#endif
	Register<DummyVideoProvider>("Dummy", true);
	Register<YUV4MPEGVideoProvider>("YUV4MPEG", true);
}
