# Code quality metrics.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/token"
require "adlint/code"
require "adlint/util"

module AdLint #:nodoc:

  class CodeMetric
    def to_a
      subclass_responsibility
    end

    def to_s
      delim = ",".to_default_external
      to_a.map { |obj| obj.to_s.to_default_external }.join(delim)
    end

    def to_csv
      to_a.map { |obj| obj ? obj.to_s.to_default_external : nil }.to_csv
    end
  end

  class FL_STMT_Metric < CodeMetric
    def initialize(fpath, statement_count)
      @fpath = fpath
      @statement_count = statement_count
    end

    def to_a
      ["MET", "FL_STMT", @fpath, @statement_count]
    end
  end

  class FL_FUNC_Metric < CodeMetric
    def initialize(fpath, function_count)
      @fpath = fpath
      @function_count = function_count
    end

    def to_a
      ["MET", "FL_FUNC", @fpath, @function_count]
    end
  end

  class FN_STMT_Metric < CodeMetric
    def initialize(function_identifier, location, statement_count)
      @function_identifier = function_identifier
      @location = location
      @statement_count = statement_count
    end

    def to_a
      ["MET", "FN_STMT",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @statement_count]
    end
  end

  class FN_UNRC_Metric < CodeMetric
    def initialize(function_identifier, location, unreached_statement_count)
      @function_identifier = function_identifier
      @location = location
      @unreached_statement_count = unreached_statement_count
    end

    def to_a
      ["MET", "FN_UNRC",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @unreached_statement_count]
    end
  end

  class FN_LINE_Metric < CodeMetric
    def initialize(function_identifier, location, total_lines)
      @function_identifier = function_identifier
      @location = location
      @total_lines = total_lines
    end

    def to_a
      ["MET", "FN_LINE",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @total_lines]
    end
  end

  class FN_PARA_Metric < CodeMetric
    def initialize(function_identifier, location, parameter_count)
      @function_identifier = function_identifier
      @location = location
      @parameter_count = parameter_count
    end

    def to_a
      ["MET", "FN_PARA",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @parameter_count]
    end
  end

  class FN_UNUV_Metric < CodeMetric
    def initialize(function_identifier, location, useless_variable_count)
      @function_identifier = function_identifier
      @location = location
      @useless_variable_count = useless_variable_count
    end

    def to_a
      ["MET", "FN_UNUV",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @useless_variable_count]
    end
  end

  class FN_CSUB_Metric < CodeMetric
    def initialize(function_identifier, location, function_call_count)
      @function_identifier = function_identifier
      @location = location
      @function_call_count = function_call_count
    end

    def to_a
      ["MET", "FN_CSUB",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @function_call_count]
    end
  end

  class FN_GOTO_Metric < CodeMetric
    def initialize(function_identifier, location, goto_count)
      @function_identifier = function_identifier
      @location = location
      @goto_count = goto_count
    end

    def to_a
      ["MET", "FN_GOTO",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @goto_count]
    end
  end

  class FN_RETN_Metric < CodeMetric
    def initialize(function_identifier, location, return_count)
      @function_identifier = function_identifier
      @location = location
      @return_count = return_count
    end

    def to_a
      ["MET", "FN_RETN",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @return_count]
    end
  end

  class FN_UELS_Metric < CodeMetric
    def initialize(function_identifier, location, if_statement_count)
      @function_identifier = function_identifier
      @location = location
      @if_statement_count = if_statement_count
    end

    def to_a
      ["MET", "FN_UELS",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @if_statement_count]
    end
  end

  class FN_NEST_Metric < CodeMetric
    def initialize(function_identifier, location, max_nest_level)
      @function_identifier = function_identifier
      @location = location
      @max_nest_level = max_nest_level
    end

    def to_a
      ["MET", "FN_NEST",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @max_nest_level]
    end
  end

  class FN_PATH_Metric < CodeMetric
    def initialize(function_identifier, location, path_count)
      @function_identifier = function_identifier
      @location = location
      @path_count = path_count
    end

    def to_a
      ["MET", "FN_PATH",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @path_count]
    end
  end

  class FN_CYCM_Metric < CodeMetric
    def initialize(function_identifier, location, cyclomatic_complexity)
      @function_identifier = function_identifier
      @location = location
      @cyclomatic_complexity = cyclomatic_complexity
    end

    def to_a
      ["MET", "FN_CYCM",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @cyclomatic_complexity]
    end
  end

  class FN_CALL_Metric < CodeMetric
    def initialize(function_identifier, location, caller_count)
      @function_identifier = function_identifier
      @location = location
      @caller_count = caller_count
    end

    def to_a
      ["MET", "FN_CALL",
        @function_identifier.name, @function_identifier.signature,
        @location.fpath, @location.line_no, @location.column_no,
        @caller_count]
    end
  end

  # == Base of metric measurement classes.
  class MetricMeasurement
    def initialize(context)
      @context = context
    end

    def execute
      do_prepare(@context)
      do_execute(@context)
    end

    private
    def do_prepare(context)
      subclass_responsibility
    end

    def do_execute(context)
      subclass_responsibility
    end

    def report
      @context.report
    end
  end

  class MetricRecord < CsvRecord
    def self.of(csv_row)
      case csv_row[0]
      when "VER"
        create_version_record(csv_row)
      when "DCL"
        create_declaration_record(csv_row)
      when "DEF"
        create_definition_record(csv_row)
      when "INI"
        create_initialization_record(csv_row)
      when "ASN"
        create_assignment_record(csv_row)
      when "DEP"
        create_dependency_record(csv_row)
      when "LIT"
        create_literal_record(csv_row)
      when "PRE"
        create_pp_directive_record(csv_row)
      when "MET"
        create_metric_record(csv_row)
      else
        raise "invalid metric record."
      end
    end

    def type
      field_at(0)
    end

    def fpath
      Pathname.new(field_at(1))
    end

    def line_no
      field_at(2)
    end

    def column_no
      field_at(3)
    end

    def location
      Location.new(fpath, line_no, column_no)
    end

    def version?; false end

    def typedef_declaration?; false end

    def struct_declaration?; false end

    def union_declaration?; false end

    def enum_declaration?; false end

    def global_variable_declaration?; false end

    def function_declaration?; false end

    def variable_definition?; false end

    def function_definition?; false end

    def macro_definition?; false end

    def label_definition?; false end

    def initialization?; false end

    def assignment?; false end

    def include?; false end

    def function_call?; false end

    def variable_xref?; false end

    def function_xref?; false end

    def literal?; false end

    def pp_directive?; false end

    def self.create_version_record(csv_row)
      VersionRecord.new(csv_row)
    end
    private_class_method :create_version_record

    def self.create_declaration_record(csv_row)
      case csv_row[4]
      when "T"
        case csv_row[5]
        when "T"
          TypedefDeclarationRecord.new(csv_row)
        when "S"
          StructDeclarationRecord.new(csv_row)
        when "U"
          UnionDeclarationRecord.new(csv_row)
        when "E"
          EnumDeclarationRecord.new(csv_row)
        else
          raise "invalid DCL record."
        end
      when "V"
        GlobalVariableDeclarationRecord.new(csv_row)
      when "F"
        FunctionDeclarationRecord.new(csv_row)
      else
        raise "invalid DCL record."
      end
    end
    private_class_method :create_declaration_record

    def self.create_definition_record(csv_row)
      case csv_row[4]
      when "V"
        VariableDefinitionRecord.new(csv_row)
      when "F"
        FunctionDefinitionRecord.new(csv_row)
      when "M"
        MacroDefinitionRecord.new(csv_row)
      when "L"
        LabelDefinitionRecord.new(csv_row)
      else
        raise "invalid DEF record."
      end
    end
    private_class_method :create_definition_record

    def self.create_initialization_record(csv_row)
      InitializationRecord.new(csv_row)
    end
    private_class_method :create_initialization_record

    def self.create_assignment_record(csv_row)
      AssignmentRecord.new(csv_row)
    end
    private_class_method :create_assignment_record

    def self.create_dependency_record(csv_row)
      case csv_row[4]
      when "I"
        IncludeRecord.new(csv_row)
      when "C"
        FunctionCallRecord.new(csv_row)
      when "X"
        case csv_row[5]
        when "V"
          VariableXRefRecord.new(csv_row)
        when "F"
          FunctionXRefRecord.new(csv_row)
        else
          raise "invalid DEP record."
        end
      else
        raise "invalid DEP record."
      end
    end
    private_class_method :create_dependency_record

    def self.create_literal_record(csv_row)
      LiteralRecord.new(csv_row)
    end
    private_class_method :create_literal_record

    def self.create_pp_directive_record(csv_row)
      PPDirectiveRecord.new(csv_row)
    end
    private_class_method :create_pp_directive_record

    def self.create_metric_record(csv_row)
      MetricRecord.new(csv_row)
    end
    private_class_method :create_metric_record

    class VersionRecord < MetricRecord
      def version?
        true
      end

      def version_number
        field_at(1)
      end

      def exec_timestamp
        field_at(2)
      end
    end
    private_constant :VersionRecord

    class TypedefDeclarationRecord < MetricRecord
      def typedef_declaration?
        true
      end

      def typedcl_type
        field_at(5)
      end

      def type_name
        field_at(6)
      end

      def type_rep
        field_at(7)
      end
    end
    private_constant :TypedefDeclarationRecord

    class StructDeclarationRecord < MetricRecord
      def struct_declaration?
        true
      end

      def typedcl_type
        field_at(5)
      end

      def type_name
        field_at(6)
      end

      def type_rep
        field_at(7)
      end
    end
    private_constant :StructDeclarationRecord

    class UnionDeclarationRecord < MetricRecord
      def union_declaration?
        true
      end

      def typedcl_type
        field_at(5)
      end

      def type_name
        field_at(6)
      end

      def type_rep
        field_at(7)
      end
    end
    private_constant :UnionDeclarationRecord

    class EnumDeclarationRecord < MetricRecord
      def enum_declaration?
        true
      end

      def typedcl_type
        field_at(5)
      end

      def type_name
        field_at(6)
      end

      def type_rep
        field_at(7)
      end
    end
    private_constant :EnumDeclarationRecord

    class GlobalVariableDeclarationRecord < MetricRecord
      def global_variable_declaration?
        true
      end

      def variable_name
        field_at(5)
      end

      def type_rep
        field_at(6)
      end
    end
    private_constant :GlobalVariableDeclarationRecord

    class FunctionDeclarationRecord < MetricRecord
      def function_declaration?
        true
      end

      def function_linkage_type
        field_at(5)
      end

      def function_scope_type
        field_at(6)
      end

      def function_identifier
        FunctionIdentifier.new(field_at(7), field_at(8))
      end
    end
    private_constant :FunctionDeclarationRecord

    class VariableDefinitionRecord < MetricRecord
      def variable_definition?
        true
      end

      def var_linkage_type
        field_at(5)
      end

      def var_scope_type
        field_at(6)
      end

      def storage_class_type
        field_at(7)
      end

      def variable_name
        field_at(8)
      end

      def type_rep
        field_at(9)
      end
    end
    private_constant :VariableDefinitionRecord

    class FunctionDefinitionRecord < MetricRecord
      def function_definition?
        true
      end

      def function_linkage_type
        field_at(5)
      end

      def function_scope_type
        field_at(6)
      end

      def function_identifier
        FunctionIdentifier.new(field_at(7), field_at(8))
      end

      def lines
        field_at(9)
      end
    end
    private_constant :FunctionDefinitionRecord

    class MacroDefinitionRecord < MetricRecord
      def macro_definition?
        true
      end

      def macro_name
        field_at(5)
      end

      def macro_type
        field_at(6)
      end
    end
    private_constant :MacroDefinitionRecord

    class LabelDefinitionRecord < MetricRecord
      def label_definition?
        true
      end

      def label_name
        field_at(5)
      end
    end
    private_constant :LabelDefinitionRecord

    class InitializationRecord < MetricRecord
      def initialization?
        true
      end

      def variable_name
        field_at(4)
      end

      def initializer_rep
        field_at(5)
      end
    end
    private_constant :InitializationRecord

    class AssignmentRecord < MetricRecord
      def assignment?
        true
      end

      def variable_name
        field_at(4)
      end

      def assignment_rep
        field_at(5)
      end
    end
    private_constant :AssignmentRecord

    class IncludeRecord < MetricRecord
      def include?
        true
      end

      def included_fpath
        Pathname.new(field_at(5))
      end
    end
    private_constant :IncludeRecord

    class FunctionCallRecord < MetricRecord
      def function_call?
        true
      end

      def caller_function
        FunctionIdentifier.new(field_at(5), field_at(6))
      end

      def callee_function
        FunctionIdentifier.new(field_at(7), field_at(8))
      end
    end
    private_constant :FunctionCallRecord

    class VariableXRefRecord < MetricRecord
      def variable_xref?
        true
      end

      def accessor_function
        FunctionIdentifier.new(field_at(6), field_at(7))
      end

      def access_type
        field_at(8)
      end

      def accessee_variable
        field_at(9)
      end
    end
    private_constant :VariableXRefRecord

    class FunctionXRefRecord < MetricRecord
      def function_xref?
        true
      end

      def accessor_function
        FunctionIdentifier.new(field_at(6), field_at(7))
      end

      def access_type
        field_at(8)
      end

      def accessee_function
        FunctionIdentifier.new(field_at(9), field_at(10))
      end
    end
    private_constant :FunctionXRefRecord

    class LiteralRecord < MetricRecord
      def literal?
        true
      end

      def literal_type
        field_at(4)
      end

      def literal_prefix
        field_at(5)
      end

      def literal_suffix
        field_at(6)
      end

      def literal_value
        field_at(7)
      end
    end
    private_constant :LiteralRecord

    class PPDirectiveRecord < MetricRecord
      def pp_directive?
        true
      end

      def pp_directive
        field_at(4)
      end

      def pp_tokens
        field_at(5)
      end
    end
    private_constant :PPDirectiveRecord
  end

end
