# Message detection classes for C preprocessor language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/report"
require "adlint/message"
require "adlint/cpp/syntax"

module AdLint #:nodoc:
module Cpp #:nodoc:

  class W0001 < PassiveMessageDetection
    # NOTE: W0001 may be duplicative when the same header which has the deeply
    #       grouped expression is included twice or more.
    ensure_uniqueness_of :W0001

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_grouped_expression += method(:enter_grouped_expression)
      visitor.leave_grouped_expression += method(:leave_grouped_expression)
      @group_depth = 0
    end

    private
    def enter_grouped_expression(node)
      @group_depth += 1
      W(:W0001, node.location) if @group_depth == 32
    end

    def leave_grouped_expression(node)
      @group_depth -= 1
    end
  end

  class W0025 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_user_header_included += method(:check_user_include)
      interp.on_system_header_included += method(:check_system_include)
      @user_header_fpaths = Set.new
      @system_header_fpaths = Set.new
    end

    private
    def check_user_include(user_include_line, user_header)
      if @user_header_fpaths.include?(user_header.fpath)
        W(:W0025, user_include_line.location)
      else
        if user_include_line.include_depth == 1
          @user_header_fpaths.add(user_header.fpath)
        end
      end
    end

    def check_system_include(system_include_line, system_header)
      if @system_header_fpaths.include?(system_header.fpath)
        W(:W0025, system_include_line.location)
      else
        if system_include_line.include_depth == 1
          @system_header_fpaths.add(system_header.fpath)
        end
      end
    end
  end

  class W0026 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_user_header_included += method(:check_user_include)
      interp.on_system_header_included += method(:check_system_include)
      @user_header_fpaths = Set.new
      @system_header_fpaths = Set.new
    end

    private
    def check_user_include(user_include_line, user_header)
      if @user_header_fpaths.include?(user_header.fpath)
        W(:W0026, user_include_line.location)
      else
        if user_include_line.include_depth > 1
          @user_header_fpaths.add(user_header.fpath)
        end
      end
    end

    def check_system_include(system_include_line, system_header)
      if @system_header_fpaths.include?(system_header.fpath)
        W(:W0026, system_include_line.location)
      else
        if system_include_line.include_depth > 1
          @system_header_fpaths.add(system_header.fpath)
        end
      end
    end
  end

  class W0053 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_user_header_included += method(:check_user_include)
    end

    private
    def check_user_include(user_include_line, user_header)
      if user_include_line.include_depth == 7
        W(:W0053, user_include_line.location, user_header.fpath)
      end
    end
  end

  class W0054 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_if_section += method(:enter_if_section)
      visitor.leave_if_section += method(:leave_if_section)
      @if_depth = 0
    end

    private
    def enter_if_section(node)
      @if_depth += 1
      W(:W0054, node.location) if @if_depth == 9
    end

    def leave_if_section(node)
      @if_depth -= 1
    end
  end

  class W0055 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_object_like_macro_defined += method(:check)
      interp.on_function_like_macro_defined += method(:check)
      interp.on_va_function_like_macro_defined += method(:check)
      @macro_num = 0
    end

    private
    def check(define_line, macro)
      unless in_initial_header?(define_line)
        @macro_num += 1
        W(:W0055, define_line.location) if @macro_num == 1025
      end
    end

    def in_initial_header?(node)
      case
      when pinit_fpath = Traits.instance.of_project.initial_header
        File.identical?(node.location.fpath, pinit_fpath)
      when cinit_fpath = Traits.instance.of_compiler.initial_header
        File.identical?(node.location.fpath, cinit_fpath)
      else
        false
      end
    end
  end

  class W0056 < PassiveMessageDetection
    # NOTE: W0056 may be duplicative when the same header which has macro
    #       definition with too many parameters is included twice or more.
    ensure_uniqueness_of :W0056

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_function_like_macro_defined += method(:check)
      interp.on_va_function_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      if macro.parameter_names.size > 31
        W(:W0056, define_line.location)
      end
    end
  end

  class W0057 < PassiveMessageDetection
    # NOTE: W0057 may be duplicative when the same header which has macro call
    #       with too many arguments is included twice or more.
    ensure_uniqueness_of :W0057

    def initialize(context)
      super
      macro_table = context[:cpp_macro_table]
      macro_table.on_function_like_macro_replacement += method(:check)
    end

    private
    def check(macro, tokens)
      W(:W0057, tokens.first.location) if macro.parameter_names.size > 31
    end
  end

  class W0059 < PassiveMessageDetection
    # NOTE: W0059 may be duplicative when the same header which has carriage
    #       return at end of lines is included twice or more.
    ensure_uniqueness_of :W0059

    def initialize(context)
      super
      context[:cpp_interpreter].on_cr_at_eol_found +=
        lambda { |location| W(:W0059, location) }
    end
  end

  class W0060 < PassiveMessageDetection
    # NOTE: W0060 may be duplicative when the same header which has the
    #       end-of-file mark is included twice or more.
    ensure_uniqueness_of :W0060

    def initialize(context)
      super
      context[:cpp_interpreter].on_eof_mark_at_eof_found +=
        lambda { |location| W(:W0060, location) }
    end
  end

  class W0061 < PassiveMessageDetection
    # NOTE: W0061 may be duplicative when the same header which has the token
    #       sequence of the compiler specific extension is included twice or
    #       more.
    ensure_uniqueness_of :W0061

    def initialize(context)
      super
      context[:c_source].on_language_extension += method(:check)
    end

    private
    def check(matched_tokens)
      W(:W0061, matched_tokens.first.location, tokens_to_str(matched_tokens))
    end

    def tokens_to_str(tokens)
      tokens.map { |token|
        token.type == :NEW_LINE ? nil : token.value
      }.compact.join(" ")
    end
  end

  class W0069 < PassiveMessageDetection
    # NOTE: W0069 may be duplicative when the same header which has the nested
    #       block comment is included twice or more.
    ensure_uniqueness_of :W0069

    def initialize(context)
      super
      context[:cpp_interpreter].on_nested_block_comment_found +=
        lambda { |location| W(:W0069, location) }
    end
  end

  class W0072 < PassiveMessageDetection
    # NOTE: W0072 may be duplicative when the same header which has non
    #       basic-source-character in the #include directive is included twice
    #       or more.
    ensure_uniqueness_of :W0072

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_user_include_line += method(:check_user_include_line)
      visitor.enter_system_include_line += method(:check_system_include_line)
    end

    private
    def check_user_include_line(node)
      unless BasicSourceCharacterSet.include?(node.header_name.value)
        W(:W0072, node.header_name.location)
      end
    end

    def check_system_include_line(node)
      unless BasicSourceCharacterSet.include?(node.header_name.value)
        W(:W0072, node.header_name.location)
      end
    end
  end

  class W0073 < PassiveMessageDetection
    # NOTE: W0073 may be duplicative when the same header without the
    #       include-guard is included twice or more.
    ensure_uniqueness_of :W0073

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_user_include_line += method(:enter_include_line)
      visitor.enter_system_include_line += method(:enter_include_line)
      visitor.enter_text_line += method(:enter_text_line)
      visitor.enter_if_section += method(:enter_if_section)
      @main_fpath = context[:sources].first.fpath
      @last_fpath = nil
      @enclosed_by_if_section = false
      @warned_files = Set.new
    end

    private
    def enter_include_line(node)
      unless node.location.fpath == @last_fpath
        @enclosed_by_if_section = false
        @last_fpath = node.location.fpath
      end

      return if in_initial_header?(node) || in_main_file?(node)

      unless @enclosed_by_if_section || in_warned_file?(node)
        W(:W0073, node.location)
        @warned_files.add(node.location.fpath)
      end
    end

    def enter_text_line(node)
      unless node.location.fpath == @last_fpath
        @enclosed_by_if_section = false
        @last_fpath = node.location.fpath
      end

      if empty_line?(node) || in_initial_header?(node) || in_main_file?(node)
        return
      end

      unless @enclosed_by_if_section || in_warned_file?(node)
        W(:W0073, node.location)
        @warned_files.add(node.location.fpath)
      end
    end

    def enter_if_section(node)
      @enclosed_by_if_section = true
      @last_fpath = node.location.fpath
    end

    def empty_line?(node)
      node.token.value.chomp.empty?
    end

    def in_main_file?(node)
      File.identical?(node.location.fpath, @main_fpath)
    end

    def in_initial_header?(node)
      case
      when pinit_fpath = Traits.instance.of_project.initial_header
        File.identical?(node.location.fpath, pinit_fpath)
      when cinit_fpath = Traits.instance.of_compiler.initial_header
        File.identical?(node.location.fpath, cinit_fpath)
      else
        false
      end
    end

    def in_warned_file?(node)
      @warned_files.include?(node.location.fpath)
    end
  end

  class W0442 < PassiveMessageDetection
    # NOTE: W0442 may be duplicative when the same header which has
    #       function-like macro definitions is included twice or more.
    ensure_uniqueness_of :W0442

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_function_like_macro_defined += method(:check)
      interp.on_va_function_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      W(:W0442, define_line.location) unless in_initial_header?(define_line)
    end

    def in_initial_header?(node)
      case
      when pinit_fpath = Traits.instance.of_project.initial_header
        File.identical?(node.location.fpath, pinit_fpath)
      when cinit_fpath = Traits.instance.of_compiler.initial_header
        File.identical?(node.location.fpath, cinit_fpath)
      else
        false
      end
    end
  end

  class W0443 < PassiveMessageDetection
    # NOTE: W0443 may be duplicative when the same header which has
    #       functionizable function-like macro definitions is included twice or
    #       more.
    ensure_uniqueness_of :W0443

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_function_like_define_line += method(:check)
      visitor.enter_va_function_like_define_line += method(:check)
    end

    private
    def check(node)
      return if in_initial_header?(node)
      return unless node.identifier_list

      if replacement_list = node.replacement_list
        should_be_function = replacement_list.tokens.all? { |pp_token|
          if keyword_or_punctuator?(pp_token)
            false
          elsif type_specifier_or_type_qualifier?(pp_token)
            false
          else
            true
          end
        }
        W(:W0443, node.location) if should_be_function
      end
    end

    def in_initial_header?(node)
      case
      when pinit_fpath = Traits.instance.of_project.initial_header
        File.identical?(node.location.fpath, pinit_fpath)
      when cinit_fpath = Traits.instance.of_compiler.initial_header
        File.identical?(node.location.fpath, cinit_fpath)
      else
        false
      end
    end

    KEYWORDS = [
      "sizeof", "typedef", "extern", "static", "auto", "register", "inline",
      "restrict", "char", "short", "int", "long", "signed", "unsigned",
      "float", "double", "const", "volatile", "void", "_Bool", "_Complex",
      "_Imaginary", "struct", "union", "enum", "case", "default", "if",
      "else", "switch", "while", "do", "for", "goto", "continue", "break",
      "return"
    ].to_set.freeze
    private_constant :KEYWORDS

    def keyword_or_punctuator?(pp_token)
      pp_token.value == "#" || pp_token.value == "##" ||
        pp_token.value == "{" || pp_token.value == "}" ||
        pp_token.value == ";" || KEYWORDS.include?(pp_token.value)
    end

    def type_specifier_or_type_qualifier?(pp_token)
      pp_token.value == "const" || pp_token.value == "volatile" ||
        pp_token.value == "restrict" ||
        @context[:c_type_table].all_type_names.include?(pp_token.value)
    end
  end

  class W0444 < PassiveMessageDetection
    # NOTE: W0444 may be duplicative when the same header which has the
    #       function-like macro definition with `#' and `##' operators is
    #       included twice or more.
    ensure_uniqueness_of :W0444

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_function_like_macro_defined += method(:check)
      interp.on_va_function_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      if replacement_list = macro.replacement_list
        sharp_op_num = 0
        sharpsharp_op_num = 0

        replacement_list.tokens.each do |pp_token|
          case pp_token.value
          when "#"
            sharp_op_num += 1
          when "##"
            sharpsharp_op_num += 1
          end
        end

        if sharp_op_num > 0 && sharpsharp_op_num > 0
          W(:W0444, define_line.location)
        end
      end
    end
  end

  class W0445 < PassiveMessageDetection
    # NOTE: W0445 may be duplicative when the same header which has the
    #       function-like macro definition with two or more `##' operators is
    #       included twice or more.
    ensure_uniqueness_of :W0445

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_function_like_macro_defined += method(:check)
      interp.on_va_function_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      if replacement_list = macro.replacement_list
        sharpsharp_op_num = 0

        replacement_list.tokens.each do |pp_token|
          if pp_token.value == "##"
            sharpsharp_op_num += 1
          end
        end

        if sharpsharp_op_num > 1
          W(:W0445, define_line.location)
        end
      end
    end
  end

  class W0477 < PassiveMessageDetection
    # NOTE: W0477 may be duplicative when the same header which has the macro
    #       definition with unbalanced grouping tokens is included twice or
    #       more.
    ensure_uniqueness_of :W0477

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_object_like_macro_defined += method(:check)
      interp.on_function_like_macro_defined += method(:check)
      interp.on_va_function_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      if replacement_list = macro.replacement_list
        paren_count = 0
        brace_count = 0
        bracket_count = 0
        replacement_list.tokens.each do |pp_token|
          case pp_token.value
          when "("
            paren_count += 1
          when ")"
            paren_count -= 1
          when "{"
            brace_count += 1
          when "}"
            brace_count -= 1
          when "["
            bracket_count += 1
          when "]"
            bracket_count -= 1
          end
        end

        unless paren_count == 0 && brace_count == 0 && bracket_count == 0
          W(:W0477, define_line.location)
        end
      end
    end
  end

  class W0478 < PassiveMessageDetection
    # NOTE: W0478 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0478

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_object_like_define_line += method(:check)
      visitor.enter_function_like_define_line += method(:check)
      visitor.enter_va_function_like_define_line += method(:check)
    end

    private
    def check(node)
      replacement_list = node.replacement_list
      if replacement_list && replacement_list.tokens.size > 1
        unless replacement_list.may_represent_expression? ||
            replacement_list.may_represent_initializer? ||
            replacement_list.may_represent_block? ||
            replacement_list.may_represent_do_while_zero_idiom? ||
            replacement_list.may_represent_specifier_qualifier_list? ||
            replacement_list.may_represent_declaration_specifiers_head?
          W(:W0478, node.location)
        end
      end
    end
  end

  class W0479 < PassiveMessageDetection
    # NOTE: W0479 may be duplicative when the same header which has
    #       typedef-able macro definition is included twice or more.
    ensure_uniqueness_of :W0479

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_object_like_define_line += method(:check)
    end

    private
    def check(node)
      if replacement_list = node.replacement_list
        if replacement_list.may_represent_specifier_qualifier_list?
          W(:W0479, node.location)
        end
      end
    end
  end

  class W0480 < PassiveMessageDetection
    # NOTE: W0480 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0480

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_object_like_define_line += method(:check)
      visitor.enter_function_like_define_line += method(:check)
      visitor.enter_va_function_like_define_line += method(:check)
    end

    private
    def check(node)
      replacement_list = node.replacement_list
      if replacement_list and
          replacement_list.may_represent_punctuator? ||
          replacement_list.may_represent_controlling_keyword?
        W(:W0480, node.location)
      end
    end
  end

  class W0481 < PassiveMessageDetection
    # NOTE: W0481 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0481

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_object_like_define_line += method(:check)
      visitor.enter_function_like_define_line += method(:check)
      visitor.enter_va_function_like_define_line += method(:check)
    end

    private
    def check(node)
      replacement_list = node.replacement_list
      if replacement_list && replacement_list.may_represent_block?
        W(:W0481, node.location)
      end
    end
  end

  class W0482 < PassiveMessageDetection
    # NOTE: W0482 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0482

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_object_like_define_line += method(:check)
    end

    private
    def check(node)
      if replacement_list = node.replacement_list
        return unless replacement_list.tokens.size == 1
        if type_specifier?(replacement_list.tokens.first)
          W(:W0482, node.location)
        end
      end
    end

    def type_specifier?(pp_token)
      case pp_token.value
      when "void", "char", "short", "int", "long", "float", "double"
        true
      when "signed", "unsigned"
        true
      when "struct", "union", "enum"
        true
      else
        false
      end
    end
  end

  class W0483 < PassiveMessageDetection
    # NOTE: W0483 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0483

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_object_like_define_line += method(:check)
    end

    private
    def check(node)
      if replacement_list = node.replacement_list
        if replacement_list.may_represent_declaration_specifiers_head?
          W(:W0483, node.location)
        end
      end
    end
  end

  class W0511 < PassiveMessageDetection
    # NOTE: W0511 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0511

    def initialize(context)
      super
      context[:cpp_interpreter].on_line_comment_found +=
        lambda { |comment, location| W(:W0511, location) }
    end
  end

  class W0528 < PassiveMessageDetection
    # NOTE: W0528 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0528

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_object_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      if replacement_list = macro.replacement_list
        octal_token = replacement_list.tokens.find { |pp_token|
          pp_token.value =~ /\A0[0-9]+[UL]*\z/i
        }
        W(:W0528, octal_token.location) if octal_token
      end
    end
  end

  class W0541 < PassiveMessageDetection
    # NOTE: W0541 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0541

    def initialize(context)
      super
      context[:c_source].on_inline_assembly += method(:check)
    end

    private
    def check(asm_tokens)
      unless asm_tokens.any? { |token| token.replaced? }
        W(:W0541, asm_tokens.first.location)
      end
    end
  end

  class W0549 < PassiveMessageDetection
    # NOTE: W0549 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0549

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_function_like_macro_defined += method(:check)
      interp.on_va_function_like_macro_defined += method(:check)
    end

    private
    def check(define_line, macro)
      return unless macro.replacement_list

      macro.parameter_names.each do |name|
        macro.replacement_list.tokens.each_with_index do |pp_token, index|
          next unless pp_token.value == name

          prev_token = macro.replacement_list.tokens[[index - 1, 0].max]
          next_token = macro.replacement_list.tokens[index + 1]

          next if prev_token && prev_token.value =~ /\A##?\z/
          next if next_token && next_token.value == "##"

          unless prev_token && prev_token.value == "("
            W(:W0549, pp_token.location)
            next
          end

          unless next_token && next_token.value == ")"
            W(:W0549, pp_token.location)
            next
          end
        end
      end
    end
  end

  class W0554 < PassiveMessageDetection
    # NOTE: W0554 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0554

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_unknown_pragma_evaled += method(:check)
    end

    private
    def check(pragma_line)
      W(:W0554, pragma_line.location,
        pragma_line.pp_tokens ? pragma_line.pp_tokens.to_s : "")
    end
  end

  class W0574 < PassiveMessageDetection
    # NOTE: W0574 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0574

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_pp_token_extracted += method(:check_pp_token)
      interp.on_object_like_macro_defined += method(:check_macro)
      interp.on_function_like_macro_defined += method(:check_macro)
      interp.on_va_function_like_macro_defined += method(:check_macro)
    end

    private
    def check_pp_token(pp_token)
      if pp_token.value =~ /\AL?'(.*)'\z/
        unless BasicSourceCharacterSet.include?($1)
          W(:W0574, pp_token.location)
        end
      end
    end

    def check_macro(define_line, macro)
      if replacement_list = macro.replacement_list
        replacement_list.tokens.each { |pp_token| check_pp_token(pp_token) }
      end
    end
  end

  class W0575 < PassiveMessageDetection
    # NOTE: W0575 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0575

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_pp_token_extracted += method(:check_pp_token)
      interp.on_object_like_macro_defined += method(:check_macro)
      interp.on_function_like_macro_defined += method(:check_macro)
      interp.on_va_function_like_macro_defined += method(:check_macro)
    end

    private
    def check_pp_token(pp_token)
      if pp_token.value =~ /\AL?"(.*)"\z/
        unless BasicSourceCharacterSet.include?($1)
          W(:W0575, pp_token.location)
        end
      end
    end

    def check_macro(define_line, macro)
      if replacement_list = macro.replacement_list
        replacement_list.tokens.each { |pp_token| check_pp_token(pp_token) }
      end
    end
  end

  class W0576 < PassiveMessageDetection
    # NOTE: W0576 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0576

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_block_comment_found += method(:check)
      interp.on_line_comment_found += method(:check)
      @warned_chars = Set.new
    end

    private
    def check(comment, location)
      not_adapted = BasicSourceCharacterSet.select_not_adapted(comment).to_set
      new_chars = not_adapted - @warned_chars

      unless new_chars.empty?
        W(:W0576, location, location.fpath)
        @warned_chars.merge(new_chars)
      end
    end
  end

  class W0577 < PassiveMessageDetection
    # NOTE: W0577 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0577

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_unlexable_char_found += method(:check)
      @warned = false
    end

    private
    def check(char, location)
      unless @warned
        W(:W0577, location, location.fpath)
        @warned = true
      end
    end
  end

  class W0632 < PassiveMessageDetection
    # NOTE: W0632 may be duplicative when the same header which has references
    #       to the undefined macro is included twice or more.
    ensure_uniqueness_of :W0632

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_system_include_line += method(:check)
      visitor.enter_system_include_next_line += method(:check)
    end

    private
    def check(include_line)
      if include_line.header_name.value =~ /['"]/
        W(:W0632, include_line.location, include_line.header_name.value)
      end
    end
  end

  class W0633 < PassiveMessageDetection
    # NOTE: W0633 may be duplicative when the same header which has references
    #       to the undefined macro is included twice or more.
    ensure_uniqueness_of :W0633

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_user_include_line += method(:check)
      visitor.enter_user_include_next_line += method(:check)
    end

    private
    def check(include_line)
      if include_line.header_name.value =~ /'/
        W(:W0633, include_line.location, include_line.header_name.value)
      end
    end
  end

  class W0634 < PassiveMessageDetection
    # NOTE: W0634 may be duplicative when the same header which has references
    #       to the undefined macro is included twice or more.
    ensure_uniqueness_of :W0634

    def initialize(context)
      super
      visitor = context[:cpp_visitor]
      visitor.enter_user_include_line += method(:check)
      visitor.enter_user_include_next_line += method(:check)
      visitor.enter_system_include_line += method(:check)
      visitor.enter_system_include_next_line += method(:check)
    end

    private
    def check(include_line)
      if include_line.header_name.value.include?("\\")
        W(:W0634, include_line.location)
      end
    end
  end

  class W0696 < PassiveMessageDetection
    # NOTE: W0696 may be duplicative when the same header which has references
    #       to the undefined macro is included twice or more.
    ensure_uniqueness_of :W0696

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_undefined_macro_referred +=
        lambda { |token| W(:W0696, token.location, token.value) }
    end
  end

  class W0804 < PassiveMessageDetection
    # NOTE: W0804 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0804

    def initialize(context)
      super
      context[:cpp_interpreter].on_illformed_defined_op_found +=
        lambda { |location| W(:W0804, location) }
    end
  end

  class W0831 < PassiveMessageDetection
    # NOTE: W0831 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0831

    def initialize(context)
      super
      interp = context[:cpp_interpreter]
      interp.on_asm_section_evaled += method(:check)
    end

    private
    def check(asm_section)
      W(:W0831, asm_section.location)
    end
  end

  class W0832 < PassiveMessageDetection
    # NOTE: W0832 may be duplicative when the same header which causes this
    #       warning is included twice or more.
    ensure_uniqueness_of :W0832

    def initialize(context)
      super
      context[:c_source].on_inline_assembly += method(:check)
    end

    private
    def check(asm_tokens)
      W(:W0832, asm_tokens.first.location)
    end
  end

  class W9002 < PassiveMessageDetection
    # NOTE: W9002 may be duplicative when the same header which has no newline
    #       at end of the file is included twice or more.
    ensure_uniqueness_of :W9002

    def initialize(context)
      super
      context[:cpp_interpreter].on_eof_newline_not_found +=
        lambda { |location| W(:W9002, location) }
    end
  end

end
end
