//  Copyright (c) 2015 John Maddock
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#pragma once

#include <vector>
#include <boost/multiprecision/cpp_int.hpp>
#include <boost/lexical_cast.hpp>

template <class T>
std::vector<T> const& fibonacci_numbers(const std::integral_constant<int, 16>&)
{
   static const std::uint16_t numbers[] = {
      static_cast<std::uint16_t>(21u),
      static_cast<std::uint16_t>(34u),
      static_cast<std::uint16_t>(55u),
      static_cast<std::uint16_t>(89u),
      static_cast<std::uint16_t>(144u),
      static_cast<std::uint16_t>(233u),
      static_cast<std::uint16_t>(377u),
      static_cast<std::uint16_t>(610u),
      static_cast<std::uint16_t>(987u),
      static_cast<std::uint16_t>(1597u),
      static_cast<std::uint16_t>(2584u),
      static_cast<std::uint16_t>(4181u),
      static_cast<std::uint16_t>(6765u),
      static_cast<std::uint16_t>(17711u),
      static_cast<std::uint16_t>(10946u),
      static_cast<std::uint16_t>(17711u),
      static_cast<std::uint16_t>(28657u),
      static_cast<std::uint16_t>(46368u)
   };
   static const std::vector<T> data(numbers, numbers + sizeof(numbers) / sizeof(numbers[0]));

   return data;
}

template <class T>
std::vector<T> const& fibonacci_numbers(const std::integral_constant<int, 32>&)
{
   static const std::uint32_t numbers[] = {
      21u,
      34u,
      55u,
      89u,
      144u,
      233u,
      377u,
      610u,
      987u,
      1597u,
      2584u,
      4181u,
      6765u,
      10946u,
      17711u,
      28657u,
      46368u,
      75025u,
      121393u,
      196418u,
      317811u,
      514229u,
      832040u,
      1346269u,
      2178309u,
      3524578u,
      5702887u,
      9227465u,
      14930352u,
      24157817u,
      39088169u,
      63245986u,
      102334155u,
      165580141u,
      267914296u,
      433494437u,
      701408733u,
      1134903170u,
      1836311903u,
      2971215073u
   };
   static const std::vector<T> data(numbers, numbers + sizeof(numbers) / sizeof(numbers[0]));

   return data;
}

template <class T>
std::vector<T> const& fibonacci_numbers(const std::integral_constant<int, 64>&)
{
   static const std::uint64_t numbers[] = {
      21uLL,
      34uLL,
      55uLL,
      89uLL,
      144uLL,
      233uLL,
      377uLL,
      610uLL,
      987uLL,
      1597uLL,
      2584uLL,
      4181uLL,
      6765uLL,
      10946uLL,
      17711uLL,
      28657uLL,
      46368uLL,
      75025uLL,
      121393uLL,
      196418uLL,
      317811uLL,
      514229uLL,
      832040uLL,
      1346269uLL,
      2178309uLL,
      3524578uLL,
      5702887uLL,
      9227465uLL,
      14930352uLL,
      24157817uLL,
      39088169uLL,
      63245986uLL,
      102334155uLL,
      165580141uLL,
      267914296uLL,
      433494437uLL,
      701408733uLL,
      1134903170uLL,
      1836311903uLL,
      2971215073uLL,
      4807526976uLL,
      7778742049uLL,
      12586269025uLL,
      20365011074uLL,
      32951280099uLL,
      53316291173uLL,
      86267571272uLL,
      139583862445uLL,
      225851433717uLL,
      365435296162uLL,
      591286729879uLL,
      956722026041uLL,
      1548008755920uLL,
      2504730781961uLL,
      4052739537881uLL,
      6557470319842uLL,
      10610209857723uLL,
      17167680177565uLL,
      27777890035288uLL,
      44945570212853uLL,
      72723460248141uLL,
      117669030460994uLL,
      190392490709135uLL,
      308061521170129uLL,
      498454011879264uLL,
      806515533049393uLL,
      1304969544928657uLL,
      2111485077978050uLL,
      3416454622906707uLL,
      5527939700884757uLL,
      8944394323791464uLL,
      14472334024676221uLL,
      23416728348467685uLL,
      37889062373143906uLL,
      61305790721611591uLL,
      99194853094755497uLL,
      160500643816367088uLL,
      259695496911122585uLL,
      420196140727489673uLL,
      679891637638612258uLL,
      1100087778366101931uLL,
      1779979416004714189uLL,
      2880067194370816120uLL,
      4660046610375530309uLL,
      7540113804746346429uLL,
      12200160415121876738uLL,
   };
   static const std::vector<T> data(numbers, numbers + sizeof(numbers) / sizeof(numbers[0]));

   return data;
}

template <class T>
std::vector<T> const& fibonacci_numbers(const std::integral_constant<int, 0>&)
{
   static const T numbers[] = {
      boost::lexical_cast<T>("21"),
      boost::lexical_cast<T>("34"),
      boost::lexical_cast<T>("55"),
      boost::lexical_cast<T>("89"),
      boost::lexical_cast<T>("144"),
      boost::lexical_cast<T>("233"),
      boost::lexical_cast<T>("377"),
      boost::lexical_cast<T>("610"),
      boost::lexical_cast<T>("987"),
      boost::lexical_cast<T>("1597"),
      boost::lexical_cast<T>("2584"),
      boost::lexical_cast<T>("4181"),
      boost::lexical_cast<T>("6765"),
      boost::lexical_cast<T>("10946"),
      boost::lexical_cast<T>("17711"),
      boost::lexical_cast<T>("28657"),
      boost::lexical_cast<T>("46368"),
      boost::lexical_cast<T>("75025"),
      boost::lexical_cast<T>("121393"),
      boost::lexical_cast<T>("196418"),
      boost::lexical_cast<T>("317811"),
      boost::lexical_cast<T>("514229"),
      boost::lexical_cast<T>("832040"),
      boost::lexical_cast<T>("1346269"),
      boost::lexical_cast<T>("2178309"),
      boost::lexical_cast<T>("3524578"),
      boost::lexical_cast<T>("5702887"),
      boost::lexical_cast<T>("9227465"),
      boost::lexical_cast<T>("14930352"),
      boost::lexical_cast<T>("24157817"),
      boost::lexical_cast<T>("39088169"),
      boost::lexical_cast<T>("63245986"),
      boost::lexical_cast<T>("102334155"),
      boost::lexical_cast<T>("165580141"),
      boost::lexical_cast<T>("267914296"),
      boost::lexical_cast<T>("433494437"),
      boost::lexical_cast<T>("701408733"),
      boost::lexical_cast<T>("1134903170"),
      boost::lexical_cast<T>("1836311903"),
      boost::lexical_cast<T>("2971215073"),
      // 64-bit:
      boost::lexical_cast<T>("4807526976"),
      boost::lexical_cast<T>("7778742049"),
      boost::lexical_cast<T>("12586269025"),
      boost::lexical_cast<T>("20365011074"),
      boost::lexical_cast<T>("32951280099"),
      boost::lexical_cast<T>("53316291173"),
      boost::lexical_cast<T>("86267571272"),
      boost::lexical_cast<T>("139583862445"),
      boost::lexical_cast<T>("225851433717"),
      boost::lexical_cast<T>("365435296162"),
      boost::lexical_cast<T>("591286729879"),
      boost::lexical_cast<T>("956722026041"),
      boost::lexical_cast<T>("1548008755920"),
      boost::lexical_cast<T>("2504730781961"),
      boost::lexical_cast<T>("4052739537881"),
      boost::lexical_cast<T>("6557470319842"),
      boost::lexical_cast<T>("10610209857723"),
      boost::lexical_cast<T>("17167680177565"),
      boost::lexical_cast<T>("27777890035288"),
      boost::lexical_cast<T>("44945570212853"),
      boost::lexical_cast<T>("72723460248141"),
      boost::lexical_cast<T>("117669030460994"),
      boost::lexical_cast<T>("190392490709135"),
      boost::lexical_cast<T>("308061521170129"),
      boost::lexical_cast<T>("498454011879264"),
      boost::lexical_cast<T>("806515533049393"),
      boost::lexical_cast<T>("1304969544928657"),
      boost::lexical_cast<T>("2111485077978050"),
      boost::lexical_cast<T>("3416454622906707"),
      boost::lexical_cast<T>("5527939700884757"),
      boost::lexical_cast<T>("8944394323791464"),
      boost::lexical_cast<T>("14472334024676221"),
      boost::lexical_cast<T>("23416728348467685"),
      boost::lexical_cast<T>("37889062373143906"),
      boost::lexical_cast<T>("61305790721611591"),
      boost::lexical_cast<T>("99194853094755497"),
      boost::lexical_cast<T>("160500643816367088"),
      boost::lexical_cast<T>("259695496911122585"),
      boost::lexical_cast<T>("420196140727489673"),
      boost::lexical_cast<T>("679891637638612258"),
      boost::lexical_cast<T>("1100087778366101931"),
      boost::lexical_cast<T>("1779979416004714189"),
      boost::lexical_cast<T>("2880067194370816120"),
      boost::lexical_cast<T>("4660046610375530309"),
      boost::lexical_cast<T>("7540113804746346429"),
      boost::lexical_cast<T>("12200160415121876738"),
      // 128-bit:
      boost::lexical_cast<T>("19740274219868223167"),
      boost::lexical_cast<T>("31940434634990099905"),
      boost::lexical_cast<T>("51680708854858323072"),
      boost::lexical_cast<T>("83621143489848422977"),
      boost::lexical_cast<T>("135301852344706746049"),
      boost::lexical_cast<T>("218922995834555169026"),
      boost::lexical_cast<T>("354224848179261915075"),
      boost::lexical_cast<T>("573147844013817084101"),
      boost::lexical_cast<T>("927372692193078999176"),
      boost::lexical_cast<T>("1500520536206896083277"),
      boost::lexical_cast<T>("2427893228399975082453"),
      boost::lexical_cast<T>("3928413764606871165730"),
      boost::lexical_cast<T>("6356306993006846248183"),
      boost::lexical_cast<T>("10284720757613717413913"),
      boost::lexical_cast<T>("16641027750620563662096"),
      boost::lexical_cast<T>("26925748508234281076009"),
      boost::lexical_cast<T>("43566776258854844738105"),
      boost::lexical_cast<T>("70492524767089125814114"),
      boost::lexical_cast<T>("114059301025943970552219"),
      boost::lexical_cast<T>("184551825793033096366333"),
      boost::lexical_cast<T>("298611126818977066918552"),
      boost::lexical_cast<T>("483162952612010163284885"),
      boost::lexical_cast<T>("781774079430987230203437"),
      boost::lexical_cast<T>("1264937032042997393488322"),
      boost::lexical_cast<T>("2046711111473984623691759"),
      boost::lexical_cast<T>("3311648143516982017180081"),
      boost::lexical_cast<T>("5358359254990966640871840"),
      boost::lexical_cast<T>("8670007398507948658051921"),
      boost::lexical_cast<T>("14028366653498915298923761"),
      boost::lexical_cast<T>("22698374052006863956975682"),
      boost::lexical_cast<T>("36726740705505779255899443"),
      boost::lexical_cast<T>("59425114757512643212875125"),
      boost::lexical_cast<T>("96151855463018422468774568"),
      boost::lexical_cast<T>("155576970220531065681649693"),
      boost::lexical_cast<T>("251728825683549488150424261"),
      boost::lexical_cast<T>("407305795904080553832073954"),
      boost::lexical_cast<T>("659034621587630041982498215"),
      boost::lexical_cast<T>("1066340417491710595814572169"),
      boost::lexical_cast<T>("1725375039079340637797070384"),
      boost::lexical_cast<T>("2791715456571051233611642553"),
      boost::lexical_cast<T>("4517090495650391871408712937"),
      boost::lexical_cast<T>("7308805952221443105020355490"),
      boost::lexical_cast<T>("11825896447871834976429068427"),
      boost::lexical_cast<T>("19134702400093278081449423917"),
      boost::lexical_cast<T>("30960598847965113057878492344"),
      boost::lexical_cast<T>("50095301248058391139327916261"),
      boost::lexical_cast<T>("81055900096023504197206408605"),
      boost::lexical_cast<T>("131151201344081895336534324866"),
      boost::lexical_cast<T>("212207101440105399533740733471"),
      boost::lexical_cast<T>("343358302784187294870275058337"),
      boost::lexical_cast<T>("555565404224292694404015791808"),
      boost::lexical_cast<T>("898923707008479989274290850145"),
      boost::lexical_cast<T>("1454489111232772683678306641953"),
      boost::lexical_cast<T>("2353412818241252672952597492098"),
      boost::lexical_cast<T>("3807901929474025356630904134051"),
      boost::lexical_cast<T>("6161314747715278029583501626149"),
      boost::lexical_cast<T>("9969216677189303386214405760200"),
      boost::lexical_cast<T>("16130531424904581415797907386349"),
      boost::lexical_cast<T>("26099748102093884802012313146549"),
      boost::lexical_cast<T>("42230279526998466217810220532898"),
      boost::lexical_cast<T>("68330027629092351019822533679447"),
      boost::lexical_cast<T>("110560307156090817237632754212345"),
      boost::lexical_cast<T>("178890334785183168257455287891792"),
      boost::lexical_cast<T>("289450641941273985495088042104137"),
      boost::lexical_cast<T>("468340976726457153752543329995929"),
      boost::lexical_cast<T>("757791618667731139247631372100066"),
      boost::lexical_cast<T>("1226132595394188293000174702095995"),
      boost::lexical_cast<T>("1983924214061919432247806074196061"),
      boost::lexical_cast<T>("3210056809456107725247980776292056"),
      boost::lexical_cast<T>("5193981023518027157495786850488117"),
      boost::lexical_cast<T>("8404037832974134882743767626780173"),
      boost::lexical_cast<T>("13598018856492162040239554477268290"),
      boost::lexical_cast<T>("22002056689466296922983322104048463"),
      boost::lexical_cast<T>("35600075545958458963222876581316753"),
      boost::lexical_cast<T>("57602132235424755886206198685365216"),
      boost::lexical_cast<T>("93202207781383214849429075266681969"),
      boost::lexical_cast<T>("150804340016807970735635273952047185"),
      boost::lexical_cast<T>("244006547798191185585064349218729154"),
      boost::lexical_cast<T>("394810887814999156320699623170776339"),
      boost::lexical_cast<T>("638817435613190341905763972389505493"),
      boost::lexical_cast<T>("1033628323428189498226463595560281832"),
      boost::lexical_cast<T>("1672445759041379840132227567949787325"),
      boost::lexical_cast<T>("2706074082469569338358691163510069157"),
      boost::lexical_cast<T>("4378519841510949178490918731459856482"),
      boost::lexical_cast<T>("7084593923980518516849609894969925639"),
      boost::lexical_cast<T>("11463113765491467695340528626429782121"),
      boost::lexical_cast<T>("18547707689471986212190138521399707760"),
      boost::lexical_cast<T>("30010821454963453907530667147829489881"),
      boost::lexical_cast<T>("48558529144435440119720805669229197641"),
      boost::lexical_cast<T>("78569350599398894027251472817058687522"),
      boost::lexical_cast<T>("127127879743834334146972278486287885163"),
      boost::lexical_cast<T>("205697230343233228174223751303346572685"),
      boost::lexical_cast<T>("332825110087067562321196029789634457848"),
      boost::lexical_cast<T>("538522340430300790495419781092981030533"),
      boost::lexical_cast<T>("871347450517368352816615810882615488381"),
      boost::lexical_cast<T>("1409869790947669143312035591975596518914"),
      boost::lexical_cast<T>("2281217241465037496128651402858212007295"),
      boost::lexical_cast<T>("3691087032412706639440686994833808526209"),
      boost::lexical_cast<T>("5972304273877744135569338397692020533504"),
      boost::lexical_cast<T>("9663391306290450775010025392525829059713"),
      boost::lexical_cast<T>("15635695580168194910579363790217849593217"),
      boost::lexical_cast<T>("25299086886458645685589389182743678652930"),
      boost::lexical_cast<T>("40934782466626840596168752972961528246147"),
      boost::lexical_cast<T>("66233869353085486281758142155705206899077"),
      boost::lexical_cast<T>("107168651819712326877926895128666735145224"),
      boost::lexical_cast<T>("173402521172797813159685037284371942044301"),
      boost::lexical_cast<T>("280571172992510140037611932413038677189525"),
      boost::lexical_cast<T>("453973694165307953197296969697410619233826"),
      boost::lexical_cast<T>("734544867157818093234908902110449296423351"),
      boost::lexical_cast<T>("1188518561323126046432205871807859915657177"),
      boost::lexical_cast<T>("1923063428480944139667114773918309212080528"),
      boost::lexical_cast<T>("3111581989804070186099320645726169127737705"),
      boost::lexical_cast<T>("5034645418285014325766435419644478339818233"),
      boost::lexical_cast<T>("8146227408089084511865756065370647467555938"),
      boost::lexical_cast<T>("13180872826374098837632191485015125807374171"),
      boost::lexical_cast<T>("21327100234463183349497947550385773274930109"),
      boost::lexical_cast<T>("34507973060837282187130139035400899082304280"),
      boost::lexical_cast<T>("55835073295300465536628086585786672357234389"),
      boost::lexical_cast<T>("90343046356137747723758225621187571439538669"),
      boost::lexical_cast<T>("146178119651438213260386312206974243796773058"),
      boost::lexical_cast<T>("236521166007575960984144537828161815236311727"),
      boost::lexical_cast<T>("382699285659014174244530850035136059033084785"),
      boost::lexical_cast<T>("619220451666590135228675387863297874269396512"),
      boost::lexical_cast<T>("1001919737325604309473206237898433933302481297"),
      boost::lexical_cast<T>("1621140188992194444701881625761731807571877809"),
      boost::lexical_cast<T>("2623059926317798754175087863660165740874359106"),
      boost::lexical_cast<T>("4244200115309993198876969489421897548446236915"),
      boost::lexical_cast<T>("6867260041627791953052057353082063289320596021"),
      boost::lexical_cast<T>("11111460156937785151929026842503960837766832936"),
      boost::lexical_cast<T>("17978720198565577104981084195586024127087428957"),
      boost::lexical_cast<T>("29090180355503362256910111038089984964854261893"),
      boost::lexical_cast<T>("47068900554068939361891195233676009091941690850"),
      boost::lexical_cast<T>("76159080909572301618801306271765994056795952743"),
      boost::lexical_cast<T>("123227981463641240980692501505442003148737643593"),
      boost::lexical_cast<T>("199387062373213542599493807777207997205533596336"),
      boost::lexical_cast<T>("322615043836854783580186309282650000354271239929"),
      boost::lexical_cast<T>("522002106210068326179680117059857997559804836265"),
      boost::lexical_cast<T>("844617150046923109759866426342507997914076076194"),
      boost::lexical_cast<T>("1366619256256991435939546543402365995473880912459"),
      boost::lexical_cast<T>("2211236406303914545699412969744873993387956988653"),
      boost::lexical_cast<T>("3577855662560905981638959513147239988861837901112"),
      boost::lexical_cast<T>("5789092068864820527338372482892113982249794889765"),
      boost::lexical_cast<T>("9366947731425726508977331996039353971111632790877"),
      boost::lexical_cast<T>("15156039800290547036315704478931467953361427680642"),
      boost::lexical_cast<T>("24522987531716273545293036474970821924473060471519"),
      boost::lexical_cast<T>("39679027332006820581608740953902289877834488152161"),
      boost::lexical_cast<T>("64202014863723094126901777428873111802307548623680"),
      boost::lexical_cast<T>("103881042195729914708510518382775401680142036775841"),
      boost::lexical_cast<T>("168083057059453008835412295811648513482449585399521"),
      boost::lexical_cast<T>("271964099255182923543922814194423915162591622175362"),
      boost::lexical_cast<T>("440047156314635932379335110006072428645041207574883"),
      boost::lexical_cast<T>("712011255569818855923257924200496343807632829750245"),
      boost::lexical_cast<T>("1152058411884454788302593034206568772452674037325128"),
      boost::lexical_cast<T>("1864069667454273644225850958407065116260306867075373"),
      boost::lexical_cast<T>("3016128079338728432528443992613633888712980904400501"),
      boost::lexical_cast<T>("4880197746793002076754294951020699004973287771475874"),
      boost::lexical_cast<T>("7896325826131730509282738943634332893686268675876375"),
      boost::lexical_cast<T>("12776523572924732586037033894655031898659556447352249"),
      boost::lexical_cast<T>("20672849399056463095319772838289364792345825123228624"),
      boost::lexical_cast<T>("33449372971981195681356806732944396691005381570580873"),
      boost::lexical_cast<T>("54122222371037658776676579571233761483351206693809497"),
      boost::lexical_cast<T>("87571595343018854458033386304178158174356588264390370"),
      boost::lexical_cast<T>("141693817714056513234709965875411919657707794958199867"),
      boost::lexical_cast<T>("229265413057075367692743352179590077832064383222590237"),
      boost::lexical_cast<T>("370959230771131880927453318055001997489772178180790104"),
      boost::lexical_cast<T>("600224643828207248620196670234592075321836561403380341"),
      boost::lexical_cast<T>("971183874599339129547649988289594072811608739584170445"),
      boost::lexical_cast<T>("1571408518427546378167846658524186148133445300987550786"),
      boost::lexical_cast<T>("2542592393026885507715496646813780220945054040571721231"),
      boost::lexical_cast<T>("4114000911454431885883343305337966369078499341559272017"),
      boost::lexical_cast<T>("6656593304481317393598839952151746590023553382130993248"),
      boost::lexical_cast<T>("10770594215935749279482183257489712959102052723690265265"),
      boost::lexical_cast<T>("17427187520417066673081023209641459549125606105821258513"),
      boost::lexical_cast<T>("28197781736352815952563206467131172508227658829511523778"),
      boost::lexical_cast<T>("45624969256769882625644229676772632057353264935332782291"),
      boost::lexical_cast<T>("73822750993122698578207436143903804565580923764844306069"),
      boost::lexical_cast<T>("119447720249892581203851665820676436622934188700177088360"),
      boost::lexical_cast<T>("193270471243015279782059101964580241188515112465021394429"),
      boost::lexical_cast<T>("312718191492907860985910767785256677811449301165198482789"),
      boost::lexical_cast<T>("505988662735923140767969869749836918999964413630219877218"),
      boost::lexical_cast<T>("818706854228831001753880637535093596811413714795418360007"),
      boost::lexical_cast<T>("1324695516964754142521850507284930515811378128425638237225"),
      boost::lexical_cast<T>("2143402371193585144275731144820024112622791843221056597232"),
      boost::lexical_cast<T>("3468097888158339286797581652104954628434169971646694834457"),
      boost::lexical_cast<T>("5611500259351924431073312796924978741056961814867751431689"),
      boost::lexical_cast<T>("9079598147510263717870894449029933369491131786514446266146"),
      boost::lexical_cast<T>("14691098406862188148944207245954912110548093601382197697835"),
      boost::lexical_cast<T>("23770696554372451866815101694984845480039225387896643963981"),
      boost::lexical_cast<T>("38461794961234640015759308940939757590587318989278841661816"),
      boost::lexical_cast<T>("62232491515607091882574410635924603070626544377175485625797"),
      boost::lexical_cast<T>("100694286476841731898333719576864360661213863366454327287613"),
      boost::lexical_cast<T>("162926777992448823780908130212788963731840407743629812913410"),
      boost::lexical_cast<T>("263621064469290555679241849789653324393054271110084140201023"),
      boost::lexical_cast<T>("426547842461739379460149980002442288124894678853713953114433"),
      boost::lexical_cast<T>("690168906931029935139391829792095612517948949963798093315456"),
      boost::lexical_cast<T>("1116716749392769314599541809794537900642843628817512046429889"),
      boost::lexical_cast<T>("1806885656323799249738933639586633513160792578781310139745345"),
      boost::lexical_cast<T>("2923602405716568564338475449381171413803636207598822186175234"),
      boost::lexical_cast<T>("4730488062040367814077409088967804926964428786380132325920579"),
      boost::lexical_cast<T>("7654090467756936378415884538348976340768064993978954512095813"),
      boost::lexical_cast<T>("12384578529797304192493293627316781267732493780359086838016392"),
      boost::lexical_cast<T>("20038668997554240570909178165665757608500558774338041350112205"),
      boost::lexical_cast<T>("32423247527351544763402471792982538876233052554697128188128597"),
      boost::lexical_cast<T>("52461916524905785334311649958648296484733611329035169538240802"),
      boost::lexical_cast<T>("84885164052257330097714121751630835360966663883732297726369399"),
      boost::lexical_cast<T>("137347080577163115432025771710279131845700275212767467264610201"),
      boost::lexical_cast<T>("222232244629420445529739893461909967206666939096499764990979600"),
      boost::lexical_cast<T>("359579325206583560961765665172189099052367214309267232255589801"),
      boost::lexical_cast<T>("581811569836004006491505558634099066259034153405766997246569401"),
      boost::lexical_cast<T>("941390895042587567453271223806288165311401367715034229502159202"),
      boost::lexical_cast<T>("1523202464878591573944776782440387231570435521120801226748728603"),
      boost::lexical_cast<T>("2464593359921179141398048006246675396881836888835835456250887805"),
      boost::lexical_cast<T>("3987795824799770715342824788687062628452272409956636682999616408"),
      boost::lexical_cast<T>("6452389184720949856740872794933738025334109298792472139250504213"),
      boost::lexical_cast<T>("10440185009520720572083697583620800653786381708749108822250120621"),
      boost::lexical_cast<T>("16892574194241670428824570378554538679120491007541580961500624834"),
      boost::lexical_cast<T>("27332759203762391000908267962175339332906872716290689783750745455"),
      boost::lexical_cast<T>("44225333398004061429732838340729878012027363723832270745251370289"),
      boost::lexical_cast<T>("71558092601766452430641106302905217344934236440122960529002115744"),
      boost::lexical_cast<T>("115783425999770513860373944643635095356961600163955231274253486033"),
      boost::lexical_cast<T>("187341518601536966291015050946540312701895836604078191803255601777"),
      boost::lexical_cast<T>("303124944601307480151388995590175408058857436768033423077509087810"),
      boost::lexical_cast<T>("490466463202844446442404046536715720760753273372111614880764689587"),
      boost::lexical_cast<T>("793591407804151926593793042126891128819610710140145037958273777397"),
      boost::lexical_cast<T>("1284057871006996373036197088663606849580363983512256652839038466984"),
      boost::lexical_cast<T>("2077649278811148299629990130790497978399974693652401690797312244381"),
      boost::lexical_cast<T>("3361707149818144672666187219454104827980338677164658343636350711365"),
      boost::lexical_cast<T>("5439356428629292972296177350244602806380313370817060034433662955746"),
      boost::lexical_cast<T>("8801063578447437644962364569698707634360652047981718378070013667111"),
      boost::lexical_cast<T>("14240420007076730617258541919943310440740965418798778412503676622857"),
      boost::lexical_cast<T>("23041483585524168262220906489642018075101617466780496790573690289968"),
      boost::lexical_cast<T>("37281903592600898879479448409585328515842582885579275203077366912825"),
      boost::lexical_cast<T>("60323387178125067141700354899227346590944200352359771993651057202793"),
      boost::lexical_cast<T>("97605290770725966021179803308812675106786783237939047196728424115618"),
      boost::lexical_cast<T>("157928677948851033162880158208040021697730983590298819190379481318411"),
      boost::lexical_cast<T>("255533968719576999184059961516852696804517766828237866387107905434029"),
      boost::lexical_cast<T>("413462646668428032346940119724892718502248750418536685577487386752440"),
      boost::lexical_cast<T>("668996615388005031531000081241745415306766517246774551964595292186469"),
      boost::lexical_cast<T>("1082459262056433063877940200966638133809015267665311237542082678938909"),
      boost::lexical_cast<T>("1751455877444438095408940282208383549115781784912085789506677971125378"),
      boost::lexical_cast<T>("2833915139500871159286880483175021682924797052577397027048760650064287"),
      boost::lexical_cast<T>("4585371016945309254695820765383405232040578837489482816555438621189665"),
      boost::lexical_cast<T>("7419286156446180413982701248558426914965375890066879843604199271253952"),
      boost::lexical_cast<T>("12004657173391489668678522013941832147005954727556362660159637892443617"),
      boost::lexical_cast<T>("19423943329837670082661223262500259061971330617623242503763837163697569"),
      boost::lexical_cast<T>("31428600503229159751339745276442091208977285345179605163923475056141186"),
      boost::lexical_cast<T>("50852543833066829834000968538942350270948615962802847667687312219838755"),
      boost::lexical_cast<T>("82281144336295989585340713815384441479925901307982452831610787275979941"),
      boost::lexical_cast<T>("133133688169362819419341682354326791750874517270785300499298099495818696"),
      boost::lexical_cast<T>("215414832505658809004682396169711233230800418578767753330908886771798637"),
      boost::lexical_cast<T>("348548520675021628424024078524038024981674935849553053830206986267617333"),
      boost::lexical_cast<T>("563963353180680437428706474693749258212475354428320807161115873039415970"),
      boost::lexical_cast<T>("912511873855702065852730553217787283194150290277873860991322859307033303"),
      boost::lexical_cast<T>("1476475227036382503281437027911536541406625644706194668152438732346449273"),
      boost::lexical_cast<T>("2388987100892084569134167581129323824600775934984068529143761591653482576"),
      boost::lexical_cast<T>("3865462327928467072415604609040860366007401579690263197296200323999931849"),
      boost::lexical_cast<T>("6254449428820551641549772190170184190608177514674331726439961915653414425"),
      boost::lexical_cast<T>("10119911756749018713965376799211044556615579094364594923736162239653346274"),
      boost::lexical_cast<T>("16374361185569570355515148989381228747223756609038926650176124155306760699"),
      boost::lexical_cast<T>("26494272942318589069480525788592273303839335703403521573912286394960106973"),
      boost::lexical_cast<T>("42868634127888159424995674777973502051063092312442448224088410550266867672"),
      boost::lexical_cast<T>("69362907070206748494476200566565775354902428015845969798000696945226974645"),
      boost::lexical_cast<T>("112231541198094907919471875344539277405965520328288418022089107495493842317"),
      boost::lexical_cast<T>("181594448268301656413948075911105052760867948344134387820089804440720816962"),
      boost::lexical_cast<T>("293825989466396564333419951255644330166833468672422805842178911936214659279"),
      boost::lexical_cast<T>("475420437734698220747368027166749382927701417016557193662268716376935476241"),
      boost::lexical_cast<T>("769246427201094785080787978422393713094534885688979999504447628313150135520"),
      boost::lexical_cast<T>("1244666864935793005828156005589143096022236302705537193166716344690085611761"),
      boost::lexical_cast<T>("2013913292136887790908943984011536809116771188394517192671163973003235747281"),
      boost::lexical_cast<T>("3258580157072680796737099989600679905139007491100054385837880317693321359042"),
      boost::lexical_cast<T>("5272493449209568587646043973612216714255778679494571578509044290696557106323"),
      boost::lexical_cast<T>("8531073606282249384383143963212896619394786170594625964346924608389878465365"),
      boost::lexical_cast<T>("13803567055491817972029187936825113333650564850089197542855968899086435571688"),
      boost::lexical_cast<T>("22334640661774067356412331900038009953045351020683823507202893507476314037053"),
      boost::lexical_cast<T>("36138207717265885328441519836863123286695915870773021050058862406562749608741"),
      boost::lexical_cast<T>("58472848379039952684853851736901133239741266891456844557261755914039063645794"),
      boost::lexical_cast<T>("94611056096305838013295371573764256526437182762229865607320618320601813254535"),
      boost::lexical_cast<T>("153083904475345790698149223310665389766178449653686710164582374234640876900329"),
      boost::lexical_cast<T>("247694960571651628711444594884429646292615632415916575771902992555242690154864"),
      boost::lexical_cast<T>("400778865046997419409593818195095036058794082069603285936485366789883567055193"),
      boost::lexical_cast<T>("648473825618649048121038413079524682351409714485519861708388359345126257210057"),
      boost::lexical_cast<T>("1049252690665646467530632231274619718410203796555123147644873726135009824265250"),
      boost::lexical_cast<T>("1697726516284295515651670644354144400761613511040643009353262085480136081475307"),
      boost::lexical_cast<T>("2746979206949941983182302875628764119171817307595766156998135811615145905740557"),
      boost::lexical_cast<T>("4444705723234237498833973519982908519933430818636409166351397897095281987215864"),
      boost::lexical_cast<T>("7191684930184179482016276395611672639105248126232175323349533708710427892956421"),
      boost::lexical_cast<T>("11636390653418416980850249915594581159038678944868584489700931605805709880172285"),
      boost::lexical_cast<T>("18828075583602596462866526311206253798143927071100759813050465314516137773128706"),
      boost::lexical_cast<T>("30464466237021013443716776226800834957182606015969344302751396920321847653300991"),
      boost::lexical_cast<T>("49292541820623609906583302538007088755326533087070104115801862234837985426429697"),
      boost::lexical_cast<T>("79757008057644623350300078764807923712509139103039448418553259155159833079730688"),
      boost::lexical_cast<T>("129049549878268233256883381302815012467835672190109552534355121389997818506160385"),
      boost::lexical_cast<T>("208806557935912856607183460067622936180344811293149000952908380545157651585891073"),
      boost::lexical_cast<T>("337856107814181089864066841370437948648180483483258553487263501935155470092051458"),
      boost::lexical_cast<T>("546662665750093946471250301438060884828525294776407554440171882480313121677942531"),
      boost::lexical_cast<T>("884518773564275036335317142808498833476705778259666107927435384415468591769993989"),
      boost::lexical_cast<T>("1431181439314368982806567444246559718305231073036073662367607266895781713447936520"),
      boost::lexical_cast<T>("2315700212878644019141884587055058551781936851295739770295042651311250305217930509"),
      boost::lexical_cast<T>("3746881652193013001948452031301618270087167924331813432662649918207032018665867029"),
      boost::lexical_cast<T>("6062581865071657021090336618356676821869104775627553202957692569518282323883797538"),
      boost::lexical_cast<T>("9809463517264670023038788649658295091956272699959366635620342487725314342549664567"),
      boost::lexical_cast<T>("15872045382336327044129125268014971913825377475586919838578035057243596666433462105"),
      boost::lexical_cast<T>("25681508899600997067167913917673267005781650175546286474198377544968911008983126672"),
      boost::lexical_cast<T>("41553554281937324111297039185688238919607027651133206312776412602212507675416588777"),
      boost::lexical_cast<T>("67235063181538321178464953103361505925388677826679492786974790147181418684399715449"),
      boost::lexical_cast<T>("108788617463475645289761992289049744844995705477812699099751202749393926359816304226"),
      boost::lexical_cast<T>("176023680645013966468226945392411250770384383304492191886725992896575345044216019675"),
      boost::lexical_cast<T>("284812298108489611757988937681460995615380088782304890986477195645969271404032323901"),
      boost::lexical_cast<T>("460835978753503578226215883073872246385764472086797082873203188542544616448248343576"),
      boost::lexical_cast<T>("745648276861993189984204820755333242001144560869101973859680384188513887852280667477"),
      boost::lexical_cast<T>("1206484255615496768210420703829205488386909032955899056732883572731058504300529011053"),
      boost::lexical_cast<T>("1952132532477489958194625524584538730388053593825001030592563956919572392152809678530"),
      boost::lexical_cast<T>("3158616788092986726405046228413744218774962626780900087325447529650630896453338689583"),
      boost::lexical_cast<T>("5110749320570476684599671752998282949163016220605901117918011486570203288606148368113"),
      boost::lexical_cast<T>("8269366108663463411004717981412027167937978847386801205243459016220834185059487057696"),
      boost::lexical_cast<T>("13380115429233940095604389734410310117100995067992702323161470502791037473665635425809"),
      boost::lexical_cast<T>("21649481537897403506609107715822337285038973915379503528404929519011871658725122483505"),
      boost::lexical_cast<T>("35029596967131343602213497450232647402139968983372205851566400021802909132390757909314"),
      boost::lexical_cast<T>("56679078505028747108822605166054984687178942898751709379971329540814780791115880392819"),
      boost::lexical_cast<T>("91708675472160090711036102616287632089318911882123915231537729562617689923506638302133"),
      boost::lexical_cast<T>("148387753977188837819858707782342616776497854780875624611509059103432470714622518694952"),
      boost::lexical_cast<T>("240096429449348928530894810398630248865816766662999539843046788666050160638129156997085"),
      boost::lexical_cast<T>("388484183426537766350753518180972865642314621443875164454555847769482631352751675692037"),
      boost::lexical_cast<T>("628580612875886694881648328579603114508131388106874704297602636435532791990880832689122"),
      boost::lexical_cast<T>("1017064796302424461232401846760575980150446009550749868752158484205015423343632508381159"),
      boost::lexical_cast<T>("1645645409178311156114050175340179094658577397657624573049761120640548215334513341070281"),
      boost::lexical_cast<T>("2662710205480735617346452022100755074809023407208374441801919604845563638678145849451440"),
      boost::lexical_cast<T>("4308355614659046773460502197440934169467600804865999014851680725486111854012659190521721"),
      boost::lexical_cast<T>("6971065820139782390806954219541689244276624212074373456653600330331675492690805039973161"),
      boost::lexical_cast<T>("11279421434798829164267456416982623413744225016940372471505281055817787346703464230494882"),
      boost::lexical_cast<T>("18250487254938611555074410636524312658020849229014745928158881386149462839394269270468043"),
      boost::lexical_cast<T>("29529908689737440719341867053506936071765074245955118399664162441967250186097733500962925"),
      boost::lexical_cast<T>("47780395944676052274416277690031248729785923474969864327823043828116713025492002771430968"),
      boost::lexical_cast<T>("77310304634413492993758144743538184801550997720924982727487206270083963211589736272393893"),
      boost::lexical_cast<T>("125090700579089545268174422433569433531336921195894847055310250098200676237081739043824861"),
      boost::lexical_cast<T>("202401005213503038261932567177107618332887918916819829782797456368284639448671475316218754"),
      boost::lexical_cast<T>("327491705792592583530106989610677051864224840112714676838107706466485315685753214360043615"),
      boost::lexical_cast<T>("529892711006095621792039556787784670197112759029534506620905162834769955134424689676262369"),
      boost::lexical_cast<T>("857384416798688205322146546398461722061337599142249183459012869301255270820177904036305984"),
      boost::lexical_cast<T>("1387277127804783827114186103186246392258450358171783690079918032136025225954602593712568353"),
      boost::lexical_cast<T>("2244661544603472032436332649584708114319787957314032873538930901437280496774780497748874337"),
      boost::lexical_cast<T>("3631938672408255859550518752770954506578238315485816563618848933573305722729383091461442690"),
      boost::lexical_cast<T>("5876600217011727891986851402355662620898026272799849437157779835010586219504163589210317027"),
      boost::lexical_cast<T>("9508538889419983751537370155126617127476264588285666000776628768583891942233546680671759717"),
      boost::lexical_cast<T>("15385139106431711643524221557482279748374290861085515437934408603594478161737710269882076744"),
      boost::lexical_cast<T>("24893677995851695395061591712608896875850555449371181438711037372178370103971256950553836461"),
      boost::lexical_cast<T>("40278817102283407038585813270091176624224846310456696876645445975772848265708967220435913205"),
      boost::lexical_cast<T>("65172495098135102433647404982700073500075401759827878315356483347951218369680224170989749666"),
      boost::lexical_cast<T>("105451312200418509472233218252791250124300248070284575192001929323724066635389191391425662871"),
      boost::lexical_cast<T>("170623807298553611905880623235491323624375649830112453507358412671675285005069415562415412537"),
      boost::lexical_cast<T>("276075119498972121378113841488282573748675897900397028699360341995399351640458606953841075408"),
      boost::lexical_cast<T>("446698926797525733283994464723773897373051547730509482206718754667074636645528022516256487945"),
      boost::lexical_cast<T>("722774046296497854662108306212056471121727445630906510906079096662473988285986629470097563353"),
      boost::lexical_cast<T>("1169472973094023587946102770935830368494778993361415993112797851329548624931514651986354051298"),
      boost::lexical_cast<T>("1892247019390521442608211077147886839616506438992322504018876947992022613217501281456451614651"),
      boost::lexical_cast<T>("3061719992484545030554313848083717208111285432353738497131674799321571238149015933442805665949"),
      boost::lexical_cast<T>("4953967011875066473162524925231604047727791871346061001150551747313593851366517214899257280600"),
      boost::lexical_cast<T>("8015687004359611503716838773315321255839077303699799498282226546635165089515533148342062946549"),
      boost::lexical_cast<T>("12969654016234677976879363698546925303566869175045860499432778293948758940882050363241320227149"),
      boost::lexical_cast<T>("20985341020594289480596202471862246559405946478745659997715004840583924030397583511583383173698"),
      boost::lexical_cast<T>("33954995036828967457475566170409171862972815653791520497147783134532682971279633874824703400847"),
      boost::lexical_cast<T>("54940336057423256938071768642271418422378762132537180494862787975116607001677217386408086574545"),
      boost::lexical_cast<T>("88895331094252224395547334812680590285351577786328700992010571109649289972956851261232789975392"),
      boost::lexical_cast<T>("143835667151675481333619103454952008707730339918865881486873359084765896974634068647640876549937"),
      boost::lexical_cast<T>("232730998245927705729166438267632598993081917705194582478883930194415186947590919908873666525329"),
      boost::lexical_cast<T>("376566665397603187062785541722584607700812257624060463965757289279181083922224988556514543075266"),
      boost::lexical_cast<T>("609297663643530892791951979990217206693894175329255046444641219473596270869815908465388209600595"),
      boost::lexical_cast<T>("985864329041134079854737521712801814394706432953315510410398508752777354792040897021902752675861"),
      boost::lexical_cast<T>("1595161992684664972646689501703019021088600608282570556855039728226373625661856805487290962276456"),
      boost::lexical_cast<T>("2581026321725799052501427023415820835483307041235886067265438236979150980453897702509193714952317"),
      boost::lexical_cast<T>("4176188314410464025148116525118839856571907649518456624120477965205524606115754507996484677228773"),
      boost::lexical_cast<T>("6757214636136263077649543548534660692055214690754342691385916202184675586569652210505678392181090"),
      boost::lexical_cast<T>("10933402950546727102797660073653500548627122340272799315506394167390200192685406718502163069409863"),
      boost::lexical_cast<T>("17690617586682990180447203622188161240682337031027142006892310369574875779255058929007841461590953"),
      boost::lexical_cast<T>("28624020537229717283244863695841661789309459371299941322398704536965075971940465647510004531000816"),
      boost::lexical_cast<T>("46314638123912707463692067318029823029991796402327083329291014906539951751195524576517845992591769"),
      boost::lexical_cast<T>("74938658661142424746936931013871484819301255773627024651689719443505027723135990224027850523592585"),
      boost::lexical_cast<T>("121253296785055132210628998331901307849293052175954107980980734350044979474331514800545696516184354"),
      boost::lexical_cast<T>("196191955446197556957565929345772792668594307949581132632670453793550007197467505024573547039776939"),
      boost::lexical_cast<T>("317445252231252689168194927677674100517887360125535240613651188143594986671799019825119243555961293"),
      boost::lexical_cast<T>("513637207677450246125760857023446893186481668075116373246321641937144993869266524849692790595738232"),
      boost::lexical_cast<T>("831082459908702935293955784701120993704369028200651613859972830080739980541065544674812034151699525"),
      boost::lexical_cast<T>("1344719667586153181419716641724567886890850696275767987106294472017884974410332069524504824747437757"),
      boost::lexical_cast<T>("2175802127494856116713672426425688880595219724476419600966267302098624954951397614199316858899137282"),
      boost::lexical_cast<T>("3520521795081009298133389068150256767486070420752187588072561774116509929361729683723821683646575039"),
      boost::lexical_cast<T>("5696323922575865414847061494575945648081290145228607189038829076215134884313127297923138542545712321"),
      boost::lexical_cast<T>("9216845717656874712980450562726202415567360565980794777111390850331644813674856981646960226192287360"),
      boost::lexical_cast<T>("14913169640232740127827512057302148063648650711209401966150219926546779697987984279570098768737999681"),
      boost::lexical_cast<T>("24130015357889614840807962620028350479216011277190196743261610776878424511662841261217058994930287041"),
      boost::lexical_cast<T>("39043184998122354968635474677330498542864661988399598709411830703425204209650825540787157763668286722"),
      boost::lexical_cast<T>("63173200356011969809443437297358849022080673265589795452673441480303628721313666802004216758598573763"),
      boost::lexical_cast<T>("102216385354134324778078911974689347564945335253989394162085272183728832930964492342791374522266860485"),
      boost::lexical_cast<T>("165389585710146294587522349272048196587026008519579189614758713664032461652278159144795591280865434248"),
      boost::lexical_cast<T>("267605971064280619365601261246737544151971343773568583776843985847761294583242651487586965803132294733"),
      boost::lexical_cast<T>("432995556774426913953123610518785740738997352293147773391602699511793756235520810632382557083997728981"),
      boost::lexical_cast<T>("700601527838707533318724871765523284890968696066716357168446685359555050818763462119969522887130023714"),
      boost::lexical_cast<T>("1133597084613134447271848482284309025629966048359864130560049384871348807054284272752352079971127752695"),
      boost::lexical_cast<T>("1834198612451841980590573354049832310520934744426580487728496070230903857873047734872321602858257776409"),
      boost::lexical_cast<T>("2967795697064976427862421836334141336150900792786444618288545455102252664927332007624673682829385529104"),
      boost::lexical_cast<T>("4801994309516818408452995190383973646671835537213025106017041525333156522800379742496995285687643305513"),
      boost::lexical_cast<T>("7769790006581794836315417026718114982822736329999469724305586980435409187727711750121668968517028834617"),
      boost::lexical_cast<T>("12571784316098613244768412217102088629494571867212494830322628505768565710528091492618664254204672140130"),
      boost::lexical_cast<T>("20341574322680408081083829243820203612317308197211964554628215486203974898255803242740333222721700974747"),
      boost::lexical_cast<T>("32913358638779021325852241460922292241811880064424459384950843991972540608783894735358997476926373114877"),
      boost::lexical_cast<T>("53254932961459429406936070704742495854129188261636423939579059478176515507039697978099330699648074089624"),
      boost::lexical_cast<T>("86168291600238450732788312165664788095941068326060883324529903470149056115823592713458328176574447204501"),
      boost::lexical_cast<T>("139423224561697880139724382870407283950070256587697307264108962948325571622863290691557658876222521294125"),
      boost::lexical_cast<T>("225591516161936330872512695036072072046011324913758190588638866418474627738686883405015987052796968498626"),
      boost::lexical_cast<T>("365014740723634211012237077906479355996081581501455497852747829366800199361550174096573645929019489792751"),
      boost::lexical_cast<T>("590606256885570541884749772942551428042092906415213688441386695785274827100237057501589632981816458291377"),
      boost::lexical_cast<T>("955620997609204752896986850849030784038174487916669186294134525152075026461787231598163278910835948084128"),
      boost::lexical_cast<T>("1546227254494775294781736623791582212080267394331882874735521220937349853562024289099752911892652406375505"),
      boost::lexical_cast<T>("2501848252103980047678723474640612996118441882248552061029655746089424880023811520697916190803488354459633"),
      boost::lexical_cast<T>("4048075506598755342460460098432195208198709276580434935765176967026774733585835809797669102696140760835138"),
      boost::lexical_cast<T>("6549923758702735390139183573072808204317151158828986996794832713116199613609647330495585293499629115294771"),
      boost::lexical_cast<T>("10597999265301490732599643671505003412515860435409421932560009680142974347195483140293254396195769876129909"),
      boost::lexical_cast<T>("17147923024004226122738827244577811616833011594238408929354842393259173960805130470788839689695398991424680"),
      boost::lexical_cast<T>("27745922289305716855338470916082815029348872029647830861914852073402148308000613611082094085891168867554589"),
      boost::lexical_cast<T>("44893845313309942978077298160660626646181883623886239791269694466661322268805744081870933775586567858979269"),
      boost::lexical_cast<T>("72639767602615659833415769076743441675530755653534070653184546540063470576806357692953027861477736726533858"),
      boost::lexical_cast<T>("117533612915925602811493067237404068321712639277420310444454241006724792845612101774823961637064304585513127"),
      boost::lexical_cast<T>("190173380518541262644908836314147509997243394930954381097638787546788263422418459467776989498542041312046985"),
      boost::lexical_cast<T>("307706993434466865456401903551551578318956034208374691542093028553513056268030561242600951135606345897560112"),
      boost::lexical_cast<T>("497880373953008128101310739865699088316199429139329072639731816100301319690449020710377940634148387209607097"),
      boost::lexical_cast<T>("805587367387474993557712643417250666635155463347703764181824844653814375958479581952978891769754733107167209"),
      boost::lexical_cast<T>("1303467741340483121659023383282949754951354892487032836821556660754115695648928602663356832403903120316774306"),
      boost::lexical_cast<T>("2109055108727958115216736026700200421586510355834736601003381505407930071607408184616335724173657853423941515"),
      boost::lexical_cast<T>("3412522850068441236875759409983150176537865248321769437824938166162045767256336787279692556577560973740715821"),
      boost::lexical_cast<T>("5521577958796399352092495436683350598124375604156506038828319671569975838863744971896028280751218827164657336"),
      boost::lexical_cast<T>("8934100808864840588968254846666500774662240852478275476653257837732021606120081759175720837328779800905373157"),
      boost::lexical_cast<T>("14455678767661239941060750283349851372786616456634781515481577509301997444983826731071749118079998628070030493"),
      boost::lexical_cast<T>("23389779576526080530029005130016352147448857309113056992134835347034019051103908490247469955408778428975403650"),
      boost::lexical_cast<T>("37845458344187320471089755413366203520235473765747838507616412856336016496087735221319219073488777057045434143"),
      boost::lexical_cast<T>("61235237920713401001118760543382555667684331074860895499751248203370035547191643711566689028897555486020837793"),
      boost::lexical_cast<T>("99080696264900721472208515956748759187919804840608734007367661059706052043279378932885908102386332543066271936"),
      boost::lexical_cast<T>("160315934185614122473327276500131314855604135915469629507118909263076087590471022644452597131283888029087109729"),
      boost::lexical_cast<T>("259396630450514843945535792456880074043523940756078363514486570322782139633750401577338505233670220572153381665"),
      boost::lexical_cast<T>("419712564636128966418863068957011388899128076671547993021605479585858227224221424221791102364954108601240491394"),
      boost::lexical_cast<T>("679109195086643810364398861413891462942652017427626356536092049908640366857971825799129607598624329173393873059"),
      boost::lexical_cast<T>("1098821759722772776783261930370902851841780094099174349557697529494498594082193250020920709963578437774634364453"),
      boost::lexical_cast<T>("1777930954809416587147660791784794314784432111526800706093789579403138960940165075820050317562202766948028237512"),
      boost::lexical_cast<T>("2876752714532189363930922722155697166626212205625975055651487108897637555022358325840971027525781204722662601965"),
      boost::lexical_cast<T>("4654683669341605951078583513940491481410644317152775761745276688300776515962523401661021345087983971670690839477"),
      boost::lexical_cast<T>("7531436383873795315009506236096188648036856522778750817396763797198414070984881727501992372613765176393353441442"),
      boost::lexical_cast<T>("12186120053215401266088089750036680129447500839931526579142040485499190586947405129163013717701749148064044280919"),
      boost::lexical_cast<T>("19717556437089196581097595986132868777484357362710277396538804282697604657932286856665006090315514324457397722361"),
      boost::lexical_cast<T>("31903676490304597847185685736169548906931858202641803975680844768196795244879691985828019808017263472521442003280"),
      boost::lexical_cast<T>("51621232927393794428283281722302417684416215565352081372219649050894399902811978842493025898332777796978839725641"),
      boost::lexical_cast<T>("83524909417698392275468967458471966591348073767993885347900493819091195147691670828321045706350041269500281728921"),
      boost::lexical_cast<T>("135146142345092186703752249180774384275764289333345966720120142869985595050503649670814071604682819066479121454562"),
      boost::lexical_cast<T>("218671051762790578979221216639246350867112363101339852068020636689076790198195320499135117311032860335979403183483"),
      boost::lexical_cast<T>("353817194107882765682973465820020735142876652434685818788140779559062385248698970169949188915715679402458524638045"),
      boost::lexical_cast<T>("572488245870673344662194682459267086009989015536025670856161416248139175446894290669084306226748539738437927821528"),
      boost::lexical_cast<T>("926305439978556110345168148279287821152865667970711489644302195807201560695593260839033495142464219140896452459573"),
      boost::lexical_cast<T>("1498793685849229455007362830738554907162854683506737160500463612055340736142487551508117801369212758879334380281101"),
      boost::lexical_cast<T>("2425099125827785565352530979017842728315720351477448650144765807862542296838080812347151296511676978020230832740674"),
      boost::lexical_cast<T>("3923892811677015020359893809756397635478575034984185810645229419917883032980568363855269097880889736899565213021775"),
      boost::lexical_cast<T>("6348991937504800585712424788774240363794295386461634460789995227780425329818649176202420394392566714919796045762449"),
      boost::lexical_cast<T>("10272884749181815606072318598530637999272870421445820271435224647698308362799217540057689492273456451819361258784224"),
      boost::lexical_cast<T>("16621876686686616191784743387304878363067165807907454732225219875478733692617866716260109886666023166739157304546673"),
      boost::lexical_cast<T>("26894761435868431797857061985835516362340036229353275003660444523177042055417084256317799378939479618558518563330897"),
      boost::lexical_cast<T>("4351663812255504798964180537314039472540720203726072973588566439865577574803495097257790926560550278529767567877570"),
      boost::lexical_cast<T>("70411399558423479787498867358975911087747238266614004739546108921832817803452035228895708644544982403856194431208467"),
      boost::lexical_cast<T>("113928037680978527777140672732116305813154440303874734475431773320488593551486986201473617910150485189153870299086037"),
      boost::lexical_cast<T>("184339437239402007564639540091092216900901678570488739214977882242321411354939021430369326554695467593010064730294504"),
      boost::lexical_cast<T>("298267474920380535341780212823208522714056118874363473690409655562810004906426007631842944464845952782163935029380541"),
      boost::lexical_cast<T>("482606912159782542906419752914300739614957797444852212905387537805131416261365029062212271019541420375173999759675045"),
      boost::lexical_cast<T>("780874387080163078248199965737509262329013916319215686595797193367941421167791036694055215484387373157337934789055586"),
      boost::lexical_cast<T>("1263481299239945621154619718651810001943971713764067899501184731173072837429156065756267486503928793532511934548730631"),
      boost::lexical_cast<T>("2044355686320108699402819684389319264272985630083283586096981924541014258596947102450322701988316166689849869337786217"),
      boost::lexical_cast<T>("3307836985560054320557439403041129266216957343847351485598166655714087096026103168206590188492244960222361803886516848"),
      boost::lexical_cast<T>("5352192671880163019960259087430448530489942973930635071695148580255101354623050270656912890480561126912211673224303065"),
      boost::lexical_cast<T>("8660029657440217340517698490471577796706900317777986557293315235969188450649153438863503078972806087134573477110819913"),
      boost::lexical_cast<T>("14012222329320380360477957577902026327196843291708621628988463816224289805272203709520415969453367214046785150335122978"),
      boost::lexical_cast<T>("22672251986760597700995656068373604123903743609486608186281779052193478255921357148383919048426173301181358627445942891"),
      boost::lexical_cast<T>("36684474316080978061473613646275630451100586901195229815270242868417768061193560857904335017879540515228143777781065869"),
      boost::lexical_cast<T>("59356726302841575762469269714649234575004330510681838001552021920611246317114918006288254066305713816409502405227008760"),
      boost::lexical_cast<T>("96041200618922553823942883360924865026104917411877067816822264789029014378308478864192589084185254331637646183008074629"),
      boost::lexical_cast<T>("155397926921764129586412153075574099601109247922558905818374286709640260695423396870480843150490968148047148588235083389"),
      boost::lexical_cast<T>("251439127540686683410355036436498964627214165334435973635196551498669275073731875734673432234676222479684794771243158018"),
      boost::lexical_cast<T>("406837054462450812996767189512073064228323413256994879453570838208309535769155272605154275385167190627731943359478241407"),
      boost::lexical_cast<T>("658276182003137496407122225948572028855537578591430853088767389706978810842887148339827707619843413107416738130721399425"),
      boost::lexical_cast<T>("1065113236465588309403889415460645093083860991848425732542338227915288346612042420944981983005010603735148681490199640832"),
      boost::lexical_cast<T>("1723389418468725805811011641409217121939398570439856585631105617622267157454929569284809690624854016842565419620921040257"),
      boost::lexical_cast<T>("2788502654934314115214901056869862215023259562288282318173443845537555504066971990229791673629864620577714101111120681089"),
      boost::lexical_cast<T>("4511892073403039921025912698279079336962658132728138903804549463159822661521901559514601364254718637420279520732041721346"),
      boost::lexical_cast<T>("7300394728337354036240813755148941551985917695016421221977993308697378165588873549744393037884583257997993621843162402435"),
      boost::lexical_cast<T>("11812286801740393957266726453428020888948575827744560125782542771857200827110775109258994402139301895418273142575204123781"),
      boost::lexical_cast<T>("19112681530077747993507540208576962440934493522760981347760536080554578992699648659003387440023885153416266764418366526216"),
      boost::lexical_cast<T>("30924968331818141950774266662004983329883069350505541473543078852411779819810423768262381842163187048834539906993570649997"),
      boost::lexical_cast<T>("50037649861895889944281806870581945770817562873266522821303614932966358812510072427265769282187072202250806671411937176213"),
      boost::lexical_cast<T>("80962618193714031895056073532586929100700632223772064294846693785378138632320496195528151124350259251085346578405507826210"),
      boost::lexical_cast<T>("131000268055609921839337880403168874871518195097038587116150308718344497444830568622793920406537331453336153249817445002423"),
      boost::lexical_cast<T>("211962886249323953734393953935755803972218827320810651410997002503722636077151064818322071530887590704421499828222952828633"),
      boost::lexical_cast<T>("342963154304933875573731834338924678843737022417849238527147311222067133521981633441115991937424922157757653078040397831056"),
      boost::lexical_cast<T>("554926040554257829308125788274680482815955849738659889938144313725789769599132698259438063468312512862179152906263350659689"),
      boost::lexical_cast<T>("897889194859191704881857622613605161659692872156509128465291624947856903121114331700554055405737435019936805984303748490745"),
      boost::lexical_cast<T>("1452815235413449534189983410888285644475648721895169018403435938673646672720247029959992118874049947882115958890567099150434"),
      boost::lexical_cast<T>("2350704430272641239071841033501890806135341594051678146868727563621503575841361361660546174279787382902052764874870847641179"),
      boost::lexical_cast<T>("3803519665686090773261824444390176450610990315946847165272163502295150248561608391620538293153837330784168723765437946791613"),
      boost::lexical_cast<T>("6154224095958732012333665477892067256746331909998525312140891065916653824402969753281084467433624713686221488640308794432792"),
      boost::lexical_cast<T>("9957743761644822785595489922282243707357322225945372477413054568211804072964578144901622760587462044470390212405746741224405"),
      boost::lexical_cast<T>("16111967857603554797929155400174310964103654135943897789553945634128457897367547898182707228021086758156611701046055535657197"),
      boost::lexical_cast<T>("26069711619248377583524645322456554671460976361889270266967000202340261970332126043084329988608548802627001913451802276881602"),
      boost::lexical_cast<T>("42181679476851932381453800722630865635564630497833168056520945836468719867699673941267037216629635560783613614497857812538799"),
      boost::lexical_cast<T>("68251391096100309964978446045087420307025606859722438323487946038808981838031799984351367205238184363410615527949660089420401"),
      boost::lexical_cast<T>("110433070572952242346432246767718285942590237357555606380008891875277701705731473925618404421867819924194229142447517901959200"),
      boost::lexical_cast<T>("178684461669052552311410692812805706249615844217278044703496837914086683543763273909969771627106004287604844670397177991379601"),
      boost::lexical_cast<T>("289117532242004794657842939580523992192206081574833651083505729789364385249494747835588176048973824211799073812844695893338801"),
      boost::lexical_cast<T>("467801993911057346969253632393329698441821925792111695787002567703451068793258021745557947676079828499403918483241873884718402"),
      boost::lexical_cast<T>("756919526153062141627096571973853690634028007366945346870508297492815454042752769581146123725053652711202992296086569778057203"),
      boost::lexical_cast<T>("1224721520064119488596350204367183389075849933159057042657510865196266522836010791326704071401133481210606910779328443662775605"),
      boost::lexical_cast<T>("1981641046217181630223446776341037079709877940526002389528019162689081976878763560907850195126187133921809903075415013440832808"),
      boost::lexical_cast<T>("3206362566281301118819796980708220468785727873685059432185530027885348499714774352234554266527320615132416813854743457103608413"),
      boost::lexical_cast<T>("5188003612498482749043243757049257548495605814211061821713549190574430476593537913142404461653507749054226716930158470544441221"),
      boost::lexical_cast<T>("8394366178779783867863040737757478017281333687896121253899079218459778976308312265376958728180828364186643530784901927648049634"),
      boost::lexical_cast<T>("13582369791278266616906284494806735565776939502107183075612628409034209452901850178519363189834336113240870247715060398192490855"),
      boost::lexical_cast<T>("21976735970058050484769325232564213583058273190003304329511707627493988429210162443896321918015164477427513778499962325840540489"),
      boost::lexical_cast<T>("35559105761336317101675609727370949148835212692110487405124336036528197882112012622415685107849500590668384026215022724033031344"),
      boost::lexical_cast<T>("57535841731394367586444934959935162731893485882113791734636043664022186311322175066312007025864665068095897804714985049873571833"),
      boost::lexical_cast<T>("93094947492730684688120544687306111880728698574224279139760379700550384193434187688727692133714165658764281830930007773906603177"),
      boost::lexical_cast<T>("150630789224125052274565479647241274612622184456338070874396423364572570504756362755039699159578830726860179635644992823780175010"),
      boost::lexical_cast<T>("243725736716855736962686024334547386493350883030562350014156803065122954698190550443767391293292996385624461466575000597686778187"),
      boost::lexical_cast<T>("394356525940980789237251503981788661105973067486900420888553226429695525202946913198807090452871827112484641102219993421466953197"),
      boost::lexical_cast<T>("638082262657836526199937528316336047599323950517462770902710029494818479901137463642574481746164823498109102568794994019153731384"),
      boost::lexical_cast<T>("1032438788598817315437189032298124708705297018004363191791263255924514005104084376841381572199036650610593743671014987440620684581"),
      boost::lexical_cast<T>("1670521051256653841637126560614460756304620968521825962693973285419332485005221840483956053945201474108702846239809981459774415965"),
      boost::lexical_cast<T>("2702959839855471157074315592912585465009917986526189154485236541343846490109306217325337626144238124719296589910824968900395100546"),
      boost::lexical_cast<T>("4373480891112124998711442153527046221314538955048015117179209826763178975114528057809293680089439598827999436150634950360169516511"),
      boost::lexical_cast<T>("7076440730967596155785757746439631686324456941574204271664446368107025465223834275134631306233677723547296026061459919260564617057"),
      boost::lexical_cast<T>("11449921622079721154497199899966677907638995896622219388843656194870204440338362332943924986323117322375295462212094869620734133568"),
      boost::lexical_cast<T>("18526362353047317310282957646406309593963452838196423660508102562977229905562196608078556292556795045922591488273554788881298750625"),
      boost::lexical_cast<T>("29976283975127038464780157546372987501602448734818643049351758757847434345900558941022481278879912368297886950485649658502032884193"),
      boost::lexical_cast<T>("48502646328174355775063115192779297095565901573015066709859861320824664251462755549101037571436707414220478438759204447383331634818"),
      boost::lexical_cast<T>("78478930303301394239843272739152284597168350307833709759211620078672098597363314490123518850316619782518365389244854105885364519011"),
      boost::lexical_cast<T>("126981576631475750014906387931931581692734251880848776469071481399496762848826070039224556421753327196738843828004058553268696153829"),
      boost::lexical_cast<T>("205460506934777144254749660671083866289902602188682486228283101478168861446189384529348075272069946979257209217248912659154060672840"),
      boost::lexical_cast<T>("332442083566252894269656048603015447982636854069531262697354582877665624295015454568572631693823274175996053045252971212422756826669"),
      boost::lexical_cast<T>("537902590501030038524405709274099314272539456258213748925637684355834485741204839097920706965893221155253262262501883871576817499509"),
      boost::lexical_cast<T>("870344674067282932794061757877114762255176310327745011622992267233500110036220293666493338659716495331249315307754855083999574326178"),
      boost::lexical_cast<T>("1408247264568312971318467467151214076527715766585958760548629951589334595777425132764414045625609716486502577570256738955576391825687"),
      boost::lexical_cast<T>("2278591938635595904112529225028328838782892076913703772171622218822834705813645426430907384285326211817751892878011594039575966151865"),
      boost::lexical_cast<T>("3686839203203908875430996692179542915310607843499662532720252170412169301591070559195321429910935928304254470448268332995152357977552"),
      boost::lexical_cast<T>("5965431141839504779543525917207871754093499920413366304891874389235004007404715985626228814196262140122006363326279927034728324129417"),
      boost::lexical_cast<T>("9652270345043413654974522609387414669404107763913028837612126559647173308995786544821550244107198068426260833774548260029880682106969"),
      boost::lexical_cast<T>("15617701486882918434518048526595286423497607684326395142504000948882177316400502530447779058303460208548267197100828187064609006236386"),
      boost::lexical_cast<T>("25269971831926332089492571135982701092901715448239423980116127508529350625396289075269329302410658276974528030875376447094489688343355"),
      boost::lexical_cast<T>("40887673318809250524010619662577987516399323132565819122620128457411527941796791605717108360714118485522795227976204634159098694579741"),
      boost::lexical_cast<T>("66157645150735582613503190798560688609301038580805243102736255965940878567193080680986437663124776762497323258851581081253588382923096"),
      boost::lexical_cast<T>("107045318469544833137513810461138676125700361713371062225356384423352406508989872286703546023838895248020118486827785715412687077502837"),
      boost::lexical_cast<T>("173202963620280415751017001259699364735001400294176305328092640389293285076182952967689983686963672010517441745679366796666275460425933"),
      boost::lexical_cast<T>("280248282089825248888530811720838040860701762007547367553449024812645691585172825254393529710802567258537560232507152512078962537928770"),
      boost::lexical_cast<T>("453451245710105664639547812980537405595703162301723672881541665201938976661355778222083513397766239269055001978186519308745237998354703"),
      boost::lexical_cast<T>("733699527799930913528078624701375446456404924309271040434990690014584668246528603476477043108568806527592562210693671820824200536283473"),
      boost::lexical_cast<T>("1187150773510036578167626437681912852052108086610994713316532355216523644907884381698560556506335045796647564188880191129569438534638176"),
      boost::lexical_cast<T>("1920850301309967491695705062383288298508513010920265753751523045231108313154412985175037599614903852324240126399573862950393639070921649"),
      boost::lexical_cast<T>("3108001074820004069863331500065201150560621097531260467068055400447631958062297366873598156121238898120887690588454054079963077605559825"),
      boost::lexical_cast<T>("5028851376129971561559036562448489449069134108451526220819578445678740271216710352048635755736142750445127816988027917030356716676481474"),
      boost::lexical_cast<T>("8136852450949975631422368062513690599629755205982786687887633846126372229279007718922233911857381648566015507576481971110319794282041299"),
      boost::lexical_cast<T>("13165703827079947192981404624962180048698889314434312908707212291805112500495718070970869667593524399011143324564509888140676510958522773"),
      boost::lexical_cast<T>("21302556278029922824403772687475870648328644520417099596594846137931484729774725789893103579450906047577158832140991859250996305240564072"),
      boost::lexical_cast<T>("34468260105109870017385177312438050697027533834851412505302058429736597230270443860863973247044430446588302156705501747391672816199086845"),
      boost::lexical_cast<T>("55770816383139792841788949999913921345356178355268512101896904567668081960045169650757076826495336494165460988846493606642669121439650917"),
      boost::lexical_cast<T>("90239076488249662859174127312351972042383712190119924607198962997404679190315613511621050073539766940753763145551995354034341937638737762"),
      boost::lexical_cast<T>("146009892871389455700963077312265893387739890545388436709095867565072761150360783162378126900035103434919224134398488960677011059078388679"),
      boost::lexical_cast<T>("236248969359639118560137204624617865430123602735508361316294830562477440340676396673999176973574870375672987279950484314711352996717126441"),
      boost::lexical_cast<T>("382258862231028574261100281936883758817863493280896798025390698127550201491037179836377303873609973810592211414348973275388364055795515120"),
      boost::lexical_cast<T>("618507831590667692821237486561501624247987096016405159341685528690027641831713576510376480847184844186265198694299457590099717052512641561"),
      boost::lexical_cast<T>("1000766693821696267082337768498385383065850589297301957367076226817577843322750756346753784720794817996857410108648430865488081108308156681"),
      boost::lexical_cast<T>("1619274525412363959903575255059887007313837685313707116708761755507605485154464332857130265567979662183122608802947888455587798160820798242"),
      boost::lexical_cast<T>("2620041219234060226985913023558272390379688274611009074075837982325183328477215089203884050288774480179980018911596319321075879269128954923"),
      boost::lexical_cast<T>("4239315744646424186889488278618159397693525959924716190784599737832788813631679422061014315856754142363102627714544207776663677429949753165"),
      boost::lexical_cast<T>("6859356963880484413875401302176431788073214234535725264860437720157972142108894511264898366145528622543082646626140527097739556699078708088"),
      boost::lexical_cast<T>("11098672708526908600764889580794591185766740194460441455645037457990760955740573933325912682002282764906185274340684734874403234129028461253"),
      boost::lexical_cast<T>("17958029672407393014640290882971022973839954428996166720505475178148733097849468444590811048147811387449267920966825261972142790828107169341"),
      boost::lexical_cast<T>("29056702380934301615405180463765614159606694623456608176150512636139494053590042377916723730150094152355453195307509996846546024957135630594"),
      boost::lexical_cast<T>("47014732053341694630045471346736637133446649052452774896655987814288227151439510822507534778297905539804721116274335258818688815785242799935"),
      boost::lexical_cast<T>("76071434434275996245450651810502251293053343675909383072806500450427721205029553200424258508447999692160174311581845255665234840742378430529"),
      boost::lexical_cast<T>("123086166487617690875496123157238888426499992728362157969462488264715948356469064022931793286745905231964895427856180514483923656527621230464"),
      boost::lexical_cast<T>("199157600921893687120946774967741139719553336404271541042268988715143669561498617223356051795193904924125069739438025770149158497269999660993"),
      boost::lexical_cast<T>("322243767409511377996442898124980028146053329132633699011731476979859617917967681246287845081939810156089965167294206284633082153797620891457"),
      boost::lexical_cast<T>("521401368331405065117389673092721167865606665536905240054000465695003287479466298469643896877133715080215034906732232054782240651067620552450"),
      boost::lexical_cast<T>("843645135740916443113832571217701196011659994669538939065731942674862905397433979715931741959073525236305000074026438339415322804865241443907"),
      boost::lexical_cast<T>("1365046504072321508231222244310422363877266660206444179119732408369866192876900278185575638836207240316520034980758670394197563455932861996357"),
      boost::lexical_cast<T>("2208691639813237951345054815528123559888926654875983118185464351044729098274334257901507380795280765552825035054785108733612886260798103440264"),
      boost::lexical_cast<T>("3573738143885559459576277059838545923766193315082427297305196759414595291151234536087083019631488005869345070035543779127810449716730965436621"),
      boost::lexical_cast<T>("5782429783698797410921331875366669483655119969958410415490661110459324389425568793988590400426768771422170105090328887861423335977529068876885"),
      boost::lexical_cast<T>("9356167927584356870497608935205215407421313285040837712795857869873919680576803330075673420058256777291515175125872666989233785694260034313506"),
      boost::lexical_cast<T>("15138597711283154281418940810571884891076433254999248128286518980333244070002372124064263820485025548713685280216201554850657121671789103190391"),
      boost::lexical_cast<T>("24494765638867511151916549745777100298497746540040085841082376850207163750579175454139937240543282326005200455342074221839890907366049137503897"),
      boost::lexical_cast<T>("39633363350150665433335490556348985189574179795039333969368895830540407820581547578204201061028307874718885735558275776690548029037838240694288"),
      boost::lexical_cast<T>("64128128989018176585252040302126085488071926335079419810451272680747571571160723032344138301571590200724086190900349998530438936403887378198185"),
      boost::lexical_cast<T>("103761492339168842018587530858475070677646106130118753779820168511287979391742270610548339362599898075442971926458625775220986965441725618892473"),
      boost::lexical_cast<T>("167889621328187018603839571160601156165718032465198173590271441192035550962902993642892477664171488276167058117358975773751425901845612997090658"),
      boost::lexical_cast<T>("271651113667355860622427102019076226843364138595316927370091609703323530354645264253440817026771386351610030043817601548972412867287338615983131"),
      boost::lexical_cast<T>("439540734995542879226266673179677383009082171060515100960363050895359081317548257896333294690942874627777088161176577322723838769132951613073789"),
      boost::lexical_cast<T>("711191848662898739848693775198753609852446309655832028330454660598682611672193522149774111717714260979387118204994178871696251636420290229056920"),
      boost::lexical_cast<T>("1150732583658441619074960448378430992861528480716347129290817711494041692989741780046107406408657135607164206366170756194420090405553241842130709"),
      boost::lexical_cast<T>("1861924432321340358923654223577184602713974790372179157621272372092724304661935302195881518126371396586551324571164935066116342041973532071187629"),
      boost::lexical_cast<T>("3012657015979781977998614671955615595575503271088526286912090083586765997651677082241988924535028532193715530937335691260536432447526773913318338"),
      boost::lexical_cast<T>("4874581448301122336922268895532800198289478061460705444533362455679490302313612384437870442661399928780266855508500626326652774489500305984505967"),
      boost::lexical_cast<T>("7887238464280904314920883567488415793864981332549231731445452539266256299965289466679859367196428460973982386445836317587189206937027079897824305"),
      boost::lexical_cast<T>("12761819912582026651843152463021215992154459394009937175978814994945746602278901851117729809857828389754249241954336943913841981426527385882330272"),
      boost::lexical_cast<T>("20649058376862930966764036030509631786019440726559168907424267534212002902244191317797589177054256850728231628400173261501031188363554465780154577"),
      boost::lexical_cast<T>("33410878289444957618607188493530847778173900120569106083403082529157749504523093168915318986912085240482480870354510205414873169790081851662484849"),
      boost::lexical_cast<T>("54059936666307888585371224524040479564193340847128274990827350063369752406767284486712908163966342091210712498754683466915904358153636317442639426"),
      boost::lexical_cast<T>("87470814955752846203978413017571327342367240967697381074230432592527501911290377655628227150878427331693193369109193672330777527943718169105124275"),
      boost::lexical_cast<T>("141530751622060734789349637541611806906560581814825656065057782655897254318057662142341135314844769422903905867863877139246681886097354486547763701"),
      boost::lexical_cast<T>("229001566577813580993328050559183134248927822782523037139288215248424756229348039797969362465723196754597099236973070811577459414041072655652887976"),
      boost::lexical_cast<T>("370532318199874315782677688100794941155488404597348693204345997904322010547405701940310497780567966177501005104836947950824141300138427142200651677"),
      boost::lexical_cast<T>("599533884777687896776005738659978075404416227379871730343634213152746766776753741738279860246291162932098104341810018762401600714179499797853539653"),
      boost::lexical_cast<T>("970066202977562212558683426760773016559904631977220423547980211057068777324159443678590358026859129109599109446646966713225742014317926940054191330"),
      boost::lexical_cast<T>("1569600087755250109334689165420751091964320859357092153891614424209815544100913185416870218273150292041697213788456985475627342728497426737907730983"),
      boost::lexical_cast<T>("2539666290732812321893372592181524108524225491334312577439594635266884321425072629095460576300009421151296323235103952188853084742815353677961922313"),
      boost::lexical_cast<T>("4109266378488062431228061757602275200488546350691404731331209059476699865525985814512330794573159713192993537023560937664480427471312780415869653296"),
      boost::lexical_cast<T>("6648932669220874753121434349783799309012771842025717308770803694743584186951058443607791370873169134344289860258664889853333512214128134093831575609"),
      boost::lexical_cast<T>("10758199047708937184349496107386074509501318192717122040102012754220284052477044258120122165446328847537283397282225827517813939685440914509701228905"),
      boost::lexical_cast<T>("17407131716929811937470930457169873818514090034742839348872816448963868239428102701727913536319497981881573257540890717371147451899569048603532804514"),
      boost::lexical_cast<T>("28165330764638749121820426564555948328015408227459961388974829203184152291905146959848035701765826829418856654823116544888961391585009963113234033419"),
      boost::lexical_cast<T>("45572462481568561059291357021725822146529498262202800737847645652148020531333249661575949238085324811300429912364007262260108843484579011716766837933"),
      boost::lexical_cast<T>("73737793246207310181111783586281770474544906489662762126822474855332172823238396621423984939851151640719286567187123807149070235069588974830000871352"),
      boost::lexical_cast<T>("119310255727775871240403140608007592621074404751865562864670120507480193354571646282999934177936476452019716479551131069409179078554167986546767709285"),
      boost::lexical_cast<T>("193048048973983181421514924194289363095619311241528324991492595362812366177810042904423919117787628092739003046738254876558249313623756961376768580637"),
      boost::lexical_cast<T>("312358304701759052661918064802296955716693715993393887856162715870292559532381689187423853295724104544758719526289385945967428392177924947923536289922"),
      boost::lexical_cast<T>("505406353675742234083432988996586318812313027234922212847655311233104925710191732091847772413511732637497722573027640822525677705801681909300304870559"),
      boost::lexical_cast<T>("817764658377501286745351053798883274529006743228316100703818027103397485242573421279271625709235837182256442099317026768493106097979606857223841160481"),
      boost::lexical_cast<T>("1323171012053243520828784042795469593341319770463238313551473338336502410952765153371119398122747569819754164672344667591018783803781288766524146031040"),
      boost::lexical_cast<T>("2140935670430744807574135096594352867870326513691554414255291365439899896195338574650391023831983407002010606771661694359511889901760895623747987191521"),
      boost::lexical_cast<T>("3464106682483988328402919139389822461211646284154792727806764703776402307148103728021510421954730976821764771444006361950530673705542184390272133222561"),
      boost::lexical_cast<T>("5605042352914733135977054235984175329081972797846347142062056069216302203343442302671901445786714383823775378215668056310042563607303080014020120414082"),
      boost::lexical_cast<T>("9069149035398721464379973375373997790293619082001139869868820772992704510491546030693411867741445360645540149659674418260573237312845264404292253636643"),
      boost::lexical_cast<T>("14674191388313454600357027611358173119375591879847487011930876842209006713834988333365313313528159744469315527875342474570615800920148344418312374050725"),
      boost::lexical_cast<T>("23743340423712176064737000986732170909669210961848626881799697615201711224326534364058725181269605105114855677535016892831189038232993608822604627687368"),
      boost::lexical_cast<T>("38417531812025630665094028598090344029044802841696113893730574457410717938161522697424038494797764849584171205410359367401804839153141953240917001738093"),
      boost::lexical_cast<T>("62160872235737806729831029584822514938714013803544740775530272072612429162488057061482763676067369954699026882945376260232993877386135562063521629425461"),
      boost::lexical_cast<T>("100578404047763437394925058182912858967758816645240854669260846530023147100649579758906802170865134804283198088355735627634798716539277515304438631163554"),
      boost::lexical_cast<T>("162739276283501244124756087767735373906472830448785595444791118602635576263137636820389565846932504758982224971301111887867792593925413077367960260589015"),
      boost::lexical_cast<T>("263317680331264681519681145950648232874231647094026450114051965132658723363787216579296368017797639563265423059656847515502591310464690592672398891752569"),
      boost::lexical_cast<T>("426056956614765925644437233718383606780704477542812045558843083735294299626924853399685933864730144322247648030957959403370383904390103670040359152341584"),
      boost::lexical_cast<T>("689374636946030607164118379669031839654936124636838495672895048867953022990712069978982301882527783885513071090614806918872975214854794262712758044094153"),
      boost::lexical_cast<T>("1115431593560796532808555613387415446435640602179650541231738132603247322617636923378668235747257928207760719121572766322243359119244897932753117196435737"),
      boost::lexical_cast<T>("1804806230506827139972673993056447286090576726816489036904633181471200345608348993357650537629785712093273790212187573241116334334099692195465875240529890"),
      boost::lexical_cast<T>("2920237824067623672781229606443862732526217328996139578136371314074447668225985916736318773377043640301034509333760339563359693453344590128218992436965627"),
      boost::lexical_cast<T>("4725044054574450812753903599500310018616794055812628615041004495545648013834334910093969311006829352394308299545947912804476027787444282323684867677495517"),
      boost::lexical_cast<T>("7645281878642074485535133205944172751143011384808768193177375809620095682060320826830288084383872992695342808879708252367835721240788872451903860114461144"),
      boost::lexical_cast<T>("12370325933216525298289036805444482769759805440621396808218380305165743695894655736924257395390702345089651108425656165172311749028233154775588727791956661"),
      boost::lexical_cast<T>("20015607811858599783824170011388655520902816825430165001395756114785839377954976563754545479774575337784993917305364417540147470269022027227492587906417805"),
      boost::lexical_cast<T>("32385933745075125082113206816833138290662622266051561809614136419951583073849632300678802875165277682874645025731020582712459219297255182003081315698374466"),
      boost::lexical_cast<T>("52401541556933724865937376828221793811565439091481726811009892534737422451804608864433348354939853020659638943036385000252606689566277209230573903604792271"),
      boost::lexical_cast<T>("84787475302008849948050583645054932102228061357533288620624028954689005525654241165112151230105130703534283968767405582965065908863532391233655219303166737"),
      boost::lexical_cast<T>("137189016858942574813987960473276725913793500449015015431633921489426427977458850029545499585044983724193922911803790583217672598429809600464229122907959008"),
      boost::lexical_cast<T>("221976492160951424762038544118331658016021561806548304052257950444115433503113091194657650815150114427728206880571196166182738507293341991697884342211125745"),
      boost::lexical_cast<T>("359165509019893999576026504591608383929815062255563319483891871933541861480571941224203150400195098151922129792374986749400411105723151592162113465119084753"),
      boost::lexical_cast<T>("581142001180845424338065048709940041945836624062111623536149822377657294983685032418860801215345212579650336672946182915583149613016493583859997807330210498"),
      boost::lexical_cast<T>("940307510200739423914091553301548425875651686317674943020041694311199156464256973643063951615540310731572466465321169664983560718739645176022111272449295251"),
      boost::lexical_cast<T>("1521449511381584848252156602011488467821488310379786566556191516688856451447942006061924752830885523311222803138267352580566710331756138759882109079779505749"),
      boost::lexical_cast<T>("2461757021582324272166248155313036893697139996697461509576233211000055607912198979704988704446425834042795269603588522245550271050495783935904220352228801000"),
      boost::lexical_cast<T>("3983206532963909120418404757324525361518628307077248076132424727688912059360140985766913457277311357354018072741855874826116981382251922695786329432008306749"),
      boost::lexical_cast<T>("6444963554546233392584652912637562255215768303774709585708657938688967667272339965471902161723737191396813342345444397071667252432747706631690549784237107749"),
      boost::lexical_cast<T>("10428170087510142513003057669962087616734396610851957661841082666377879726632480951238815619001048548750831415087300271897784233814999629327476879216245414498"),
      boost::lexical_cast<T>("16873133642056375905587710582599649871950164914626667247549740605066847393904820916710717780724785740147644757432744668969451486247747335959167429000482522247"),
      boost::lexical_cast<T>("27301303729566518418590768252561737488684561525478624909390823271444727120537301867949533399725834288898476172520044940867235720062746965286644308216727936745"),
      boost::lexical_cast<T>("44174437371622894324178478835161387360634726440105292156940563876511574514442122784660251180450620029046120929952789609836687206310494301245811737217210458992"),
      boost::lexical_cast<T>("71475741101189412742769247087723124849319287965583917066331387147956301634979424652609784580176454317944597102472834550703922926373241266532456045433938395737"),
      boost::lexical_cast<T>("115650178472812307066947725922884512209954014405689209223271951024467876149421547437270035760627074346990718032425624160540610132683735567778267782651148854729"),
      boost::lexical_cast<T>("187125919574001719809716973010607637059273302371273126289603338172424177784400972089879820340803528664935315134898458711244533059056976834310723828085087250466"),
      boost::lexical_cast<T>("302776098046814026876664698933492149269227316776962335512875289196892053933822519527149856101430603011926033167324082871785143191740712402088991610736236105195"),
      boost::lexical_cast<T>("489902017620815746686381671944099786328500619148235461802478627369316231718223491617029676442234131676861348302222541583029676250797689236399715438821323355661"),
      boost::lexical_cast<T>("792678115667629773563046370877591935597727935925197797315353916566208285652046011144179532543664734688787381469546624454814819442538401638488707049557559460856"),
      boost::lexical_cast<T>("1282580133288445520249428042821691721926228555073433259117832543935524517370269502761209208985898866365648729771769166037844495693336090874888422488378882816517"),
      boost::lexical_cast<T>("2075258248956075293812474413699283657523956490998631056433186460501732803022315513905388741529563601054436111241315790492659315135874492513377129537936442277373"),
      boost::lexical_cast<T>("3357838382244520814061902456520975379450185046072064315551019004437257320392585016666597950515462467420084841013084956530503810829210583388265552026315325093890"),
      boost::lexical_cast<T>("5433096631200596107874376870220259036974141537070695371984205464938990123414900530571986692045026068474520952254400747023163125965085075901642681564251767371263"),
      boost::lexical_cast<T>("8790935013445116921936279326741234416424326583142759687535224469376247443807485547238584642560488535894605793267485703553666936794295659289908233590567092465153"),
      boost::lexical_cast<T>("14224031644645713029810656196961493453398468120213455059519429934315237567222386077810571334605514604369126745521886450576830062759380735191550915154818859836416"),
      boost::lexical_cast<T>("23014966658090829951746935523702727869822794703356214747054654403691485011029871625049155977166003140263732538789372154130496999553676394481459148745385952301569"),
      boost::lexical_cast<T>("37238998302736542981557591720664221323221262823569669806574084338006722578252257702859727311771517744632859284311258604707327062313057129673010063900204812137985"),
      boost::lexical_cast<T>("60253964960827372933304527244366949193044057526925884553628738741698207589282129327908883288937520884896591823100630758837824061866733524154469212645590764439554"),
      boost::lexical_cast<T>("97492963263563915914862118965031170516265320350495554360202823079704930167534387030768610600709038629529451107411889363545151124179790653827479276545795576577539"),
      boost::lexical_cast<T>("157746928224391288848166646209398119709309377877421438913831561821403137756816516358677493889646559514426042930512520122382975186046524177981948489191386341017093"),
      boost::lexical_cast<T>("255239891487955204763028765174429290225574698227916993274034384901108067924350903389446104490355598143955494037924409485928126310226314831809427765737181917594632"),
      boost::lexical_cast<T>("412986819712346493611195411383827409934884076105338432187865946722511205681167419748123598380002157658381536968436929608311101496272839009791376254928568258611725"),
      boost::lexical_cast<T>("668226711200301698374224176558256700160458774333255425461900331623619273605518323137569702870357755802337031006361339094239227806499153841600804020665750176206357"),
      boost::lexical_cast<T>("1081213530912648191985419587942084110095342850438593857649766278346130479286685742885693301250359913460718567974798268702550329302771992851392180275594318434818082"),
      boost::lexical_cast<T>("1749440242112949890359643764500340810255801624771849283111666609969749752892204066023263004120717669263055598981159607796789557109271146692992984296260068611024439"),
      boost::lexical_cast<T>("2830653773025598082345063352442424920351144475210443140761432888315880232178889808908956305371077582723774166955957876499339886412043139544385164571854387045842521"),
      boost::lexical_cast<T>("4580094015138547972704707116942765730606946099982292423873099498285629985071093874932219309491795251986829765937117484296129443521314286237378148868114455656866960"),
      boost::lexical_cast<T>("7410747788164146055049770469385190650958090575192735564634532386601510217249983683841175614862872834710603932893075360795469329933357425781763313439968842702709481"),
      boost::lexical_cast<T>("11990841803302694027754477586327956381565036675175027988507631884887140202321077558773394924354668086697433698830192845091598773454671712019141462308083298359576441"),
      boost::lexical_cast<T>("19401589591466840082804248055713147032523127250367763553142164271488650419571061242614570539217540921408037631723268205887068103388029137800904775748052141062285922"),
      boost::lexical_cast<T>("31392431394769534110558725642041103414088163925542791541649796156375790621892138801387965463572209008105471330553461050978666876842700849820046238056135439421862363"),
      boost::lexical_cast<T>("50794020986236374193362973697754250446611291175910555094791960427864441041463200044002536002789749929513508962276729256865734980230729987620951013804187580484148285"),
      boost::lexical_cast<T>("82186452381005908303921699339795353860699455101453346636441756584240231663355338845390501466361958937618980292830190307844401857073430837440997251860323019906010648"),
      boost::lexical_cast<T>("132980473367242282497284673037549604307310746277363901731233717012104672704818538889393037469151708867132489255106919564710136837304160825061948265664510600390158933"),
      boost::lexical_cast<T>("215166925748248190801206372377344958168010201378817248367675473596344904368173877734783538935513667804751469547937109872554538694377591662502945517524833620296169581"),
      boost::lexical_cast<T>("348147399115490473298491045414894562475320947656181150098909190608449577072992416624176576404665376671883958803044029437264675531681752487564893783189344220686328514"),
      boost::lexical_cast<T>("563314324863738664099697417792239520643331149034998398466584664204794481441166294358960115340179044476635428350981139309819214226059344150067839300714177840982498095"),
      boost::lexical_cast<T>("911461723979229137398188463207134083118652096691179548565493854813244058514158710983136691744844421148519387154025168747083889757741096637632733083903522061668826609"),
      boost::lexical_cast<T>("1474776048842967801497885880999373603761983245726177947032078519018038539955325005342096807085023465625154815505006308056903103983800440787700572384617699902651324704"),
      boost::lexical_cast<T>("2386237772822196938896074344206507686880635342417357495597572373831282598469483716325233498829867886773674202659031476803986993741541537425333305468521221964320151313"),
      boost::lexical_cast<T>("3861013821665164740393960225205881290642618588143535442629650892849321138424808721667330305914891352398829018164037784860890097725341978213033877853138921866971476017"),
      boost::lexical_cast<T>("6247251594487361679290034569412388977523253930560892938227223266680603736894292437992563804744759239172503220823069261664877091466883515638367183321660143831291627330"),
      boost::lexical_cast<T>("10108265416152526419683994794618270268165872518704428380856874159529924875319101159659894110659650591571332238987107046525767189192225493851401061174799065698263103347"),
      boost::lexical_cast<T>("16355517010639888098974029364030659245689126449265321319084097426210528612213393597652457915404409830743835459810176308190644280659109009489768244496459209529554730677"),
      boost::lexical_cast<T>("26463782426792414518658024158648929513854998967969749699940971585740453487532494757312352026064060422315167698797283354716411469851334503341169305671258275227817834024"),
      boost::lexical_cast<T>("42819299437432302617632053522679588759544125417235071019025069011950982099745888354964809941468470253059003158607459662907055750510443512830937550167717484757372564701"),
      boost::lexical_cast<T>("69283081864224717136290077681328518273399124385204820718966040597691435587278383112277161967532530675374170857404743017623467220361778016172106855838975759985190398725"),
      boost::lexical_cast<T>("112102381301657019753922131204008107032943249802439891737991109609642417687024271467241971909001000928433174016012202680530522970872221529003044406006693244742562963426"),
      boost::lexical_cast<T>("181385463165881736890212208885336625306342374187644712456957150207333853274302654579519133876533531603807344873416945698153990191233999545175151261845669004727753362151"),
      boost::lexical_cast<T>("293487844467538756644134340089344732339285623990084604194948259816976270961326926046761105785534532532240518889429148378684513162106221074178195667852362249470316325577"),
      boost::lexical_cast<T>("474873307633420493534346548974681357645627998177729316651905410024310124235629580626280239662068064136047863762846094076838503353340220619353346929698031254198069687728"),
      boost::lexical_cast<T>("768361152100959250178480889064026089984913622167813920846853669841286395196956506673041345447602596668288382652275242455523016515446441693531542597550393503668386013305"),
      boost::lexical_cast<T>("1243234459734379743712827438038707447630541620345543237498759079865596519432586087299321585109670660804336246415121336532361519868786662312884889527248424757866455701033"),
      boost::lexical_cast<T>("2011595611835338993891308327102733537615455242513357158345612749706882914629542593972362930557273257472624629067396578987884536384233104006416432124798818261534841714338"),
      boost::lexical_cast<T>("3254830071569718737604135765141440985245996862858900395844371829572479434062128681271684515666943918276960875482517915520246056253019766319301321652047243019401297415371"),
      boost::lexical_cast<T>("5266425683405057731495444092244174522861452105372257554189984579279362348691671275244047446224217175749585504549914494508130592637252870325717753776846061280936139129709"),
      boost::lexical_cast<T>("8521255754974776469099579857385615508107448968231157950034356408851841782753799956515731961891161094026546380032432410028376648890272636645019075428893304300337436545080"),
      boost::lexical_cast<T>("13787681438379834200595023949629790030968901073603415504224340988131204131445471231759779408115378269776131884582346904536507241527525506970736829205739365581273575674789"),
      boost::lexical_cast<T>("22308937193354610669694603807015405539076350041834573454258697396983045914199271188275511370006539363802678264614779314564883890417798143615755904634632669881611012219869"),
      boost::lexical_cast<T>("36096618631734444870289627756645195570045251115437988958483038385114250045644742420035290778121917633578810149197126219101391131945323650586492733840372035462884587894658"),
      boost::lexical_cast<T>("58405555825089055539984231563660601109121601157272562412741735782097295959844013608310802148128456997381488413811905533666275022363121794202248638475004705344495600114527"),
      boost::lexical_cast<T>("94502174456823500410273859320305796679166852272710551371224774167211546005488756028346092926250374630960298563009031752767666154308445444788741372315376740807380188009185"),
      boost::lexical_cast<T>("152907730281912555950258090883966397788288453429983113783966509949308841965332769636656895074378831628341786976820937286433941176671567238990990010790381446151875788123712"),
      boost::lexical_cast<T>("247409904738736056360531950204272194467455305702693665155191284116520387970821525665002988000629206259302085539829969039201607330980012683779731383105758186959255976132897"),
      boost::lexical_cast<T>("400317635020648612310790041088238592255743759132676778939157794065829229936154295301659883075008037887643872516650906325635548507651579922770721393896139633111131764256609"),
      boost::lexical_cast<T>("647727539759384668671321991292510786723199064835370444094349078182349617906975820966662871075637244146945958056480875364837155838631592606550452777001897820070387740389506"),
      boost::lexical_cast<T>("1048045174780033280982112032380749378978942823968047223033506872248178847843130116268322754150645282034589830573131781690472704346283172529321174170898037453181519504646115"),
      boost::lexical_cast<T>("1695772714539417949653434023673260165702141888803417667127855950430528465750105937234985625226282526181535788629612657055309860184914765135871626947899935273251907245035621"),
      boost::lexical_cast<T>("2743817889319451230635546056054009544681084712771464890161362822678707313593236053503308379376927808216125619202744438745782564531197937665192801118797972726433426749681736"),
      boost::lexical_cast<T>("4439590603858869180288980079727269710383226601574882557289218773109235779343341990738294004603210334397661407832357095801092424716112702801064428066697907999685333994717357"),
      boost::lexical_cast<T>("7183408493178320410924526135781279255064311314346347447450581595787943092936578044241602383980138142613787027035101534546874989247310640466257229185495880726118760744399093"),
      boost::lexical_cast<T>("11622999097037189591213506215508548965447537915921230004739800368897178872279920034979896388583348477011448434867458630347967413963423343267321657252193788725804094739116450"),
      boost::lexical_cast<T>("18806407590215510002138032351289828220511849230267577452190381964685121965216498079221498772563486619625235461902560164894842403210733983733578886437689669451922855483515543"),
      boost::lexical_cast<T>("30429406687252699593351538566798377185959387146188807456930182333582300837496418114201395161146835096636683896770018795242809817174157327000900543689883458177726950222631993"),
      boost::lexical_cast<T>("49235814277468209595489570918088205406471236376456384909120564298267422802712916193422893933710321716261919358672578960137652220384891310734479430127573127629649805706147536"),
      boost::lexical_cast<T>("79665220964720909188841109484886582592430623522645192366050746631849723640209334307624289094857156812898603255442597755380462037559048637735379973817456585807376755928779529"),
      boost::lexical_cast<T>("128901035242189118784330680402974787998901859899101577275171310930117146442922250501047183028567478529160522614115176715518114257943939948469859403945029713437026561634927065"),
      boost::lexical_cast<T>("208566256206910027973171789887861370591332483421746769641222057561966870083131584808671472123424635342059125869557774470898576295502988586205239377762486299244403317563706594"),
      boost::lexical_cast<T>("337467291449099146757502470290836158590234343320848346916393368492084016526053835309718655151992113871219648483672951186416690553446928534675098781707516012681429879198633659"),
      boost::lexical_cast<T>("546033547656009174730674260178697529181566826742595116557615426054050886609185420118390127275416749213278774353230725657315266848949917120880338159470002311925833196762340253"),
      boost::lexical_cast<T>("883500839105108321488176730469533687771801170063443463474008794546134903135239255428108782427408863084498422836903676843731957402396845655555436941177518324607263075960973912"),
      boost::lexical_cast<T>("1429534386761117496218850990648231216953367996806038580031624220600185789744424675546498909702825612297777197190134402501047224251346762776435775100647520636533096272723314165"),
      boost::lexical_cast<T>("2313035225866225817707027721117764904725169166869482043505633015146320692879663930974607692130234475382275620027038079344779181653743608431991212041825038961140359348684288077"),
      boost::lexical_cast<T>("3742569612627343313925878711765996121678537163675520623537257235746506482624088606521106601833060087680052817217172481845826405905090371208426987142472559597673455621407602242"),
      boost::lexical_cast<T>("6055604838493569131632906432883761026403706330545002667042890250892827175503752537495714293963294563062328437244210561190605587558833979640418199184297598558813814970091890319"),
      boost::lexical_cast<T>("9798174451120912445558785144649757148082243494220523290580147486639333658127841144016820895796354650742381254461383043036431993463924350848845186326770158156487270591499492561"),
      boost::lexical_cast<T>("15853779289614481577191691577533518174485949824765525957623037737532160833631593681512535189759649213804709691705593604227037581022758330489263385511067756715301085561591382880"),
      boost::lexical_cast<T>("25651953740735394022750476722183275322568193318986049248203185224171494491759434825529356085556003864547090946166976647263469574486682681338108571837837914871788356153090875441"),
      boost::lexical_cast<T>("41505733030349875599942168299716793497054143143751575205826222961703655325391028507041891275315653078351800637872570251490507155509441011827371957348905671587089441714682258321"),
      boost::lexical_cast<T>("67157686771085269622692645021900068819622336462737624454029408185875149817150463332571247360871656942898891584039546898753976729996123693165480529186743586458877797867773133762"),
      boost::lexical_cast<T>("108663419801435145222634813321616862316676479606489199659855631147578805142541491839613138636187310021250692221912117150244483885505564704992852486535649258045967239582455392083"),
      boost::lexical_cast<T>("175821106572520414845327458343516931136298816069226824113885039333453954959691955172184385997058966964149583805951664048998460615501688398158333015722392844504845037450228525845"),
      boost::lexical_cast<T>("284484526373955560067962271665133793452975295675716023773740670481032760102233447011797524633246276985400276027863781199242944501007253103151185502258042102550812277032683917928"),
      boost::lexical_cast<T>("460305632946475974913289730008650724589274111744942847887625709814486715061925402183981910630305243949549859833815445248241405116508941501309518517980434947055657314482912443773"),
      boost::lexical_cast<T>("744790159320431534981252001673784518042249407420658871661366380295519475164158849195779435263551520934950135861679226447484349617516194604460704020238477049606469591515596361701"),
      boost::lexical_cast<T>("1205095792266907509894541731682435242631523519165601719548992090110006190226084251379761345893856764884499995695494671695725754734025136105770222538218911996662126905998508805474"),
      boost::lexical_cast<T>("1949885951587339044875793733356219760673772926586260591210358470405525665390243100575540781157408285819450131557173898143210104351541330710230926558457389046268596497514105167175"),
      boost::lexical_cast<T>("3154981743854246554770335465038655003305296445751862310759350560515531855616327351955302127051265050703950127252668569838935859085566466816001149096676301042930723403512613972649"),
      boost::lexical_cast<T>("5104867695441585599646129198394874763979069372338122901969709030921057521006570452530842908208673336523400258809842467982145963437107797526232075655133690089199319901026719139824"),
      boost::lexical_cast<T>("8259849439295832154416464663433529767284365818089985212729059591436589376622897804486145035259938387227350386062511037821081822522674264342233224751809991132130043304539333112473"),
      boost::lexical_cast<T>("13364717134737417754062593861828404531263435190428108114698768622357646897629468257016987943468611723750750644872353505803227785959782061868465300406943681221329363205566052252297"),
      boost::lexical_cast<T>("21624566574033249908479058525261934298547801008518093327427828213794236274252366061503132978728550110978101030934864543624309608482456326210698525158753672353459406510105385364770"),
      boost::lexical_cast<T>("34989283708770667662541652387090338829811236198946201442126596836151883171881834318520120922197161834728851675807218049427537394442238388079163825565697353574788769715671437617067"),
      boost::lexical_cast<T>("56613850282803917571020710912352273128359037207464294769554425049946119446134200380023253900925711945706952706742082593051847002924694714289862350724451025928248176225776822981837"),
      boost::lexical_cast<T>("91603133991574585233562363299442611958170273406410496211681021886098002618016034698543374823122873780435804382549300642479384397366933102369026176290148379503036945941448260598904"),
      boost::lexical_cast<T>("148216984274378502804583074211794885086529310613874790981235446936044122064150235078566628724048585726142757089291383235531231400291627816658888527014599405431285122167225083580741"),
      boost::lexical_cast<T>("239820118265953088038145437511237497044699584020285287192916468822142124682166269777110003547171459506578561471840683878010615797658560919027914703304747784934322068108673344179645"),
      boost::lexical_cast<T>("388037102540331590842728511723032382131228894634160078174151915758186246746316504855676632271220045232721318561132067113541847197950188735686803230319347190365607190275898427760386"),
      boost::lexical_cast<T>("627857220806284678880873949234269879175928478654445365367068384580328371428482774632786635818391504739299880032972750991552462995608749654714717933624094975299929258384571771940031"),
      boost::lexical_cast<T>("1015894323346616269723602460957302261307157373288605443541220300338514618174799279488463268089611549972021198594104818105094310193558938390401521163943442165665536448660470199700417"),
      boost::lexical_cast<T>("1643751544152900948604476410191572140483085851943050808908288684918842989603282054121249903908003054711321078627077569096646773189167688045116239097567537140965465707045041971640448"),
      boost::lexical_cast<T>("2659645867499517218328078871148874401790243225231656252449508985257357607778081333609713171997614604683342277221182387201741083382726626435517760261510979306631002155705512171340865"),
      boost::lexical_cast<T>("4303397411652418166932555281340446542273329077174707061357797670176200597381363387730963075905617659394663355848259956298387856571894314480633999359078516447596467862750554142981313"),
      boost::lexical_cast<T>("6963043279151935385260634152489320944063572302406363313807306655433558205159444721340676247903232264078005633069442343500128939954620940916151759620589495754227470018456066314322178"),
      boost::lexical_cast<T>("11266440690804353552193189433829767486336901379581070375165104325609758802540808109071639323808849923472668988917702299798516796526515255396785758979668012201823937881206620457303491"),
      boost::lexical_cast<T>("18229483969956288937453823586319088430400473681987433688972410981043317007700252830412315571712082187550674621987144643298645736481136196312937518600257507956051407899662686771625669"),
      boost::lexical_cast<T>("29495924660760642489647013020148855916737375061568504064137515306653075810241060939483954895520932111023343610904846943097162533007651451709723277579925520157875345780869307228929160"),
      boost::lexical_cast<T>("47725408630716931427100836606467944347137848743555937753109926287696392817941313769896270467233014298574018232891991586395808269488787648022660796180183028113926753680531994000554829"),
      boost::lexical_cast<T>("77221333291477573916747849626616800263875223805124441817247441594349468628182374709380225362753946409597361843796838529492970802496439099732384073760108548271802099461401301229483989"),
      boost::lexical_cast<T>("124946741922194505343848686233084744611013072548680379570357367882045861446123688479276495829986960708171380076688830115888779071985226747755044869940291576385728853141933295230038818"),
      boost::lexical_cast<T>("202168075213672079260596535859701544874888296353804821387604809476395330074306063188656721192740907117768741920485668645381749874481665847487428943700400124657530952603334596459522807"),
      boost::lexical_cast<T>("327114817135866584604445222092786289485901368902485200957962177358441191520429751667933217022727867825940121997174498761270528946466892595242473813640691701043259805745267891689561625"),
      boost::lexical_cast<T>("529282892349538663865041757952487834360789665256290022345566986834836521594735814856589938215468774943708863917660167406652278820948558442729902757341091825700790758348602488149084432"),
      boost::lexical_cast<T>("856397709485405248469486980045274123846691034158775223303529164193277713115165566524523155238196642769648985914834666167922807767415451037972376570981783526744050564093870379838646057"),
      boost::lexical_cast<T>("1385680601834943912334528737997761958207480699415065245649096151028114234709901381381113093453665417713357849832494833574575086588364009480702279328322875352444841322442472867987730489"),
      boost::lexical_cast<T>("2242078311320349160804015718043036082054171733573840468952625315221391947825066947905636248691862060483006835747329499742497894355779460518674655899304658879188891886536343247826376546"),
      boost::lexical_cast<T>("3627758913155293073138544456040798040261652432988905714601721466249506182534968329286749342145527478196364685579824333317072980944143469999376935227627534231633733208978816115814107035"),
      boost::lexical_cast<T>("5869837224475642233942560174083834122315824166562746183554346781470898130360035277192385590837389538679371521327153833059570875299922930518051591126932193110822625095515159363640483581"),
      boost::lexical_cast<T>("9497596137630935307081104630124632162577476599551651898156068247720404312895003606479134932982917016875736206906978166376643856244066400517428526354559727342456358304493975479454590616"),
      boost::lexical_cast<T>("15367433362106577541023664804208466284893300766114398081710415029191302443255038883671520523820306555555107728234131999436214731543989331035480117481491920453278983400009134843095074197"),
      boost::lexical_cast<T>("24865029499737512848104769434333098447470777365666049979866483276911706756150042490150655456803223572430843935141110165812858587788055731552908643836051647795735341704503110322549664813"),
      boost::lexical_cast<T>("40232462861844090389128434238541564732364078131780448061576898306103009199405081373822175980623530127985951663375242165249073319332045062588388761317543568249014325104512245165644739010"),
      boost::lexical_cast<T>("65097492361581603237233203672874663179834855497446498041443381583014715955555123863972831437426753700416795598516352331061931907120100794141297405153595216044749666809015355488194403823"),
      boost::lexical_cast<T>("105329955223425693626361637911416227912198933629226946103020279889117725154960205237795007418050283828402747261891594496311005226452145856729686166471138784293763991913527600653839142833"),
      boost::lexical_cast<T>("170427447585007296863594841584290891092033789126673444144463661472132441110515329101767838855477037528819542860407946827372937133572246650870983571624734000338513658722542956142033546656"),
      boost::lexical_cast<T>("275757402808432990489956479495707119004232722755900390247483941361250166265475534339562846273527321357222290122299541323683942360024392507600669738095872784632277650636070556795872689489"),
      boost::lexical_cast<T>("446184850393440287353551321079998010096266511882573834391947602833382607375990863441330685129004358886041832982707488151056879493596639158471653309720606784970791309358613512937906236145"),
      boost::lexical_cast<T>("721942253201873277843507800575705129100499234638474224639431544194632773641466397780893531402531680243264123105007029474740821853621031666072323047816479569603068959994684069733778925634"),
      boost::lexical_cast<T>("1168127103595313565197059121655703139196765746521048059031379147028015381017457261222224216531536039129305956087714517625797701347217670824543976357537086354573860269353297582671685161779"),
      boost::lexical_cast<T>("1890069356797186843040566922231408268297264981159522283670810691222648154658923659003117747934067719372570079192721547100538523200838702490616299405353565924176929229347981652405464087413"),
      boost::lexical_cast<T>("3058196460392500408237626043887111407494030727680570342702189838250663535676380920225341964465603758501876035280436064726336224548056373315160275762890652278750789498701279235077149249192"),
      boost::lexical_cast<T>("4948265817189687251278192966118519675791295708840092626373000529473311690335304579228459712399671477874446114473157611826874747748895075805776575168244218202927718728049260887482613336605"),
      boost::lexical_cast<T>("8006462277582187659515819010005631083285326436520662969075190367723975226011685499453801676865275236376322149753593676553210972296951449120936850931134870481678508226750540122559762585797"),
      boost::lexical_cast<T>("12954728094771874910794011976124150759076622145360755595448190897197286916346990078682261389264946714250768264226751288380085720045846524926713426099379088684606226954799801010042375922402"),
      boost::lexical_cast<T>("20961190372354062570309830986129781842361948581881418564523381264921262142358675578136063066130221950627090413980344964933296692342797974047650277030513959166284735181550341132602138508199"),
      boost::lexical_cast<T>("33915918467125937481103842962253932601438570727242174159971572162118549058705665656818324455395168664877858678207096253313382412388644498974363703129893047850890962136350142142644514430601"),
      boost::lexical_cast<T>("54877108839480000051413673948383714443800519309123592724494953427039811201064341234954387521525390615504949092187441218246679104731442473022013980160407007017175697317900483275246652938800"),
      boost::lexical_cast<T>("88793027306605937532517516910637647045239090036365766884466525589158360259770006891772711976920559280382807770394537471560061517120086971996377683290300054868066659454250625417891167369401"),
      boost::lexical_cast<T>("143670136146085937583931190859021361489039609345489359608961479016198171460834348126727099498445949895887756862581978689806740621851529445018391663450707061885242356772151108693137820308201"),
      boost::lexical_cast<T>("232463163452691875116448707769659008534278699381855126493428004605356531720604355018499811475366509176270564632976516161366802138971616417014769346741007116753309016226401734111028987677602"),
      boost::lexical_cast<T>("376133299598777812700379898628680370023318308727344486102389483621554703181438703145226910973812459072158321495558494851173542760823145862033161010191714178638551372998552842804166807985803"),
      boost::lexical_cast<T>("608596463051469687816828606398339378557597008109199612595817488226911234902043058163726722449178968248428886128535011012540344899794762279047930356932721295391860389224954576915195795663405"),
      boost::lexical_cast<T>("984729762650247500517208505027019748580915316836544098698206971848465938083481761308953633422991427320587207624093505863713887660617908141081091367124435474030411762223507419719362603649208"),
      boost::lexical_cast<T>("1593326225701717188334037111425359127138512324945743711294024460075377172985524819472680355872170395569016093752628516876254232560412670420129021724057156769422272151448461996634558399312613"),
      boost::lexical_cast<T>("2578055988351964688851245616452378875719427641782287809992231431923843111069006580781633989295161822889603301376722022739968120221030578561210113091181592243452683913671969416353921002961821"),
      boost::lexical_cast<T>("4171382214053681877185282727877738002857939966728031521286255891999220284054531400254314345167332218458619395129350539616222352781443248981339134815238749012874956065120431412988479402274434"),
      boost::lexical_cast<T>("6749438202405646566036528344330116878577367608510319331278487323923063395123537981035948334462494041348222696506072562356190473002473827542549247906420341256327639978792400829342400405236255"),
      boost::lexical_cast<T>("10920820416459328443221811072207854881435307575238350852564743215922283679178069381290262679629826259806842091635423101972412825783917076523888382721659090269202596043912832242330879807510689"),
      boost::lexical_cast<T>("17670258618864975009258339416537971760012675183748670183843230539845347074301607362326211014092320301155064788141495664328603298786390904066437630628079431525530236022705233071673280212746944"),
      boost::lexical_cast<T>("28591079035324303452480150488745826641447982758987021036407973755767630753479676743616473693722146560961906879776918766301016124570307980590326013349738521794732832066618065314004160020257633"),
      boost::lexical_cast<T>("46261337654189278461738489905283798401460657942735691220251204295612977827781284105942684707814466862116971667918414430629619423356698884656763643977817953320263068089323298385677440233004577"),
      boost::lexical_cast<T>("74852416689513581914218640394029625042908640701722712256659178051380608581260960849559158401536613423078878547695333196930635547927006865247089657327556475114995900155941363699681600253262210"),
      boost::lexical_cast<T>("121113754343702860375957130299313423444369298644458403476910382346993586409042244955501843109351080285195850215613747627560254971283705749903853301305374428435258968245264662085359040486266787"),
      boost::lexical_cast<T>("195966171033216442290175770693343048487277939346181115733569560398374194990303205805061001510887693708274728763309080824490890519210712615150942958632930903550254868401206025785040640739528997"),
      boost::lexical_cast<T>("317079925376919302666132900992656471931647237990639519210479942745367781399345450760562844620238773993470578978922828452051145490494418365054796259938305331985513836646470687870399681225795784"),
      boost::lexical_cast<T>("513046096410135744956308671685999520418925177336820634944049503143741976389648656565623846131126467701745307742231909276542036009705130980205739218571236235535768705047676713655440321965324781"),
      boost::lexical_cast<T>("830126021787055047622441572678655992350572415327460154154529445889109757788994107326186690751365241695215886721154737728593181500199549345260535478509541567521282541694147401525840003191120565"),
      boost::lexical_cast<T>("1343172118197190792578750244364655512769497592664280789098578949032851734178642763891810536882491709396961194463386647005135217509904680325466274697080777803057051246741824115181280325156445346"),
      boost::lexical_cast<T>("2173298139984245840201191817043311505120070007991740943253108394921961491967636871217997227633856951092177081184541384733728399010104229670726810175590319370578333788435971516707120328347565911"),
      boost::lexical_cast<T>("3516470258181436632779942061407967017889567600656021732351687343954813226146279635109807764516348660489138275647928031738863616520008909996193084872671097173635385035177795631888400653504011257"),
      boost::lexical_cast<T>("5689768398165682472981133878451278523009637608647762675604795738876774718113916506327804992150205611581315356832469416472592015530113139666919895048261416544213718823613767148595520981851577168"),
      boost::lexical_cast<T>("9206238656347119105761075939859245540899205209303784407956483082831587944260196141437612756666554272070453632480397448211455632050122049663112979920932513717849103858791562780483921635355588425"),
      boost::lexical_cast<T>("14896007054512801578742209818310524063908842817951547083561278821708362662374112647765417748816759883651768989312866864684047647580235189330032874969193930262062822682405329929079442617207165593"),
      boost::lexical_cast<T>("24102245710859920684503285758169769604808048027255331491517761904539950606634308789203030505483314155722222621793264312895503279630357238993145854890126443979911926541196892709563364252562754018"),
      boost::lexical_cast<T>("38998252765372722263245495576480293668716890845206878575079040726248313269008421436968448254300074039373991611106131177579550927210592428323178729859320374241974749223602222638642806869769919611"),
      boost::lexical_cast<T>("63100498476232642947748781334650063273524938872462210066596802630788263875642730226171478759783388195096214232899395490475054206840949667316324584749446818221886675764799115348206171122332673629"),
      boost::lexical_cast<T>("102098751241605365210994276911130356942241829717669088641675843357036577144651151663139927014083462234470205844005526668054605134051542095639503314608767192463861424988401337986848977992102593240"),
      boost::lexical_cast<T>("165199249717838008158743058245780420215766768590131298708272645987824841020293881889311405773866850429566420076904922158529659340892491762955827899358214010685748100753200453335055149114435266869"),
      boost::lexical_cast<T>("267298000959443373369737335156910777158008598307800387349948489344861418164945033552451332787950312664036625920910448826584264474944033858595331213966981203149609525741601791321904127106537860109"),
      boost::lexical_cast<T>("432497250677281381528480393402691197373775366897931686058221135332686259185238915441762738561817163093603045997815370985113923815836525621551159113325195213835357626494802244656959276220973126978"),
      boost::lexical_cast<T>("699795251636724754898217728559601974531783965205732073408169624677547677350183948994214071349767475757639671918725819811698188290780559480146490327292176416984967152236404035978863403327510987087"),
      boost::lexical_cast<T>("1132292502314006136426698121962293171905559332103663759466390760010233936535422864435976809911584638851242717916541190796812112106617085101697649440617371630820324778731206280635822679548484114065"),
      boost::lexical_cast<T>("1832087753950730891324915850521895146437343297309395832874560384687781613885606813430190881261352114608882389835267010608510300397397644581844139767909548047805291930967610316614686082875995101152"),
      boost::lexical_cast<T>("2964380256264737027751613972484188318342902629413059592340951144698015550421029677866167691172936753460125107751808201405322412504014729683541789208526919678625616709698816597250508762424479215217"),
      boost::lexical_cast<T>("4796468010215467919076529823006083464780245926722455425215511529385797164306636491296358572434288868069007497587075212013832712901412374265385928976436467726430908640666426913865194845300474316369"),
      boost::lexical_cast<T>("7760848266480204946828143795490271783123148556135515017556462674083812714727666169162526263607225621529132605338883413419155125405427103948927718184963387405056525350365243511115703607724953531586"),
      boost::lexical_cast<T>("12557316276695672865904673618496355247903394482857970442771974203469609879034302660458884836041514489598140102925958625432987838306839478214313647161399855131487433991031670424980898453025427847955"),
      boost::lexical_cast<T>("20318164543175877812732817413986627031026543038993485460328436877553422593761968829621411099648740111127272708264842038852142963712266582163241365346363242536543959341396913936096602060750381379541"),
      boost::lexical_cast<T>("32875480819871550678637491032482982278929937521851455903100411081023032472796271490080295935690254600725412811190800664285130802019106060377555012507763097668031393332428584361077500513775809227496"),
      boost::lexical_cast<T>("53193645363047428491370308446469609309956480560844941363428847958576455066558240319701707035338994711852685519455642703137273765731372642540796377854126340204575352673825498297174102574526190607037"),
      boost::lexical_cast<T>("86069126182918979170007799478952591588886418082696397266529259039599487539354511809782002971029249312578098330646443367422404567750478702918351390361889437872606746006254082658251603088301999834533"),
      boost::lexical_cast<T>("139262771545966407661378107925422200898842898643541338629958106998175942605912752129483710006368244024430783850102086070559678333481851345459147768216015778077182098680079580955425705662828190441570"),
      boost::lexical_cast<T>("225331897728885386831385907404374792487729316726237735896487366037775430145267263939265712977397493337008882180748529437982082901232330048377499158577905215949788844686333663613677308751130190276103"),
      boost::lexical_cast<T>("364594669274851794492764015329796993386572215369779074526445473035951372751180016068749422983765737361439666030850615508541761234714181393836646926793920994026970943366413244569103014413958380717673"),
      boost::lexical_cast<T>("589926567003737181324149922734171785874301532096016810422932839073726802896447280008015135961163230698448548211599144946523844135946511442214146085371826209976759788052746908182780323165088570993776"),
      boost::lexical_cast<T>("954521236278588975816913938063968779260873747465795884949378312109678175647627296076764558944928968059888214242449760455065605370660692836050793012165747204003730731419160152751883337579046951711449"),
      boost::lexical_cast<T>("1544447803282326157141063860798140565135175279561812695372311151183404978544074576084779694906092198758336762454048905401589449506607204278264939097537573413980490519471907060934663660744135522705225"),
      boost::lexical_cast<T>("2498969039560915132957977798862109344396049027027608580321689463293083154191701872161544253851021166818224976696498665856655054877267897114315732109703320617984221250891067213686546998323182474416674"),
      boost::lexical_cast<T>("4043416842843241290099041659660249909531224306589421275694000614476488132735776448246323948757113365576561739150547571258244504383875101392580671207240894031964711770362974274621210659067317997121899"),
      boost::lexical_cast<T>("6542385882404156423057019458522359253927273333617029856015690077769571286927478320407868202608134532394786715847046237114899559261142998506896403316944214649948933021254041488307757657390500471538573"),
      boost::lexical_cast<T>("10585802725247397713156061118182609163458497640206451131709690692246059419663254768654192151365247897971348454997593808373144063645018099899477074524185108681913644791617015762928968316457818468660472"),
      boost::lexical_cast<T>("17128188607651554136213080576704968417385770973823480987725380770015630706590733089062060353973382430366135170844640045488043622906161098406373477841129323331862577812871057251236725973848318940199045"),
      boost::lexical_cast<T>("27713991332898951849369141694887577580844268614029932119435071462261690126253987857716252505338630328337483625842233853861187686551179198305850552365314432013776222604488073014165694290306137408859517"),
      boost::lexical_cast<T>("44842179940550505985582222271592545998230039587853413107160452232277320832844720946778312859312012758703618796686873899349231309457340296712224030206443755345638800417359130265402420264154456349058562"),
      boost::lexical_cast<T>("72556171273449457834951363966480123579074308201883345226595523694539010959098708804494565364650643087041102422529107753210418996008519495018074582571758187359415023021847203279568114554460593757918079"),
      boost::lexical_cast<T>("117398351213999963820533586238072669577304347789736758333755975926816331791943429751272878223962655845744721219215981652559650305465859791730298612778201942705053823439206333544970534818615050106976641"),
      boost::lexical_cast<T>("189954522487449421655484950204552793156378655991620103560351499621355342751042138555767443588613298932785823641745089405770069301474379286748373195349960130064468846461053536824538649373075643864894720"),
      boost::lexical_cast<T>("307352873701449385476018536442625462733683003781356861894107475548171674542985568307040321812575954778530544860961071058329719606940239078478671808128162072769522669900259870369509184191690693971871361"),
      boost::lexical_cast<T>("497307396188898807131503486647178255890061659772976965454458975169527017294027706862807765401189253711316368502706160464099788908414618365227045003478122202833991516361313407194047833564766337836766081"),
      boost::lexical_cast<T>("804660269890348192607522023089803718623744663554333827348566450717698691837013275169848087213765208489846913363667231522429508515354857443705716811606284275603514186261573277563557017756457031808637442"),
      boost::lexical_cast<T>("1301967666079246999739025509736981974513806323327310792803025425887225709131040982032655852614954462201163281866373391986529297423769475808932761815084406478437505702622886684757604851321223369645403523"),
      boost::lexical_cast<T>("2106627935969595192346547532826785693137550986881644620151591876604924400968054257202503939828719670691010195230040623508958805939124333252638478626690690754041019888884459962321161869077680401454040965"),
      boost::lexical_cast<T>("3408595602048842192085573042563767667651357310208955412954617302492150110099095239235159792443674132892173477096414015495488103362893809061571240441775097232478525591507346647078766720398903771099444488"),
      boost::lexical_cast<T>("5515223538018437384432120575390553360788908297090600033106209179097074511067149496437663732272393803583183672326454639004446909302018142314209719068465787986519545480391806609399928589476584172553485453"),
      boost::lexical_cast<T>("8923819140067279576517693617954321028440265607299555446060826481589224621166244735672823524716067936475357149422868654499935012664911951375780959510240885218998071071899153256478695309875487943652929941"),
      boost::lexical_cast<T>("14439042678085716960949814193344874389229173904390155479167035660686299132233394232110487256988461740058540821749323293504381921966930093689990678578706673205517616552290959865878623899352072116206415394"),
      boost::lexical_cast<T>("23362861818152996537467507811299195417669439511689710925227862142275523753399638967783310781704529676533897971172191948004316934631842045065771638088947558424515687624190113122357319209227560059859345335"),
      boost::lexical_cast<T>("37801904496238713498417322004644069806898613416079866404394897802961822885633033199893798038692991416592438792921515241508698856598772138755762316667654231630033304176481072988235943108579632176065760729"),
      boost::lexical_cast<T>("61164766314391710035884829815943265224568052927769577329622759945237346639032672167677108820397521093126336764093707189513015791230614183821533954756601790054548991800671186110593262317807192235925106064"),
      boost::lexical_cast<T>("98966670810630423534302151820587335031466666343849443734017657748199169524665705367570906859090512509718775557015222431021714647829386322577296271424256021684582295977152259098829205426386824411990866793"),
      boost::lexical_cast<T>("160131437125022133570186981636530600256034719271619021063640417693436516163698377535248015679488033602845112321108929620534730439060000506398830226180857811739131287777823445209422467744194016647915972857"),
      boost::lexical_cast<T>("259098107935652557104489133457117935287501385615468464797658075441635685688364082902818922538578546112563887878124152051556445086889386828976126497605113833423713583754975704308251673170580841059906839650"),
      boost::lexical_cast<T>("419229545060674690674676115093648535543536104887087485861298493135072201852062460438066938218066579715409000199233081672091175525949387335374956723785971645162844871532799149517674140914774857707822812507"),
      boost::lexical_cast<T>("678327652996327247779165248550766470831037490502555950658956568576707887540426543340885860756645125827972888077357233723647620612838774164351083221391085478586558455287774853825925814085355698767729652157"),
      boost::lexical_cast<T>("1097557198057001938453841363644415006374573595389643436520255061711780089392489003778952798974711705543381888276590315395738796138788161499726039945177057123749403326820574003343599955000130556475552464664"),
      boost::lexical_cast<T>("1775884851053329186233006612195181477205611085892199387179211630288487976932915547119838659731356831371354776353947549119386416751626935664077123166568142602335961782108348857169525769085486255243282116821"),
      boost::lexical_cast<T>("2873442049110331124686847975839596483580184681281842823699466692000268066325404550898791458706068536914736664630537864515125212890415097163803163111745199726085365108928922860513125724085616811718834581485"),
      boost::lexical_cast<T>("4649326900163660310919854588034777960785795767174042210878678322288756043258320098018630118437425368286091440984485413634511629642042032827880286278313342328421326891037271717682651493171103066962116698306"),
      boost::lexical_cast<T>("7522768949273991435606702563874374444365980448455885034578145014289024109583724648917421577143493905200828105615023278149636842532457129991683449390058542054506691999966194578195777217256719878680951279791"),
      boost::lexical_cast<T>("12172095849437651746526557151909152405151776215629927245456823336577780152842044746936051695580919273486919546599508691784148472174499162819563735668371884382928018891003466295878428710427822945643067978097"),
      boost::lexical_cast<T>("19694864798711643182133259715783526849517756664085812280034968350866804262425769395853473272724413178687747652214531969933785314706956292811247185058430426437434710890969660874074205927684542824324019257888"),
      boost::lexical_cast<T>("31866960648149294928659816867692679254669532879715739525491791687444584415267814142789524968305332452174667198814040661717933786881455455630810920726802310820362729781973127169952634638112365769967087235985"),
      boost::lexical_cast<T>("51561825446860938110793076583476206104187289543801551805526760038311388677693583538642998241029745630862414851028572631651719101588411748442058105785232737257797440672942788044026840565796908594291106493873"),
      boost::lexical_cast<T>("83428786095010233039452893451168885358856822423517291331018551725755973092961397681432523209335078083037082049842613293369652888469867204072869026512035048078160170454915915213979475203909274364258193729858"),
      boost::lexical_cast<T>("134990611541871171150245970034645091463044111967318843136545311764067361770654981220075521450364823713899496900871185925021371990058278952514927132297267785335957611127858703258006315769706182958549300223731"),
      boost::lexical_cast<T>("218419397636881404189698863485813976821900934390836134467563863489823334863616378901508044659699901796936578950713799218391024878528146156587796158809302833414117781582774618471985790973615457322807493953589"),
      boost::lexical_cast<T>("353410009178752575339944833520459068284945046358154977604109175253890696634271360121583566110064725510836075851584985143412396868586425109102723291106570618750075392710633321729992106743321640281356794177320"),
      boost::lexical_cast<T>("571829406815633979529643697006273045106845980748991112071673038743714031497887739023091610769764627307772654802298784361803421747114571265690519449915873452164193174293407940201977897716937097604164288130909"),
      boost::lexical_cast<T>("925239415994386554869588530526732113391791027107146089675782213997604728132159099144675176879829352818608730653883769505215818615700996374793242741022444070914268567004041261931970004460258737885521082308229"),
      boost::lexical_cast<T>("1497068822810020534399232227533005158498637007856137201747455252741318759630046838167766787649593980126381385456182553867019240362815567640483762190938317523078461741297449202133947902177195835489685370439138"),
      boost::lexical_cast<T>("2422308238804407089268820758059737271890428034963283291423237466738923487762205937312441964529423332944990116110066323372235058978516564015277004931960761593992730308301490464065917906637454573375206452747367"),
      boost::lexical_cast<T>("3919377061614427623668052985592742430389065042819420493170692719480242247392252775480208752179017313071371501566248877239254299341332131655760767122899079117071192049598939666199865808814650408864891823186505"),
      boost::lexical_cast<T>("6341685300418834712936873743652479702279493077782703784593930186219165735154458712792650716708440646016361617676315200611489358319848695671037772054859840711063922357900430130265783715452104982240098275933872"),
      boost::lexical_cast<T>("10261062362033262336604926729245222132668558120602124277764622905699407982546711488272859468887457959087733119242564077850743657661180827326798539177758919828135114407499369796465649524266755391104990099120377"),
      boost::lexical_cast<T>("16602747662452097049541800472897701834948051198384828062358553091918573717701170201065510185595898605104094736918879278462233015981029522997836311232618760539199036765399799926731433239718860373345088375054249"),
      boost::lexical_cast<T>("26863810024485359386146727202142923967616609318986952340123175997617981700247881689338369654483356564191827856161443356312976673642210350324634850410377680367334151172899169723197082763985615764450078474174626"),
      boost::lexical_cast<T>("43466557686937456435688527675040625802564660517371780402481729089536555417949051890403879840079255169295922593080322634775209689623239873322471161642996440906533187938298969649928516003704476137795166849228875"),
      boost::lexical_cast<T>("70330367711422815821835254877183549770181269836358732742604905087154537118196933579742249494562611733487750449241765991088186363265450223647106012053374121273867339111198139373125598767690091902245245323403501")
   };
   static const std::vector<T> data(numbers, numbers + sizeof(numbers) / sizeof(numbers[0]));

   return data;
}

template <class T>
std::vector<std::pair<T, T> > const& fibonacci_numbers_permution_1()
{
   typedef std::integral_constant<int,
      ((std::numeric_limits<T>::digits <= 16) ? 16 : ((std::numeric_limits<T>::digits <= 32) ? 32 : (std::numeric_limits<T>::digits <= 64) ? 64 : 0))
   > tag_type;

   static std::vector<std::pair<T, T> > data;
   static bool init = false;
   if(!init)
   {
      init = true;
      for(unsigned i = 0; i < fibonacci_numbers<T>(tag_type()).size() - 1; ++i)
         data.push_back(std::make_pair(fibonacci_numbers<T>(tag_type())[i], fibonacci_numbers<T>(tag_type())[i + 1]));
   }
   return data;
}

template <class T>
std::vector<std::pair<T, T> > const& fibonacci_numbers_permution_2()
{
   typedef std::integral_constant<int,
      ((std::numeric_limits<T>::digits <= 16) ? 16 : ((std::numeric_limits<T>::digits <= 32) ? 32 : (std::numeric_limits<T>::digits <= 64) ? 64 : 0))
   > tag_type;

   static std::vector<std::pair<T, T> > data;
   static bool init = false;
   if(!init)
   {
      init = true;
      for(unsigned i = 0; i < fibonacci_numbers<T>(tag_type()).size(); ++i)
         for(unsigned j = 0; j < fibonacci_numbers<T>(tag_type()).size(); ++j)
            data.push_back(std::make_pair(fibonacci_numbers<T>(tag_type())[i], fibonacci_numbers<T>(tag_type())[j]));
   }
   return data;
}

template <class T>
std::vector<std::pair<T, T> > const & trivial_gcd_test_cases()
{
   static const std::pair<T, T> a[] = {
      std::pair<T, T>(0, 0),
      std::pair<T, T>(0, 832040u),
      std::pair<T, T>(832040u, 0),
      std::pair<T, T>(1u << 15, 1u << 4),
      std::pair<T, T>(1u << 3, 1u << 14),
      std::pair<T, T>(832040u, 832040u),
      std::pair<T, T>(991u * 3 * 7, 991u),
      std::pair<T, T>(991u * 2, 991u),
   };
   static const std::vector<std::pair<T, T> > data(a, a + sizeof(a) / sizeof(a[0]));
   return data;
}

