# frozen_string_literal: true

require "helper"

class TestCollections < JekyllUnitTest
  context "an evil collection" do
    setup do
      @collection = Jekyll::Collection.new(fixture_site, "../../etc/password")
    end

    should "sanitize the label name" do
      assert_equal @collection.label, "....etcpassword"
    end

    should "have a sanitized relative path name" do
      assert_equal @collection.relative_directory, "_....etcpassword"
    end

    should "have a sanitized full path" do
      assert_equal @collection.directory, source_dir("_....etcpassword")
    end
  end

  context "a simple collection" do
    setup do
      @collection = Jekyll::Collection.new(fixture_site, "methods")
    end

    should "sanitize the label name" do
      assert_equal @collection.label, "methods"
    end

    should "have default URL template" do
      assert_equal @collection.url_template, "/:collection/:path:output_ext"
    end

    should "contain no docs when initialized" do
      assert_empty @collection.docs
    end

    should "know its relative directory" do
      assert_equal @collection.relative_directory, "_methods"
    end

    should "know the full path to itself on the filesystem" do
      assert_equal @collection.directory, source_dir("_methods")
    end

    context "when turned into Liquid" do
      should "have a label attribute" do
        assert_equal @collection.to_liquid["label"], "methods"
      end

      should "have a docs attribute" do
        assert_equal @collection.to_liquid["docs"], []
      end

      should "have a files attribute" do
        assert_equal @collection.to_liquid["files"], []
      end

      should "have a directory attribute" do
        assert_equal @collection.to_liquid["directory"], source_dir("_methods")
      end

      should "have a relative_directory attribute" do
        assert_equal @collection.to_liquid["relative_directory"], "_methods"
      end

      should "have a output attribute" do
        assert_equal @collection.to_liquid["output"], false
      end
    end

    should "know whether it should be written or not" do
      assert_equal @collection.write?, false
      @collection.metadata["output"] = true
      assert_equal @collection.write?, true
      @collection.metadata.delete "output"
    end
  end

  context "with no collections specified" do
    setup do
      @site = fixture_site
      @site.process
    end

    should "contain only the default collections" do
      refute_equal @site.collections, {}
      refute_nil @site.collections
    end
  end

  context "a collection with permalink" do
    setup do
      @site = fixture_site({
        "collections" => {
          "methods" => {
            "permalink" => "/awesome/:path/",
          },
        },
      })
      @site.process
      @collection = @site.collections["methods"]
    end

    should "have custom URL template" do
      assert_equal @collection.url_template, "/awesome/:path/"
    end
  end

  context "with a collection" do
    setup do
      @site = fixture_site({
        "collections" => ["methods"],
      })
      @site.process
      @collection = @site.collections["methods"]
    end

    should "create a Hash mapping label to Collection instance" do
      assert @site.collections.is_a?(Hash)
      refute_nil @site.collections["methods"]
      assert @site.collections["methods"].is_a? Jekyll::Collection
    end

    should "collects docs in an array on the Collection object" do
      assert @site.collections["methods"].docs.is_a? Array
      @site.collections["methods"].docs.each do |doc|
        assert doc.is_a? Jekyll::Document
        assert_includes %w(
          _methods/configuration.md
          _methods/sanitized_path.md
          _methods/collection/entries
          _methods/site/generate.md
          _methods/site/initialize.md
          _methods/um_hi.md
          _methods/escape-+\ #%20[].md
          _methods/yaml_with_dots.md
          _methods/3940394-21-9393050-fifif1323-test.md
        ), doc.relative_path
      end
    end

    should "not include files from base dir which start with an underscore" do
      refute_includes @collection.filtered_entries, "_do_not_read_me.md"
    end

    should "not include files which start with an underscore in a subdirectory" do
      refute_includes @collection.filtered_entries, "site/_dont_include_me_either.md"
    end

    should "not include the underscored files in the list of docs" do
      refute_includes @collection.docs.map(&:relative_path), "_methods/_do_not_read_me.md"
      refute_includes @collection.docs.map(&:relative_path),
                                           "_methods/site/_dont_include_me_either.md"
    end
  end

  context "with a collection with metadata" do
    setup do
      @site = fixture_site({
        "collections" => {
          "methods" => {
            "foo" => "bar",
            "baz" => "whoo",
          },
        },
      })
      @site.process
      @collection = @site.collections["methods"]
    end

    should "extract the configuration collection information as metadata" do
      assert_equal @collection.metadata, { "foo" => "bar", "baz" => "whoo" }
    end
  end

  context "in safe mode" do
    setup do
      @site = fixture_site({
        "collections" => ["methods"],
        "safe"        => true,
      })
      @site.process
      @collection = @site.collections["methods"]
    end

    should "include the symlinked file as it resolves to inside site.source" do
      assert_includes @collection.filtered_entries, "um_hi.md"
      refute_includes @collection.filtered_entries, "/um_hi.md"
    end

    should "include the symlinked file from site.source in the list of docs" do
      # no support for including symlinked file on Windows
      skip_if_windows "Jekyll does not currently support symlinks on Windows."

      assert_includes @collection.docs.map(&:relative_path), "_methods/um_hi.md"
    end
  end

  context "with dots in the filenames" do
    setup do
      @site = fixture_site({
        "collections" => ["with.dots"],
        "safe"        => true,
      })
      @site.process
      @collection = @site.collections["with.dots"]
    end

    should "exist" do
      refute_nil @collection
    end

    should "contain one document" do
      assert_equal 4, @collection.docs.size
    end

    should "allow dots in the filename" do
      assert_equal "_with.dots", @collection.relative_directory
    end

    should "read document in subfolders with dots" do
      assert(
        @collection.docs.any? { |d| d.path.include?("all.dots") }
      )
    end
  end

  context "a collection with included dotfiles" do
    setup do
      @site = fixture_site({
        "collections" => {
          "methods" => {
            "permalink" => "/awesome/:path/",
          },
        },
        "include"     => %w(.htaccess .gitignore),
      })
      @site.process
      @collection = @site.collections["methods"]
    end

    should "contain .htaccess file" do
      assert(@collection.files.any? { |d| d.name == ".htaccess" })
    end

    should "contain .gitignore file" do
      assert(@collection.files.any? { |d| d.name == ".gitignore" })
    end

    should "have custom URL in static file" do
      assert(
        @collection.files.any? { |d| d.url.include?("/awesome/with.dots/") }
      )
    end
  end
end
